﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundPlayer.h
 *
 * @file snd_SoundPlayer.h
 */

#ifndef NW_SND_SOUND_PLAYER_H_
#define NW_SND_SOUND_PLAYER_H_

#include <nw/ut/ut_LinkList.h>
#include <nw/snd/snd_Global.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_PlayerHeap.h>

namespace nw {
namespace snd {


//---------------------------------------------------------------------------
//! @brief    サウンドを再生するためのプレイヤークラスです。
//!
//!           SoundPlayer クラスのインスタンスは
//!           @ref SoundArchivePlayer::GetSoundPlayer を呼び出して取得してください。
//!
//!           一つの SoundPlayer で複数のサウンドを再生することができます。
//!           同時に再生できるサウンド数を設定することができ、
//!           そのサウンド数を超えた場合には、
//!           各サウンドに設定されたプレイヤープライオリティを比較して、
//!           もっとも優先度が低いサウンドが停止します。
//!
//!         @n
//!
//!         ■ 出力先別パラメータ
//!
//!         Cafe では、[1] メイン出力 (TV 出力) [2] DRC 出力 [3] リモコン出力の
//!         3 系統の出力があります。
//!         SoundPlayer では、このプレイヤーで再生するサウンドに関して、
//!         下記の影響を与えることができます。
//!
//!         (全体)
//!         - LPF ... @ref SetLpfFreq
//!         - Biquad フィルタ ... @ref SetBiquadFilter
//!         - 出力先 ... SetDefaultOutputLine
//!
//!         (TV 出力向け)
//!         - 音量 ... @ref SetVolume, @ref StopAllSound, @ref PauseAllSound, @ref SetMainOutVolume
//!         - メインセンド ... @ref SetMainSend
//!         - エフェクトセンド ... @ref SetFxSend
//!
//!         (DRC 出力向け)
//!         - 音量 ... @ref SetVolume, @ref StopAllSound, @ref PauseAllSound, @ref SetDrcOutVolume
//!
//!         (リモコン出力向け)
//!         - 音量 ... @ref SetVolume, @ref StopAllSound, @ref PauseAllSound, @ref SetDrcOutVolume
//!
//!         @n
//!
//!         各パラメータは、上記の影響の他に、下記の影響を受けます。
//!         - SoundMaker で指定したデータの影響
//!         - @ref SoundHandle や @ref SoundActor による影響
//!         - 3D サウンド計算 (@ref Sound3DEngine) による影響
//!
//! @see SoundArchivePlayer クラス
//! @see SoundArchivePlayer::GetSoundPlayer
//!
//! @date 2011/11/25 SetDrcOutVolume などの追加
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundPlayer
{
public:
    //! @briefprivate
    typedef ut::LinkList<
        internal::PlayerHeap,
        offsetof(internal::PlayerHeap,m_Link)
    > PlayerHeapList;

    //! @briefprivate
    typedef ut::LinkList<
        internal::BasicSound,
        offsetof(internal::BasicSound,m_SoundPlayerPlayLink)
    > SoundList;

    //! @briefprivate
    typedef ut::LinkList<
        internal::BasicSound,
        offsetof(internal::BasicSound,m_SoundPlayerPriorityLink)
    > PriorityList;

public:

    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundPlayer();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    ~SoundPlayer();
    //@}

    //----------------------------------------
    //! @name 更新
    //@{
    //---------------------------------------------------------------------------
    //! @brief        サウンドプレイヤーの更新処理を行います。
    //!
    //!               @ref SoundArchivePlayer クラスを使用している場合は、
    //!               この関数は @ref SoundArchivePlayer::Update から呼び出されます。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see SoundArchivePlayer::Update
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Update();
    //@}

    //----------------------------------------
    //! @name 再生制御
    //@{
    //---------------------------------------------------------------------------
    //! @brief        すべてのサウンドを停止します。
    //!
    //!               プレイヤーで再生中のすべてのサウンドを停止します。
    //!               各サウンドに対して、
    //!               ハンドルクラスを通して停止させたときと同様の処理を行います。
    //!
    //!               fadeFrames で指定したフレーム数をかけて、
    //!               フェードアウトさせることができます。
    //!               0 を指定した場合は、フェードアウトを行いません。
    //!               ただし、シーケンスサウンドで発音中の音は、
    //!               エンベロープのリリースを発音しすべての減衰が完了した後に
    //!               サウンドが停止します。
    //!
    //!               フェードアウトの音量制御は、フェードインと共有されます。
    //!               フェードアウトにかかるフレーム数は、
    //!               最大音量から音が消えるまでにかかる変化速度を表しますので、
    //!               フェードイン中にフェードアウトを指定した時などは、
    //!               指定したフレーム数よりも短い時間で
    //!               フェードアウトが完了する可能性があります。
    //!
    //! @param[in]    fadeFrames  フェードアウトにかけるフレーム数です。
    //!                           フレーム数は @ref SoundArchivePlayer::Update
    //!                           の呼び出し回数で換算されます。
    //!                           fadeFrames には 0 以上の値を指定してください。
    //!                           負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void StopAllSound( int fadeFrames );

    //---------------------------------------------------------------------------
    //! @brief        すべてのサウンドを一時停止・再開します。
    //!
    //!               プレイヤーで再生中のすべてのサウンドを一時停止または再開します。
    //!               各サウンドに対して、
    //!               ハンドルクラスを通して一時停止・再開させたときと同様の処理を行います。
    //!
    //!               一時停止・再開時のフェードは、再生開始時のフェードイン、
    //!               停止時のフェードアウトとは独立してはたらきます。
    //!
    //! @param[in]    flag        true なら一時停止、false なら再開。
    //! @param[in]    fadeFrames  フェードイン・フェードアウトにかけるフレーム数。
    //!                           フレーム数は @ref SoundArchivePlayer::Update
    //!                           の呼び出し回数で換算されます。
    //!                           fadeFrames には 0 以上の値を指定してください。
    //!                           負値を指定した場合は、0 を指定した場合と同じ挙動になります。
    //!
    //! @date 2014/03/13 fadeFrames の範囲について追記
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void PauseAllSound( bool flag, int fadeFrames );
    //@}

    //----------------------------------------
    //! @name パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    プレイヤーの音量を変更します。
    //!
    //!           プレイヤーの音量は、
    //!           プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!           この関数で指定する値は、
    //!           サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!           それらはすべて重ね合わされます。
    //!
    //!           音量 volume は、0.0 以上の倍率で指定します。
    //!           すなわち、1.0 を指定すると音量に影響を与えません。
    //!           0.0 を指定すると発音されなくなります。
    //!           デフォルト値は 1.0 です。
    //!
    //!           他の音量パラメータと重ね合わされたあと、
    //!           最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!           この関数で 2.0 を設定したとしても、
    //!           元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in]    volume    プレイヤーの音量の倍率（ 0.0 ～ ）です。
    //!
    //! @see GetVolume
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVolume( f32 volume );

    //---------------------------------------------------------------------------
    //! @brief    プレイヤーの音量を取得します。
    //!
    //! @return   プレイヤーに設定されている現在の音量を返します。
    //!
    //! @see SetVolume
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetVolume() const { return m_Volume; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーのローパスフィルタカットオフ値を変更します。
    //!
    //!         プレイヤーのローパスフィルタは、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、サウンドに設定される
    //!         他のどのローパスフィルタカットオフパラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         lpfFreq は、カットオフの相対変化の値を指定します。
    //!         0.0 を指定するとカットオフの値を変更しません。
    //!         -1.0 を指定すると、フィルタがかかっていない状態から、
    //!         もっともフィルタがかかっている状態（カットオフ周波数が下がる方向）に変更します。
    //!
    //! @param[in] lpfFreq    0.0 を基準としたフィルタカットオフの相対変化の値です。
    //!
    //! @see GetLpfFreq
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetLpfFreq( f32 lpfFreq );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されているローパスフィルタのカットオフ周波数を取得します。
    //!
    //! @return プレイヤーに設定されている、
    //!         現在のローパスフィルタのカットオフ周波数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetLpfFreq() const { return m_LpfFreq; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーの biquad フィルタの設定を変更します。
    //!
    //!         プレイヤーの biquad フィルタは、プレイヤーで再生中の
    //!         すべてのサウンドに対して効果があります。
    //!
    //!         biquad フィルタは複数の箇所での設定が重ね合わされず、
    //!         以下の優先度に従って設定されます。
    //!         優先度が高い箇所でパラメータの設定がされた場合、
    //!         それより下位の設定は上書きされます。
    //!
    //!         -# サウンドハンドルでの設定
    //!         -# サウンドプレイヤーでの設定
    //!         -# アンビエントパラメータ構造体での設定
    //!         -# シーケンスデータでの設定
    //!
    //!         フィルタの種類 type は @ref BiquadFilterType の値を使用します。
    //!         プリセットで用意されているフィルタの種類の他、
    //!         ユーザーが登録したフィルタを選択することができます。
    //!
    //!         フィルタのかかり具合を指定する value は、
    //!         0.0f ～ 1.0f の値で指定します。
    //!         値の意味はフィルタの係数の種類によって変化します。
    //!
    //! @param[in] type     フィルタの種類です (BIQUAD_FILTER_TYPE_MIN ～ BIQUAD_FILTER_TYPE_MAX)。
    //! @param[in] value    フィルタのかかり具合です (0.0f ～ 1.0f)。
    //!
    //! @see BiquadFilterType
    //! @see GetBiquadFilterType
    //! @see GetBiquadFilterValue
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetBiquadFilter( int type, f32 value );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されている biquad フィルタの種類を取得します。
    //!
    //!         返される値については @ref BiquadFilterType を参照してください。
    //!
    //! @return 現在の biquad フィルタの種類を返します。
    //!
    //! @see BiquadFilterType
    //! @see SetBiquadFilter
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetBiquadFilterType() const { return m_BiquadType; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されている biquad フィルタの設定値を取得します。
    //!
    //! @return 現在の biquad フィルタの設定値を返します。
    //!
    //! @see SetBiquadFilter
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    f32 GetBiquadFilterValue() const { return m_BiquadValue; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで再生するサウンドの出力先を設定します。
    //!
    //!         サウンドを再生する際、サウンドの出力先はこの関数で設定された値になります。
    //!         @ref SoundHandle::SetOutputLine を呼び出して、
    //!         サウンドごとに異なる出力先を設定することも可能です。
    //!
    //!         この関数は、プレイヤーで再生中のサウンドの出力先は変更しません。
    //!         この関数で出力先を設定した後にプレイヤーで再生されるサウンドについて、
    //!         新しい出力先が反映されます。
    //!         プレイヤーで再生中のサウンドの出力先を変更するには、
    //!         それぞれのサウンドに対して @ref SoundHandle::SetOutputLine を呼び出してください。
    //!
    //!         出力先は @ref OutputLine のビットフラグの論理和で指定します。
    //!         指定した値は、それまで設定されていた値を上書きします。
    //!         初期状態では、@ref nw::snd::OUTPUT_LINE_MAIN のみが指定されています。
    //!
    //! @param[in] lineFlag     出力先のビットフラグです。
    //!
    //! @see OutputLine
    //! @see GetDefaultOutputLine
    //! @see SoundHandle::SetOutputLine
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetDefaultOutputLine( u32 lineFlag );
    //RVL:  サウンドの出力に関しての詳細は「解説：サウンド出力の流れ」を参照してください。

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されている、サウンドの出力先を取得します。
    //!
    //! @return 現在設定されているサウンドの出力先を返します。
    //!
    //! @see OutputLine
    //! @see SetDefaultOutputLine
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetDefaultOutputLine() const { return m_OutputLineFlag; }

    //@}

    //----------------------------------------
    //! @name メイン (TV) 出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プレイヤーのメイン (TV) 出力に対する音量を変更します。
    //!
    //!         プレイヤーのメイン出力に対する音量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどのメイン出力音量とも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         メイン出力音量は、メイン出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!         0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!         他のメイン出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] volume   変更する音量の倍率です (0.0～)。
    //!
    //! @see GetMainOutVolume
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetMainOutVolume( f32 volume ) );
    //RVL:      詳しくは「解説：サウンド出力の流れ」を参照してください。

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されているメイン (TV) 出力に対する音量を取得します。
    //!
    //! @return 現在のメイン出力に対する音量を返します。
    //!
    //! @see SetMainOutVolume
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetMainOutVolume() const ) { return m_TvParam.volume; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーのメイン (TV) 出力へのメインセンド量を変更します。
    //!
    //!         プレイヤーのメイン出力へのセンド量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         メインセンドは、メイン出力に送るサウンドの音量を調節するパラメータです。
    //!         主に、エフェクトのドライ・ウェット成分のバランスを調整するために使用されます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         -1.0 を指定するとメインバスに最大のセンド量で送られていたサウンドが
    //!         メインバスに送られないようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] send     0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see GetMainSend
    //! @see SetFxSend
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetMainSend( f32 send ) ) { m_TvParam.mainSend = send; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されているメイン (TV) 出力へのメインセンド量を取得します。
    //!
    //! @return 現在のメイン出力へのセンド量を返します。
    //!
    //! @see SetMainSend
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetMainSend() const ) { return m_TvParam.mainSend; }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーのメイン (TV) 出力へのエフェクトセンド量を変更します。
    //!
    //!         プレイヤーのエフェクトへのセンド量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         センド量 send は、相対変化の値を指定します。
    //!         すなわち、0.0 を指定するとセンド量を変更しません。
    //!         1.0 を指定すると AUX バスに送られていなかったサウンドが
    //!         最大のセンド量で送られるようになります。 デフォルト値は 0.0 です。
    //!
    //! @param[in] bus      センド量を設定する AUX バスです。
    //! @param[in] send     0.0 を基準としたセンド量の相対変化の値です。
    //!
    //! @see AuxBus
    //! @see GetFxSend
    //! @see SetMainSend
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetFxSend( AuxBus bus, f32 send ) )
    {
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        m_TvParam.fxSend[bus] = send;
    }

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されているメイン (TV) 出力へのエフェクトセンド量を取得します。
    //!
    //! @return 現在のエフェクトへのセンド量を返します。
    //!
    //! @param[in] bus      センド量を設定する AUX バスです。
    //!
    //! @see AuxBus
    //! @see SetFxSend
    //!
    //! @date 2011/11/25 文言の微調整
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetFxSend( AuxBus bus ) const )
    {
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        return m_TvParam.fxSend[ bus ];
    }
    //@}

    //----------------------------------------
    //! @name DRC 出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プレイヤーの DRC 出力音量を変更します。
    //!
    //!         プレイヤーの DRC 出力音量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         DRC 出力音量は、
    //!         指定した番号の DRC 出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!         0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!         他の DRC 出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] volume       変更する音量の倍率です (0.0～)。
    //! @param[in] drcIndex     DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @see GetDrcOutVolume
    //!
    //! @date 2011/11/25 drcIndex 引数を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( void SetDrcOutVolume( f32 volume, u32 drcIndex=0 ) );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されている DRC 出力音量を取得します。
    //!
    //! @param[in] drcIndex   DRC の番号です。現状 0 以外を指定しても効果はありません。
    //!
    //! @return 現在の DRC 出力音量を返します。
    //!
    //! @see SetDrcOutVolume
    //!
    //! @date 2011/11/25 drcIndex 引数を追加
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    NW_DEPRECATED_FUNCTION( f32 GetDrcOutVolume( u32 drcIndex=0 ) const )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_DrcParam[drcIndex].volume;
    }

    //! @briefprivate
    //! @param send :private
    //! @param drcIndex :private
    NW_DEPRECATED_FUNCTION( void SetDrcMainSend( f32 send, u32 drcIndex = 0 ) )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        m_DrcParam[drcIndex].mainSend = send;
    }
    //! @briefprivate
    //! @param drcIndex :private
    //! @return :private
    NW_DEPRECATED_FUNCTION( f32 GetDrcMainSend( u32 drcIndex = 0 ) const )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        return m_DrcParam[drcIndex].mainSend;
    }

    //! @briefprivate
    //! @param bus :private
    //! @param send :private
    //! @param drcIndex :private
    NW_DEPRECATED_FUNCTION( void SetDrcFxSend( AuxBus bus, f32 send, u32 drcIndex = 0 ) )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        m_DrcParam[drcIndex].fxSend[bus] = send;
    }
    //! @briefprivate
    //! @param bus :private
    //! @param drcIndex :private
    //! @return :private
    NW_DEPRECATED_FUNCTION( f32 GetDrcFxSend( AuxBus bus, u32 drcIndex = 0 ) const )
    {
        NW_ASSERT_MAXLT( drcIndex, DRC_OUT_COUNT );
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        return m_DrcParam[drcIndex].fxSend[bus];
    }
    //@}

    //----------------------------------------
    //! @name 出力先別パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  指定した出力先について、プレイヤーの音量を変更します。
    //!
    //!         プレイヤーの音量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         この出力音量は、
    //!         指定した番号の指定した出力先に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!         0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!         他の出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] device       出力先のデバイスです。
    //! @param[in] volume       変更する音量の倍率です (0.0～)。
    //!
    //! @see GetOutputVolume
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    void SetOutputVolume( OutputDevice device,f32 volume );

    //---------------------------------------------------------------------------
    //! @brief  指定した出力先について、プレイヤーに設定されている音量を取得します。
    //!
    //! @param[in] device   出力先のデバイスです。
    //!
    //! @return 指定した出力先の現在の音量を返します。
    //!         出力先のデバイスが存在しない場合、0.0f を返します。
    //!
    //! @see SetOutputVolume
    //!
    //! @date 2012/12/19 初版
    //---------------------------------------------------------------------------
    f32 GetOutputVolume( OutputDevice device ) const
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch ( device )
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            return m_TvParam.volume;
        case nw::snd::OUTPUT_DEVICE_DRC:
            return m_DrcParam[0].volume;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
        return 0.0f;
    }

    //! @briefprivate
    //! @param device :private
    //! @param send :private
    void SetOutputMainSend( OutputDevice device, f32 send )
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch ( device )
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            m_TvParam.mainSend = send;
            break;
        case nw::snd::OUTPUT_DEVICE_DRC:
            m_DrcParam[0].mainSend = send;
            break;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
    }
    //! @briefprivate
    //! @param device :private
    //! @return :private
    f32 GetOutputMainSend( OutputDevice device ) const
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        switch ( device )
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            return m_TvParam.mainSend;
        case nw::snd::OUTPUT_DEVICE_DRC:
            return m_DrcParam[0].mainSend;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
        return 0.0f;
    }

    //! @briefprivate
    //! @param device :private
    //! @param bus :private
    //! @param send :private
    void SetOutputFxSend( OutputDevice device, AuxBus bus, f32 send )
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        switch ( device )
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            m_TvParam.fxSend[bus] = send;
            break;
        case nw::snd::OUTPUT_DEVICE_DRC:
            m_DrcParam[0].fxSend[bus] = send;
            break;
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
    }
    //! @briefprivate
    //! @param device :private
    //! @param bus :private
    //! @return :private
    f32 GetOutputFxSend( OutputDevice device, AuxBus bus ) const
    {
        NW_ASSERT_MINMAXLT( device, 0, OUTPUT_DEVICE_COUNT );
        NW_MINMAXLT_ASSERT( bus, 0, AUX_BUS_NUM );
        switch ( device )
        {
        case nw::snd::OUTPUT_DEVICE_MAIN:
            return m_TvParam.fxSend[bus];
        case nw::snd::OUTPUT_DEVICE_DRC:
            return m_DrcParam[0].fxSend[bus];
        default:
            NW_ASSERTMSG( false, "Invalid device\n" );
            break;
        }
        return 0.0f;
    }
    //@}

    //----------------------------------------
    //! @name リモコン出力用パラメータ
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プレイヤーのリモコン出力音量を変更します。
    //!
    //!         プレイヤーのリモコン出力音量は、
    //!         プレイヤーで再生中のすべてのサウンドに対して効果があります。
    //!
    //!         この関数で指定する値は、
    //!         サウンドに設定される他のどの音量パラメータとも独立して動作し、
    //!         それらはすべて重ね合わされます。
    //!
    //!         リモコン出力音量は、
    //!         指定したチャンネルのリモコン出力に対してのみ影響する音量です。
    //!
    //!         音量 volume は、0.0 以上の倍率で指定します。
    //!         すなわち、1.0 を指定すると音量に影響を与えません。
    //!         0.0 を指定すると発音されなくなります。デフォルト値は 1.0 です。
    //!
    //!         他のリモコン出力音量と重ね合わされたあと、
    //!         最終的な音量は 0.0 ～ 2.0 の範囲でクランプされます。
    //!         この関数で 2.0 を設定したとしても、
    //!         元の音量の 2 倍にならない可能性があることに注意してください。
    //!
    //! @param[in] remoteIndex  リモコンのチャンネル番号です。
    //! @param[in] volume       変更する音量の倍率です (0.0～)。
    //!
    //! @see GetRemoteOutVolume
    //!
    //! @date 2011/08/19 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    void SetRemoteOutVolume( int remoteIndex, f32 volume );
#else
    NW_DEPRECATED_FUNCTION_MSG( void SetRemoteOutVolume( int remoteIndex, f32 volume ), "This API is only supported for Cafe.");
#endif
    //RVL: 詳しくは「解説：サウンド出力の流れ」を参照してください。

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーに設定されているリモコン出力音量を取得します。
    //!
    //! @param[in] remoteIndex  リモコンのチャンネル番号です。
    //!
    //! @return 現在のリモコン出力音量を返します。
    //!
    //! @see SetRemoteOutVolume
    //!
    //! @date 2011/08/19 初版
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    f32 GetRemoteOutVolume( int remoteIndex ) const;
#else
    NW_DEPRECATED_FUNCTION_MSG( f32 GetRemoteOutVolume( int remoteIndex ) const, "This API is only supported for Cafe.");
#endif

    //@}

    //----------------------------------------
    //! @name サウンド数
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで現在再生中のサウンドの個数を取得します。
    //!
    //! @return プレイヤーで再生中のサウンド数を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetPlayingSoundCount() const { return static_cast<int>( m_SoundList.GetSize()); }

    //---------------------------------------------------------------------------
    //! @brief  同時に再生可能なサウンド数を設定します。
    //!
    //!         設定したサウンド数を超えるサウンドを再生しようとすると、
    //!         各サウンドに設定されたプレイヤープライオリティを比較して、
    //!         もっとも優先度が低いサウンドが停止します。
    //!
    //!         @ref SoundArchivePlayer クラスを使用する場合は、
    //!         サウンドアーカイブ中で指定されている同時再生数が初期化時に自動的に設定されます。
    //!
    //!         プレイヤーヒープを使用している場合、
    //!         この関数で指定できる同時再生数の上限値は、
    //!         セットアップ時に設定された同時再生数の値に制限されます。
    //!         上限値を超える値を指定しても上限値に丸め込まれます。
    //!
    //! @param[in] count        同時に再生可能なサウンド数です。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see GetPlayableSoundCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPlayableSoundCount( int count );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで同時に再生可能なサウンド数を取得します。
    //!
    //! @return プレイヤーで同時に再生可能なサウンド数を返します。
    //!
    //! @see SetPlayableSoundCount
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetPlayableSoundCount() const { return m_PlayableCount; }
    //@}

    //----------------------------------------
    //! @name その他
    //@{
    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで再生中のすべてのサウンドに対して処理を行います。
    //!
    //!         @ref ForEachSound のFunctionの参照渡し版となります。
    //!
    //! @param[in] function 関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse  処理順を逆にする場合は true を指定します。
    //!
    //! @see SoundHandle クラス
    //! @see ForEachSoundPriorityOrderRef
    //!
    //! @date 2012/02/10 初版
    //---------------------------------------------------------------------------
    template< class Function >
    void ForEachSoundRef( Function& function, bool reverse = false );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで再生中のすべてのサウンドに対して処理を行います。
    //!
    //!         プレイヤーで再生中のすべてのサウンドに対して、
    //!
    //! @code
    //! function( nw::snd::SoundHandle& handle )
    //! @endcode
    //!
    //!         を呼び出します。 function には、
    //!         再生中のサウンドに関連付けられたサウンドハンドル handle が渡されます。
    //!         これは一時的なハンドルですので、ハンドルを後で使用することはできません。
    //!
    //!         関数 function は、サウンドの再生が古い順に呼び出されます。
    //!         reverse に true を指定すると、サウンドの再生が新しい順に呼び出されます。
    //!
    //!         function には関数ポインタ、または関数オブジェクトを渡します。
    //!         関数ポインタを渡す例を以下に示します。
    //!
    //! @code
    //! void ReportSoundId( nw::snd::SoundHandle& handle )
    //! {
    //!     NW_LOG( "%d\n", handle.GetId() );
    //! }
    //! soundPlayer.ForEachSound( ReportSoundId );
    //! @endcode
    //!
    //! @param[in] function 関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse  処理順を逆にする場合は true を指定します。
    //!
    //! @return 引数に指定された関数ポインタ、または関数オブジェクトを返します。
    //!
    //! @see SoundHandle クラス
    //! @see ForEachSoundPriorityOrder
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    template< class Function >
    Function ForEachSound( Function function, bool reverse = false );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで再生中のすべてのサウンドに対してプレイヤープライオリティ順で処理を行います。
    //!
    //!         その他の動作仕様は @ref ForEachSoundRef と同じです。
    //!
    //! @param[in] function 関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse  処理順を逆にする場合は true を指定します。
    //!
    //! @see SoundHandle クラス
    //! @see ForEachSoundRef
    //!
    //! @date 2012/02/10 初版
    //---------------------------------------------------------------------------
    template< class Function >
    void ForEachSoundPriorityOrderRef( Function& function, bool reverse = false );

    //---------------------------------------------------------------------------
    //! @brief  プレイヤーで再生中のすべてのサウンドに対してプレイヤープライオリティ順で処理を行います。
    //!
    //!         その他の動作仕様は @ref ForEachSound と同じです。
    //!
    //! @param[in] function 関数ポインタ、または関数オブジェクトです。
    //! @param[in] reverse  処理順を逆にする場合は true を指定します。
    //!
    //! @return 引数に指定された関数ポインタ、または関数オブジェクトを返します。
    //!
    //! @see SoundHandle クラス
    //! @see ForEachSound
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    template< class Function >
    Function ForEachSoundPriorityOrder( Function function, bool reverse = false );
    //@}

    //-----------------------------------------------------------------------------
    //  internal functions

    //! @briefprivate
    //! @param limit :private
    void detail_SetPlayableSoundLimit( int limit );

    // サウンドの登録
    //! @briefprivate
    //! @param startPriority :private
    //! @return :private
    bool detail_CanPlaySound( int startPriority );
    //! @briefprivate
    //! @param pSound :private
    //! @return :private
    bool detail_AppendSound( internal::BasicSound* pSound );
    //! @briefprivate
    //! @param pSound :private
    void detail_RemoveSound( internal::BasicSound* pSound );

    // プライオリティリスト
    //! @briefprivate
    //! @param reverse :private
    void detail_SortPriorityList( bool reverse = false );
    //! @briefprivate
    //! @param pSound :private
    void detail_SortPriorityList( internal::BasicSound* pSound );

    // プレイヤーヒープ
    //! @briefprivate
    //! @param pHeap :private
    void detail_AppendPlayerHeap( internal::PlayerHeap* pHeap );

    // void detail_RemovePlayerHeap( internal::PlayerHeap* pHeap );

    //! @briefprivate
    //! @return :private
    bool detail_CanUsePlayerHeap() const
    {
        if (m_PlayerHeapCount > 0)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    //! @briefprivate
    //! @return :private
    internal::PlayerHeap* detail_AllocPlayerHeap();
    //! @briefprivate
    //! @param pHeap :private
    void detail_FreePlayerHeap(internal::PlayerHeap* pHeap);

    //! @briefprivate
    //! @return :private
    bool IsFirstComeBased() { return m_IsFirstComeBased; }
    //! @briefprivate
    //! @param enabled :private
    void SetFirstComeBased( bool enabled )
    {
        if ( m_IsFirstComeBased != enabled )
        {
            m_IsFirstComeBased = enabled;
            detail_SortPriorityList( true );
        }
    }

private:
    //-----------------------------------------------------------------------------
    //  private functions

    internal::BasicSound* GetLowestPrioritySound() { return &m_PriorityList.GetFront(); }
    void InsertPriorityList( internal::BasicSound* pSound );
    void RemovePriorityList( internal::BasicSound* pSound );
    void RemoveSoundList( internal::BasicSound* pSound );

    void DoFreePlayerHeap();

    //-----------------------------------------------------------------------------
    //  private variables

    SoundList       m_SoundList;
    PriorityList    m_PriorityList;
    PlayerHeapList  m_PlayerHeapFreeList;
    PlayerHeapList  m_PlayerHeapFreeReqList;

    int m_PlayableCount;
    int m_PlayableLimit;
    u32 m_PlayerHeapCount;

    f32 m_Volume;
    f32 m_LpfFreq;
    int m_BiquadType;
    f32 m_BiquadValue;

    u32 m_OutputLineFlag;

    struct OutputParam
    {
        f32 volume;
        f32 mainSend;
        f32 fxSend[AUX_BUS_NUM];

        void Initialize()
        {
            volume = 1.0f;
            mainSend = 0.0f;
            for ( int i = 0; i < AUX_BUS_NUM; i++ )
            {
                fxSend[i] = 0.0f;
            }
        }
    };
    OutputParam m_TvParam;
    OutputParam m_DrcParam[DRC_OUT_COUNT];

    f32 m_RemoteOutVolume[ REMOTE_OUT_COUNT ];

    bool m_IsFirstComeBased;
};

template< class Function >
inline void SoundPlayer::ForEachSoundRef( Function& function, bool reverse )
{
    if ( reverse )
    {
        // 再生の新しい順
        for ( SoundList::ReverseIterator itr = m_SoundList.GetBeginReverseIter();
              itr != m_SoundList.GetEndReverseIter();
            )
        {
            SoundList::ReverseIterator curItr = itr;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
            if ( handle.IsAttachedSound() ) itr++;
        }
    }
    else
    {
        // 再生の古い順
        for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
              itr != m_SoundList.GetEndIter();
            )
        {
            SoundList::Iterator curItr = itr++;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
        }
    }
}

template< class Function >
inline Function SoundPlayer::ForEachSound( Function function, bool reverse )
{
    if ( reverse )
    {
        // 再生の新しい順
        for ( SoundList::ReverseIterator itr = m_SoundList.GetBeginReverseIter();
              itr != m_SoundList.GetEndReverseIter();
            )
        {
            SoundList::ReverseIterator curItr = itr;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
            if ( handle.IsAttachedSound() ) itr++;
        }
    }
    else
    {
        // 再生の古い順
        for ( SoundList::Iterator itr = m_SoundList.GetBeginIter();
              itr != m_SoundList.GetEndIter();
            )
        {
            SoundList::Iterator curItr = itr++;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
        }
    }
    return function;
}

template< class Function >
void SoundPlayer::ForEachSoundPriorityOrderRef( Function& function, bool reverse )
{
    if ( reverse )
    {
        // プライオリティの低い順
        for ( PriorityList::Iterator itr = m_PriorityList.GetBeginIter();
              itr != m_PriorityList.GetEndIter();
            )
        {
            PriorityList::Iterator curItr = itr++;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
        }
    }
    else
    {
        // プライオリティの高い順
        for ( PriorityList::ReverseIterator itr = m_PriorityList.GetBeginReverseIter();
              itr != m_PriorityList.GetEndReverseIter();
            )
        {
            PriorityList::ReverseIterator curItr = itr;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
            if ( handle.IsAttachedSound() ) itr++;
        }
    }
}

template< class Function >
Function SoundPlayer::ForEachSoundPriorityOrder( Function function, bool reverse )
{
    if ( reverse )
    {
        // プライオリティの低い順
        for ( PriorityList::Iterator itr = m_PriorityList.GetBeginIter();
              itr != m_PriorityList.GetEndIter();
            )
        {
            PriorityList::Iterator curItr = itr++;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
        }
    }
    else
    {
        // プライオリティの高い順
        for ( PriorityList::ReverseIterator itr = m_PriorityList.GetBeginReverseIter();
              itr != m_PriorityList.GetEndReverseIter();
            )
        {
            PriorityList::ReverseIterator curItr = itr;
            SoundHandle handle;
            handle.detail_AttachSoundAsTempHandle( &( *curItr ) );
            function( handle );
            if ( handle.IsAttachedSound() ) itr++;
        }
    }
    return function;
}

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_PLAYER_H_ */

