﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SoundHeap.h
 *
 * @file snd_SoundHeap.h
 */

#ifndef NW_SND_SOUND_HEAP_H_
#define NW_SND_SOUND_HEAP_H_

#include <nw/snd/snd_SoundMemoryAllocatable.h>
#include <nw/snd/snd_FrameHeap.h>
#include <nw/snd/snd_Util.h>
#include <nw/ut.h>

namespace nw {
namespace snd {

//---------------------------------------------------------------------------
//! @brief    サウンドデータ用のヒープクラスです。
//!
//!           サウンドヒープでは、
//!           メモリブロックが解放された時に呼び出されるコールバックを
//!           設定することができます。
//!           サウンドライブラリではこの機能を用いて、
//!           サウンドデータを含むメモリブロックが解放された時に、
//!           安全にデータを解放するための処理を行うように設計されています。
//!
//! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class SoundHeap : public SoundMemoryAllocatable
{
public:
    //---------------------------------------------------------------------------
    //! @brief    メモリブロックが破棄された時に呼び出されるコールバックです。
    //!
    //!           コールバック関数は @ref SoundHeap::Alloc
    //!           の引数としてメモリブロックに渡し、
    //!           @ref SoundHeap::Clear または @ref SoundHeap::LoadState によって、
    //!           メモリブロックが解放されたときに呼びだされます。
    //!
    //! @param[in]    mem         解放されたメモリブロックの先頭アドレスです。
    //! @param[in]    size        解放されたメモリブロックのサイズです。
    //! @param[in]    userArg     ユーザ引数です。
    //!
    //! @see Alloc
    //! @see Clear
    //! @see LoadState
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    typedef void (*DisposeCallback)( void* mem, unsigned long size, void* userArg );


public:
    //----------------------------------------
    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    SoundHeap();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~SoundHeap();
    //@}

    //----------------------------------------
    //! @name ヒープ操作
    //@{
    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープを作成します。
    //!
    //!               メモリ領域のサイズ size が十分に無いと、関数は失敗します。
    //!               ヒープの管理領域が必要なため、
    //!               作成されたメモリのヒープサイズは size よりも小さくなります。
    //!
    //!               この関数に渡したメモリ領域を再利用するには、
    //!               @ref nw::snd::SoundHeap::Destroy
    //!               でヒープを破棄する必要があります。
    //!
    //! @param[in]    startAddress    ヒープとして使用するメモリの先頭アドレス。
    //! @param[in]    size            ヒープとして使用するメモリのサイズ。
    //!
    //! @return       ヒープの作成に成功したら true を、
    //!               失敗したら false を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool Create(
        void* startAddress,
        size_t size
    );

    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープを破棄します。
    //!
    //!               確保済みのメモリブロックそれぞれに対して、
    //!               @ref Alloc で設定したコールバック関数が呼び出されます。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Destroy();

    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープからメモリを確保します。
    //!
    //!               各メモリブロックには管理領域が必要になります。
    //!               そのため、実際には確保するメモリサイズ size
    //!               よりも少し大きな空き容量が必要になります。
    //!               空き容量が足りない場合は、関数は失敗します。
    //!
    //! @param[in]    size            確保するメモリサイズ。
    //!
    //! @return       確保したメモリブロックの先頭アドレスを返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual void* Alloc( size_t size );

    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープからメモリを確保します。
    //!
    //!               各メモリブロックには管理領域が必要になります。
    //!               そのため、実際には確保するメモリサイズ size
    //!               よりも少し大きな空き容量が必要になります。
    //!               空き容量が足りない場合は、関数は失敗します。
    //!
    //!               コールバック関数 callback は、
    //!               @ref Clear または @ref LoadState によって、
    //!               メモリブロックが解放されたときに呼び出されます。
    //!               コールバック関数が不要の時は、callback に NULL を入れます。
    //!
    //! @param[in]    size            確保するメモリサイズ。
    //! @param[in]    callback        メモリブロックが解放されるときに呼び出される
    //!                               コールバック関数。
    //! @param[in]    callbackArg     コールバック関数の引数用のユーザーデータ。
    //!
    //! @return       確保したメモリブロックの先頭アドレスを返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void* Alloc(
        size_t size,
        SoundHeap::DisposeCallback callback,
        void* callbackArg
    );

    //---------------------------------------------------------------------------
    //! @brief        確保したメモリブロックを全て解放します。
    //!
    //!               解放したメモリブロックそれぞれに対して、@ref Alloc で設定した
    //!               コールバック関数が呼び出されます。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Clear();

    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープが有効かどうかを調べます。
    //!
    //!               @ref Create を呼び出してサウンドヒープが作成され、
    //!               メモリブロックの確保が可能な状態であれば、true を返します。
    //!
    //! @return       サウンドヒープが有効なら true を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    bool IsValid() const { return m_FrameHeap.IsValid(); }
    //@}

    //----------------------------------------
    //! @name 階層管理
    //---------------------------------------------------------------------------
    //! @brief        サウンドヒープの現在の状態を保存します。
    //!
    //!               ヒープ作成直後の階層レベルは 0 で、
    //!               この関数を呼ぶ毎に階層レベルがひとつ増えます。
    //!               @ref LoadState を呼びだすことで、
    //!               指定した階層レベルの保存直後の状態に復元させることができます。
    //!
    //!               状態の保存にはヒープを少し消費します。
    //!               ヒープの空き容量が足りない場合は、関数の呼びだしは失敗します。
    //!
    //! @return       状態保存後の階層レベルを返します。
    //!               状態の保存に失敗すると -1 を返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int SaveState();

    //---------------------------------------------------------------------------
    //! @brief        保存したサウンドヒープの状態を復元します。
    //!
    //!               サウンドヒープを @ref SaveState で保存した直後の状態に戻します。
    //!               状態を保存してから現在までに確保されたメモリブロックは、
    //!               全て解放されます。 解放したメモリブロックそれぞれに対して、
    //!               @ref Alloc で設定したコールバック関数が呼びだされます。
    //!
    //!               階層レベル level は現在の階層レベルの値と同じか、
    //!               より小さい値を指定します。
    //!               0 を指定すると、@ref Clear と同じ意味になります。
    //!               ヒープは、状態が保存された直後の状態に戻り、
    //!               現在の階層レベルも指定した値になります。
    //!
    //! @param[in]    level   復元する階層レベル。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void LoadState( int level );

    //---------------------------------------------------------------------------
    //! @brief        現在のサウンドヒープの階層レベルを取得します。
    //!
    //!               ヒープ作成直後の階層レベルは 0 です。
    //!               @ref SaveState を呼びだす毎に、階層レベルがひとつ増えます。
    //!               @ref LoadState で状態を復元すると、
    //!               現在の階層レベルも指定した階層レベルに戻ります。
    //!
    //! @return       現在の階層レベルを返します。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    int GetCurrentLevel() const
    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
        return m_FrameHeap.GetCurrentLevel();
    }
    //@}

    //----------------------------------------
    //! @name 情報取得
    //---------------------------------------------------------------------------
    //! @brief        メモリ上のヒープのサイズを取得します。
    //!
    //! @return       メモリ上のヒープのサイズ。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    size_t GetSize() const
    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
        return m_FrameHeap.GetSize();
    }

    //---------------------------------------------------------------------------
    //! @brief        メモリ上のヒープの空き容量を取得します。
    //!
    //! @return       メモリ上のヒープの空き容量のサイズ。
    //!
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    size_t GetFreeSize() const
    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
        return m_FrameHeap.GetFreeSize();
    }

    //---------------------------------------------------------------------------
    //! @brief      ヒープの内容をダンプします。
    //!
    //!             arc には、@ref SoundDataManager::Initialize 関数に渡した
    //!             サウンドアーカイブを指定する必要があります。
    //!
    //!             サウンドヒープに複数のサウンドアーカイブのデータをロードしていると、
    //!             本関数は正常に動作しません。
    //!
    //! @param[in]   mgr  サウンドデータマネージャです。
    //! @param[in]   arc  サウンドアーカイブです。
    //!
    //! @date 2012/02/10 const を追加 (void Dump(); -> voice Dump() const;)
    //! @date 2011/07/06 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Dump( nw::snd::SoundDataManager& mgr, nw::snd::SoundArchive& arc ) const
    {
        ut::ScopedLock<ut::CriticalSection> lock( m_CriticalSection );
        m_FrameHeap.Dump( mgr, arc );
    }
    //@}

private:
    static void DisposeCallbackFunc( void* mem, unsigned long size, void* arg );

    mutable ut::CriticalSection m_CriticalSection;
    internal::FrameHeap m_FrameHeap;       // メインメモリ管理ヒープ
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_HEAP_H_ */

