﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_ARCHIVE_LOADER_H_
#define NW_SND_SOUND_ARCHIVE_LOADER_H_

#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_SoundArchive.h>

namespace nw {
namespace snd {

class SoundMemoryAllocatable;

namespace internal {

struct LoadItemInfo
{
    nw::snd::SoundArchive::ItemId itemId;
    const void* address;

    LoadItemInfo()
    : itemId( nw::snd::SoundArchive::INVALID_ID ),
      address( NULL )
    {}
};

/* ========================================================================
        class difinition
   ======================================================================== */

//! @briefprivate
class SoundArchiveLoader
{
public:
    //! @briefprivate
    static const u32 SIGNATURE_INDIVIDUAL_WAVE = NW_UT_MAKE_SIGWORD( 'F', 'I', 'W', 'V' );

    // Cafe 向けのファイル読み込み用アライメント
    static const u32 FILE_ALIGNMENT = 64;

    // 個別ロード波形のメモリブロック先頭に置かれる情報
    struct IndividualWaveInfo
    {
        // データ
        u32 signature;
        u32 fileId;         //< 所属する波形アーカイブファイル ID
        u32 waveIndex;      //< 波形アーカイブ内インデックス
        u32 padding[5];     // 32バイトアライメントのためのパディング
        u32 padding2[8];    // 64バイトアライメントのためのパディング

        // メソッド
        IndividualWaveInfo( u32 id, u32 index )
        : signature( SoundArchiveLoader::SIGNATURE_INDIVIDUAL_WAVE ),
          fileId( id ),
          waveIndex( index )
        {}
    };


    //---------------------------------------------------------------------------
    //! @brief  ロードフラグです。
    //---------------------------------------------------------------------------
    enum LoadFlag
    {
        // サウンド
        LOAD_SEQ    = ( 1 << 0 ),   //!< シーケンスサウンドをロードします
        LOAD_WSD    = ( 1 << 1 ),   //!< ウェーブサウンドをロードします

        // バンク
        LOAD_BANK   = ( 1 << 2 ),   //!< バンクをロードします

        // 波形アーカイブ
        LOAD_WARC   = ( 1 << 3 ),   //!< 波形アーカイブをロードします

        LOAD_ALL    = 0xffffffff    //!< 関連するファイルをすべてロードします
    };

    SoundArchiveLoader();
    virtual ~SoundArchiveLoader();

    //----------------------------------------
    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief  サウンドデータマネージャが利用可能な状態かどうかを調べます。
    //!
    //! @return   サウンドデータマネージャが利用可能な状態なら true を、
    //!           そうでなければ false を返します。
    //---------------------------------------------------------------------------
    bool IsAvailable() const;
    //@}


    //----------------------------------------
    //! @name ロード
    //@{
    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のデータをロードします。
    //!
    //!           ロードに必要なメモリはアロケータ pAllocator から確保されます。
    //!           十分なメモリが確保できない場合は false を返します。
    //!
    //!           この関数はサウンドデータの同期ロードを行います。
    //!           非同期でサウンドデータをロードする場合は、
    //!           ロード用のスレッドでこの関数を呼び出してください。
    //!           関連付けられているサウンドアーカイブや、
    //!           引数で渡すサウンドヒープに対する操作をロード中に行わない限り、
    //!           この関数はスレッドセーフです。
    //!
    //! @param[in]    id              ロードするデータのアイテム ID です。
    //! @param[in]    pAllocator      ロードするメモリを確保するためのアロケータです。
    //! @param[in]    loadFlag        関連するデータをロードするためのフラグです。
    //! @param[in]    loadBlockSize   データを分割ロードする際の分割サイズを、
    //!                               バイト単位で指定します。
    //!
    //! @return   ロードに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2010/12/17 参照から「Initialize」「IsAvailable」を削除、「IsDataLoaded」追加
    //! @date 2010/09/06 loadBlockSize 指定の対応にともない、引数の文言を調整
    //! @date 2010/04/09 文言の修正 (グループ→サウンドデータ)、「参照」に IsDataLoaded 追加
    //! @date 2010/01/15 初版
    //---------------------------------------------------------------------------
    bool LoadData(
        SoundArchive::ItemId id,
        SoundMemoryAllocatable* pAllocator,
        u32 loadFlag = LOAD_ALL, // LoadFlag の論理和を渡す
        size_t loadBlockSize = 0
    );

    //---------------------------------------------------------------------------
    //! @brief    サウンドアーカイブ中のデータをロードします。
    //!
    //!           ロードに必要なメモリはアロケータ pAllocator から確保されます。
    //!           十分なメモリが確保できない場合は false を返します。
    //!
    //!           この関数はサウンドデータの同期ロードを行います。
    //!           非同期でサウンドデータをロードする場合は、
    //!           ロード用のスレッドでこの関数を呼び出してください。
    //!           関連付けられているサウンドアーカイブや、
    //!           引数で渡すサウンドヒープに対する操作をロード中に行わない限り、
    //!           この関数はスレッドセーフです。
    //!
    //! @param[in]    pItemName       ロードするデータのラベル文字列です。
    //! @param[in]    pAllocator      ロードするメモリを確保するためのアロケータです。
    //! @param[in]    loadFlag        関連するデータをロードするためのフラグです。
    //! @param[in]    loadBlockSize   データを分割ロードする際の分割サイズを、
    //!                               バイト単位で指定します。
    //!
    //! @return   ロードに成功したら true を、失敗したら false を返します。
    //!
    //! @date 2010/12/17 参照から「Initialize」「IsAvailable」を削除
    //! @date 2010/09/06 loadBlockSize 指定の対応にともない、引数の文言を調整
    //! @date 2010/04/09 文言の修正 (グループ→サウンドデータ)、「参照」に IsDataLoaded 追加
    //! @date 2010/01/15 初版
    //---------------------------------------------------------------------------
    bool LoadData(
        const char* pItemName,
        SoundMemoryAllocatable* pAllocator,
        u32 loadFlag = LOAD_ALL, // LoadFlag の論理和を渡す
        size_t loadBlockSize = 0
    );

    //---------------------------------------------------------------------------
    //! @brief    指定したデータがロードされているか調べます。
    //!
    //!           loadFlag を利用することで、確認するデータを制限することができます。
    //!           デフォルトはすべての関連アイテムがロードされているかを確認します。
    //!
    //! @param[in] itemId     ロードされているか確認するサウンドデータのアイテム ID です。
    //! @param[in] loadFlag   関連するデータを確認するかを示すフラグです。
    //!
    //! @return   ロードされていれば true を、されていなければ false を返します。
    //!
    //! @date 2010/04/09 初版
    //---------------------------------------------------------------------------
    bool IsDataLoaded( SoundArchive::ItemId itemId, u32 loadFlag = LOAD_ALL ) const;

    //---------------------------------------------------------------------------
    //! @brief    指定したデータがロードされているか調べます。
    //!
    //!           loadFlag を利用することで、確認するデータを制限することができます。
    //!           デフォルトはすべての関連アイテムがロードされているかを確認します。
    //!
    //! @param[in] pItemName  ロードされているか確認するデータのラベル文字列です。
    //! @param[in] loadFlag   関連するデータを確認するかを示すフラグです。
    //!
    //! @return   ロードされていれば true を、されていなければ false を返します。
    //!
    //! @date 2010/04/09 初版
    //---------------------------------------------------------------------------
    bool IsDataLoaded( const char* pItemName, u32 loadFlag = LOAD_ALL ) const;

    //---------------------------------------------------------------------------
    //! @brief  実行中の @ref LoadData をキャンセルします。
    //!
    //!         @ref LoadData では、loadBlockSize に非ゼロの値を指定すると、
    //!         そのサイズごとに分割ロードされますが、
    //!         この分割のタイミングでキャンセルされ、LoadData 関数から返ってきます。
    //!
    //!         ロードがキャンセルされても、本来ロードに必要であったメモリは占有されたままです。
    //!         そのため、ヒープを有効に使うには、適切に解放処理
    //!         (@ref SoundHeap::LoadState など) を呼び出す必要があります。
    //!
    //! @return   現在ロード中のデータがあれば true を、ロード中でなければ false を返します。
    //!
    //! @date 2011/02/17 初版
    //---------------------------------------------------------------------------
    bool CancelLoading();
    //@}

    //! @briefprivate
    //! @param itemId :private
    //! @return :private
    const void* detail_GetFileAddressByItemId( SoundArchive::ItemId itemId ) const;

    //! @briefprivate
    //! @param bankFile :private
    //! @param pAllocator :private
    //! @return :private
    bool detail_LoadWaveArchiveByBankFile(
            const void* bankFile,
            SoundMemoryAllocatable* pAllocator );

    //! @briefprivate
    //! @param wsdFile :private
    //! @param waveSoundIndex :private
    //! @param pAllocator :private
    //! @return :private
    bool detail_LoadWaveArchiveByWaveSoundFile(
            const void* wsdFile,
            s32 waveSoundIndex,
            SoundMemoryAllocatable* pAllocator );

protected:
    //! @briefprivate
    //! @param arc :private
    void SetSoundArchive( const SoundArchive* arc );

    //! @briefprivate
    //! @return :private
    const SoundArchive* GetSoundArchive() const { return m_pSoundArchive; }

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    const void* GetFileAddressFromSoundArchive( SoundArchive::FileId fileId ) const;

    //! @briefprivate
    //! @param fileId :private
    //! @param address :private
    //! @return :private
    virtual const void* SetFileAddressToTable(
            SoundArchive::FileId fileId, const void* address ) = 0;
    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* GetFileAddressFromTable( SoundArchive::FileId fileId ) const = 0;

    //! @briefprivate
    //! @param fileId :private
    //! @return :private
    virtual const void* GetFileAddressImpl( SoundArchive::FileId fileId ) const = 0;
        // テーブルを含め、アクセス可能な箇所にデータが読み込まれているかを確認する

private:
    bool LoadSequenceSound(
            SoundArchive::ItemId soundId,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize );
    bool LoadWaveSound(
            SoundArchive::ItemId soundId,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize,
            SoundArchive::ItemId waveSoundSetId = SoundArchive::INVALID_ID );
    bool LoadStreamSoundPrefetch(
            SoundArchive::ItemId soundId,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize );
    bool LoadBank(
            SoundArchive::ItemId bankId,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize );
    bool LoadWaveArchive(
            SoundArchive::ItemId warcId,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize );
    const void* LoadWaveArchiveTable(
            SoundArchive::ItemId warcId,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize );
    bool LoadIndividualWave(
            SoundArchive::ItemId warcId,
            u32 waveIndex,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize );
    bool LoadGroup(
            SoundArchive::ItemId groupId,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize );
    bool LoadSoundGroup(
            SoundArchive::ItemId soundGroupId,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize );
    const void* LoadImpl(
            SoundArchive::FileId fileId,
            SoundMemoryAllocatable* pAllocator,
            size_t loadBlockSize,
            bool needDeviceMemory = false );
    bool LoadWaveArchiveImpl(
            SoundArchive::ItemId warcId,
            u32 waveIndex,
            SoundMemoryAllocatable* pAllocator,
            u32 loadFlag,
            size_t loadBlockSize = 0 );

    bool PostProcessForLoadedGroupFile(
            const void* pGroupFile,
            SoundMemoryAllocatable* pAllocator = NULL,
            size_t loadBlockSize = 0 );

    bool IsSequenceSoundDataLoaded( SoundArchive::ItemId itemId, u32 loadFlag ) const;
    bool IsWaveSoundDataLoaded( SoundArchive::ItemId itemId, u32 loadFlag ) const;
    bool IsBankDataLoaded( SoundArchive::ItemId itemId, u32 loadFlag ) const;
    bool IsWaveArchiveDataLoaded( SoundArchive::ItemId itemId, u32 waveIndex ) const;
    bool IsGroupDataLoaded( SoundArchive::ItemId itemId ) const;
    bool IsSoundGroupDataLoaded( SoundArchive::ItemId itemId, u32 loadFlag ) const;

    void* LoadFile(
        SoundArchive::FileId fileId,
        SoundMemoryAllocatable* allocator,
        size_t loadBlockSize,
        bool needDeviceMemory
    );
    s32 ReadFile(
            SoundArchive::FileId fileId,
            void* buffer,
            size_t size,
            s32 offset,
            size_t loadBlockSize );

    void SetWaveArchiveTableWithSeqInEmbeddedGroup(
            SoundArchive::ItemId seqId, SoundMemoryAllocatable* pAllocator );
    void SetWaveArchiveTableWithBankInEmbeddedGroup(
            SoundArchive::ItemId bankId, SoundMemoryAllocatable* pAllocator );
    void SetWaveArchiveTableWithWsdInEmbeddedGroup(
            SoundArchive::ItemId wsdId, SoundMemoryAllocatable* pAllocator );
    void SetWaveArchiveTableInEmbeddedGroupImpl(
            SoundArchive::ItemId warcId, SoundMemoryAllocatable* pAllocator );

    const SoundArchive* m_pSoundArchive;
    u32 m_StreamArea[ 128 ];

    u16 m_LoadDataCallCount;
    bool m_IsCancelLoading;
};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_ARCHIVE_LOADER_H_ */

