﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SOUND_ARCHIVE_FILE_H_
#define NW_SND_SOUND_ARCHIVE_FILE_H_

#if defined(NW_PLATFORM_CAFE)
#include <nw/ut/ut_BinaryFileFormat.h>      // ut::BinaryFileHeader など
#else
#include <nw/snd/snd_BinaryFileFormat.h>
#endif
#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_Global.h>              // PanMode など
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_ElementType.h>
#include <nw/snd/snd_StreamSoundFile.h>

namespace nw {
namespace snd {
namespace internal {

struct SoundArchiveFile
{
    // ------------------------------------------------------------------------
    //  ファイルヘッダー

    // 前方宣言
    struct StringBlock;
    struct InfoBlock;
    struct FileBlock;

    // 定数
    static const int BLOCK_SIZE = 3;

    // メモ : SoundArchiveFile::FileHeader はヘッダだけ個別にロードするので、
    //        Util::SoundFileHeader を継承できない。
#if defined(NW_PLATFORM_CAFE)
    struct FileHeader : public ut::BinaryFileHeader
#else
    struct FileHeader : public BinaryFileHeader
#endif
    {
        Util::ReferenceWithSize toBlocks[ BLOCK_SIZE ];

#if 0
        // ファイルを全ロードした場合
        const StringBlock*  GetStringBlock() const;
        const InfoBlock*    GetInfoBlock() const;
        const FileBlock*    GetFileBlock() const;
#endif

        // 各ブロックのサイズ
        u32 GetStringBlockSize() const;
        u32 GetInfoBlockSize() const;
        u32 GetFileBlockSize() const;

        // 各ブロックのオフセット
        s32 GetStringBlockOffset() const;
        s32 GetInfoBlockOffset() const;
        s32 GetFileBlockOffset() const;

    private:
        const Util::ReferenceWithSize* GetReferenceBy( u16 typeId ) const;
    };


    // ------------------------------------------------------------------------
    //  文字列ブロック

    struct StringTable;
    struct PatriciaTree;
    struct StringBlockBody
    {
    private:
        enum Sections
        {
            Sections_StringTable,
            Sections_PatriciaTree,
            Sections_Max = Sections_PatriciaTree
        };

    public:
        // データ
        Util::Reference toSection[ Sections_Max ];

        // アクセサ
        const char* GetString( SoundArchive::ItemId stringId ) const;
        u32 GetStringCount() const { return GetStringTable()->GetCount(); }

        NW_INLINE u32 GetItemId( const char* str ) const
        {
            return GetItemIdImpl( Sections_PatriciaTree, str );
        }

        // デバッグ
        void DumpTree() const;

    private:
        const void* GetSection( Sections section ) const;
        NW_INLINE const StringTable* GetStringTable() const
        {
            return reinterpret_cast<const StringTable*>(
                    GetSection( Sections_StringTable ) );
        }
        NW_INLINE const PatriciaTree* GetPatriciaTree( Sections section ) const
        {
            return reinterpret_cast<const PatriciaTree*>(
                    GetSection( section ) );
        }
        u32 GetItemIdImpl( Sections section, const char* str ) const;
    };

    struct PatriciaTree
    {
        struct NodeData
        {
#if defined(NW_PLATFORM_CAFE)
            ut::ResU32 stringId;        // 文字列 ID
            ut::ResU32 itemId;          // アイテム ID (サウンドID、バンクIDなど)
#else
            u32 stringId;        // 文字列 ID
            u32 itemId;          // アイテム ID (サウンドID、バンクIDなど)
#endif
        };
        struct Node
        {
            static const u16 FLAG_LEAF = ( 1 << 0 );
#if defined(NW_PLATFORM_CAFE)
            ut::ResU16  flags;
            ut::ResU16  bit;
            ut::ResU32  leftIdx;
            ut::ResU32  rightIdx;
#else
            u16  flags;
            u16  bit;
            u32  leftIdx;
            u32  rightIdx;
#endif
            NodeData    nodeData;
        };

#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 rootIdx;
#else
        u32 rootIdx;
#endif
        Util::Table<Node> nodeTable;

        const NodeData* GetNodeDataBy( const char* str, std::size_t len = 0 ) const;

        // ut_ResDictionary.h の ResDicPatricia のインタフェイスを踏襲
        // のちのち、役立つことがあるかもしれない。
        void* operator[](int idx) const
        {
            NW_MINMAX_ASSERT( idx, 0L, static_cast<int>( nodeTable.count - 1 ));
            // 辞書引き関連についてはconst correctnessを維持しなくても問題ないだろう
            return const_cast<void*>(
                    reinterpret_cast<const void*>( &nodeTable.item[ idx ] ) );
        }
        void* operator[](u32 idx) const
        {
            return operator[]( static_cast<int>(idx) );
        }
        void* operator[](const char* s) const
        {
            return operator()( s, 0 );
        }
        void* operator()(const char* s, size_t len) const
        {
            return const_cast<void*>(
                    reinterpret_cast<const void*>( GetNodeDataBy( s, len ) ) );
        }
        u32 GetNumData() const { return nodeTable.count; }
        u32 GetCount() const { return GetNumData(); }
    };

    struct StringBlock
    {
        ut::BinaryBlockHeader   header;
        StringBlockBody         body;
    };

    struct StringTable
    {
        // データ
        Util::ReferenceWithSizeTable table;

        // アクセサ
        const char* GetString( int stringId ) const
        {
            return reinterpret_cast<const char*>(
                    ut::AddOffsetToPtr( this, table.item[ stringId ].offset ) );
        }
        u32 GetCount() const
        {
            return table.count;
        }
    };


    // ------------------------------------------------------------------------
    //  情報ブロック

    // 前方宣言
    struct SoundInfo;
    struct BankInfo;
    struct PlayerInfo;
    struct SoundGroupInfo;
    struct GroupInfo;
    struct WaveArchiveInfo;
    struct FileInfo;
    struct SoundArchivePlayerInfo;

    struct InfoBlockBody
    {
        //
        // データ
        //
        Util::Reference toSoundInfoReferenceTable;
        Util::Reference toSoundGroupInfoReferenceTable;
        Util::Reference toBankInfoReferenceTable;
        Util::Reference toWaveArchiveInfoReferenceTable;
        Util::Reference toGroupInfoReferenceTable;
        Util::Reference toPlayerInfoReferenceTable;
        Util::Reference toFileInfoReferenceTable;
        Util::Reference toSoundArchivePlayerInfo;

        //
        // アクセサ
        //
        NW_INLINE u32 GetSoundCount() const       { return GetSoundInfoReferenceTable().count; }
        NW_INLINE u32 GetBankCount() const        { return GetBankInfoReferenceTable().count; }
        NW_INLINE u32 GetPlayerCount() const      { return GetPlayerInfoReferenceTable().count; }
        NW_INLINE u32 GetSoundGroupCount() const  { return GetSoundGroupInfoReferenceTable().count; }
        NW_INLINE u32 GetGroupCount() const       { return GetGroupInfoReferenceTable().count; }
        NW_INLINE u32 GetWaveArchiveCount() const { return GetWaveArchiveInfoReferenceTable().count; }
        NW_INLINE u32 GetFileCount() const        { return GetFileInfoReferenceTable().count; }

        const SoundInfo*        GetSoundInfo( SoundArchive::ItemId soundId ) const;
        const BankInfo*         GetBankInfo( SoundArchive::ItemId bankId ) const;
        const PlayerInfo*       GetPlayerInfo( SoundArchive::ItemId playerId ) const;
        const SoundGroupInfo*   GetSoundGroupInfo( SoundArchive::ItemId soundGroupId ) const;
        const GroupInfo*        GetGroupInfo( SoundArchive::ItemId groupId ) const;
        const WaveArchiveInfo*  GetWaveArchiveInfo( SoundArchive::ItemId warcId ) const;
        const FileInfo*         GetFileInfo( SoundArchive::FileId fileId ) const;

        const SoundArchivePlayerInfo* GetSoundArchivePlayerInfo() const;

        SoundArchive::FileId    GetItemFileId( SoundArchive::ItemId id ) const;
        SoundArchive::StringId  GetItemStringId( SoundArchive::ItemId id ) const;
        SoundArchive::FileId    GetItemPrefetchFileId( SoundArchive::ItemId id ) const;

    private:
        const Util::ReferenceTable& GetSoundInfoReferenceTable() const;
        const Util::ReferenceTable& GetBankInfoReferenceTable() const;
        const Util::ReferenceTable& GetPlayerInfoReferenceTable() const;
        const Util::ReferenceTable& GetSoundGroupInfoReferenceTable() const;
        const Util::ReferenceTable& GetGroupInfoReferenceTable() const;
        const Util::ReferenceTable& GetWaveArchiveInfoReferenceTable() const;
        const Util::ReferenceTable& GetFileInfoReferenceTable() const;
    };

    struct InfoBlock
    {
        ut::BinaryBlockHeader   header;
        InfoBlockBody           body;
    };


    struct StreamSoundInfo;
    struct WaveSoundInfo;
    struct SequenceSoundInfo;
    struct Sound3DInfo;

    struct SoundInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      fileId;
        ut::ResU32      playerId;
        ut::ResU8       volume;
        ut::ResU8       remoteFilter;
#else
        u32      fileId;
        u32      playerId;
        u8       volume;
        u8       remoteFilter;
#endif
        u16             padding1;
        Util::Reference toDetailSoundInfo;  // {Sequence,Stream,Wave}SoundInfo への参照
        Util::BitFlag   optionParameter;

        // アクセサ
        SoundArchive::SoundType     GetSoundType() const;
        const StreamSoundInfo&      GetStreamSoundInfo() const;
        const WaveSoundInfo&        GetWaveSoundInfo() const;
        const SequenceSoundInfo&    GetSequenceSoundInfo() const;
        const Sound3DInfo*          GetSound3DInfo() const;
            // いつか、3D サウンド情報を削ることもありそうなので、ポインタを返すようにする

        u32         GetStringId() const;
        PanMode     GetPanMode() const;
        PanCurve    GetPanCurve() const;
        u8          GetPlayerPriority() const;
        u8          GetActorPlayerId() const;
        u32         GetUserParam() const;       // 通常の u32 ユーザーパラメータ
        bool        ReadUserParam( int index, u32& value ) const;    // index:0-3 まで利用可能
        bool        IsFrontBypass() const;

        // TODO: SEND/MOD パラメータの取得関数
    };

    struct StreamTrackInfo;
    struct StreamTrackInfoTable
    {
        // データ
        Util::ReferenceTable table;

        // アクセサ
        const StreamTrackInfo* GetTrackInfo( u32 index ) const;
        u32 GetTrackCount() const { return table.count; }
    };

    struct SendValue
    {
#if defined(NW_PLATFORM_CAFE)
        ut::ResU8 mainSend;
        ut::ResU8 fxSend[AUX_BUS_NUM];
#else
        u8 mainSend;
        u8 fxSend[AUX_BUS_NUM];
#endif
    };

    struct StreamTrackInfo
    {
        // データ
        u8 volume;
        u8 pan;
        u8 span;
        u8 flags;   // enum SurroundMode が入る
        Util::Reference toGlobalChannelIndexTable;
        Util::Reference toSendValue;
        u8 lpfFreq;
        u8 biquadType;
        u8 biquadValue;
        u8 padding;

        // アクセサ
        NW_INLINE u32 GetTrackChannelCount() const
        {
            return GetGlobalChannelIndexTable().GetCount();
        }
        NW_INLINE u8 GetGlobalChannelIndex( u32 index ) const
        {
            return GetGlobalChannelIndexTable().GetGlobalIndex( index );
        }
        const SendValue& GetSendValue() const
        {
            NW_ASSERT( toSendValue.typeId ==
                    nw::snd::internal::ElementType_SoundArchiveFile_SendInfo );
            return *reinterpret_cast<const SendValue*>(
                    ut::AddOffsetToPtr( this, toSendValue.offset ) );
        }

    private:
        const StreamSoundFile::GlobalChannelIndexTable& GetGlobalChannelIndexTable() const
        {
            return *reinterpret_cast<const StreamSoundFile::GlobalChannelIndexTable*>(
                    ut::AddOffsetToPtr(
                        this,
                        toGlobalChannelIndexTable.offset ) );
        }
    };

    struct StreamSoundExtension
    {
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 streamTypeInfo;
        ut::ResU32 loopStartFrame;
        ut::ResU32 loopEndFrame;
#else
        u32 streamTypeInfo;
        u32 loopStartFrame;
        u32 loopEndFrame;
#endif

        SoundArchive::StreamFileType GetStreamFileType() const
        {
            return static_cast<SoundArchive::StreamFileType>( Util::DevideBy8bit( streamTypeInfo, 0 ) );
        }

        bool IsLoop() const
        {
            return Util::DevideBy8bit( streamTypeInfo, 1 ) == 1;
        }
    };

    struct StreamSoundInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU16 allocateTrackFlags;  // 有効トラックを表すビットフラグ (マルチトラックの設定)
        ut::ResU16 allocateChannelCount;// マルチトラックを加味した総チャンネル数
#else
        u16 allocateTrackFlags;  // 有効トラックを表すビットフラグ (マルチトラックの設定)
        u16 allocateChannelCount;// マルチトラックを加味した総チャンネル数
#endif
        Util::Reference toTrackInfoTable;

#if defined(NW_PLATFORM_CAFE)
        ut::ResF32 pitch;
#else
        f32 pitch;
#endif
        Util::Reference toSendValue;

        Util::Reference toStreamSoundExtension;

#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 prefetchFileId;
#else
        u32 prefetchFileId;
#endif

        // トラック情報テーブル取得
        const StreamTrackInfoTable* GetTrackInfoTable() const;
        // アクセサ
        f32 GetPitch() const;
        const SendValue& GetSendValue() const;
        const StreamSoundExtension*  GetStreamSoundExtension() const;
    };

    struct WaveSoundInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 index;           // ウェーブサウンドセットファイル (bxwsd) 内で何番目か
        ut::ResU32 allocateTrackCount; // 必要なトラック数
#else
        u32 index;              // ウェーブサウンドセットファイル (bxwsd) 内で何番目か
        u32 allocateTrackCount; // 必要なトラック数
#endif
        Util::BitFlag optionParameter;

        // アクセサ
        u8 GetChannelPriority() const;
        u8 GetIsReleasePriorityFix() const;

    };

    struct SequenceSoundInfo
    {
        // データ
        Util::Reference toBankIdTable;
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      allocateTrackFlags;
#else
        u32      allocateTrackFlags;
#endif
        Util::BitFlag   optionParameter;

        // アクセサ
        void GetBankIds( u32* bankIds ) const;  // SoundArchive::SEQ_BANK_MAX だけコピー
        u32 GetStartOffset() const;
        u8  GetChannelPriority() const;
        bool IsReleasePriorityFix() const;
    private:
#if defined(NW_PLATFORM_CAFE)
        const Util::Table<ut::ResU32>& GetBankIdTable() const;
#else
        const Util::Table<u32>& GetBankIdTable() const;
#endif
    };

    struct Sound3DInfo
    {
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32 flags;       // [0]Vol [1]Priotity [2]Pan [3]SPan [4]Filter
                                // (SoundArchive::Sound3DInfo::FLAG_CTRL_*** に対応しています)
        ut::ResF32 decayRatio;  // 減衰率
#else
        u32 flags;              // [0]Vol [1]Priotity [2]Pan [3]SPan [4]Filter
                                // (SoundArchive::Sound3DInfo::FLAG_CTRL_*** に対応しています)
        f32 decayRatio;         // 減衰率
#endif
        u8 decayCurve;          // 減衰カーブタイプ (nw::snd::DecayCurve が入る)
        u8 dopplerFactor;       // ドップラーファクター
        u8 padding[2];          // padding
        Util::BitFlag optionParameter;      // 予備
    };

    struct BankInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      fileId;
#else
        u32      fileId;
#endif
        Util::Reference toWaveArchiveItemIdTable;
        Util::BitFlag   optionParameter;

        // アクセサ
        u32 GetStringId() const;

#if defined(NW_PLATFORM_CAFE)
        NW_INLINE const Util::Table<ut::ResU32>*
#else
        NW_INLINE const Util::Table<u32>*
#endif
        GetWaveArchiveItemIdTable() const
        {
#if defined(NW_PLATFORM_CAFE)
            return reinterpret_cast<const Util::Table<ut::ResU32>*>(
#else
            return reinterpret_cast<const Util::Table<u32>*>(
#endif
                    ut::AddOffsetToPtr( this, toWaveArchiveItemIdTable.offset ) );
        }
    };

    struct PlayerInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      playableSoundMax;   // プレイヤー最大再生数
#else
        u32      playableSoundMax;          // プレイヤー最大再生数
#endif
        Util::BitFlag   optionParameter;

        // アクセサ
        u32 GetStringId() const;
        u32 GetPlayerHeapSize() const;
    };

    struct WaveSoundGroupInfo;
    struct SoundGroupInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      startId;    // アイテム ID (startId と endId の ItemType は同一)
        ut::ResU32      endId;
#else
        u32      startId;           // アイテム ID (startId と endId の ItemType は同一)
        u32      endId;
#endif
        Util::Reference toFileIdTable;
        Util::Reference toDetailSoundGroupInfo;
        Util::BitFlag   optionParameter;

        // アクセサ
        u32 GetStringId() const;

#if defined(NW_PLATFORM_CAFE)
        NW_INLINE const Util::Table<ut::ResU32>* GetFileIdTable() const
#else
        NW_INLINE const Util::Table<u32>* GetFileIdTable() const
#endif
        {
#if defined(NW_PLATFORM_CAFE)
            return reinterpret_cast<const Util::Table<ut::ResU32>*>(
#else
            return reinterpret_cast<const Util::Table<u32>*>(
#endif
                    ut::AddOffsetToPtr( this, toFileIdTable.offset ) );
        }
        NW_INLINE const WaveSoundGroupInfo* GetWaveSoundGroupInfo() const
        {
            if ( toDetailSoundGroupInfo.typeId !=
                    ElementType_SoundArchiveFile_WaveSoundGroupInfo )
            {
                return NULL;
            }

            return reinterpret_cast<const WaveSoundGroupInfo*>(
                    ut::AddOffsetToPtr( this, toDetailSoundGroupInfo.offset ) );
        }

#if 0   // 必要になるかもしれない
        u32 GetFileIdCount() const { return GetFileIdTable().count; }
        u32 GetFileId( u32 index ) const
        {
            if ( index >= GetFileIdCount() ) return SoundArchive::INVALID_ID;
            return GetFileIdTable().item[ index ];
        }
#endif
    };

    struct WaveSoundGroupInfo
    {
        // データ
        Util::Reference toWaveArchiveItemIdTable;
        Util::BitFlag   optionParameter;    // 仮。なにが割り当てられるかは未定。

        // アクセサ
#if defined(NW_PLATFORM_CAFE)
        NW_INLINE const Util::Table<ut::ResU32>*
#else
        NW_INLINE const Util::Table<u32>*
#endif
        GetWaveArchiveItemIdTable() const
        {
#if defined(NW_PLATFORM_CAFE)
            return reinterpret_cast<const Util::Table<ut::ResU32>*>(
#else
            return reinterpret_cast<const Util::Table<u32>*>(
#endif
                    ut::AddOffsetToPtr( this, toWaveArchiveItemIdTable.offset ) );
        }
    };

    struct GroupInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      fileId;
#else
        u32      fileId;
#endif
        Util::BitFlag   optionParameter;

        // アクセサ
        u32 GetStringId() const;
    };

    struct WaveArchiveInfo
    {
        // データ
#if defined(NW_PLATFORM_CAFE)
        ut::ResU32      fileId;
#else
        u32      fileId;
#endif
        bool            isLoadIndividual;
        u8              padding1;
        u16             padding2;
        Util::BitFlag   optionParameter;

        // アクセサ
        u32 GetStringId() const;
        u32 GetWaveCount() const;
    };

    enum FileLocationType
    {
        FILE_LOCATION_TYPE_INTERNAL,    // サウンドアーカイブ内に配置
        FILE_LOCATION_TYPE_EXTERNAL,    // サウンドアーカイブ外に配置
        FILE_LOCATION_TYPE_NONE
            // ファイル単体にアクセスできない (＝ロードできない)
            //  * バンクファイルかウェーブサウンドセットファイルのうち、
            //    次の条件を満たすものが該当
            //      * 波形アーカイブ設定が【自動(共有)】で、
            //      * 出力設定が【埋め込み】のグループに属している
    };

    struct InternalFileInfo;    // サウンドアーカイブ FILE ブロック内に格納
    struct ExternalFileInfo;    // サウンドアーカイブ外に格納

    struct FileInfo
    {
        // データ
        Util::Reference toFileLocation;
        Util::BitFlag   optionParameter;    // 仮。なにが割り当てられるかは未定。

        // アクセサ
        FileLocationType GetFileLocationType() const;
        const InternalFileInfo* GetInternalFileInfo() const;
        const ExternalFileInfo* GetExternalFileInfo() const;

#if 0
        // サウンドアーカイブファイル上のアドレスを取得
        // ROM 上のバイナリサウンドアーカイブファイルの先頭をアドレス 0 とする。
        const void* GetFileAddress( const void* dataBlockBodyAddress ) const;
            // TODO: ↑ これは必要ない？
#endif
    };

    struct InternalFileInfo
    {
        static const u32 INVALID_OFFSET = 0xffffffff;
        static const u32 INVALID_SIZE = 0xffffffff;

        // データ
        Util::ReferenceWithSize toFileImageFromFileBlockBody;
        Util::Reference         toAttachedGroupIdTable; // 所属するグループIDのテーブル

        // アクセサ
        NW_INLINE u32 GetFileSize() const
        {
            return toFileImageFromFileBlockBody.size;
        }
        NW_INLINE u32 GetOffsetFromFileBlockHead() const
        {
            if(toFileImageFromFileBlockBody.offset == INVALID_OFFSET)
            {
                return INVALID_OFFSET;
            }
            return toFileImageFromFileBlockBody.offset + sizeof(ut::BinaryBlockHeader);
        }
#if defined(NW_PLATFORM_CAFE)
        NW_INLINE const Util::Table<ut::ResU32>* GetAttachedGroupTable() const
#else
        NW_INLINE const Util::Table<u32>* GetAttachedGroupTable() const
#endif
        {
            // どのグループに属しているかを示すテーブルを返す
            // (どこにも属していない場合は、count = 0 のテーブルが返される)
#if defined(NW_PLATFORM_CAFE)
            return reinterpret_cast<const Util::Table<ut::ResU32>*>(
#else
            return reinterpret_cast<const Util::Table<u32>*>(
#endif
                    ut::AddOffsetToPtr( this, toAttachedGroupIdTable.offset ) );
        }
    };

    struct ExternalFileInfo
    {
        // データ
        // TODO: u32 fileSize や u16? filePathLength が入るかも
        char filePath[1];
    };

    struct SoundArchivePlayerInfo
    {
#if defined(NW_PLATFORM_CAFE)
        nw::ut::ResU16  sequenceSoundMax;
        nw::ut::ResU16  sequenceTrackMax;
        nw::ut::ResU16  streamSoundMax;
        nw::ut::ResU16  streamTrackMax;
        nw::ut::ResU16  streamChannelMax;
        nw::ut::ResU16  waveSoundMax;
        nw::ut::ResU16  waveTrackmax;
#else
        u16  sequenceSoundMax;
        u16  sequenceTrackMax;
        u16  streamSoundMax;
        u16  streamTrackMax;
        u16  streamChannelMax;
        u16  waveSoundMax;
        u16  waveTrackmax;
#endif
        u8              streamBufferTimes;
        u8              padding;
#if defined(NW_PLATFORM_CAFE)
        nw::ut::ResU32  options;        // TODO: 仮
#else
        u32  options;
#endif
    };


    // ------------------------------------------------------------------------
    //  ファイルブロック

    struct FileBlockBody
    {
        // ボディのアドレスを取るためだけ
    };

    struct FileBlock
    {
        ut::BinaryBlockHeader   header;
        FileBlockBody           body;
    };

};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif /* NW_SND_SOUND_ARCHIVE_FILE_H_ */

