﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_Sound3DActor.h
 *
 * @file snd_Sound3DActor.h
 */

#ifndef NW_SND_SOUND_3D_ACTOR_H_
#define NW_SND_SOUND_3D_ACTOR_H_

#include <nw/snd/snd_SoundActor.h>
#include <nw/snd/snd_BasicSound.h>
#include <nw/math/math_Types.h>         // nw::math::VEC3

namespace nw {
namespace snd {

class Sound3DManager;
class SoundArchivePlayer;
class SoundHandle;

/* ========================================================================
        class difinition
   ======================================================================== */

//---------------------------------------------------------------------------
//! @brief    3D 空間内での音源を表す 3D サウンドアクタークラスです。
//!
//!           1 つのアクターは複数のサウンドを同時に管理することができます。
//!
//!           アクターのインスタンスが存在する間、
//!           アクターに設定された位置情報を元に @ref nw::snd::Sound3DManager
//!           クラスがサウンドのパラメータを計算して設定します。
//!           アクターのインスタンスが無くなっても、
//!           そのアクターのサウンド再生は続きますが、
//!           3D のパラメータの更新は行われなくなります。
//!
//! @see Sound3DManager クラス
//!
//! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
class Sound3DActor
: public SoundActor,
  public internal::BasicSound::AmbientArgUpdateCallback
{
public:

    //! @name コンストラクタ/デストラクタ
    //@{
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           3D サウンドアクターは、
    //!           引数で渡されたサウンドアーカイブプレイヤーを使用してサウンドを再生します。
    //!           また、3D サウンドのパラメータ計算には、
    //!           引数で渡された 3D サウンドマネージャーを使用します。
    //!
    //!           引数付きのコンストラクタの場合は、
    //!           @ref Initialize を呼ぶ必要はありません
    //!           (呼んでも問題ありません)。
    //!
    //! @param[in] player     アクターが使用するサウンドアーカイブプレイヤーです。
    //! @param[in] manager    アクターが使用する 3D サウンドマネージャーです。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see Sound3DManager クラス
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    Sound3DActor( SoundArchivePlayer& player, Sound3DManager& manager );

    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //!           引数なしのコンストラクタは、@ref Initialize を呼ぶ必要があります。
    //!
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    Sound3DActor();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    virtual ~Sound3DActor();
    //@}

    //! @name 初期化
    //@{
    //---------------------------------------------------------------------------
    //! @brief    3D サウンドアクターを初期化します。
    //!
    //!           3D サウンドアクターを使用する前に、初期化を行う必要があります。
    //!           ただし、引数付きコンストラクタを呼んでいる場合は、
    //!           呼ぶ必要はありません (呼んでも問題ありません)。
    //!
    //!           @ref SoundActor::Initialize は本関数内で呼ばれます。
    //!
    //!           同じ Sound3DActor インスタンスを再利用するには、
    //!           あらかじめ @ref Finalize 関数を呼んでから、
    //!           再度、本関数を呼び出してください。
    //!
    //!           本関数の引数で渡した Sound3DManager について、
    //!           Sound3DActor で再生したサウンドが鳴り終わる前に破棄を行うと
    //!           SoundArchivePlayer::Update() で例外で停止する可能性があります。
    //!
    //!           これは Sound3DActor で再生したサウンドが
    //!           Sound3DManager をコールバックとして参照しているためです。
    //!           Sound3DManager を破棄する場合は
    //!           必ず参照しているサウンドが全て再生完了してから行ってください。
    //!
    //! @param[in] player     アクターが使用するサウンドアーカイブプレイヤーです。
    //! @param[in] manager    アクターが使用する 3D サウンドマネージャーです。
    //!
    //! @see SoundArchivePlayer クラス
    //! @see Sound3DManager クラス
    //! @see Sound3DActor()
    //! @see Finalize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //! @date 2013/06/28 Sound3DManager の破棄に関する注意点を追記
    //---------------------------------------------------------------------------
    void Initialize( SoundArchivePlayer& player, Sound3DManager& manager );

    //---------------------------------------------------------------------------
    //! @brief    3D サウンドアクターを破棄します。
    //!
    //!           デストラクタの中で呼ばれますが、
    //!           明示的に呼んでも問題ありません。
    //!
    //!           同じ Sound3DActor インスタンスを再利用するには、
    //!           本関数を呼んだ後、再度 @ref Initialize 関数を呼び出してください。
    //!
    //! @see ~Sound3DActor
    //! @see Initialize
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void Finalize();
    //@}

    //! @name 位置・速度情報の設定と取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief    アクターの位置情報を設定します。
    //!
    //!           3D サウンドアクターの位置情報を設定します。
    //!           設定された座標は、3D サウンドのパラメータ計算で使用されます。
    //!
    //!           新しく位置情報を設定した際、前回設定された位置情報との差分を計算し、
    //!           3D サウンドアクターの速度が自動的に設定されます。
    //!           アクターの速度は、ドップラー効果による音程変化に反映されます。
    //!           速度は @ref nw::snd::Sound3DActor::SetVelocity
    //!           で明示的に設定することも可能です。
    //!
    //!           座標が別の位置に飛んだ場合、飛んだ距離の差分で速度が計算されてしまうため、
    //!           極めて高速で移動したように処理され、急激な音程変化が発生します。
    //!           このような場合、この関数を呼び出した後で、
    //!           @ref nw::snd::Sound3DActor::SetVelocity を呼び出し、
    //!           速度を明示的に設定する必要があることに注意してください。
    //!
    //! @param[in] position   アクターの位置座標です。
    //!
    //! @see SetVelocity
    //! @see GetPosition
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetPosition( const nw::math::VEC3& position );

    //---------------------------------------------------------------------------
    //! @brief    アクターの位置情報を取得します。
    //!
    //! @return   現在設定されているアクターの位置情報を返します。
    //!
    //! @see SetPosition
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetPosition() const { return m_Position; }

    //---------------------------------------------------------------------------
    //! @brief    アクターの位置情報をリセットします。
    //!
    //!           3D サウンドアクターの位置情報と速度をゼロクリアします。
    //!
    //!           この関数でリセットした直後の @ref nw::snd::Sound3DActor::SetPosition
    //!           の呼び出しでは、3D サウンドアクターの速度は設定されません。
    //!
    //! @see SetPosition
    //! @see SetVelocity
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void ResetPosition();

    //---------------------------------------------------------------------------
    //! @brief    アクターの速度を設定します。
    //!
    //!           設定された速度は、ドップラー効果による音程変化に反映されます。
    //!
    //!           3D サウンドアクターの速度は @ref nw::snd::Sound3DActor::SetPosition
    //!           で座標を設定した際、前回の登録座標との差分から自動的に設定されますが、
    //!           この関数を使用すると 3D サウンドアクターの速度を手動で設定することができます。
    //!
    //! @param[in] velocity   アクターの速度です。
    //!
    //! @see GetVelocity
    //! @see SetPosition
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetVelocity( const nw::math::VEC3& velocity );

    //---------------------------------------------------------------------------
    //! @brief    アクターの速度を取得します。
    //!
    //! @return   現在設定されているアクターの速度を返します。
    //!
    //! @see SetVelocity
    //! @see SetPosition
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    const nw::math::VEC3& GetVelocity() const { return m_Velocity; }
    //@}

    //! @name パラメータ設定・取得
    //@{
    //---------------------------------------------------------------------------
    //! @brief    アクターにユーザーパラメータを設定します。
    //!
    //!           設定されたユーザーパラメータは、
    //!           @ref nw::snd::Sound3DParam 構造体の actorUserParam に反映されます。
    //!
    //! @param[in] param  ユーザーパラメータです。
    //!
    //! @see GetUserParam
    //! @see Sound3DParam 構造体
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    void SetUserParam( u32 param ) { m_UserParam = param; }

    //---------------------------------------------------------------------------
    //! @brief    アクターのユーザーパラメータを取得します。
    //!
    //! @return   現在設定されているアクターのユーザーパラメータを返します。
    //!
    //! @see SetUserParam
    //!
    //! @date 2011/07/05 NW4F 1.0.0 PR 公開に向けた調整
    //---------------------------------------------------------------------------
    u32 GetUserParam() const { return m_UserParam; }
    //@}

protected:
    // リファレンスは SoundActor::Setup に書かれている
    virtual SoundStartable::StartResult SetupSound(
        SoundHandle* handle,
        u32 soundId,
        const StartInfo* startInfo,
        void* setupArg
    );

    //! @briefprivate
    //! @param arg :private
    //! @param sound :private
    virtual void detail_UpdateAmbientArg( void* arg, const internal::BasicSound* sound );

private:
    static void ClearUpdateCallback( SoundHandle& handle );

    Sound3DManager* m_p3dManager;
    SoundArchivePlayer* m_pArchivePlayer;
    u32 m_UserParam;
    nw::math::VEC3 m_Position;
    nw::math::VEC3 m_Velocity;
    bool m_ResetPositionFlag;
    bool m_IsInitialized;
    bool m_IsFinalized;
};

} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SOUND_3D_ACTOR_H_ */

