﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/snd_SequenceSoundPlayer.h
 *
 * @file snd_SequenceSoundPlayer.h
 */

#ifndef NW_SND_SEQUENCE_SOUND_PLAYER_H_
#define NW_SND_SEQUENCE_SOUND_PLAYER_H_

#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_DisposeCallback.h>
#include <nw/snd/snd_SoundThread.h>
#include <nw/snd/snd_NoteOnCallback.h>      // NoteOnCallback, NoteOnInfo
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_BankFileReader.h>
#include <nw/snd/snd_WaveArchiveFileReader.h>
#include <nw/snd/snd_LoaderManager.h>
#include <nw/snd/snd_Task.h>

namespace nw {
namespace snd {

// シーケンスコマンド'userproc'用コールバック
//---------------------------------------------------------------------------
//! @brief    'userproc' コマンドで呼び出されるコールバックで使用される構造体です。
//!
//!           'userproc' コマンドの詳細については、
//!           シーケンスデータマニュアルを参照してください。
//!
//! @see SequenceUserprocCallback
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
struct SequenceUserprocCallbackParam
{
    //---------------------------------------------------------------------------
    //! @brief    ローカル変数の配列です。
    //!           localVariable[0] から localVariable[15] まで指定できます。
    //!           変数の値を参照、あるいは変更することができます。
    //---------------------------------------------------------------------------
    vs16* localVariable;

    //---------------------------------------------------------------------------
    //! @brief    グローバル変数の配列です。
    //!           globalVariable[0] から globalVariable[15] まで指定できます。
    //!           変数の値を参照、あるいは変更することができます。
    //---------------------------------------------------------------------------
    vs16* globalVariable;

    //---------------------------------------------------------------------------
    //! @brief    トラック変数の配列です。
    //!           trackVariable[0] から trackVariable[15] まで指定できます。
    //!           変数の値を参照、あるいは変更することができます。
    //---------------------------------------------------------------------------
    vs16* trackVariable;

    //---------------------------------------------------------------------------
    //! @brief    比較コマンドで設定される真偽のフラグです。
    //!           フラグの状態を参照、あるいは変更することができます。
    //---------------------------------------------------------------------------
    bool cmpFlag;
};

//---------------------------------------------------------------------------
//! @brief    シーケンスサウンドで使用可能なユーザープロシージャのコールバックです。
//!
//!           このコールバックはシーケンスデータ内の 'userproc'
//!           コマンドによって呼び出されます。
//!           'userproc' コマンドが処理されたフレームで、
//!           サウンドスレッドから呼び出されます。
//!
//!           procId は、シーケンスデータで 'userproc'
//!           コマンドのパラメータとして指定された値が渡されます。
//!
//!           param は、コールバック内で利用できるパラメータです。
//!           データで設定されてる値を参照することができます。
//!           また、この値を書き換えることでサウンドデータに反映することが出来ます。
//!
//!           'userproc' コマンドの詳細については、
//!           シーケンスデータマニュアルを参照してください。
//!
//! @param[in] procId     シーケンスデータで指定されたプロシージャ ID です。
//! @param[in] param      コールバック内で利用できるパラメータです。
//! @param[in] arg        ユーザー引数です。
//!
//! @see SoundArchivePlayer::SetSequenceUserprocCallback
//! @see SequenceUserprocCallbackParam
//!
//! @date 2011/07/07 NW4F 1.0.0 PR 公開に向けた調整
//---------------------------------------------------------------------------
typedef void (*SequenceUserprocCallback)(
    u16 procId,
    SequenceUserprocCallbackParam* param,
    void* arg
);

class SoundDataManager;
class SoundPlayer;

namespace internal {

class PlayerHeap;

namespace driver {

/* ========================================================================
        typename declaration
   ======================================================================== */

class SequenceTrack;
class SequenceTrackAllocator;

/* ========================================================================
        class definition
   ======================================================================== */
class SequenceSoundLoader;
typedef LoaderManager<SequenceSoundLoader> SequenceSoundLoaderManager;

// プレイヤーヒープへのロードをつかさどる
class SequenceSoundLoader
{
public:
    struct LoadInfo
    {
        const SoundArchive* soundArchive;
        const SoundDataManager* soundDataManager;
        LoadItemInfo* loadInfoSeq;
        LoadItemInfo* loadInfoBanks[SoundArchive::SEQ_BANK_MAX];
        SoundPlayer* soundPlayer;

        LoadInfo(
            const SoundArchive* arc,
            const SoundDataManager* mgr,
            LoadItemInfo* seq,
            LoadItemInfo* banks,
            SoundPlayer* player);
    };

    struct Data
    {
        const void* seqFile;
        const void* bankFiles[SoundArchive::SEQ_BANK_MAX];   // TODO: バンクの数は SoundArchive への依存ではなく、SequenceSound 側で定義すべき
        const void* warcFiles[SoundArchive::SEQ_BANK_MAX];
        bool warcIsIndividuals[SoundArchive::SEQ_BANK_MAX];

        Data() { Initialize(); }
        void Initialize()
        {
            seqFile = NULL;
            for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
            {
                bankFiles[i] = NULL;
                warcFiles[i] = NULL;
                warcIsIndividuals[i] = false;
            }
        }
    };

    struct Arg
    {
        const SoundArchive* soundArchive;
        const SoundDataManager* soundDataManager;
        SoundPlayer* soundPlayer;
        LoadItemInfo loadInfoSeq;
        LoadItemInfo loadInfoBanks[SoundArchive::SEQ_BANK_MAX];
        // LoadItemInfo loadInfoWarcs[SoundArchive::SEQ_BANK_MAX];
        // bool warcIsIndividuals は必要？？？

        Arg() : soundArchive(NULL), soundDataManager(NULL), soundPlayer(NULL)
        {}
    };

    class DataLoadTask : public Task
    {
    public:
        void Initialize();
        /* override */ void Execute();
        bool TryAllocPlayerHeap();
        void FreePlayerHeap();

        Arg m_Arg;
        Data m_Data;
        PlayerHeap* m_pPlayerHeap;
        bool m_IsLoadSuccess;
        u8 padding[3];
    };

    bool IsInUse();
    void Initialize(const Arg& arg);
    void Finalize();
    bool TryWait();
    bool IsLoadSuccess() const { return m_Task.m_IsLoadSuccess; }

    const Data& GetData() const { return m_Task.m_Data; }

private:
    DataLoadTask m_Task;

public:
    ut::LinkListNode m_LinkForLoaderManager; // for SequenceSoundLoaderManager
};


class SequenceSoundPlayer : public BasicSoundPlayer, public DisposeCallback, public SoundThread::PlayerCallback
{
    /* ------------------------------------------------------------------------
            constant variable
       ------------------------------------------------------------------------ */
public:
    static const int PLAYER_VARIABLE_NUM    = 16;
    static const int GLOBAL_VARIABLE_NUM    = 16;
    static const int TRACK_NUM_PER_PLAYER   = 16;
    static const u32 ALL_TRACK_BIT_FLAG     = (1 << TRACK_NUM_PER_PLAYER) - 1; // 全てのトラックを含むビットフラグ
    static const int VARIABLE_DEFAULT_VALUE = -1;

    static const int DEFAULT_TIMEBASE       = 48; /* 四分音符分解能 */
    static const int DEFAULT_TEMPO          = 120;

    // オフセット再生時、1 オーディオフレームあたりのデフォルトスキップ幅 (tick単位)
    static const s32 DEFAULT_SKIP_INTERVAL_TICK = 48*4*4;

    /* ------------------------------------------------------------------------
            type definition
       ------------------------------------------------------------------------ */
public:
    struct ParserPlayerParam
    {
        u8 priority;
        u8 timebase;
        u16 tempo;
        MoveValue<u8,s16> volume;

        NoteOnCallback* callback;

        ParserPlayerParam()
        : priority(64),
          timebase(DEFAULT_TIMEBASE), tempo(DEFAULT_TEMPO),
          callback(NULL)
        {
            volume.InitValue(127);
        }
    };

    enum StartOffsetType
    {
        START_OFFSET_TYPE_TICK,
        START_OFFSET_TYPE_MILLISEC
    };

    struct StartInfo
    {
        s32 seqOffset;
        StartOffsetType startOffsetType;
        int startOffset;
        s32 delayTime;
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    static void InitSequenceSoundPlayer();

    static void SetSkipIntervalTick( s32 intervalTick );
    static s32 GetSkipIntervalTick();

    SequenceSoundPlayer();
    virtual ~SequenceSoundPlayer();

    /* override */ void Initialize();
    /* override */ void Finalize();

    void SetLoaderManager(SequenceSoundLoaderManager* manager)
    {
        m_pLoaderManager = manager;
    }

    struct SetupArg
    {
        SequenceTrackAllocator* trackAllocator;
        u32 allocTracks;
        NoteOnCallback* callback;

        SetupArg() : trackAllocator(NULL), allocTracks(0), callback(NULL)
        {}
    };
    void Setup(const SetupArg& arg);

    bool IsPrepared() const
    {
        if (m_ResState >= RES_STATE_ASSIGNED)
        {
            return true;
        }
        return false;
    }

    struct PrepareArg
    {
        const void* seqFile; // シーケンスファイル
        const void* bankFiles[SEQ_BANK_MAX];  // バンクファイル
        const void* warcFiles[SEQ_BANK_MAX];  // 波形アーカイブファイル
        bool warcIsIndividuals[SEQ_BANK_MAX];
        s32 seqOffset;
        s32 delayTime;

        PrepareArg() : seqFile(NULL), seqOffset(NULL), delayTime(0)
        {
            for (int i = 0; i < SEQ_BANK_MAX; i++)
            {
                bankFiles[i] = NULL;
                warcFiles[i] = NULL;
                warcIsIndividuals[i] = false;
            }
        }
    };
    void Prepare(const PrepareArg& arg);
    void RequestLoad(const StartInfo& info, const SequenceSoundLoader::Arg& arg);

    virtual void Start();
    virtual void Stop();
    virtual void Pause( bool flag );
    void Skip( StartOffsetType offsetType, int offset );

    Channel* NoteOn(
        u8 bankIndex,
        const NoteOnInfo& noteOnInfo
    );

    void SetSequenceUserprocCallback( SequenceUserprocCallback callback, void* arg );
    void CallSequenceUserprocCallback( u16 procId, SequenceTrack* track );

    //------------------------------------------------------------------
    // プレイヤーパラメータ
    void SetTempoRatio( f32 tempoRatio );
    void SetPanRange( f32 panRange );
    void SetChannelPriority( int priority );
    void SetReleasePriorityFix( bool fix );

    f32 GetTempoRatio() const { return m_TempoRatio; }
    f32 GetPanRange() const { return m_PanRange; }
    int GetChannelPriority() const { return m_ParserParam.priority; }

    bool IsReleasePriorityFix() const { return m_ReleasePriorityFixFlag; }

    //------------------------------------------------------------------
    // トラックパラメータ
    void SetTrackMute( u32 trackBitFlag, SeqMute mute );
    void SetTrackSilence( unsigned long trackBitFlag, bool silenceFlag, int fadeTimes );
    void SetTrackVolume( u32 trackBitFlag, f32 volume );
    void SetTrackPitch( u32 trackBitFlag, f32 pitch );
    void SetTrackLpfFreq( u32 trackBitFlag, f32 lpfFreq );
    void SetTrackBiquadFilter( u32 trackBitFlag, int type, f32 value );
    bool SetTrackBankIndex( u32 trackBitFlag, int bankIndex );
    void SetTrackTranspose( u32 trackBitFlag, s8 transpose );
    void SetTrackVelocityRange( u32 trackBitFlag, u8 range );
    void SetTrackOutputLine( u32 trackBitFlag, u32 outputLine );
    void ResetTrackOutputLine( u32 trackBitFlag );

    const BankFileReader& GetBankFileReader( u8 bankIndex ) const
    {
        return m_BankFileReader[bankIndex];
    }
    const WaveArchiveFileReader& GetWaveArchiveFileReader( u8 bankIndex ) const
    {
        // 波形アーカイブは、バンクと１：１対応している
        return m_WarcFileReader[bankIndex];
    }

    void SetTrackTvVolume( u32 trackBitFlag, f32 volume );
    void SetTrackChannelTvMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& param );
    void SetTrackTvPan( u32 trackBitFlag, f32 pan );
    void SetTrackTvSurroundPan( u32 trackBitFlag, f32 span );
    void SetTrackTvMainSend( u32 trackBitFlag, f32 send );
    void SetTrackTvFxSend( u32 trackBitFlag, AuxBus bus, f32 send );

    void SetTrackDrcVolume( u32 drcIndex, u32 trackBitFlag, f32 volume );
    void SetTrackChannelDrcMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& param );
    void SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan );
    void SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span );
    void SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send );
    void SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send );

    //------------------------------------------------------------------
    // シーケンス変数
    s16 GetLocalVariable( int varNo ) const;
    static s16 GetGlobalVariable( int varNo );
    void SetLocalVariable( int varNo, s16 var );
    static void SetGlobalVariable( int varNo, s16 var );

    vs16* GetVariablePtr( int varNo );

    //------------------------------------------------------------------
    // invalidate
    virtual void InvalidateData( const void* start, const void* end );

    //------------------------------------------------------------------
    // info
    const ParserPlayerParam& GetParserPlayerParam() const { return m_ParserParam; }
    ParserPlayerParam& GetParserPlayerParam() { return m_ParserParam; }
    u32 GetTickCounter() const { return m_TickCounter; }

    //------------------------------------------------------------------
    SequenceTrack* GetPlayerTrack( int trackNo );
    const SequenceTrack* GetPlayerTrack( int trackNo ) const;
    void SetPlayerTrack( int trackNo, SequenceTrack* track );

    const SequenceTrackAllocator* GetTrackAllocator() { return m_pSequenceTrackAllocator; }

    void Update();

    virtual void ChannelCallback( Channel* channel ) { (void)channel; }

protected:
    // バンクと波形アーカイブのみを対象とした Prepare() です。
    void PrepareForMidi(
        const void* banks[],    // バンクファイル (4つ)
        const void* warcs[],    // 波形アーカイブファイル (4つ)
        bool warcIsIndividuals[] );

private:
    virtual void OnUpdateFrameSoundThread() { Update(); }
    virtual void OnShutdownSoundThread() { Stop(); }

    void PrepareForPlayerHeap(const PrepareArg& arg);
    bool TryAllocLoader();
    void FreeLoader();

    template< typename T >
    void SetTrackParam( u32 trackBitFlag, void (SequenceTrack::*func)( T ), T param );

    template< typename T1, typename T2 >
    void SetTrackParam(
            u32 trackBitFlag,
            void (SequenceTrack::*func)( T1, T2 ),
            T1 t1, T2 t2 );

    template< typename T1, typename T2, typename T3 >
    void SetTrackParam(
            u32 trackBitFlag,
            void (SequenceTrack::*func)( T1, T2, T3 ),
            T1 t1, T2 t2, T3 t3 );

    template< typename T1, typename T2, typename T3, typename T4 >
    void SetTrackParam(
        u32 trackBitFlag,
        void (SequenceTrack::*func)( T1, T2, T3, T4 ),
        T1 t1, T2 t2, T3 t3, T4 t4 );

    int  ParseNextTick( bool doNoteOn );

    void UpdateChannelParam();
    void UpdateTick();
    void SkipTick();

    void CloseTrack( int trackNo );
    void FinishPlayer();

    f32 CalcTickPerMinute() const { return m_ParserParam.timebase * m_ParserParam.tempo * m_TempoRatio; }
    f32 CalcTickPerMsec() const { return CalcTickPerMinute() / ( 60 * 1000.0f ); }

    static vs16 m_GlobalVariable[ GLOBAL_VARIABLE_NUM ];
    static vs32 m_SkipIntervalTickPerFrame;

    bool m_ReleasePriorityFixFlag;
    bool m_IsPrepared;

    f32 m_PanRange;
    f32 m_TempoRatio;
    f32 m_TickFraction;
    u32 m_SkipTickCounter;
    f32 m_SkipTimeCounter;

    s32 m_DelayCount;

    ParserPlayerParam m_ParserParam;
    SequenceTrackAllocator* m_pSequenceTrackAllocator;

    SequenceUserprocCallback m_SequenceUserprocCallback;
    void* m_pSequenceUserprocCallbackArg;

    SequenceTrack* m_pTracks[ TRACK_NUM_PER_PLAYER ];

    vs16 m_LocalVariable[ PLAYER_VARIABLE_NUM ];
    vu32 m_TickCounter;

    WaveArchiveFileReader m_WarcFileReader[SoundArchive::SEQ_BANK_MAX];
    BankFileReader m_BankFileReader[SoundArchive::SEQ_BANK_MAX];

    enum ResState
    {
        RES_STATE_INVALID,          // 未初期化
        RES_STATE_RECV_LOAD_REQ,    // ロードリクエストを受けた
        RES_STATE_APPEND_LOAD_TASK, // ロードタスクを投げた
        RES_STATE_ASSIGNED          // ロードされ、SequenceSoundPlayer にセットされた
    };
    u8 m_ResState; // enum ResState
    bool m_IsInitialized;
    bool m_IsRegisterPlayerCallback;
    u8 padding;
    StartInfo m_StartInfo;

    SequenceSoundLoaderManager* m_pLoaderManager;
    SequenceSoundLoader* m_pLoader;
    SequenceSoundLoader::Arg m_LoaderArg;
}; // class SequenceSoundPlayer



template< typename T >
void SequenceSoundPlayer::SetTrackParam( u32 trackBitFlag, void (SequenceTrack::*func)( T ), T param )
{
    for( int trackNo = 0;
         trackNo < TRACK_NUM_PER_PLAYER && trackBitFlag != 0 ;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 ) continue;
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track != NULL ) (track->*func)( param );
    }
}

template< typename T1, typename T2 >
void SequenceSoundPlayer::SetTrackParam(
        u32 trackBitFlag,
        void (SequenceTrack::*func)( T1, T2 ),
        T1 t1, T2 t2 )
{
    for( int trackNo = 0;
         trackNo < TRACK_NUM_PER_PLAYER && trackBitFlag != 0 ;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 ) continue;
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track != NULL ) (track->*func)( t1, t2 );
    }
}

template< typename T1, typename T2, typename T3 >
void SequenceSoundPlayer::SetTrackParam(
        u32 trackBitFlag,
        void (SequenceTrack::*func)( T1, T2, T3 ),
        T1 t1, T2 t2, T3 t3 )
{
    for( int trackNo = 0;
         trackNo < TRACK_NUM_PER_PLAYER && trackBitFlag != 0 ;
         trackNo++, trackBitFlag >>= 1
    )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 ) continue;
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track != NULL ) (track->*func)( t1, t2, t3 );
    }
}

template< typename T1, typename T2, typename T3, typename T4 >
void SequenceSoundPlayer::SetTrackParam(
    u32 trackBitFlag,
    void (SequenceTrack::*func)( T1, T2, T3, T4 ),
    T1 t1, T2 t2, T3 t3, T4 t4 )
{
    for( int trackNo = 0;
        trackNo < TRACK_NUM_PER_PLAYER && trackBitFlag != 0 ;
        trackNo++, trackBitFlag >>= 1
        )
    {
        if ( ( trackBitFlag & 0x01 ) == 0 ) continue;
        SequenceTrack* track = GetPlayerTrack( trackNo );
        if ( track != NULL ) (track->*func)( t1, t2, t3, t4 );
    }
}

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SEQUENCE_SOUND_PLAYER_H_ */

