﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_SEQUENCE_SOUND_H_
#define NW_SND_SEQUENCE_SOUND_H_

#include <nw/snd/snd_BasicSound.h>
#include <nw/snd/snd_BasicSoundPlayer.h>
#include <nw/snd/snd_SequenceSoundPlayer.h>
#include <nw/snd/snd_SoundInstanceManager.h>
#include <nw/snd/snd_Task.h>
#include <nw/snd/snd_Debug.h>
#include <nw/snd/snd_PlayerHeapDataManager.h>

namespace nw {
namespace snd {

/* ========================================================================
        type declarataion
   ======================================================================== */

class SequenceSoundHandle;

//! @briefprivate
struct SequenceSoundResourceStatus
{
    bool isLoadBanks[SoundArchive::SEQ_BANK_MAX];
    bool isLoadWarcs[SoundArchive::SEQ_BANK_MAX];

    SequenceSoundResourceStatus()
    {
        for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
        {
            isLoadBanks[i] = isLoadWarcs[i] = false;
        }
    }

    void Dump() const
    {
        NW_LOG("[SEQ] isLoaded : bank(%d:%d:%d:%d) warc(%d:%d:%d:%d)\n",
                isLoadBanks[0], isLoadBanks[1], isLoadBanks[2], isLoadBanks[3],
                isLoadWarcs[0], isLoadWarcs[1], isLoadWarcs[2], isLoadWarcs[3] );
    }
};

namespace internal {

namespace driver {

class NoteOnCallback;

} // namespace nw::snd:internal::driver

class SequenceSound;

typedef SoundInstanceManager<SequenceSound> SequenceSoundInstanceManager;


class SequenceSound : public BasicSound
{
    friend class nw::snd::SequenceSoundHandle;

public:
    NW_UT_RUNTIME_TYPEINFO(BasicSound)


    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    static const int BANK_INDEX_MIN = 0;
    static const int BANK_INDEX_MAX = 3;
    static const s8 TRANSPOSE_MIN = -64;
    static const s8 TRANSPOSE_MAX = 63;
    static const u8 VELOCITY_RANGE_MIN = 0;
    static const u8 VELOCITY_RANGE_MAX = 127;

    explicit SequenceSound( SequenceSoundInstanceManager& manager );

    void Setup(
        driver::SequenceTrackAllocator* trackAllocator,
        u32 allocTracks,
        driver::NoteOnCallback* callback,
        int channelPriority,
        bool isReleasePriorityFix,
        SequenceUserprocCallback userproc,
        void* userprocArg
    );

    struct Resource
    {
        const void* seq;
        const void* banks[SoundArchive::SEQ_BANK_MAX];   // TODO: バンクの数は SoundArchive への依存ではなく、SequenceSound 側で定義すべき
        const void* warcs[SoundArchive::SEQ_BANK_MAX];
        bool warcIsIndividuals[SoundArchive::SEQ_BANK_MAX];

        Resource() { Initialize(); }
        void Initialize()
        {
            seq = NULL;
            for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
            {
                banks[i] = NULL;
                warcs[i] = NULL;
                warcIsIndividuals[i] = false;
            }
        }
    };
    void Prepare( const Resource& res,
            const driver::SequenceSoundPlayer::StartInfo& startInfo );

    void RegisterDataLoadTask(
            const driver::SequenceSoundLoader::LoadInfo& loadInfo,
            const driver::SequenceSoundPlayer::StartInfo& startInfo );


    /* override */ void Initialize();
    /* override */ void Finalize();
    /* override */ bool IsPrepared() const
    {
        if ( m_IsCalledPrepare ) return true;

        if (IsPlayerAvailable() == false)
        {
            return false;
        }
        return m_PlayerInstance.IsPrepared();
    }

    // パラメータ設定
    void SetTempoRatio( f32 tempoRatio );
    void SetChannelPriority( int priority );

    // パラメータ取得
    u32 GetTick() const;

    // トラックパラメータ設定
    void SetTrackMute( u32 trackBitFlag, SeqMute mute );
    void SetTrackMute( u32 trackBitFlag, bool muteFlag );
    void SetTrackSilence( u32 trackBitFlag, bool silenceFlag, int fadeTimes );

    void SetTrackBiquadFilter( u32 trackBitFlag, int type, f32 value );
    void SetTrackBankIndex( u32 trackBitFlag, int bankIndex );

    void SetTrackVolume( u32 trackBitFlag, f32 volume );
    void SetTrackPitch( u32 trackBitFlag, f32 pitch );
    void SetTrackLpfFreq( u32 trackBitFlag, f32 lpfFreq );
    void SetTrackTranspose( u32 trackBitFlag, s8 transpose );
    void SetTrackVelocityRange( u32 trackBitFlag, u8 range );
    void SetTrackOutputLine( u32 trackBitFlag, u32 lineFlag );
    void ResetTrackOutputLine( u32 trackBitFlag );

    void SetTrackChannelMixParameter( u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam );
    void SetTrackMainOutVolume( u32 trackBitFlag, f32 volume );
    void SetTrackPan( u32 trackBitFlag, f32 pan );
    void SetTrackSurroundPan( u32 trackBitFlag, f32 surroundPan );
    void SetTrackMainSend( u32 trackBitFlag, f32 send );
    void SetTrackFxSend( u32 trackBitFlag, AuxBus bus, f32 send );

    void SetTrackDrcChannelMixParameter( u32 drcIndex, u32 trackBitFlag, u32 srcChNo, const MixParameter& mixParam );
    void SetTrackDrcOutVolume( u32 drcIndex, u32 trackBitFlag, f32 volume );
    void SetTrackDrcPan( u32 drcIndex, u32 trackBitFlag, f32 pan );
    void SetTrackDrcSurroundPan( u32 drcIndex, u32 trackBitFlag, f32 span );
    void SetTrackDrcMainSend( u32 drcIndex, u32 trackBitFlag, f32 send );
    void SetTrackDrcFxSend( u32 drcIndex, u32 trackBitFlag, AuxBus bus, f32 send );

    // シーケンス変数
    bool ReadVariable( int varNo, s16* var ) const;
    static bool ReadGlobalVariable( int varNo, s16* var );
    bool ReadTrackVariable( int trackNo, int varNo, s16* var ) const;
    void WriteVariable( int varNo, s16 var );
    static void WriteGlobalVariable( int varNo, s16 var );
    void WriteTrackVariable( int trackNo, int varNo, s16 var );

    void SetLoaderManager(driver::SequenceSoundLoaderManager& manager)
    {
        m_PlayerInstance.SetLoaderManager(&manager);
    }

    // デバッグ関数
    DebugSoundType GetSoundType() const { return DEBUG_SOUND_TYPE_SEQSOUND; }

private:
    /* override */ bool IsAttachedTempSpecialHandle();
    /* override */ void DetachTempSpecialHandle();
    /* override */ void OnUpdatePlayerPriority();

    /* override */ void OnUpdateParam() {} // あとで消す
    /* override */ driver::BasicSoundPlayer* GetBasicSoundPlayerHandle()
    {
        return &m_PlayerInstance;
    }

    void Skip( driver::SequenceSoundPlayer::StartOffsetType offsetType, int offset );

    SequenceSoundHandle* m_pTempSpecialHandle;
    SequenceSoundInstanceManager& m_Manager;

    bool m_InitializeFlag;
    bool m_CanUseTask;
    bool m_IsCalledPrepare; // Prepare() が呼ばれたかどうか
    u8 padding[1];

    driver::SequenceSoundPlayer m_PlayerInstance;

};

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_SEQUENCE_SOUND_H_ */

