﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_MIDI_SEQUENCE_PLAYER_H_
#define NW_SND_MIDI_SEQUENCE_PLAYER_H_

#include <nw/snd/snd_SequenceSoundPlayer.h>
#include <nw/snd/snd_Channel.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        class definition
   ======================================================================== */

class MidiSequencePlayer : public SequenceSoundPlayer
{
    /* ------------------------------------------------------------------------
            type definition
       ------------------------------------------------------------------------ */
public:
    enum {
        MIDI_MSG_NOTE_OFF = 0x80,
        MIDI_MSG_NOTE_ON = 0x90,
        MIDI_MSG_POLY_KEY_PRESSURE = 0xa0,
        MIDI_MSG_CONTROL_CHANGE = 0xb0,
        MIDI_MSG_PROGRAM_CHANGE = 0xc0,
        MIDI_MSG_CHANNEL_PRESSURE = 0xd0,
        MIDI_MSG_PITCH_BEND = 0xe0,
        MIDI_MSG_SYSTEM = 0xf0
    };

    enum {
        MIDI_CTRLCHG_BANK_SELECT = 0,
        MIDI_CTRLCHG_MOD_DEPTH = 1,
        MIDI_CTRLCHG_INIT_PAN = 3,
        MIDI_CTRLCHG_PORTA_TIME = 5,
        MIDI_CTRLCHG_DATA_ENTRY = 6,
        MIDI_CTRLCHG_VOLUME = 7,
        MIDI_CTRLCHG_SPAN = 9,
        MIDI_CTRLCHG_PAN = 10,
        MIDI_CTRLCHG_EXPRESSION = 11,
        MIDI_CTRLCHG_MAIN_VOLUME = 12,
        MIDI_CTRLCHG_TRANSPOSE = 13,
        MIDI_CTRLCHG_PRIORITY = 14,
        MIDI_CTRLCHG_SETVAR_0 = 16,
        MIDI_CTRLCHG_SETVAR_1 = 17,
        MIDI_CTRLCHG_SETVAR_2 = 18,
        MIDI_CTRLCHG_SETVAR_3 = 19,
        MIDI_CTRLCHG_BEND_RANGE = 20,
        MIDI_CTRLCHG_MOD_SPEED = 21,
        MIDI_CTRLCHG_MOD_TYPE = 22,
        MIDI_CTRLCHG_MOD_RANGE = 23,
        MIDI_CTRLCHG_MOD_DELAY = 26,
        MIDI_CTRLCHG_MOD_DELAY_10 = 27,
        MIDI_CTRLCHG_SWEEP_PITCH = 28,
        MIDI_CTRLCHG_SWEEP_PITCH_24 = 29,
        MIDI_CTRLCHG_BIQUAD_TYPE = 30,
        MIDI_CTRLCHG_BIQUAD_VALUE = 31,
        MIDI_CTRLCHG_DAMPER = 64,
        MIDI_CTRLCHG_PORTA = 65,
        MIDI_CTRLCHG_MONOPHONIC = 68,
        MIDI_CTRLCHG_CUTOFF = 74,
        MIDI_CTRLCHG_ENV_HOLD = 79,
        MIDI_CTRLCHG_PORTA_CONTROL = 84,
        MIDI_CTRLCHG_ATTACK = 85,
        MIDI_CTRLCHG_DECAY = 86,
        MIDI_CTRLCHG_SUSTAIN = 87,
        MIDI_CTRLCHG_RELEASE = 88,
        MIDI_CTRLCHG_FXSEND_A = 91,
        MIDI_CTRLCHG_FXSEND_B = 92,
        MIDI_CTRLCHG_FXSEND_C = 93,
        MIDI_CTRLCHG_MAINSEND = 95,
        MIDI_CTRLCHG_NRPN_LSB = 98,
        MIDI_CTRLCHG_NRPN_MSB = 99,
        MIDI_CTRLCHG_RPN_LSB = 100,
        MIDI_CTRLCHG_RPN_MSB = 101,
        MIDI_CTRLCHG_FRONTBYPASS =119
    };

    enum {
        MIDI_MODEMSG_ALL_SOUND_OFF = 120,
        MIDI_MODEMSG_RESET_ALL_CONTROLER = 121,
        MIDI_MODEMSG_LOCAL_CONTROL = 122,
        MIDI_MODEMSG_ALL_NOTE_OFF = 123,
        MIDI_MODEMSG_OMNI_OFF = 124,
        MIDI_MODEMSG_OMNI_ON = 125,
        MIDI_MODEMSG_MONO_MODE = 126,
        MIDI_MODEMSG_POLY_MODE = 127
    };

    enum {
        // ( msb << 8 ) + lsb
        MIDI_RPN_PITCHBEND_SENSITIVITY = ( 0 << 8 ) + 0,
        MIDI_RPN_NULL = ( 127 << 8 ) + 127
    };

    enum {
        // ( msb << 8 ) + lsb
        MIDI_NRPN_ENV_RESET = ( 0 << 8 ) + 0,
        MIDI_NRPN_NULL = ( 127 << 8 ) + 127
    };

    static const int NOTE_INFO_COUNT = Channel::CHANNEL_NUM;

    struct NoteInfo
    {
        Channel* channel;
        int channelIndex;
        int key;
    };

    struct ParameterControlInfo
    {
        enum Mode { RPN, NRPN };

        Mode mode;
        u8 rpnLsb;
        u8 rpnMsb;
        u8 nrpnLsb;
        u8 nrpnMsb;

        ParameterControlInfo()
        : mode(RPN),
          rpnLsb(127),
          rpnMsb(127),
          nrpnLsb(127),
          nrpnMsb(127)
        {}
    };

    /* ------------------------------------------------------------------------
            class member
       ------------------------------------------------------------------------ */
public:
    MidiSequencePlayer();
    virtual ~MidiSequencePlayer();

    void Prepare(
        const void* banks[],    // バンクファイル (4つ)
        const void* warcs[],    // 波形アーカイブファイル (4つ)
        bool warcIsIndividuals[] );

    virtual void Start();

    void SendMessage( u8 status, u8 data1, u8 data2 );
    void Reset();

    void SetProgramNumber( int channelIndex, int prgNo );

protected:
    virtual void ChannelCallback( Channel* channel );

private:
    void NoteOn( u8 channelIndex, u8 key, u8 velocity );
    void NoteOff( u8 channelIndex, u8 key, u8 velocity );

    void HandleControlChangeMessage( u8 channelIndex, u8 control, u8 value );
    void HandleRpnMessage( u8 channelIndex, u8 value );
    void HandleNrpnMessage( u8 channelIndex, u8 value );
    void HandleChannelModeMessage( u8 control, u8 value );
    void HandleProgramChangeMessage( u8 channelIndex, u8 program );
    void HandlePitchBendMessage( u8 channelIndex, u8 lsb, u8 msb );

    void NoteOffAll();
    void StopAllSound();
    void ResetAllController();

    NoteInfo* FindFreeNoteInfo();
    NoteInfo* FindNoteInfo( int channelIndex, int key );
    NoteInfo* FindNoteInfo( int channelIndex );

    NoteInfo m_NoteInfo[ NOTE_INFO_COUNT ];
    ParameterControlInfo m_ParameterControlInfo[ TRACK_NUM_PER_PLAYER ];
};

} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_MIDI_SEQUENCE_PLAYER_H_ */

