﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_LOADER_MANAGER_H_
#define NW_SND_LOADER_MANAGER_H_

#include <nw/ut/ut_LinkList.h>

namespace nw {
namespace snd {
namespace internal {

template < typename Instance >
class LoaderManager : public driver::SoundThread::SoundFrameCallback
{
public:
    typedef ut::LinkList<Instance, offsetof(Instance, m_LinkForLoaderManager)> List;
    typedef typename List::Iterator Iterator;

    LoaderManager() : m_pBuffer(NULL), m_BufferSize(0) {}
    ~LoaderManager() {}

    static size_t GetRequiredMemSize(u32 count)
    {
        return sizeof(Instance) * count;
    }

    u32 Create(void* buffer, size_t size)
    {
        NW_NULL_ASSERT( buffer );

        char* ptr = static_cast<char*>(buffer);
        const u32 objectCount = size / sizeof(Instance);
        for ( u32 i = 0; i < objectCount; i++ )
        {
            Instance* instance = new( ptr ) Instance;
            m_FreeList.PushBack( instance );
            ptr += sizeof(Instance);
        }

        m_pBuffer = buffer;
        m_BufferSize = size;
        return objectCount;
    }

    void Destroy()
    {
        char* ptr = static_cast<char*>(m_pBuffer);
        const u32 objectCount = m_BufferSize / sizeof(Instance);
        for ( unsigned long i = 0; i < objectCount; i++ )
        {
            Instance* instance = reinterpret_cast<Instance*>(ptr);
            instance->~Instance();
            ptr += sizeof(Instance);
        }

        m_FreeList.Clear();
    }

    Instance* Alloc()
    {
        if (m_FreeList.IsEmpty())
        {
            return NULL;
        }

        Instance* instance = &m_FreeList.GetFront();
        m_FreeList.PopFront();

        return instance;
    }

    void Free(Instance* instance)
    {
        NW_ASSERT_NOT_NULL(instance);
        if (instance->IsInUse())
        {
            m_FreeReqList.PushBack(instance);
        }
        else
        {
            instance->Finalize();
            m_FreeList.PushBack(instance);
        }
    }

    void UpdateFreeReqList()
    {
        for (Iterator itr = m_FreeReqList.GetBeginIter();
             itr != m_FreeReqList.GetEndIter(); )
        {
            Iterator curItr = itr++;
            if (curItr->IsInUse() == false)  // 使用していない
            {
                curItr->Finalize();
                m_FreeReqList.Erase(&(*curItr));
                m_FreeList.PushBack(&(*curItr));
            }
        }
    }

    u32 GetFreeCount() const { return m_FreeList.GetSize(); }

    /* override */ void OnBeginSoundFrame() { UpdateFreeReqList(); }

private:
    void* m_pBuffer;
    u32 m_BufferSize;
    List m_FreeList;
    List m_FreeReqList;
}; // class LoaderManager

} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw


#endif /* NW_SND_LOADER_MANAGER_H_ */


