﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_FND_LEGACY_STREAM_ADAPTOR_H_
#define NW_SND_FND_LEGACY_STREAM_ADAPTOR_H_

#include <nw/ut/ut_Inlines.h>
#ifdef NW_PLATFORM_CTR
#include <nw/io/io_FileStream.h>
#else
#include <nw/ut/ut_FileStream.h>
#endif
#include <nw/snd/fnd/basis/sndfnd_Config.h>
#include <nw/snd/fnd/io/sndfnd_FileStream.h>

namespace nw {
namespace snd {
namespace internal {
namespace fnd {

#ifdef NW_PLATFORM_CTR
typedef nw::io::FileStream LegacyFileStream;
typedef nw::io::IOStream::IOStreamCallback LegacyIoStreamCallback;
enum
{
    FILE_STREAM_SEEK_BEGIN   = io::FILE_STREAM_SEEK_BEGIN,
    FILE_STREAM_SEEK_CURRENT = io::FILE_STREAM_SEEK_CURRENT,
    FILE_STREAM_SEEK_END     = io::FILE_STREAM_SEEK_END
};

#else
typedef nw::ut::FileStream LegacyFileStream;
typedef nw::ut::IoStream::IoStreamCallback LegacyIoStreamCallback;
enum
{
    FILE_STREAM_SEEK_BEGIN   = ut::FILE_STREAM_SEEK_BEGIN,
    FILE_STREAM_SEEK_CURRENT = ut::FILE_STREAM_SEEK_CURRENT,
    FILE_STREAM_SEEK_END     = ut::FILE_STREAM_SEEK_END
};
#endif

//---------------------------------------------------------------------------
//! @brief  LegacyFileStream へのアダプタです。
//---------------------------------------------------------------------------
class LegacyFileStreamAdaptor : public LegacyFileStream
{
    NW_DISALLOW_COPY_AND_ASSIGN(LegacyFileStreamAdaptor);

public:
    explicit LegacyFileStreamAdaptor(fnd::FileStream* sourceStream) :
      m_SourceStream(sourceStream)
    {
        NW_ASSERT_NOT_NULL(sourceStream);
    }

    virtual ~LegacyFileStreamAdaptor() { }

public:
    virtual void Close()
    {
        NW_ASSERT(IsAvailable());
        return m_SourceStream->Close();
    }

    virtual s32 Read(void* buf, u32 length)
    {
        NW_ASSERT(CanRead());
        return m_SourceStream->Read(buf, length);
    }

    virtual bool ReadAsync(void* buf, u32 length, LegacyIoStreamCallback callback, void* arg)
    {
        (void)buf;
        (void)length;
        (void)callback;
        (void)arg;

        NW_FATAL_ERROR("not implemented.\n");
        return false;
    }

    virtual s32 Write(const void* buf, u32 length)
    {
        NW_ASSERT(CanWrite());
        return m_SourceStream->Write(buf, length);
    }

    virtual bool WriteAsync(const void* buf, u32 length, LegacyIoStreamCallback callback, void* arg)
    {
        (void)buf;
        (void)length;
        (void)callback;
        (void)arg;

        NW_FATAL_ERROR("not implemented.\n");
        return false;
    }

    virtual s32  WaitAsync()
    {
        NW_FATAL_ERROR("not implemented.\n");
        return 0;
    }

    virtual bool IsBusy() const
    {
        NW_FATAL_ERROR("not implemented.\n");
        return false;
    }

    virtual bool CanAsync () const { return false; }

    virtual bool CanRead  () const { return IsAvailable() && m_SourceStream->CanRead(); }

    virtual bool CanWrite () const { return IsAvailable() && m_SourceStream->CanWrite(); }

    virtual bool CanSeek () const { return IsAvailable() && m_SourceStream->CanSeek(); }

    virtual bool IsAvailable() const
    {
        return m_SourceStream != NULL && m_SourceStream->IsOpened();
    }

    virtual u32  GetSize() const
    {
        return m_SourceStream != NULL ? m_SourceStream->GetSize() : 0;
    }

#ifdef NW_PLATFORM_CTR
    virtual void Seek( s32 offset, u32 origin )
#else
    virtual bool Seek( s32 offset, u32 origin )
#endif
    {
        NW_ASSERT(CanSeek());

        Stream::SeekOrigin seekOrigin = Stream::SEEK_ORIGIN_BEGIN;

        switch(origin)
        {
        case FILE_STREAM_SEEK_BEGIN:
            seekOrigin = Stream::SEEK_ORIGIN_BEGIN;
            break;

        case FILE_STREAM_SEEK_END:
            seekOrigin = Stream::SEEK_ORIGIN_END;
            break;

        case FILE_STREAM_SEEK_CURRENT:
            seekOrigin = Stream::SEEK_ORIGIN_CURRENT;
            break;
        }

#ifdef NW_PLATFORM_CTR
        m_SourceStream->Seek(offset, seekOrigin);
#else
        return m_SourceStream->Seek(offset, seekOrigin).IsTrue();
#endif
    }

    virtual void Cancel()
    {
        NW_FATAL_ERROR("not implemented.\n");
    }

    virtual bool CancelAsync( LegacyIoStreamCallback callback, void* arg )
    {
        (void)callback;
        (void)arg;

        NW_FATAL_ERROR("not implemented.\n");
        return false;
    }

    virtual bool CanCancel() const { return false; }

    virtual u32 Tell() const
    {
        NW_ASSERT(IsAvailable());
        return m_SourceStream->GetCurrentPosition();
    }

private:
    fnd::FileStream* m_SourceStream;
};

} // namespace nw::snd::internal::fnd
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_FND_LEGACY_STREAM_ADAPTOR_H_
