﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_SOUND_EDIT_SESSION_H_
#define NW_SND_EDIT_SOUND_EDIT_SESSION_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/fnd/basis/sndfnd_Time.h>
#include <nw/snd/edit/hio/sndedit_HioManager.h>
#include <nw/snd/edit/hio/sndedit_HioSyncChannel.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/edit/handler/sndedit_SyncReplyHandler.h>
#include <nw/snd/edit/handler/sndedit_QueryItemsReplyHandler.h>

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_suppress 1301 // padding inserted in struct.
#pragma diag_suppress 2530 // padding added to end of struct.
#endif
#endif

namespace nw {
namespace snd {

namespace internal {
namespace fnd {
class FrameHeap;
}
}

namespace edit {

class SoundArchiveEditor;
class SoundEditSession;

//---------------------------------------------------------------------------
//! @brief  【β版】サウンド編集のセッションを管理します。
//!
//!         セッションを開くと、SoundMaker に定期的に接続要求を投げます。
//!         @n
//!         SoundMaker から接続を許可されると、指定 SoundArchiveEditor に
//!         関連付けられた SoundArchive の内容を SoundMaker 上のプロジェクト内容に置き換えます。
//!
//!         SoundMaker との通信には MCS を利用します。
//---------------------------------------------------------------------------
class SoundEditSession
{
    NW_DISALLOW_COPY_AND_ASSIGN(SoundEditSession);

public: // 定数の定義
    static const u32 DEFAULT_CHANNEL_STREAM_BUFFER_SIZE = 64 * 1024;   //!< チャンネルバッファサイズのデフォルト値です。
    static const u32 DEFAULT_SYNC_TIMEOUT               =  3 * 1000;   //!< SYNC タイムアウトのデフォルト値です。
    static const u32 DEFAULT_CACHE_SYNC_INTERVAL        =  3 * 1000;   //!< キャッシュの同期間隔のデフォルト値です。
    static const u32 DEFAULT_SEND_TIMEOUT               =  5 * 1000;   //!< Send タイムアウトのデフォルト値です。
    static const u32 DEFAULT_MAX_ITEM_NAME              = 256;         //!< アイテム名の長さのデフォルト上限値です。

public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  SoundEditSession 設定を格納する構造体です。
    //!
    //!         各パラメータには、コンストラクタでデフォルト値が設定されます。
    //!
    //! @see    Initialize()
    //! @see    GetRequiredMemorySize()
    //---------------------------------------------------------------------------
    struct Configs
    {
        //---------------------------------------------------------------------------
        //! @brief  コンストラクタです。
        //---------------------------------------------------------------------------
        Configs() :
            channelStreamBufferSize(DEFAULT_CHANNEL_STREAM_BUFFER_SIZE),
            syncTimeout(DEFAULT_SYNC_TIMEOUT),
            cacheSyncInterval(DEFAULT_CACHE_SYNC_INTERVAL),
            sendTimeout(DEFAULT_SEND_TIMEOUT),
            maxItemName(DEFAULT_MAX_ITEM_NAME),
            port0(internal::HIO_SNDEDIT_SYNC_CHANNEL),
            port1(internal::HIO_SNDEDIT_FUNC_CHANNEL)
        {
        }

        u32 channelStreamBufferSize;    //!< チャンネル毎の受信用ストリームのバッファサイズです。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_CHANNEL_STREAM_BUFFER_SIZE が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 syncTimeout;                //!< SYNC タイムアウト（ミリ秒）です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_SYNC_TIMEOUT が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 cacheSyncInterval;          //!< キャッシュの同期間隔（ミリ秒）です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_CACHE_SYNC_INTERVAL が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 sendTimeout;                //!< 送信タイムアウト（ミリ秒）です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_SEND_TIMEOUT が使用されます。
                                        //!< @n
                                        //!< 通常の利用において、値を変更する必要はありません。
        u32 maxItemName;                //!< アイテム名の長さの上限値です。
                                        //!< @n
                                        //!< 指定しないとデフォルト値 DEFAULT_MAX_ITEM_NAME が使用されます。
                                        //!< @n
                                        //!< 非常に長いアイテム名を対象にしたい場合や、使用する名前が短いことが事前にわかっていて、
                                        //!< 少しでもメモリ使用量を減らしたい場合に設定を変更してください。

        PORT port0;                     //!< 内部で使用するポートです。
        PORT port1;                     //!< 内部で使用するポートです。
    };

private: // 型の定義
    enum State
    {
        STATE_NOT_INITIALIZED = 0,  //!< 未初期化状態です。
        STATE_STOPPED,              //!< 停止状態です。
        STATE_SYNC_REQUESTING,      //!< SYNC 要求中（オープンされていない状態）です。
        STATE_SYNC_UPDATING,        //!< SYNC 更新中（オープンされている状態）です。
        STATE_SYNC_COMPLETED        //!< SYNC 済み（オープンされている状態）です。
    };

public: // コンストラクタ
    //===========================================================================
    //! @name コンストラクタ/デストラクタ
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    SoundEditSession();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    ~SoundEditSession();

    //@}
    // コンストラクタ/デストラクタ
    //===========================================================================

public: // メソッド
    //===========================================================================
    //! @name 初期化
    //@{

#if defined(NW_PLATFORM_CAFE)

    //---------------------------------------------------------------------------
    //! @brief      SoundEditSession を初期化します。
    //!
    //!             この関数は、SoundEditSession で利用するメモリを初期化し、
    //!             SoundMaker との接続を開始するための準備を行います。
    //!             @n
    //!             SoundEditSession を利用する前に、必ずこの関数を呼び出してください。
    //!
    //!             引数 buffer は、通信の作業領域として利用されます。
    //!             @n
    //!             SoundEditSession::GetRequiredMemorySize() が返すより大きいサイズを指定する必要があります。
    //!
    //! @param[in]  buffer           通信処理の作業バッファです。
    //! @param[in]  bufferLength     buffer の長さを指定します。
    //! @param[in]  fsClient         FSClient です。
    //!                              @n
    //!                              HostFileIO 経由で PC 上のファイルを読み込むために使用します。
    //! @param[in]  hfioVolumePath   HostFileIO ボリュームパスです。
    //!                              @n
    //!                              "/dev/hfio01" のマウント先ボリューム（例 : "/vol/pc"）を指定してください。
    //! @param[in]  configs          SoundEditSession の設定を指定します。
    //!                              GetRequiredMemorySize() に渡した Configs を指定します。
    //!
    //! @return     処理結果を返します。
    //!
    //! @see        Configs
    //! @see        GetRequiredMemorySize()
    //! @see        Finalize()
    //---------------------------------------------------------------------------
    Result Initialize(
        void* buffer,
        u32 bufferLength,
        FSClient* fsClient,
        const char* hfioVolumePath,
        const Configs& configs);
#elif defined(NW_PLATFORM_CTR)

    Result Initialize(
        void* buffer,
        u32 bufferLength,
        const Configs& configs);


#elif defined(NW_PLATFORM_WIN32)

    Result Initialize(
        void* buffer,
        u32 bufferLength,
        const Configs& configs);

#elif defined(NW_USE_NINTENDO_SDK)
    // TODO: independent
    Result Initialize(
        void* buffer,
        u32 bufferLength,
        const Configs& configs);

#endif

    //---------------------------------------------------------------------------
    //! @brief  SoundEditSession の終了処理を行います。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief      SoundEditSession の利用に必要なメモリサイズを取得します。
    //!
    //!             SoundEditSession を利用するには、
    //!             この関数で取得したサイズ分のバッファを確保し、
    //!             Initialize() の引数 buffer、bufferLength に指定する必要があります。
    //!             @n
    //!             その際 Initialize() の引数 configs には、
    //!             GetRequiredMemorySize() に渡した Configs と同じものを指定してください。
    //!
    //! @param[in]  configs  SoundEditSession の設定を指定します。
    //!
    //! @return     必要なメモリサイズを返します。
    //!
    //! @see        Configs
    //! @see        Initialize()
    //---------------------------------------------------------------------------
    u32 GetRequiredMemorySize(const Configs& configs) const;

    //@}
    // 初期化
    //===========================================================================

    //===========================================================================
    //! @name 編集対象の登録
    //@{

    //---------------------------------------------------------------------------
    //! @brief      指定 SoundArchiveEditor を登録します。
    //!
    //!             ここで指定した SoundArchiveEditor が SoundMaker と接続する対象となります。
    //!
    //!             【★注意】
    //!             @n
    //!             現状では、SoundArchiveEditorを１つだけ登録でき、
    //!             複数登録しようとすると失敗します。
    //!             @n
    //!             今後、SoundMaker上のサウンドプロジェクトとサウンドアーカイブバイナリの
    //!             マッチングを可能する予定で、その際に、
    //!             複数登録が可能になる（マルチサウンドアーカイブ対応）予定です。
    //!
    //! @param[in]  soundArchiveEditor  編集対象の SoundArchiveEditor を指定します。
    //!
    //! @return     結果を返します。
    //!
    //! @see        UnregisterSoundArchiveEditor()
    //---------------------------------------------------------------------------
    Result RegisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor);

    //---------------------------------------------------------------------------
    //! @brief      指定 SoundArchiveEditor の登録を解除します。
    //!
    //!             【★注意】
    //!             @n
    //!             現状では、SoundArchiveEditorを１つだけ登録でき、
    //!             複数登録しようとすると失敗します。
    //!             @n
    //!             今後、SoundMaker上のサウンドプロジェクトとサウンドアーカイブバイナリの
    //!             マッチングを可能する予定で、その際に、
    //!             複数登録が可能になる（マルチサウンドアーカイブ対応）予定です。
    //!
    //! @param[in]  soundArchiveEditor  編集対象の SoundArchiveEditor を指定します。
    //!
    //! @see        RegisterSoundArchiveEditor()
    //---------------------------------------------------------------------------
    void UnregisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor);

    //@}
    // 編集対象の登録
    //===========================================================================

    //===========================================================================
    //! @name 状態の取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief   初期化の有無を取得します。
    //!
    //! @return  初期化済みの場合は true、初期化されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsInitialized() const { return m_State > STATE_NOT_INITIALIZED; }

    //---------------------------------------------------------------------------
    //! @brief  セッションが開かれているかどうかを取得します。
    //!
    //! @return セッションが開かれている場合は true、開らかれていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsOpened() const
    {
        return IsInitialized() && m_State > STATE_STOPPED;
    }

    //---------------------------------------------------------------------------
    //! @brief   SoundMaker との接続の有無を取得します。
    //!
    //! @return  SoundMaker と接続済みの場合は true、接続されていない場合は false を返します。
    //---------------------------------------------------------------------------
    bool IsConnected() const
    {
        return IsInitialized() && m_Connection.IsOpened();
    }

    //@}
    // 状態の取得
    //===========================================================================

    //===========================================================================
    //! @name 開始と終了
    //@{

    //---------------------------------------------------------------------------
    //! @brief  サウンド編集セッションを開きます。
    //!
    //!         この関数が呼び出されると、Update() のタイミングで
    //!         SoundMaker との通信が行われるようになります。
    //!
    //! @see    Close()
    //! @see    Update()
    //---------------------------------------------------------------------------
    void Open();

    //---------------------------------------------------------------------------
    //! @brief  サウンド編集セッションを閉じます。
    //!
    //!         SoundMakerとの通信も終了します。
    //!
    //! @see    Open()
    //---------------------------------------------------------------------------
    void Close();

    //---------------------------------------------------------------------------
    //! @brief  サウンド編集セッションを更新します。
    //!
    //!         セッションが開かれている間、SoundMaker への接続要求を投げたり、
    //!         SoundMaker に編集のあったアイテムを問い合わせたりします。
    //!
    //!         セッションが開かれていない場合は、この関数を呼び出しても何も処理しません。
    //!
    //! @see    Open()
    //! @see    Close()
    //---------------------------------------------------------------------------
    void Update();

    //@}
    // 開始と終了
    //===========================================================================

    // HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
    //---------------------------------------------------------------------------
    //! @briefprivate
    //! @brief  SoundPlayer モードの設定を行います。
    //!         ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コードです。
    //! @param value :private
    //---------------------------------------------------------------------------
    void Interim_SetIsSoundPlayer(bool value)
    {
        m_Interim_IsSoundPlayer = value;
    }
#endif

private: // メソッド
    //! @brief  SYNC 用チャンネルを初期化します。
    Result InitializeSyncChannel(
        snd::internal::fnd::FrameHeap& allocator,
        u32 recvStreamBufferSize,
        u32 recvPacketBufferSize);

    //! @brief  SYNC 用チャンネルの終了処理を行います。
    void FinalizeSyncChannel();

    //! @brief  SYNC 用メッセージハンドラを初期化します。
    void InitializeSyncHandlers();

    //! @brief  SYNC 用メッセージハンドラの終了処理を行います。
    void FinalizeSyncHandlers();

    //! @brief  実機->PCチャンネルを初期化します。
    Result InitializeFuncChannel(
        snd::internal::fnd::FrameHeap& allocator,
        u32 recvStreamBufferSize,
        u32 recvPacketBufferSize);

    //! @brief  実機->PCチャンネルの終了処理を行います。
    void FinalizeFuncChannel();

    //! @brief  実機->PCメッセージハンドラを初期化します。
    void InitializeFuncHandlers();

    //! @brief  実機->PCメッセージハンドラの終了処理を行います。
    void FinalizeFuncHandlers();

    //! @brief  SYNC チャンネルの受信パケットバッファサイズを取得します。
    u32 GetSyncChannelRecvPacketBufferSize(u32 maxItemName) const;

    //! @brief  FUNC チャンネルの受信パケットバッファサイズを取得します。
    u32 GetFuncChannelRecvPacketBufferSize(u32 maxItemName, u32 maxItemInfoSize) const;

    //! @brief  接続します。
    Result Connect();

    //! @brief  切断します。
    void Disconnect();

    //! @brief  通信バッファをクリアします。
    void ClearBuffer();

    bool RequestSync();

    //! @brief 内部で使用する作業バッファのサイズを取得します。
    u32 GetRequiredWorkBufferSize() const;

    //! @brief チャンネル識別番号からプラットフォームごとに必要なチャンネル情報を取得します。
    internal::HioStream::ChannelType GetChannelInfo(internal::HioChannelType channel) const;

private: // メンバ変数
    // HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
    bool m_Interim_IsSoundPlayer;
#endif

    State m_State;                                          //!< セッションの状態です。
    PORT m_Port0;                                           //!< 内部で使用するポートです。
    PORT m_Port1;                                           //!< 内部で使用するポートです。

    internal::HioManager       m_HioManager;                //!< HostIOマネージャです。
    internal::HioAsyncChannel  m_SyncChannel;               //!< SYNC 用非同期チャンネルです。
    internal::HioSyncChannel   m_FuncChannel;               //!< 機能チャンネルです。
    internal::SyncReplyHandler m_SyncReplyHandler;          //!< SyncReplyPacket のハンドラです。

    u32                           m_SyncTimeout;            //!< SYNC タイムアウト時間です。
    snd::internal::fnd::StopWatch m_SyncStopWatch;          //!< SYNC 間隔を示すストップウォッチです。

    internal::SoundEditConnection  m_Connection;            //!< サウンド編集コネクションです。
    SoundArchiveEditor*            m_SoundArchiveEditor;    //!< サウンドアーカイブエディタです。

    friend class internal::SyncReplyHandler;
};

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#if defined(NW_PLATFORM_CTR)
#if NN_CURRENT_VERSION_NUMBER >= NN_VERSION_NUMBER(4,0,0,0)
#pragma diag_default 1301 // padding inserted in struct.
#pragma diag_default 2530 // padding added to end of struct.
#endif
#endif

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_SOUND_EDIT_SESSION_H_
