﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_RESOURCE_MANAGER_H_
#define NW_SND_EDIT_RESOURCE_MANAGER_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_LinkList.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>
#include <nw/snd/edit/res/sndedit_CacheManager.h>
#include <nw/snd/edit/res/sndedit_ResourceAllocator.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//---------------------------------------------------------------------------
//! @brief  リソースを管理するクラスです。
//---------------------------------------------------------------------------
class ResourceManager
{
private: // 定数
    static const size_t RESOURCE_ALLOCATION_THRESHOLD_SIZE = 64 * 1024;

public: // 型の定義
    //! @brief  リソースの状態です。
    enum RESOURCE_STATE
    {
        RESOURCE_STATE_UNKNOWN,     //!< 無効値
        RESOURCE_STATE_UPDATING,    //!< 更新中
        RESOURCE_STATE_REMOVING     //!< 削除中
    };

    class ErrorItem
    {
    public: // 型の定義
        //! @brief LinkList 用の参照クラスです。
        struct Reference
        {
            ErrorItem* value;            //!< Item です。
            ut::LinkListNode node;  //!< LinkListNode です。
        };
        typedef ut::LinkList<Reference, offsetof(Reference, node)> ReferenceList;

    public:
        ErrorItem();
        ~ErrorItem();

    public:
        const char* GetItemName() const { return m_ItemName; }
        void SetItemName(const char* itemName);

        char* AttachItemNameBuffer(char* itenNameBuffer, u32 itemNameLength);

        nw::snd::edit::Result GetResult() const { return m_Result; }
        void SetResult(nw::snd::edit::Result result) { m_Result = result; }

        Reference& GetReference() { return m_Reference; }

    private:
        char* m_ItemName;
        u32 m_ItemNameLength;
        nw::snd::edit::Result m_Result;

        Reference m_Reference;          //!< LinkList 用の参照です。
    };

private: // 型の定義
    typedef void (*CACHE_ACTION)(CacheManager& cacheManager, const char* name);

    //----------------------------------------------------------
    //! @brief  キャッシュ名関数オブジェクトへのアダプタです。
    template<class FunctorType>
    class ItemNameFunctorAdaptor
    {
        NW_DISALLOW_COPY_AND_ASSIGN(ItemNameFunctorAdaptor);

    public:
        explicit ItemNameFunctorAdaptor(FunctorType& functor) : m_Functor(functor) { }

    public:
        void operator()(CacheManager::Item& item)
        {
            m_Functor(item.GetName().GetName());
        }

    private:
        FunctorType& m_Functor;
    };

public: // コンストラクタ
    //! @brief  コンストラクタ
    ResourceManager();

    //! @brief  デストラクタ
    ~ResourceManager() {}

public: // メソッド
    //! @brief  初期化します。
    //! @param heap TBD
    //! @param baseID TBD
    //! @param maxItems TBD
    //! @param maxFiles TBD
    //! @param maxName TBD
    //! @param isMaxItemsExpandable TBD
    //! @param isMaxFilesExpandable TBD
    //! @return TBD
    Result Initialize(
        snd::internal::fnd::FrameHeap& heap,
        u32 baseID,
        u32 maxItems,
        u32 maxFiles,
        u32 maxName,
        bool isMaxItemsExpandable,
        bool isMaxFilesExpandable);

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  初期化の有無を取得します。
    //! @return TBD
    bool IsInitialized() const { return m_DataCacheHeap.IsInitialized(); }

    //! @brief  現在のメモリ使用量を取得します。
    //! @return TBD
    u32 GetMemoryUsage() const;

    //! @brief  アイテム数を取得します。
    //! @return TBD
    u32 GetItemInfoCount() const { return m_ItemInfoManager.GetItemCount(); }

    //! @brief  アイテム数の最大値を取得します。
    //! @return TBD
    u32 GetMaxItemInfoCount() const { return m_ItemInfoManager.GetMaxItemCount(); }

    //! @brief  アイテム編集情報を取得します。
    //! @param index TBD
    //! @param pItemInfo TBD
    //! @return TBD
    bool TryGetEditItemInfo(u32 index, EditItemInfo* pItemInfo) const;

    //! @brief  アイテム編集情報を取得します。
    //! @param name TBD
    //! @param pItemInfo TBD
    //! @return TBD
    bool TryGetEditItemInfo(const char* name, EditItemInfo* pItemInfo) const;

    //! @brief  アイテム編集情報を取得します。
    //! @param id TBD
    //! @param pItemInfo TBD
    //! @return TBD
    bool TryGetEditItemInfoFromID(u32 id, EditItemInfo* pItemInfo) const;

    //! @brief  ファイル数を取得します。
    //! @return TBD
    u32 GetFileCount() const { return m_FileManager.GetItemCount(); }

    //! @brief  ファイル数の最大値を取得します。
    //! @return TBD
    u32 GetMaxFileCount() const { return m_FileManager.GetMaxItemCount(); }

    //! @brief  アイテム辞書拡張の有無を取得します。
    //! @return TBD
    bool IsItemInfoDictionaryExpandable() const { return m_IsItemInfoDictionaryExpandable; }

    //! @brief  ファイル辞書拡張の有無を取得します。
    //! @return TBD
    bool IsFileDictionaryExpandable() const { return m_IsFileDictionaryExpandable; }

    //! @brief  指定した名前のアイテム情報の有無を確認します。
    //! @param name TBD
    //! @return TBD
    bool ContainsItemInfo(const char* name) const;

    //! @brief  指定した名前のアイテムIDを取得します。
    //! @param name TBD
    //! @param id TBD
    //! @return TBD
    bool TryGetItemID(const char* name, u32* id) const;

    //! @brief  指定した名前のアイテムのキャッシュ状態を取得します。
    //! @param name TBD
    //! @return TBD
    CacheState GetItemCacheState(const char* name) const;

    //! @brief  指定した名前のアイテム情報を取得します。
    //! @param name TBD
    //! @param itemInfoDataType TBD
    //! @param allowUnfreezedData TBD
    //! @return TBD
    const void* GetItemInfo(
        const char* name,
        ResDataType* itemInfoDataType = NULL,
        bool allowUnfreezedData = false) const;

    //! @brief 指定した名前のアイテムデータタイプを取得します。
    //! @param itemName TBD
    //! @param result TBD
    //! @return TBD
    ResDataType GetItemInfoDataType(const char* itemName, Result* result) const;

    //! @brief  指定した名前のアイテム情報を設定します。
    //! @param soundName TBD
    //! @param originalId TBD
    //! @param itemType TBD
    //! @param itemInfo TBD
    //! @return TBD
    Result SetItemInfo(const char* soundName, u32 originalId, ResDataType itemType, const void* itemInfo);

    //! @brief  指定したファイルデータを取得します。
    //! @param filePath TBD
    //! @param fileDataType TBD
    //! @param hashCode TBD
    //! @return TBD
    const void* GetFile(
        const char* filePath,
        ResDataType fileDataType,
        const Hash32** hashCode = NULL) const;

    //! @brief  指定したファイルの参照カウントをインクリメントします。
    //! @param filePath TBD
    //! @return TBD
    Result IncrementFileReferenceCount(const char* filePath);

    //! @brief  指定したファイルの参照カウントをデクリメントします。
    //! @param filePath TBD
    //! @return TBD
    Result DecrementFileReferenceCount(const char* filePath);

    //! @brief  新しいアイテム情報エントリーを追加ます。
    //! @param name TBD
    //! @param originalId TBD
    //! @return TBD
    Result NewItemInfoEntry(const char* name, u32 originalId);

    //! @brief  新しいファイルエントリーを作成します。
    //! @param filePath TBD
    //! @param fileSize TBD
    //! @param fileDataType TBD
    //! @param filebuffer TBD
    //! @param hashCode TBD
    //! @return TBD
    Result NewFileEntry(
        const char* filePath,
        u32 fileSize,
        ResDataType fileDataType,
        void** filebuffer,
        Hash32* hashCode = NULL);

    //! @brief  すべてのアイテム情報を削除します。
    void RemoveAllItemInfos();

    //! @brief  すべてのアイテムファイルを削除します。
    void RemoveAllItemFiles();

    //! @brief  すべての未使用アイテムファイルを削除します。
    void RemoveAllGarbageItemFiles();

    //! @brief  すべてのアイテムファイルの参照カウントをクリアします。
    void ClearAllItemFileReferenceCounts();

    //! @brief  指定した名前のアイテム情報と関連するファイルを削除します。
    //! @param name TBD
    //! @return TBD
    Result RemoveItemData(const char* name);

    //! @brief  指定した名前のアイテムに関連するファイルを削除します。
    //! @param name TBD
    //! @return TBD
    Result RemoveItemFiles(const char* name);

    //! @brief  指定した名前のアイテムに関連するファイルの参照カウントをデクリメントします。
    //! @param name TBD
    //! @return TBD
    Result DecrementReferenceCountForItemFiles(const char* name);

    //! @brief  指定ファイルデータを削除します。
    //! @param filePath TBD
    //! @return TBD
    void RemoveFile(const char* filePath);

    //! @brief  すべてのアイテムに指定関数オブジェクトを適用します。
    //! @param functor TBD
    //! @tparam FunctorType TBD
    template<class FunctorType>
    void ForEachItemInfo(FunctorType& functor)
    {
        ItemNameFunctorAdaptor<FunctorType> adaptor(functor);
        m_ItemInfoManager.ForEach(adaptor);
    }

    //! @brief  最初の非フリーズアイテム名を取得します。
    //! @param state TBD
    //! @return TBD
    const char* GetFirstUnfreezedItemName(RESOURCE_STATE* state = NULL) const;

    //! @brief  指定した名前のアイテム情報をフリーズします。
    //! @param itemName TBD
    //! @param state TBD
    //! @return TBD
    Result FreezeItemInfo(const char* itemName, RESOURCE_STATE* state = NULL);

    //! @brief  すべてのアイテム情報のフリーズ状態を解除します。
    //! @param state TBD
    //! @return TBD
    Result UnfreezeAllItemInfos(RESOURCE_STATE state);

    //! @brief  指定したアイテム情報のフリーズ状態を解除します。
    //! @param itemName TBD
    //! @param state TBD
    //! @return TBD
    Result UnfreezeItemInfo(const char* itemName, RESOURCE_STATE state);

    //! @brief  すべてのアイテムを対象とした状態を取得します。
    //! @return TBD
    RESOURCE_STATE GetAllItemsState() const { return static_cast<RESOURCE_STATE>(m_AllItemsState); }

    //! @brief  すべてのアイテムを対象とした状態を設定します。
    //! @param value TBD
    void SetAllItemsState(RESOURCE_STATE value) { m_AllItemsState = value; }

    //! @brief  アイテム情報辞書を更新します。
    void UpdateItemInfoDictionary();

    //! @brief  ファイル辞書を更新します。
    void UpdateFileDictionary();

    //! @briefprivate
    //! @param itemName :private
    //! @param result :private
    //! @return :private
    ErrorItem* CreateErrorItem(const char* itemName, nw::snd::edit::Result result);

    //! @briefprivate
    //! @param item :private
    void DestroyErrorItem(ErrorItem* item);

    //! @briefprivate
    //! @param item :private
    void PushErrorItem(ErrorItem* item);

    //! @briefprivate
    //! @return :private
    ErrorItem* PopErrorItem();

    //! @briefprivate
    void DestroyAllErrorItems();

private: // メソッド
    //! @brief  アイテム情報管理に必要なメモリサイズを取得します。
    u32 GetItemInfoManagerRequiredMemorySize(u32 maxItems, u32 maxFiles, u32 maxName) const;

    //! @brief  アイテム情報管理クラスを初期化します。
    Result InitializeItemInfoManager(u32 baseID, u32 maxEntries, u32 maxName);

    //! @brief  ファイル管理クラスを初期化します。
    Result InitializeFileManager(u32 maxEntries);

    //! @brief  アイテム情報辞書を拡張します。
    Result RebuildItemInfoDictionary(u32 maxEntries);

    //! @brief  ファイル辞書を拡張します。
    Result RebuildFileDictionary(u32 maxEntries);

    //! @brief  アイテム情報データのバッファサイズを取得します。
    u32 GetItemInfoDataBufferLength(u32 maxEntries, u32 maxName, u32 maxDataLength) const;

    //! @brief  指定したアイテム名に関連するすべてのファイルキャッシュに対して指定アクションを実行します。
    Result ForeachFileCache(const char* name, CACHE_ACTION action);

    //! @briefprivate
    u32 GetErrorItemHeapRequiredMemorySize(u32 maxName) const;

private: // メンバ変数
    CacheManager                     m_ItemInfoManager;             //!< アイテム情報の管理クラスです。
    CacheManager                     m_FileManager;                 //!< アイテムファイル情報の管理クラスです。
    void*                            m_pMemoryForItemInfoManager;   //!< @briefprivate
    void*                            m_pMemoryForFileManager;       //!< @briefprivate

    ErrorItem::ReferenceList         m_ErrorItemList;               //!< @briefprivate
    void*                            m_pMemoryForErrotItemList;     //!< @briefprivate

    snd::internal::fnd::ExpandedHeap m_DataCacheHeap;        //!< データキャッシュ用のヒープです。
    snd::internal::fnd::ExpandedHeap m_ErrorItemHeap;        //!< エラーアイテム用のヒープです。
    ResourceAllocator                m_ResourceAllocator;    //!< アイテム情報のアロケータです。(m_DataCacheHeap の後ろから割り当て）

    u32  m_AllItemsState;                     //!< すべてのアイテムを対象とした状態です。
    u32  m_MaxName;
    bool m_IsItemInfoDictionaryExpandable;    //!< アイテム情報辞書拡張の有無です。
    bool m_IsFileDictionaryExpandable;        //!< ファイル辞書拡張の有無です。
    u16  m_Padding;
};


} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_RESOURCE_MANAGER_H_
