﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_CACHE_MANAGER_H_
#define NW_SND_EDIT_CACHE_MANAGER_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_LinkList.h>
#include <nw/snd/edit/sndedit_Types.h>
#include <nw/snd/edit/sndedit_Result.h>
#include <nw/snd/edit/res/sndedit_ResTypes.h>
#include <nw/snd/edit/util/sndedit_PatriciaDictionary.h>

namespace nw {
namespace snd {

namespace internal {
namespace fnd {
class IAllocator;
}
}

namespace edit {
namespace internal {

class CacheManager;

//---------------------------------------------------------------------------
//! @brief    アイテムデータを管理するクラスです。
//---------------------------------------------------------------------------
class CacheManager
{
private:
    static u32 INVALID_ID;

public: // 型の定義
    //---------------------------------------------------------------------------
    //! @brief  初期化パラメータです。
    //---------------------------------------------------------------------------
    struct InitializeArgs
    {
        InitializeArgs();

        void*                           buffer;                //!< CacheManager が使用するバッファです。
        u32                             bufferLength;          //!< CacheManager が使用するバッファの長さです。
        snd::internal::fnd::IAllocator* dataBufferAllocator;
        u32                             baseID;
        s32                             maxEntries;
        u32                             maxName;
    };

    //---------------------------------------------------------------------------
    //! @brief  アイテム情報を管理するクラスです。
    //!         アイテム名、アイテムデータは Item クラスの直後に続きます。
    //!         →Item - ResName(アイテム名) - (データ)
    //---------------------------------------------------------------------------
    class Item
    {
    public: // 型の定義
        //! @brief LinkList 用の参照クラスです。
        struct Reference
        {
            Item* value;            //!< Item です。
            ut::LinkListNode node;  //!< LinkListNode です。
        };

        typedef ut::LinkList<Reference, offsetof(Reference, node)> ReferenceList;

    public: // コンストラクタ
        Item(
            CacheManager& owner,
            u32 id,
            ResName& name,
            bool isOverrideOriginal,
            u32 bufferSize,
            u32 dataSize,
            const void* data);

    public: // メソッド
        CacheManager& GetOwner() { return m_Owner; }

        const CacheManager& GetOwner() const { return m_Owner; }

        bool IsFreezed() const { return m_IsFreezed; }

        bool IsOverrideOriginal() const { return m_IsOverrideOriginal; }

        u32 GetID() const
        {
            return m_ID;
        }

        const ResName& GetName() const
        {
            const ResName* result = m_Name.GetResName();
            NW_ASSERT_NOT_NULL(result);
            return *result;
        }

        u32 GetDataType() const { return m_DataType; }

        void SetDataType(u32 value)
        {
            NW_ASSERTMSG(!m_IsFreezed, "Data is already freezed.\n");
            m_DataType = value;
        }

        void* GetData()
        {
            NW_ASSERTMSG(!m_IsFreezed, "Data is already freezed.\n");
            return m_DataOffset.to_ptr();
        }

        const void* GetData() const
        {
            return m_DataOffset.to_ptr();
        }

        //! @brief  データサイズを取得します。
        //! @return TBD
        u32 GetDataSize() const { return m_DataSize; }

        //! @brief  キャシュのバッファサイズを取得します。
        //! @return TBD
        u32 GetBufferSize() const { return m_BufferSize; }

        //! @brief  参照カウントを取得します。
        //! @return TBD
        u32 GetReferenceCount() const { return m_ReferenceCount; }

        //! @brief  データのハッシュ値を取得します。
        //! @return TBD
        const Hash32& GetHashCode() const { return m_HashCode; }

        //! @brief  データのハッシュ値を設定します。
        //! @param value TBD
        void SetHashCode(Hash32& value)
        {
            NW_ASSERTMSG(!m_IsFreezed, "Data is already freezed.\n");
            m_HashCode = value;
        }

        //! @brief  ユーザーパラメータを取得します。
        //! @return TBD
        u32 GetUserParameter() const { return m_UserParameter; }

        //! @brief  ユーザーパラメータを設定します。
        //! @param value TBD
        void SetUserParameter(u32 value) { m_UserParameter = value; }

        //! @brief  データを固定化します。
        void Freeze();

        //! @brief  データの固定化を解除します。
        void Unfreeze();

        //! @brief  参照カウントをインクリメントします。
        void IncrementReferenceCount();

        //! @brief  参照カウントをデクリメントします。
        void DecrementReferenceCount();

        //! @brief  参照カウントをクリアします。
        void ClearReferenceCount();

        //! @brief  キャッシュを破棄します。
        void Dispose();

        //! @brief  LinkList 用の参照を取得します。
        //! @return TBD
        Reference& GetReference() { return m_Reference; }

        //! @brief  非フリーズデータ LinkList 用の参照を取得します。
        //! @return TBD
        Reference& GetUnfreezedReference() { return m_UnfreezedReference; }

        //! @brief  未使用データ LinkList 用の参照を取得します。
        //! @return TBD
        Reference& GetUnusedReference() { return m_UnusedReference; }

    private: // メンバ変数
        CacheManager& m_Owner;

        bool      m_IsFreezed;          //!< データの固定化フラグです。
                                        //!< データを後から設定するために利用します。
        bool      m_IsDisposed;         //!< キャッシュが破棄されたことを示します。
        bool      m_IsOverrideOriginal;      //!< オリジナルデータ オーバーライドの有無を保持します。
        u8        m_Padding;

        u32       m_ID;                 //!< アイテムIDです。
        BinString m_Name;               //!< アイテム名です。
        u32       m_ReferenceCount;     //!< 参照カウントです。
        u32       m_DataType;           //!< データ種別です。
        Offset    m_DataOffset;         //!< Item, ResName(アイテム名) に続くデータへのオフセットです。
        u32       m_DataSize;           //!< データの長さです。
        u32       m_BufferSize;         //!< Item, ResName(アイテム名), データをまとめたバッファの長さです。
        Hash32    m_HashCode;           //!< データのハッシュ値です。
        u32       m_UserParameter;      //!< ユーザーパラメータです。

        Reference m_Reference;          //!< LinkList 用の参照です。
        Reference m_UnfreezedReference; //!< 非フリーズデータ LinkList 用の参照です。
        Reference m_UnusedReference;    //!< 未使用データ LinkList 用の参照です。
    };

    typedef bool (*FILTER_FUNC)(Item& item);

    typedef void (*ACTION)(Item& item);

public: // コンストラクタ
    //! @brief  コンストラクタ
    CacheManager();

    //! @brief  デストラクタ
    ~CacheManager() {}

public: // メソッド
    //! @brief    初期化します。
    //! @details  maxEntries と maxDataSize は 両方 0 以上か、両方 0 以下である必要があります。
    //!           両方 0 以下の場合は、一切キャッシュされません。
    //! @param args TBD
    //! @return   TBD
    Result Initialize(const InitializeArgs& args);

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  初期化の有無を調べます。
    //! @return TBD
    bool IsInitialized() const { return m_DataBufferAllocator != NULL; }

    //! @brief  必要なメモリサイズを取得します。
    //! @param maxEntries TBD
    //! @return TBD
    u32 GetRequiredMemorySize(u32 maxEntries) const;

    //! @brief  アイテムを作成するのに必要なメモリサイズを取得します。
    //! @param maxName TBD
    //! @param maxDataLength TBD
    //! @return TBD
    u32 GetRequiredMemorySizeForItem(u32 maxName, u32 maxDataLength) const;

    //! @brief  現在のメモリ使用量を取得します。
    //! @return 現在のメモリ使用量を返します。
    u32 GetMemoryUsage() const { return m_MemoryUsage; }

    //! @brief  空きIDを取得します。
    //! @param id TBD
    //! @return TBD
    bool TryGetFreeID(u32* id) const;

    //! @brief  キャッシュされているデータを取得します。
    //! @param name TBD
    //! @param dataType TBD
    //! @param allowUnfreezedData TBD
    //! @return TBD
    const void* GetData(
        const char* name,
        u32* dataType = NULL,
        bool allowUnfreezedData = false) const;

    //! @brief  キャッシュアイテムを取得します。
    //! @param name TBD
    //! @return TBD
    Item* GetItem(const char* name);

    //! @brief  キャッシュアイテムを取得します。
    //! @param name TBD
    //! @return TBD
    const Item* GetItem(const char* name) const;

    //! @brief  キャッシュアイテムを取得します。
    //! @param index TBD
    //! @return TBD
    const Item* GetItem(u32 index) const;

    //! @brief  キャッシュアイテムを取得します。
    //! @param id TBD
    //! @return TBD
    const Item* GetItemFromID(u32 id) const;

    //! @brief  指定データを追加します。
    //! @param id TBD
    //! @param name TBD
    //! @param isOverrideOriginal TBD
    //! @param dataType TBD
    //! @param data TBD
    //! @param dataSize TBD
    //! @return 結果を返します。
    Result AddItem(
        u32 id,
        const char* name,
        bool isOverrideOriginal,
        u32 dataType,
        const void* data,
        u32 dataSize);

    //! @brief  CreateItem() で作成したアイテムを追加します。
    //! @param item TBD
    //! @return TBD
    Result AddItem(Item& item);

    //! @brief  指定した名前のアイテムを削除します。
    //! @param name TBD
    //! @param action TBD
    //! @return TBD
    Result RemoveItem(const char* name, ACTION action = NULL);

    //! @brief  すべてのアイテムを削除します。
    //! @param action TBD
    void RemoveAllItems(ACTION action = NULL);

    //! @brief  すべての未使用アイテムを削除します。
    void RemoveGarbages();

    //! @brief  すべてのアイテムの参照カウントを 0 に設定します。
    void ClearAllItemReferenceCounts();

    //! @brief  すべてのアイテムに指定関数オブジェクトを適用します。
    //! @param functor TBD
    template<class FunctorType>
    void ForEach(FunctorType& functor)
    {
        for(Item::ReferenceList::Iterator it = m_Items.begin(); it != m_Items.end(); ++it)
        {
            NW_ASSERT_NOT_NULL(it->value);
            functor(*it->value);
        }
    }

    //! @brief  アイテムの変更を辞書に反映します。
    void UpdateDictionary();

    //! @brief  辞書を再構築します。
    //! @param buffer TBD
    //! @param bufferLength TBD
    //! @param maxEntries TBD
    //! @param oldBuffer TBD
    //! @return TBD
    Result RebuildDictionary(void* buffer, u32 bufferLength, s32 maxEntries, void** oldBuffer);

    //! @brief  アイテムエントリーを作成します。
    //!         この関数で作成したアイテムのデータはフリーズするまでの間、編集が可能です。
    //! @param item TBD
    //! @param id TBD
    //! @param name TBD
    //! @param isOverrideOriginal TBD
    //! @param dataSize TBD
    //! @return TBD
    Result CreateItemEntry(Item*& item, u32 id, const char* name, bool isOverrideOriginal, u32 dataSize)
    {
        return CreateItem(item, id, name, isOverrideOriginal, NULL, dataSize);
    }

    //! @brief  アイテムを作成します。
    //! @param item TBD
    //! @param id TBD
    //! @param name TBD
    //! @param isOverrideOriginal TBD
    //! @param data TBD
    //! @param dataSize TBD
    //! @return TBD
    Result CreateItem(Item*& item, u32 id, const char* name, bool isOverrideOriginal, const void* data, u32 dataSize);

    //! @brief  アイテム数を取得します。
    //! @return TBD
    u32 GetItemCount() const { return m_ValidItemCount; }

    //! @brief  アイテム数の最大値を取得します。
    //! @return TBD
    u32 GetMaxItemCount() const { return m_MaxItemCount; }

    //! @brief  最初の非フリーズアイテムを取得します。
    //! @param[in] filter TBD
    //! @return TBD
    Item* GetFirstUnfreezedItem(FILTER_FUNC filter = NULL) const;

    //! @brief  非フリーズアイテム数を取得します。
    //! @return TBD
    u32 GetUnfreezedItemCount() const { return m_UnfreezedItems.GetSize(); }

    //! @brief  最初の未使用フリーズアイテムを取得します。
    //! @param filter TBD
    //! @return TBD
    Item* GetFirstUnusedItem(FILTER_FUNC filter = NULL) const;

    //! @brief  未使用アイテム数を取得します。
    //! @return TBD
    u32 GetUnusedItemCount() const { return m_UnusedItems.GetSize(); }

private: // メソッド
    u32 GetIDTableSize(u32 maxEntries) const
    {
        return sizeof(Item*) * maxEntries;
    }

    //! @brief  パトリシア木辞書を取得します。
    PatriciaDictionary* GetPatriciaDictionary() const
    {
        return reinterpret_cast<PatriciaDictionary*>(m_ItemDictinoaryData);
    }

    inline Result CanAddItem() const;

    s32 FindEmptyNodeIndex();

    //! @brief  アイテムを取得します。
    Item* GetItemImpl(const char* name) const;

    //! @brief  指定したアイテムを削除します。
    void RemoveItem(PatriciaDictionary::Node& node, ACTION action = NULL);

    //! @brief  アイテムを破棄します。
    void DisposeItem(Item& item, ACTION action = NULL);

    void DumpItems(const char* tag) const;

private: // メンバ変数
    snd::internal::fnd::IAllocator* m_DataBufferAllocator;  //!< アイテムデータバッファのアロケータです。
    PatriciaDictionaryData*         m_ItemDictinoaryData;   //!< アイテム辞書データです。
    s32                             m_MaxItemCount;         //!< アイテム数の最大値です。
    s32                             m_ValidItemCount;       //!< 有効なアイテム数です。
    u32                             m_MaxName;              //!< 名前の長さの最大値です。
    u32                             m_MemoryUsage;          //!< メモリ使用量です。

    mutable Item::ReferenceList     m_Items;                //!< アイテムリストです。
    Item::ReferenceList             m_UnfreezedItems;       //!< 非フリーズアイテムリストです。
    Item::ReferenceList             m_UnusedItems;          //!< 未使用アイテムリストです。

    u32                             m_BaseID;               //!< インデックス化のための基準アイテムIDです。
    Item**                          m_IDTable;              //!< アイテムIDテーブルです。
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_CACHE_MANAGER_H_
