﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SND_EDIT_HIO_CHANNEL_H_
#define NW_SND_EDIT_HIO_CHANNEL_H_

#include <nw/snd/snd_Config.h>
#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_LinkList.h>
#include <nw/snd/edit/hio/sndedit_HioStream.h>
#include <nw/snd/edit/hio/sndedit_HioPacketStream.h>
#include <nw/snd/edit/hio/sndedit_HioMessageHandler.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

class HioManager;

//---------------------------------------------------------------------------
//! @brief    HIOチャンネルを示す基本クラスです。
//---------------------------------------------------------------------------
class HioChannel
{
#if !defined(NW_SND_EDIT_USE_MCS)
public:
    static const u32 DEFAULT_WORK_BUFFER_LENGTH =  1 * 1024;   //!< HIO通信で使用する作業バッファの大きさのデフォルト値です。基本的に変更する必要はありません。
#endif

private:
    typedef ut::LinkList<HioMessageHandler::Reference, offsetof(HioMessageHandler::Reference, node)> HioMessageHandlerRefList;

public:
    //! @brief LinkList 用の参照クラスです。
    struct Reference
    {
        HioChannel* value;      //!< HioChannel です。
        ut::LinkListNode node;  //!< LinkListNode です。
    };

    //! @brief HioManager と HioChannel の関連付け用クラスです。
    class HioManagerLinker
    {
    private:
        static void SetHioManager(HioChannel& channel, HioManager* hioManager)
        {
            channel.m_HioManager = hioManager;
        }

        friend class HioManager;
    };

protected:
    /* ctor */ HioChannel();
    /* dtor */ virtual ~HioChannel();

public:
    //! @brief  初期化します。
    //! @param[in] recvStreamBuffer 受信ストリームバッファです。
    //! @param[in] recvStreamBufferLength 受信ストリームバッファのサイズです。
    //! @param[in] recvPacketBuffer 受信パケットバッファです。
    //! @param[in] recvPacketBufferLength 受信パケットバッファのサイズです。
    //! @return 結果を返します。
#if !defined(NW_SND_EDIT_USE_MCS)
    //! @param[in] workBuffer 作業バッファです。
    //! @param[in] workBufferLength 作業バッファのサイズです。通常は DEFAULT_WORK_BUFFER_LENGTH を指定します。
    HioResult Initialize(
        void* recvStreamBuffer,
        u32 recvStreamBufferLength,
        void* recvPacketBuffer,
        u32 recvPacketBufferLength,
        void* workBuffer,
        u32 workBufferLength
    );
#else
    HioResult Initialize(
        void* recvStreamBuffer,
        u32 recvStreamBufferLength,
        void* recvPacketBuffer,
        u32 recvPacketBufferLength);
#endif

    //! @brief  終了処理を行います。
    void Finalize();

    //! @brief  初期化の有無を取得します。
    //! @return 初期化済みであれば true を返します。
    bool IsInitialized() const { return m_RecvStreamBufferLength > 0; }

    //! @brief  チャンネルが開かれているかどうかを取得します。
    //! @return チャンネルが開かれていれば true を返します。
    bool IsOpened() const { return m_Stream.IsAvailable(); }

    //! @brief  非同期チャンネルかどうかを取得します。
    //! @return 非同期チャンネルであれば true を返します。
    virtual bool IsAsync() const = 0;

    //! @brief  指定チャンネルを開きます。
    //! @param[in] channel チャンネルの指定です。
    //! @return 開くことができた場合は true を返します。
    bool Open(HioStream::ChannelType channel);

    //! @brief  開かれているチャンネルを閉じます。
    void Close();

    //! @brief  ストリームを取得します。
    //! @return ストリームへの参照を返します。
    HioPacketStream& GetStream() { return m_PacketStream; }

    //! @brief  バッファをクリアします。
    void ClearBuffer();

    //! @brief  現在のメモリ使用量を取得します。
    //! @return 現在のメモリ使用量を返します。
    u32 GetMemoryUsage() const
    {
        return m_RecvStreamBufferLength + m_RecvPacketBufferLength;
    }

    //! @brief  ポーリングします。
    //! @return チャンネルが切断された時は false を返します。
    bool Polling();

    //---------------------------------------------------------------------------
    //! @brief         LinkList 用の参照を取得します。
    //!
    //! @details       この関数の戻り値である HioChannel::Reference を利用して、
    //!                HioChannel を格納する LinkList を構築できます。
    //!
    //! @return        HioChannel::Reference を返します。
    //---------------------------------------------------------------------------
    Reference& GetReference() { return m_Reference; }

protected:
    HioManager* GetHioManager() { return m_HioManager; }

    HioPacket* GetRecvPacketBuffer() { return m_RecvPacketBuffer; }

    u32 GetRecvPacketBufferLength() const { return m_RecvPacketBufferLength; }

private:
    HioStream        m_Stream;              //!< 送受信用ストリームです。
    HioPacketStream  m_PacketStream;        //!< パケット送受信用ストリームです。

    void* m_RecvStreamBuffer;               //!< 受信用ストリームのバッファです。
    u32   m_RecvStreamBufferLength;         //!< 受信用ストリームのバッファサイズです。

    HioPacket* m_RecvPacketBuffer;          //!< 受信用 HioPacket のバッファです。
    u32        m_RecvPacketBufferLength;    //!< 受信用 HioPacket のバッファサイズです。

    HioManager* m_HioManager;               //!< 関連する HioManager です。
    Reference   m_Reference;                //!< LinkList 用の参照です。
};

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV

#endif // NW_SND_EDIT_HIO_CHANNEL_H_
