﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * :include nw/snd/dw/snddw_SoundWindowSystem.h
 *
 * @file snddw_SoundWindowSystem.h
 */

#ifndef NW_SND_DW_SOUND_WINDOW_SYSTEM_H_
#define NW_SND_DW_SOUND_WINDOW_SYSTEM_H_

#ifdef NW_PLATFORM_CAFE
#include <nw/dev/cafe/dev_PadCafe.h>
#else
#include <nw/dev/win/dev_PadWin.h>
#endif
#include <nw/font/font_ResFont.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/dw/window/dw_WindowManager.h>
#include <nw/dw/system/dw_NwUIRenderer.h>
#include <nw/dw/system/dw_NwTextRenderer.h>
#include <nw/snd/dw/snddw_Config.h>
#include <nw/snd/dw/system/snddw_Allocator.h>

namespace nw {
namespace snd {
namespace dw {
    class SoundWindow;
}
}
}

namespace nw {
namespace snd {
namespace dw {

//---------------------------------------------------------------------------
//! @brief  サウンドデバッグウィンドウを管理するクラスです。
//!
//!         サウンドデバッグウィンドウを利用する場合まずこのクラスのオブジェクトを生成し、必要なウィンドウを登録します。
//!
//!         このクラスのオブジェクト初期化は下記の手順で行います。
//!         - インスタンスを生成します。
//!         - @ref GetRequiredSize で必要メモリを取得し、作成したバッファや、フォントバイナリなど初期化に必要な情報を@ref Param 構造体に格納します。
//!         - @ref Initialize に @ref Param 構造体を渡します。
//!
//!         利用したいサウンドデバッグウィンドウは@ref AddWindow で登録することで、各ウィンドウが管理されるようになります。
//!
//!         管理された各ウィンドウには、以下の操作を行うことができます。
//!
//!         - ウィンドウリストを出す              : (修飾キー) + ↑ or ↓
//!         - ウィンドウリスト上の操作方法
//!           - カーソル移動                      : (修飾キー) + ↑ or ↓
//!           - 描画順序の変更                    : (修飾キー) + A + ↑ or ↓
//!           - 表示/非表示の切り替え             : (修飾キー) + B
//!
//!         - ポインター移動                      : (修飾キー) + 左スティック（+ B で移動速度アップ）
//!         - ウィンドウの透過率の変更            : (修飾キー) + 右スティックの上下
//!         - 対象ウィンドウ上での操作方法
//!           - ウィンドウのリサイズ                       : (修飾キー) + X + 左スティック
//!           - ウィンドウシェード (タイトルバーだけ出す)  : (修飾キー) + Y
//!           - ウィンドウの場所を変える                   : (修飾キー) + A + 左スティック or 十字ボタン
//!
//!         - 全ウィンドウの表示/非表示の切り替え : (修飾キー) + ZR
//!
//!         デフォルトの修飾キーは R です。
//!
//!         具体的な使い方については、サンプルデモを確認してください。
//!
//! @see VoicesWindow クラス
//! @see PerformanceWindow クラス
//! @see MasterOutputWindow クラス
//! @see SoundWindowSystem::SetModifierKey
//! @see SoundWindowSystem::GetModifierKey
//!
//! @date 2012/03/09 SetModifierKey、GetModifierKeyを追加
//! @date 2012/03/09 キー操作に関する説明を修正
//! @date 2012/02/15 初版
//---------------------------------------------------------------------------
class SoundWindowSystem
{
public:
    //---------------------------------------------------------------------------
    //! @brief  SoundWindowSystem の初期化に使用するパラメータの構造体です。
    //!
    //! @date 2012/03/09 フォントバイナリの説明を修正
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    struct Param
    {
        //---------------------------------------------------------------------------
        //! @brief SoundWindowSystem に渡すメモリの先頭アドレスです。
        //---------------------------------------------------------------------------
        void* memory;

        //---------------------------------------------------------------------------
        //! @brief SoundWindowSystem に渡すメモリのサイズです。
        //---------------------------------------------------------------------------
        u32 memorySize;

        //---------------------------------------------------------------------------
        //! @brief フォントバイナリです。
        //!        内部でフォントバイナリの一部を書き換えるため、constではありません。
        //---------------------------------------------------------------------------
        void* fontBinary;

        //---------------------------------------------------------------------------
        //! @brief フォントバイナリのサイズです。
        //---------------------------------------------------------------------------
        u32 fontBinarySize;

        //---------------------------------------------------------------------------
        //! @brief ウィンドウの幅です。
        //---------------------------------------------------------------------------
        f32 width;

        //---------------------------------------------------------------------------
        //! @brief ウィンドウの高さです。
        //---------------------------------------------------------------------------
        f32 height;
    };

    //! @name コンストラクタ/デストラクタ
    //@{
public:
    //---------------------------------------------------------------------------
    //! @brief    コンストラクタです。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    SoundWindowSystem();

    //---------------------------------------------------------------------------
    //! @brief    デストラクタです。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    ~SoundWindowSystem();
    //@}

public:
    //---------------------------------------------------------------------------
    //! @brief    サウンドデバッグウィンドウシステムの開始処理を行います。
    //!
    //! @param param     システムの初期化に使用するパラメータです。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void Initialize(const Param& param);

    //---------------------------------------------------------------------------
    //! @brief    サウンドデバッグウィンドウシステムの終了処理を行います。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief    SoundWindowSystem を作るために必要なメモリを取得します。
    //!
    //! @return 必要とするメモリです。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    u32  GetRequiredSize() const;

    //---------------------------------------------------------------------------
    //! @brief    デバッグコントローラなどの入力の更新を行います。
    //!
    //! @param padStatus   デバッグコントローラの状態です。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void UpdateInputs(const nw::dev::RawPadStatus& padStatus);

    //---------------------------------------------------------------------------
    //! @brief    更新処理を行います。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void Update();

    //---------------------------------------------------------------------------
    //! @brief    描画処理を行います。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void Draw();

    //---------------------------------------------------------------------------
    //! @brief    システムにウィンドウを追加します。
    //!
    //! @param posX   追加するウィンドウの初期X座標です。
    //! @param posY   追加するウィンドウの初期Y座標です。
    //! @param window   追加するウィンドウです。
    //!
    //! @date 2012/02/15 初版
    //---------------------------------------------------------------------------
    void AddWindow(f32 posX, f32 posY, SoundWindow& window);

    //---------------------------------------------------------------------------
    //! @brief    修飾キーを設定します。
    //!
    //!           修飾キーは nw::dev::Pad::PadMask 列挙型の値の論理和で指定します。
    //!           ここに指定したキーすべてが押されているかどうかを判定します。
    //!
    //! @param modifier   設定する修飾キーです。
    //!
    //! @date 2012/03/09 初版
    //---------------------------------------------------------------------------
    void SetModifierKey(u32 modifier);

    //---------------------------------------------------------------------------
    //! @brief    修飾キーを取得します。
    //!
    //! @return 設定されている修飾キーです。
    //!
    //! @date 2012/03/09 初版
    //---------------------------------------------------------------------------
    u32 GetModifierKey() const;

private:
    void InitializePrimitiveRenderer(const Param& param);
    void FinalizePrimitiveRenderer();
    void InitializeTextRenderer(const Param& param);
    void FinalizeTextRenderer();
    void InitializeUIRenderer(const Param& param);
    void FinalizeUIRenderer();

private:
    friend void Config::Seal();

public:

    //---------------------------------------------------------------------------
    //! @brief    ファイルアクセスを行うときの最小のアラインメントです。
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    static const int BUFFER_MIN_ALIGNMENT = 64;
#else
    static const int BUFFER_MIN_ALIGNMENT = 32;
#endif

private:
    nw::internal::dw::WindowManager m_DWMgr;
    nw::font::ResFont m_Font;
    nw::dev::PrimitiveRenderer* m_pRenderer;
    nw::internal::dw::NwUIRenderer m_UIRenderer;
    nw::internal::dw::NwTextRenderer m_TextRenderer;
    internal::Allocator m_Allocator;
#ifdef NW_PLATFORM_CAFE
    nw::dev::PadCafe m_Pad;
    GX2ContextState* m_ContextState;
#else
    nw::dev::PadWin m_Pad;
#endif
    nw::dev::PadStatus m_PadStatus;
    u32 m_FontShaderBinarySize;
    u32 m_FontBinarySize;
    void* m_FontShaderBinary;
    void* m_FontBinary;
    nw::math::Matrix44 m_ProjectionMatrix;
    nw::math::Matrix34 m_ViewMatrix;
};

} // dw
} // snd
} // nw

#endif // NW_SND_DW_SOUND_WINDOW_SYSTEM_H_
