﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_SCFONT_TEXTURE_CACHE_H_
#define NW_SCFONT_TEXTURE_CACHE_H_

#include <nw/types.h>
#include <nw/ut/ut_LinkList.h>
#include <nw/ut/ut_Memory.h>
#include <nw/font/font_Font.h>
#include <nw/scfont/scfont_GlyphTreeMap.h>

namespace scffnd
{
    class ScalableFontEngine;
    typedef s32 ScFixed;
}

namespace nw
{
namespace scfont
{

//---------------------------------------------------------------------------
//! @brief スケーラブルフォントのテクスチャキャッシュを表すクラスです。
//!
//! @details
//! 複数のスケラーブルフォントクラスが一つのテクスチャキャッシュを
//! 利用することができます。その場合、複数のサイズのフォントが一つの
//! テクスチャキャッシュに混在することになります。
//---------------------------------------------------------------------------
class TextureCache
{
public:
    //! @brief フォントデータの形式を表す列挙型です。
    //!
    enum FontDataType
    {
        FONT_DATA_TYPE_TTF,                         //!< ttfそのままのデータです。
        FONT_DATA_TYPE_OTF = FONT_DATA_TYPE_TTF,    //!< otfそのままのデータです。
        FONT_DATA_TYPE_BFTTF                        //!< bfttf形式のデータです。
    };

    //! @brief ラインの種類を表す列挙型です。
    //!
    enum LineHeight
    {
        LINE_HEIGHT_16 = 4,     //!< 高さが16のラインです。
        LINE_HEIGHT_32 = 5,     //!< 高さが32のラインです。
        LINE_HEIGHT_64 = 6,     //!< 高さが64のラインです。
        LINE_HEIGHT_128 = 7,    //!< 高さが128のラインです。
        LINE_HEIGHT_256 = 8,    //!< 高さが256のラインです。
        LINE_HEIGHT_512 = 9     //!< 高さが512のラインです。
    };

    //----------------------------------------------------
    //! @name コンストラクタ／デストラクタ/初期化/終了処理
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 実質的な初期化はInitializeで行われます。ここでは、
    //! メンバ変数の初期化のみ行っています。
    //!
    TextureCache();

    //! @brief デストラクタです。
    //!
    //! @details
    //! このクラスでは何も行いません。メモリやWindowsでのGLのオブジェクトの
    //! 破棄を行うにはFinalizeを呼び出す必要がありますので、ご注意ください。
    //!
    virtual ~TextureCache();

    //! @brief 初期化のパラメータを指定するための構造体です。
    //!
    struct InitializeArg {
        //! テクスチャキャッシュの横幅です。必ず2の累乗としてください。
        //! デフォルトでは1024です。
        u32 textureCacheWidth;
        //! テクスチャキャッシュの縦幅です。
        //! デフォルトでは1024です。
        u32 textureCacheHeight;
        //! メモリを確保するためのアロケータです。メモリの確保はInitializeの中のみで行われます。
        //! 確保したメモリはFinalizeで解放されます。
        //! デフォルト値はありません。必ず指定してください。
        nw::ut::IAllocator* allocator;
        //! フォントデータへのポインタの配列を渡します。(ポインタが示す先はttf(otf)ファイルもしくはbfttfファイルのバイナリデータです。)
        //! 要素数はfontNumに等しくなるようにしてください。
        //! デフォルト値はありません。必ず指定してください。
        void** fontDatas;
        //! フォントデータのサイズの配列を渡します。
        //! 要素数はfontNumに等しくなるようにしてください。
        //! このメンバは、フォントの形式がFONT_DATA_TYPE_BFTTFのときのみ設定する必要があります。
        //! デフォルト値はNULLになっています。
        u32* fontDataSizes;
        //! ボールドエフェクトの重み値です。
        //! 0.0 ～ 1.0 の間で指定します。NULL を設定するとすべて 0.0 が入力されたと扱います。
        //! デフォルト値はNULLになっています。
        f32* boldWeights;
        //! 縁取りの幅です。
        //! 0 ～ 10 の間で指定します。NULL を設定するとすべて 0 が入力されたと扱います。
        //! デフォルト値はNULLになっています。
        u8* borderWidths;
        //! フォントデータの形式を指定します。
        //! NULLを指定すると、全てのフォントがFONT_DATA_TYPE_TTFであると見なします。
        //! デフォルト値はNULLになっています。
        FontDataType* fontDataTypes;
        //! 扱うフォントの数です。1～32の数を渡してください。
        //! デフォルトでは1です。
        u32 fontNum;
        //! スケーラブルフォントエンジンに渡すワークメモリのサイズです。大きいほどエンジン内で文字を
        //! キャッシュしてくれます。
        //! デフォルトでは512KBです。
        u32 workMemorySize;
        //! プロットと並行して処理ができるように、プロット用とは別に用意しているスケーラブルフォントエンジン
        //! のインスタンスのためのワークメモリのサイズです。通常はデフォルトのサイズで問題ありませんが、
        //! フォントによってはサイズが足りなくなる場合がありますので、その場合は増やしてください。
        //! サイズが足りなくなった場合、アサートに失敗し、"error: 201"というメッセージが表示されます。
        //! デフォルトでは50KBです。
        u32 noPlotWorkMemorySize;
        //! テクスチャキャッシュ内でグリフを管理するためのノードの最大数です。
        //! 管理するグリフの数が多いほど(テクスチャキャッシュが大きいほど)大きい数を指定する必要があります。
        //! デフォルトでは1024個です。
        u32 glyphNodeNumMax;
        //! このテクスチャキャッシュを使用して、マルチコアのdrawを行うか否かです。
        //! マルチコアのdrawを行う場合は、並列処理のためにスケーラブルフォントエンジンのインスタンスをコア数
        //! ぶん持たなければいけないため、消費メモリが増加します。デフォルトではfalseです。
        bool isDrawMultiCore;
        //! ヒンティングを強制するか否かです。
        //! 通常は常に false に設定されています。
        //! 特定のフォントで描画が崩れたりクラッシュする場合、
        //! true に設定すると直ります。
        //! ただし true にすることで、描画結果がピクセル単位で若干変わります。
        //! 本来フォントが持つヒント情報に基づいて
        //! フォント描画の計算していますが、そのときの演算の丸め誤差が原因して
        //! 特定のフォントでクラッシュすることが確認されています。
        //! true に設定することでヒント情報をランタイム側で計算するようになり、
        //! これらのクラッシュを回避することができます。デフォルトではfalseです。
        bool isAutoHint;

        //! @brief コンストラクタです。デフォルト値で初期化します。
        InitializeArg();
    };

    //! @brief 初期化を行います。
    //!
    //! @details
    //! テクスチャの初期化なども行いますので、描画コマンドを発行可能な場所で呼んでください。
    //! また、このメソッドを呼んでからフォントでGetGlyphを利用して描画を行うまでに、
    //! 一回はUpdateTextureCache及びCompleteTextureCacheを行うようにしてください。
    //!
    //! @param arg  パラメータを指定する構造体
    //!
    void Initialize(const InitializeArg& arg);

    //! @brief 終了処理を行います。
    //!
    //! @details
    //! Initializeで確保したメモリを開放します。
    //!
    //! @param allocator    メモリを解放するアロケータ
    //!
    void Finalize(nw::ut::IAllocator* allocator);
    //@}

    //----------------------------------------------------
    //! @name テクスチャキャッシュの操作
    //@{

    //! @brief 指定した文字を、描画する文字として登録します。
    //!
    //! @details
    //! 描画する文字として登録した文字は、テクスチャキャッシュになかった場合は、新たに
    //! プロットする必要がある文字として登録されます。この文字は、UpdateTextureCacheと
    //! CompleteTextureCacheを行った段階で使えるようになります。
    //!
    //! 描画する文字として登録した文字が、テクスチャキャッシュに既にあった場合は、
    //! プロット等は行われず、引き続き使用できます。また、その文字は必要な文字として
    //! 登録されますので、削除対象になりません。
    //!
    //! 文字をロックする際のグループは、0から31までの32個あります。
    //! ロックはグループ毎に行われますので、例えばグル―プ0で登録された文字がグループ1
    //! でもう一度登録された場合、その両方でロックされたことになり、両方についてロック
    //! がクリアされないと削除対象になりません。
    //!
    //! @param code         描画する文字として登録する文字
    //! @param fontSize     フォントサイズ
    //! @param fontFace     フォント字形を指定する数値
    //! @param lockGroup    文字をロックする際のグループ。0から31までの数値で指定してください。-1を指定した場合はロックしません。
    //! @return 登録した文字のプロットが必要な場合にtrueが返されます。つまり、falseが返った場合はその文字は
    //!         UpdateTextureCacheを行わなくても描画できます。
    //!
    bool RegisterGlyph(char16 code, u32 fontSize, u16 fontFace, s32 lockGroup);

    //! @brief 指定した文字列に含まれる指定した文字数だけの文字を、描画する文字として登録します。
    //!
    //! @details
    //! 登録する文字を文字列として指定できる以外は @ref RegisterGlyph と同じです。
    //!
    //! @param codes        描画する文字として登録する文字列
    //! @param codeLength   文字列の文字数。この文字数より短い文字数でNULL終端が来た場合は、登録はそこで終了します。
    //! @param fontSize     フォントサイズ
    //! @param fontFace     フォント字形を指定する数値
    //! @param lockGroup    文字をロックする際のグループ。0から31までの数値で指定してください。-1を指定した場合はロックしません。
    //! @return プロットが必要な文字が何文字あったかを返します。つまり0が返された場合は、
    //!         全ての文字がUpdateTextureCacheを行わなくても描画できることを示します。
    //!
    u32 RegisterGlyphsWithLength(const char16* codes, u32 codeLength, u32 fontSize, u16 fontFace, s32 lockGroup);

    //! @brief テクスチャキャッシュを更新し、新規にプロットが必要な場合はプロットを行います。
    //!
    //! @details
    //! この関数の処理は非常に長くなる可能性があるため、暗転していない、ゲームが動いている
    //! タイミングでは、処理落ちが発生しないようにサブスレッドで呼び出す等の対策を取ってください。
    //!
    //! UpdateTextureCacheとGetGlyph（及び描画）は並行して行うことができます。ただし、
    //! RegisterGlyph、RegisterGlyphWithLength、CompleteTextureCacheはUpdateTextureCacheを
    //! 行っている最中は呼び出さないようにしてください。
    //!
    //! このメソッドは最大8KBのスタックを消費しますので、スレッドで動作させる場合はスタック
    //! サイズにご注意ください。
    //!
    void UpdateTextureCache();

    //! @brief テクスチャキャッシュの更新を完了します。
    //!
    //! @details
    //! このメソッドはUpdateTextureCacheが完了した後に呼び出してください。
    //! 描画と同時に行うことはできません。また、描画コマンドを積む処理が入ります
    //! ので、描画が可能な場所で呼んでください。
    //!
    //! またこのメソッドを呼ぶと、以下の条件に一つも当てはまらない文字が使用不可能となります。
    //!
    //! - 前回CompleteTextureCacheを呼んでから一回以上描画されている(GetGlyphが呼ばれている)
    //! - 前回CompleteTextureCacheを呼んだ後にRegisterGlyphなどで描画する文字として登録された
    //! - 何れかのグループでロックされている
    //!
    void CompleteTextureCache();

    //! @brief 指定したグループのロックを解除します。
    //!
    //! @param lockGroup    ロックを解除するグループ。0から31までの数値で指定してください。
    //!
    void ClearLockAllGlyphs(s32 lockGroup);

    //! @brief 指定した長さの文字列に含まれる文字のロックを解除します。
    //!
    //! @param codes        ロックを解除する文字が含まれる文字列。
    //! @param codeLength   文字列の文字数。この文字数より短い文字数でNULL終端が来た場合は、そこで終了します。
    //! @param fontSize     フォントサイズ
    //! @param fontFace     フォント字形を指定する数値
    //! @param lockGroup    ロックを解除するグループ。0から31までの数値で指定してください。
    //!
    void ClearLockGlyphsWithLength(const char16* codes, u32 codeLength, u32 fontSize, u16 fontFace, s32 lockGroup);

    //! @brief テクスチャキャッシュの内容を初期化し、Initialize直後の状態に戻します。
    //!
    //! @details
    //! このメソッドを呼び出すと、フォントの初期化時に登録された代替文字も消えてしまいます。
    //! フォントの描画を呼び出す前に、このキャッシュを使用している各フォントで
    //! RegisterAlternateCharGlyphメソッドを呼び出すようにしてください。
    //!
    //! その後、Initializeと同様、フォントでGetGlyphを利用して描画を行うまでに、
    //! 一回はUpdateTextureCache及びCompleteTextureCacheを行うようにしてください。
    //!
    void ResetTextureCache();

    //! @brief テクスチャキャッシュ内に新しいラインを作成します。
    //!
    //! @details
    //! テクスチャキャッシュ内のラインは、足りなくなったら自動的に作られるため、
    //! 通常はこのメソッドを使用する必要はありません。
    //!
    //! このメソッドが必要になるのは、ほぼ32のラインが入るが、稀に64のラインが必要に
    //! なるというような場合です。(例えばセディーユ付きのラテン文字など、背の高い文字が
    //! 来た場合。)
    //! このような場合にテクスチャキャッシュが既に32のラインだけで埋まっていると、
    //! 64のラインを新たに作ることができず、64のラインを必要とする文字を描画できません。
    //!
    //! このような場合に、TextureCacheの初期化直後にこの関数で64のラインを一つ作って
    //! 置くことで、問題を回避することができます。
    //!
    //! 詳しくは、オーバービューのFAQの「テクスチャキャッシュに消すことのできる文字が
    //! 十分あるにも関わらず、文字を展開できないことがある」をご参照ください。
    //!
    //! @param[in] lineHeight   新しく作るラインの高さ
    //! @return ラインを作成できたらtrue。スペースが足りず、作成できなかったらfalse。
    //!
    bool CreateNewLine(LineHeight lineHeight)
    {
        return (CreateNewLineImpl(lineHeight) != NULL);
    }
    //@}

    //----------------------------------------------------
    //! @name テクスチャ情報の取得
    //@{

    //! @brief  テクスチャオブジェクトバッファへのポインタを取得します。
    //!
    //! @return テクスチャオブジェクトバッファへのポインタを返します。
    //!
    font::internal::TextureObject* GetTextureObject()
    {
        return &m_TexObj;
    }

    //! @brief  テクスチャオブジェクトバッファへのポインタを取得します。
    //!
    //! @return テクスチャオブジェクトバッファへのポインタを返します。
    //!
    const font::internal::TextureObject* GetTextureObject() const
    {
        return &m_TexObj;
    }

    //! @brief  テクスチャキャッシュの幅を取得します。
    //!
    //! @return テクスチャキャッシュの幅
    //!
    u32 GetTextureCacheWidth() const
    {
        return mTextureCacheWidth;
    }

    //! @brief  テクスチャキャッシュの高さを取得します。
    //!
    //! @return テクスチャキャッシュの高さ
    //!
    u32 GetTextureCacheHeight() const
    {
        return mTextureCacheHeight;
    }

    bool IsBorderEffectEnabled(u16 fontFace) const
    {
        return 0 < mBorderWidths[fontFace];
    }
    //@}

    //----------------------------------------------------
    //! @name 開発向け機能
    //@{

    //! @brief  テクスチャキャッシュ内の平衡二分木の内容をダンプします。
    //!
    //! @details
    //! このメソッドを実行すると、以下のような文字列が、キャッシュに含まれている文字数と
    //! 同じ行数だけコンソールに出力されます。
    //!
    //!   0 0x32 40 rp lock:0x1800
    //!
    //! この一行が一文字を表しています。一つ目の数値はフォントの種類(0～31の数値)、
    //! 二つ目の数値が文字コード、三つ目の数値がサイズ、次の文字列が内部的なフラグ値、
    //! 最後のlock:の後に続く数値は、ロック状態を32ビットの数値で表します。
    //! 例えばこの場合は、グループ11と12でロックされていることになります。
    //!
    //! インデントは平衡二分木にどのように格納されているかを表しています。
    //!
    //! また、最後に
    //!
    //! dump 102 nodes.
    //!
    //! のように表示されます。これは、テクスチャキャッシュで管理しているグリフの数を
    //! 示します。
    //!
    //! この数が、InitializeArgのglyphNodeNumMaxで指定した数値を超えると新たにグリフを
    //! 登録できなくなってしまいますので、ある程度余裕があるようにしてください。
    //!
    //! Releaseビルドでは何もしません。
    //!
    void DumpGlyphTreeMap()
    {
        return mGlyphTreeMap.Dump();
    }

    //! @brief  フォントの中に指定した文字コードの文字が含まれているかを返します。
    //!
    //! @details
    //! このメソッドが返す値は、文字がプロットされているかどうかには関係しません。
    //! 初期化時に与えたフォントデータ内に文字が含まれているかのみに依存します。
    //!
    //! @param code         幅を求めたい文字
    //! @param fontFace     フォント字形を指定する数値
    //! @return 含まれるか否か
    //!
    bool IsGlyphExistInFont(char16 code, u16 fontFace);

    //! @brief このテクスチャキャッシュを利用して、マルチコアでの描画が行えるか否かを返します。
    //!
    //! @return マルチコアでの描画が行えるか否か
    //!
    bool IsDrawMultiCore() const { return mIsDrawMultiCore; }
    //@}

    //----------------------------------------------------
    //! @name エラー状態の取得・クリア
    //@{

    //! @brief  テクスチャキャッシュに空きがないエラーが発生したか否かを取得します。
    //!
    //! @details
    //! このエラーは、発生した後はClearNoSpaceErrorを呼ばない限りクリアされません。
    //!
    //! @return エラーが発生した場合はtrue
    //!
    bool GetNoSpaceError()
    {
        return mNoSpaceError;
    }

    //! @brief  テクスチャキャッシュに空きがないエラーが発生したか否かをクリアします。
    //!
    void ClearNoSpaceError()
    {
        mNoSpaceError = false;
    }
    //@}

    //----------------------------------------------------
    //! @name ScalableFontクラスの内部実装用
    //@{

    // フォントの属性値です。

    //! @brief 内部用機能のため使用禁止です。
    struct FontMetrics {
        float ascent_ratio;     // アセントの割合。ベースラインとも呼びます。
        float height_ratio;     // 高さの割合。これからアセントを引いた値がディセントになります。
    };

    // フォントの属性値を取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in] fontFace フォント字形を指定する数値
    //!
    //! @return フォントの属性値です。
    //!
    const FontMetrics& GetFontMetrics(u16 fontFace);

    // 指定した文字の幅を求めます。
    // スケーラブルフォントエンジンから文字の情報を検索して求めますので、処理コストがかかる
    // ことにご注意ください。
    // 指定された文字がフォントに含まれていない場合は0が返されます。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param code         幅を求めたい文字
    //! @param fontSize     フォントサイズ
    //! @param fontFace     フォント字形を指定する数値
    //! @return 文字の幅
    //!
    int CalcCharWidth(char16 code, u32 fontSize, u16 fontFace);

    // カーニングの値を求めます。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param[in]  c0      カーニングの値を求める、進行方向と逆側の文字です。
    //! @param[in]  c1      カーニングの値を求める、進行方向側の文字です。
    //! @param[in] fontSize フォントサイズです。
    //! @param[in] fontFace フォント字形を指定する数値です。
    //!
    //! @return     カーニングの値を返します。
    //!
    int CalcKerning(char16 c0, char16 c1, u32 fontSize, u16 fontFace);

    // 文字の平衡二分木内のノードを取得します。
    // 文字のノードをプロット中に書き換えると誤動作する可能性がありますので、
    // このメソッドを使用するときは、UpdateTextureCacheを行っていないときに
    // するか、値を参照するだけにするよう注意してください。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @param code         幅を求めたい文字
    //! @param fontSize     フォントサイズ
    //! @param fontFace     フォント字形を指定する数値
    //! @return 文字のノード。平衡二分木内にノードがない場合はNULL
    //!
    GlyphNode* FindGlyphNode(char16 code, u32 fontSize, u16 fontFace)
    {
        return mGlyphTreeMap.Find(code, static_cast<u16>(fontSize), fontFace);
    }

    // テクスチャキャッシュの実メモリへのポインタを取得します。

    //! @brief 内部用機能のため使用禁止です。
    //!
    //! @return テクスチャキャッシュの実メモリへのポインタ
    //!
    const void* GetTextureCacheBitMap() const
    {
        return mTextureCacheBitMap;
    }
    //@}

protected:
    enum {
        FONT_NAME_LEN_MAX = 50,
        FONT_FACE_NUM_MAX = 32,
        WORK_MEMORY_SIZE_DEFAULT = 512*1024,
        WORK_MEMORY_SIZE_NO_PLOT_DEFAULT = 50*1024,
        GLYPH_NODE_NUM_MAX_DEFALUT = 1024,
        TEXTURE_CACHE_LINE_NUM_MAX = 256,
        GLYPH_PADDING = 2,
        CORE_NUM = 3
    };

    struct LineInfo {
        GlyphLineList list;
        u16 y;
        u16 currentX;
        u8 kind;
        u8 no;
        u8 dummy[2];
    };

    scffnd::ScalableFontEngine* mFsState;
    scffnd::ScalableFontEngine* mFsStatesNoPlot;
    char* mFontNameBufs;
    u8* mTextureCacheBitMap;
    u32 mTextureCacheWidth;
    u32 mTextureCacheHeight;
    GlyphTreeMap mGlyphTreeMap;
    GlyphList mNeedPlotGlyphList;
    GlyphList mNeedEraseGlyphList;
    GlyphList mNotInFontGlyphList;
    LineInfo mLineInfos[TEXTURE_CACHE_LINE_NUM_MAX];
    u32 mLineCurrentPos;
    u32 mFontNum;
    u32 mCurrentFontFace;
    u32 mCurrentFontFacesNoPlot[CORE_NUM];
    bool mNoSpaceError;
    bool mIsDrawMultiCore;
    FontMetrics* mFontMetrics;
    scffnd::ScFixed* mBoldWeights;
    u8* mBorderWidths;

    font::internal::TextureObject m_TexObj;       //!< テクスチャオブジェクト

    LineInfo* CreateNewLineImpl(LineHeight lineHeight);
    GlyphNode* FindAndReserveEraseGlyph(u8 lineKind, u16 glyphWidth);
    void SetFontFace(u32 fontFace);
    void SetFontFaceNoPlot(u32 fontFace, u32 coreId);
    void AssertFsError(const char* api_name);
    void AssertFsErrorNoPlot(const char* api_name, u32 no);

    //! @brief フォントにない文字のグリフ情報を削除します。
    //!
    //! @details
    //! TextureCacheは、登録されているフォントにない文字がリクエストされた場合、
    //! RegisterGlyphのタイミングではいったんプロットが必要な文字として受付して
    //! おき、UpdateTextureCacheの中で改めて文字がフォント内にあるか判別して、
    //! ない場合は「フォントにない文字」としてフラグを立て、情報を保持します。
    //!
    //! これは、RegisterGlyphのときにフォントに文字があるかどうかを判別して
    //! しまうと、判別の処理によってRegisterGlyphの処理量が増えてしまうためです。
    //! つまり、グリフ情報の中に「この文字はフォントに含まれない」という情報を
    //! キャッシュすることになります。
    //!
    //! ただ、この情報は、テクスチャキャッシュを圧迫してしまうため、大量にたまる
    //! と(つまり、フォントにない文字をたくさんRegisterGlyphすると)グリフ情報を
    //! 保持するためのノードが足りなくなる可能性があります。
    //!
    //! この関数は、そのような状況でノードが足りなくなった場合に、フォントにない文字
    //! のグリフ情報を削除するためのものです。
    //!
    //! 今のところは、RegisterGlyph内でグリフの数が足りなくなった場合に内部的に
    //! 呼ぶ仕組みになっています。
    //!
    //! 必要な状況があれば、公開する可能性があります。
    //!
    //! @return 削除した文字数
    //!
    u32 EraseNotInFontGlyphs();

    char* GetFontNameBuf(u32 fontFace)
    {
        return &mFontNameBufs[fontFace * FONT_NAME_LEN_MAX];
    }

    u32 GetFsStateNoPlotNum() const
    {
        return (mIsDrawMultiCore ? CORE_NUM : 1);
    }

    u32 GetCoreId() const
    {
#if defined(NW_PLATFORM_CAFE)
        return (mIsDrawMultiCore ? OSGetCoreId() : 0);
#elif defined(NW_USE_NINTENDO_SDK)
        // TODO: 正式な実装に置き換える
        return 0;
#else
        return 0;
#endif
    }

};

} // namespace nw::scfont
} // namespace nw

#endif // NW_SCFONT_TEXTURE_CACHE_H_
