﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef SCFFND_SCALABLEFONTENGINE_H_
#define SCFFND_SCALABLEFONTENGINE_H_

#include <types.h>

//! @brief スケーラブルフォントエンジンの名前空間です。
namespace scffnd
{

//! @brief  16.16 固定小数(32ビット中、16ビットを仮数とする)です。
typedef s32 ScFixed;

//---------------------------------------------------------------------------
// リザルトコード
//---------------------------------------------------------------------------

//! @brief  成功を表すリザルトコードです。
const s32 SUCCESS = 0;

//! @brief  成功を表すリザルトコードです。
const s32 NO_ERR  = 0;

//! @brief  フォントパスが不正です。
const s32 ERR_FONT_NOT_FOUND  = 301;

//! @brief  現在のフォントが読み込まれていません。
const s32 ERR_NO_CURRENT_LFNT = 303;

//! @brief  メモリの確保に失敗しました。
const s32 ERR_MALLOC_FAIL  = 201;

//! @brief  指定されたフォントは正しい TrueType フォントではありません。
const s32 ERR_NOT_A_TTF = 403;

//! @brief  別のフォントが、同一名称で登録されています。
const s32 ERR_FONT_NAME_IN_USE = 314;

//! @brief  同じフォントが、別の名称で登録されています。
const s32 ERR_FONT_NAME_NOT_UNIQUE = 311;

//! @brief  指定したスケール値が、制限値（+/- 16384）を超えています。
const s32 ERR_SCALE_LIMIT = 502;

//! @brief  指定したスケール値が、小さすぎます。
const s32 ERR_SCALE_DEGENERATE = 503;

//! @brief  指定したパーセント指定が範囲(0-100%)を超えています。
const s32 ERR_BAD_PERCENT = 613;

//---------------------------------------------------------------------------


//! @brief  内部定数です。
const s32 MAX_MAPPINGS_LEN = 20;

//! @brief  フォント名の最大文字数です。
const s32 MAX_FONT_NAME_LEN = 50;

//! @brief  フォントファミリー名の最大文字数です。
const s32 MAX_FONT_FAMILY_NAME_LEN = 50;

//! @brief  コピーライト文字列の最大文字数です。
const s32 MAX_COPYRIGHT_LEN = 2048;

//! @brief  8ビットグレーマップフォーマットです。
const s32 MAP_GRAYMAP8 = 0x0008;

//---------------------------------------------------------------------------
//! @brief  スケーラブルフォントエンジンの管理情報です。
//!
struct ScHandle
{
    //! @brief  内部メンバーです。
    u8          pad[92];
    //! @brief  内部バッファへのポインタです。
    void*       pInternalBuffer;
};

//---------------------------------------------------------------------------
//! @brief  内部情報を管理する構造体です。
//!
struct ScCharacterMapping {
    //! @brief  内部メンバーです。
    u16 platform;
    //! @brief  内部メンバーです。
    u16 encoding;
    //! @brief  内部メンバーです。
    u32 offset;
};

//---------------------------------------------------------------------------
//! @brief  境界領域情報です。
//!
struct ScBoundingBox {
    //! @brief  Xの最小値です。
    s16 xMin;
    //! @brief  Yの最小値です。
    s16 yMin;
    //! @brief  Xの最大値です。
    s16 xMax;
    //! @brief  Yの最大値です。
    s16 yMax;
};


//---------------------------------------------------------------------------
//! @brief  グリフ情報です。
//!
struct ScGlyphMap{
    //! @brief  内部バッファへのポインタです。
    void*     pInternal;
    //! @brief  グリフ情報構造体のバイトサイズです。グリフビットマップのバイト配列を含むサイズです。
    s32 size;
    //! @brief  グリフ原点(通常、ベースライン付近)を原点とした、左上位置のX値です。
    s16 loX;
    //! @brief  グリフ原点(通常、ベースライン付近)を原点とした、左上位置のY値です。
    s16 hiY;
    //! @brief  整数でのX方向の文字送り量です。
    s16 idX;
    //! @brief  整数でのY方向の文字送り量です。
    s16 idY;
    //! @brief  固定少数でのX方向の文字送り量です。
    ScFixed dX;
    //! @brief  固定少数でのY方向の文字送り量です。
    ScFixed dY;
    //! @brief  グリフビットマップのピクセル幅です。
    s16 width;
    //! @brief  グリフビットマップのピクセル高さです。
    s16 height;
    //! @brief  グリフビットマップの横1ラインのバイトサイズです。
    s16 bpl;
    //! @brief  内部メンバーです。
    u8 embedded;
    //! @brief  ピクセルあたりのビット数です。
    s16 bitsPerPixel;
    //! @brief  内部メンバーです。
    u16 type;
    //! @brief  グリフビットマップデータを格納するバイト配列です。
    u8 bits[1];
};

//---------------------------------------------------------------------------
//! @brief  フォント情報です。
//!
struct ScMetrics {

    //! @brief デザイン単位を emサイズ（通常アルファベット M の大きさ）で割った値です。
    u16 unitsPerEm;

    //! @brief head テーブルにおける MacStyle フラグ（表示形式に関するフラグ）です。
    u16 headMacStyle;

    //! @brief hhea テーブルにおける Windows形式のアセント（ベースラインからグリフ上端まで）量です。デザイン単位における値が格納されています。
    s16 os2WinAscent;
    //! @brief hhea テーブルにおけるWindows形式のディセント（ベースラインからグリフ下端まで）量です。デザイン単位における値が格納されています。
    s16 os2WinDescent;
    //! @brief os2 テーブルにおけるアセント（ベースラインからグリフ上端まで）量です。デザイン単位における値が格納されています。
    s16 os2Ascent;
    //! @brief os2 テーブルにおけるディセント（ベースラインからグリフ下端まで）量です。デザイン単位における値が格納されています。
    s16 os2Descent;
    //! @brief os2 テーブルにおけるフォントセレクションフラグ（表示形式に関するフラグ）です。
    u16 os2FsSelection;
    //! @brief os2 テーブルにおけるレディング（行と行の間隔）量です。デザイン単位における値が格納されています。
    u16 os2Leading;
    //! @brief hhea テーブルにおけるアセント（ベースラインからグリフ上端まで）量です。デザイン単位における値が格納されています。
    s16 hheaAscent;
    //! @brief hhea テーブルにおけるディセント（ベースラインからグリフ下端まで）量です。デザイン単位における値が格納されています。
    s16 hheaDescent;
    //! @brief hhea テーブルにおけるレディング（行と行の間隔）量です。デザイン単位における値が格納されています。
    u16 hheaLeading;
    //! @brief  予約されたメンバです。
    u16 embeddingBits;
    //! @brief  境界矩形情報です。デザイン単位における値が格納されています。
    ScBoundingBox fontBoundingBox;

    //! @brief  フォントの種類を表す内部メンバーです。
    u16 fontType;

    //! @brief  内部メンバーです。
    u16 num_cmap_tables;
    //! @brief  内部メンバーです。
    ScCharacterMapping mappings[MAX_MAPPINGS_LEN];
    //! @brief  TrueType collection 中に含まれるフォント数です。
    u32 numFontsInTTC;
    //! @brief  TrueType collection でのフォント番号です。
    u32 indexInTTC;

    //! @brief  フォント名です。
    char font_name[MAX_FONT_NAME_LEN];
    //! @brief  フォントファミリー名です。
    char font_family_name[MAX_FONT_FAMILY_NAME_LEN];
    //! @brief  著作権情報文字列です。
    char copyright[MAX_COPYRIGHT_LEN];
    //! @brief  フォント解像度です。デザイン単位で格納されている値に乗算してピクセルサイズに変換する場合に利用します。
    u16 metricsResolution;
    //! @brief  フォントが持つグリフ数です。
    u32 numGlyphs;
    //! @brief  フォントが持つアイコン数です。
    u32 numIcons;
};


//---------------------------------------------------------------------------
//! @brief スケーラブルフォントエンジンを表すクラスです。
//!
//! @details
//---------------------------------------------------------------------------
class ScalableFontEngine
{
public:

    //! @brief 文字効果フラグです。
    enum Flags
    {
        //! @brief 効果無し
        FLG_NO_EFFECT,
        //! @brief 縁付きフォント、塗りつぶし無し
        FLG_OUTLINED_UNFILLED,
        //! @brief 縁付きフォント、塗りつぶし有り
        FLG_OUTLINED_FILLED,
    };

    //----------------------------------------------------
    //! @name コンストラクタ／デストラクタ/初期化/終了処理
    //@{

    //! @brief コンストラクタです。
    //!
    //! @details
    //! 実質的な初期化はInitializeで行われます。ここではメンバ変数の初期化のみ行っています。<br>
    //! <br>
    //! 本クラスは複数スレッドからの利用を考慮した排他制御を内部で行いません。<br>
    //! 複数スレッドから同時に利用する必要がある場合、ユーザコード側で排他制御を行うかスレッド毎にインスタンスを作成するようにしてください。<br>
    //!
    ScalableFontEngine();

    //! @brief デストラクタです。
    //!
    ~ScalableFontEngine();

    //! @brief 初期化を行います。<br>
    //!
    //! @param[in] pHeap    ワークメモリへのポインタ
    //! @param[in] heapSize ワークメモリのサイズ
    //!
    //! @details
    //! ワークメモリは以下の用途に利用されます。<br>
    //! <br>
    //! ・読み込まれたフォントのリスト<br>
    //! ・読み込まれたフォント情報<br>
    //! ・スケール処理された、フォント情報とキャッシュ<br>
    //! ・グリフ（文字）ビットマップのキャッシュ<br>
    //! <br>
    //! メモリが不足した場合は、自動的に内部キャッシュを開放して柔軟にやりくりをします。<br>
    //! ワークメモリの適切サイズは利用状況（フォントの種類、同時利用文字数、フォントサイズ）によるので一概に記述しにくいです。<br>
    //! 64ピクセル程度のサイズなら 最低 100kバイト以下でも動作します。<br>
    //! パフォーマンスを実測しながら試行錯誤されることをお勧めします。<br>
    //!
    //! @return リザルトコードです。<br>
    //! 成功すると @ref SUCCESS が返ります。
    //!
    s32 Initialize(void* pHeap, u32 heapSize);

    //! @brief 終了処理を行います。
    //!
    //! @details
    //! ワークメモリの解放は行いません。@ref GetPointerToWorkBuffer で取得し別途解放してください。
    //!
    //! @return リザルトコードです。必ず @ref SUCCESS が返ります。
    //!
    s32 Finalize();
    //@}

    //----------------------------------------------------
    //! @name フォント設定
    //@{

    //! @brief フォントを読み込みます。
    //!
    //! @param[in] pPath    フォントファイルパス。現在は利用できませんので必ず NULL を指定してください。
    //! @param[in] pMem     フォントファイルデータへのポインタ。
    //! @param[in] index    フォントの番号。必ず 0 を指定します。
    //! @param[in] nameBufferLen    フォント名文字列を出力するバッファ（pNameBuffer）の長さです。最小で2、最大で MAX_FONT_NAME_LEN を指定します。
    //! @param[out] pNameBuffer 内部管理用のnull終端されたフォント名文字列を出力するバッファです。
    //!
    //! @details 対応するスケーラブルフォントの種類は以下です。
    //!      - 本体内蔵フォント(bfttf)
    //!      - TreuType(ttf)
    //!      - OpenType(otf)
    //!
    //! <br>
    //! bfttf ファイルを読み込む場合は、@ref ScalableFontEngineHelper::Decode でデコードしてから渡すようにしてください。
    //!
    //! @return リザルトコードです。<br>
    //! @ref SUCCESS 以外では、以下のエラーコードが返ります。<br>
    //! @ref ERR_FONT_NOT_FOUND <br>
    //! @ref ERR_FONT_NAME_IN_USE <br>
    //! @ref ERR_FONT_NAME_NOT_UNIQUE <br>
    //!
    s32 LoadFont(char* pPath, void* pMem, u32 index, u32 nameBufferLen, char* pNameBuffer);

    //! @brief フォントを設定します。
    //!
    //! @param[in] name 内部管理用のフォント名
    //!
    //! @return リザルトコードです。<br>
    //! @ref SUCCESS 以外では、以下のエラーコードが返ります。<br>
    //! @ref ERR_FONT_NOT_FOUND <br>
    //! @ref ERR_MALLOC_FAIL <br>
    //! @ref ERR_NOT_A_TTF <br>
    //!
    s32 SetFont(char* name);

    //! @brief フォントスケールを設定します。
    //!
    //! @param[in] s00  2X2 変換行列の (0,0) 要素。固定小数で指定します。
    //! @param[in] s01  2X2 変換行列の (0,1) 要素。固定小数で指定します。
    //! @param[in] s10  2X2 変換行列の (1,0) 要素。固定小数で指定します。
    //! @param[in] s11  2X2 変換行列の (1,1) 要素。固定小数で指定します。
    //!
    //! @return リザルトコードです。<br>
    //! @ref SUCCESS 以外では、以下のエラーコードが返ります。<br>
    //!   @ref ERR_NO_CURRENT_LFNT <br>
    //!   @ref ERR_SCALE_LIMIT <br>
    //!   @ref ERR_SCALE_DEGENERATE <br>
    //!
    s32 SetScale(ScFixed s00, ScFixed s01, ScFixed s10, ScFixed s11);

    //! @brief 文字のウエイト（太さ）を指定します。
    //!
    //! @param[in] weight   文字のウエイト（太さ）です。増加量を(0 - 100%)で指定します。
    //!
    //! @return リザルトコードです。<br>
    //! @ref SUCCESS 以外では、以下のエラーコードが返ります。<br>
    //! @ref ERR_BAD_PERCENT <br>
    //!
    s32 SetBoldWeight(ScFixed weight);

    //! @brief 各種描画効果フラグを設定します
    //!
    //! @param[in] flag 描画効果フラグ @ref ScalableFontEngine::Flags です。
    //!
    //! @return リザルトコードです。
    //!
    s32 SetFlags(Flags flag);

    //! @brief 縁取り機能における縁の幅を設定します
    //!
    //! @param[in] width 縁の幅です。ピクセル値で指定します。
    //!
    //! @return リザルトコードです。成功すると @ref SUCCESS が返ります。
    //!
    s32 SetOutlineWidth(u16 width);
    //@}

    //----------------------------------------------------
    //! @name グリフの取得
    //@{

    //! @brief グリフ情報を取得します。
    //!
    //! @param[in] id グリフ情報を取得する文字コードです。
    //! @param[in] type フォーマット種類です。@ref scffnd::MAP_GRAYMAP8 だけが指定できます。
    //!
    //! @return グリフ情報へのポインタです。エラーが起こった場合は NULL が返ります。
    //!
    ScGlyphMap* GetGlyphmap(u32 id, u16 type);

    //! @brief グリフ情報の有無を調査します。
    //!
    //! @param[in] id グリフ情報の有無を調査する文字コードです。
    //!
    //! @return グリフ情報が存在すれば true を返します。
    //!
    bool CheckGlyphExist(u32 id);

    //! @brief グリフ情報の有無を調査します。
    //!
    //! 今後この関数は削除されます。互換性維持のため、残されています。
    //! 以後は、CheckGlyphExist をご利用ください。
    //!
    //! @param[in] id グリフ情報の有無を調査する文字コードです。
    //!
    //! @return グリフ情報が存在すれば true を返します。
    //!
#ifdef WIN32
    __declspec(deprecated) bool CheckGlpyhExsist(u32 id);
#else
    bool CheckGlpyhExsist(u32 id) __attribute__ ((deprecated));
#endif

    //! @brief グリフ情報をエンジン内部から解放します。
    //!
    //! @param[in] pGlyph グリフ情報へのポインタです。
    //!
    //! @return リザルトコードです。成功すると @ref SUCCESS が返ります。
    //!
    s32 ReleasesGlyph(void * pGlyph);

    //! @brief ヒンティングを強制するかどうかを設定します。
    //!
    //! 通常は常に false に設定されています。
    //! 特定のフォントで描画が崩れたりクラッシュする場合、
    //! GetGlyphmapメソッドの直前で true に設定すると直ります。
    //! ただし true にすることで、描画結果がピクセル単位で若干変わります。
    //! 本来フォントが持つヒント情報に基づいて
    //! フォント描画の計算していますが、そのときの演算の丸め誤差が原因して
    //! 特定のフォントでクラッシュすることが確認されています。
    //! このメソッドに true を渡すことで
    //! ヒント情報をランタイム側で計算するようになり、
    //! これらのクラッシュを回避することができます。
    //!
    //! @param[in] isAutoHint ヒンティングを強制するかどうかです。
    //!
    void SetAutoHint(bool isAutoHint);

    //@}

    //----------------------------------------------------
    //! @name 情報の取得
    //@{

    //! @brief 内部ワークメモリのポインタを取得します。
    //!
    //! @return 内部ワークメモリのポインタです。
    //!
    void* GetPointerToWorkBuffer();

    //! @brief フォント情報を取得します。
    //!
    //! @param[out] pM フォント情報へのポインタです。
    //!
    //! @return リザルトコードです。成功すると @ref SUCCESS が返ります。
    //!
    s32 GetFontMetrics(ScMetrics* pM);

    //! @brief 次の文字への文字送り量を取得します。
    //!
    //! @param[in] id 情報を取得する文字コードです。
    //! @param[in] type グリフの種類です。必ず @ref scffnd::MAP_GRAYMAP8 を指定します。
    //! @param[out] idx 整数での文字送り量のX値です。
    //! @param[out] idy 整数での文字送り量のY値です。
    //! @param[out] dx 固定小数での文字送り量のX値です。
    //! @param[out] dy 固定小数での文字送り量のY値です。
    //!
    //! @return リザルトコードです。成功すると @ref SUCCESS が返ります。
    //!
    s32 GetAdvance(u32 id, u16 type, s16* idx, s16* idy, ScFixed* dx, ScFixed* dy);

    //! @brief 二つの文字間のカーニング情報を取得します。
    //!
    //! @param[in] id1 1文字目の文字コードです。
    //! @param[in] id2 2文字目の文字コードです。
    //! @param[out] dx 2文字間のカーニング情報の X 値です。
    //! @param[out] dy 2文字間のカーニング情報の Y 値です。
    //!
    //! @return リザルトコードです。成功すると @ref SUCCESS が返ります。
    //!
    s32 GetKerning(u32 id1, u32 id2, ScFixed* dx, ScFixed* dy);

    //! @brief エラーコードを取得します。
    //!
    //! @return エラーコードです。
    //!
    s32 GetError();
    //@}

private:
    ScHandle        mScHandle;
};

//---------------------------------------------------------------------------
//! @brief スケーラブルフォントエンジンのヘルパークラスです。
//!
//---------------------------------------------------------------------------
class ScalableFontEngineHelper
{
public:
    //! @brief bfttf 形式のフォントファイルデータを復号して、利用できるようにします。
    //!
    //! @param[in] pData 復号前のデータへのポインタです。
    //! @param[in] dataSize 復号前のデータのサイズです。
    //!
    //! @return    復号されたフォントファイルデータです。
    //!
    static void* Decode(void* pData, u32 dataSize);
};

} // namespace scffnd

#endif // SCFFND_SCALABLEFONTENGINE_H_
