﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MCS_INPUTDEVICE_H_
#define NW_MCS_INPUTDEVICE_H_

#if defined(__CWCC__)
#pragma once
#endif

#include <nw/mcs/mcs_Base.h>
#include <nw/mcs/mcs_Common.h>

namespace nw
{
namespace mcs
{

/* ========================================================================
    constant definitions
   ======================================================================== */

namespace internal
{

const ChannelType   MCSHID_CHANNEL     = 0xFF7E;

}   // namespace internal

//! @name 入力デバイス
//@{

//---------------------------------------------------------------------------
//! @brief キーボードの仮想キーコードです。
//!
//! @details
//!   値は Windows の仮想キーと同値となります。
//!   使用しているキーボードによっては、ここに定義されていない入力値をとることがあります。
//---------------------------------------------------------------------------
enum Keycode
{
    KC_BACKSPACE            = 0x08, //!< BackSpace キー
    KC_TAB                  = 0x09, //!< Tab キー
    KC_LINEFEED             = 0x0A, //!< ライン フィード キー
    KC_CLEAR                = 0x0C, //!< Clear キー
    KC_ENTER                = 0x0D, //!< Enter キー
    KC_SHIFT                = 0x10, //!< Shift キー
    KC_CONTROL              = 0x11, //!< Ctrl キー
    KC_ALT                  = 0x12, //!< Alt キー
    KC_PAUSE                = 0x13, //!< Pause キー
    KC_CAPS_LOCK            = 0x14, //!< CapsLock キー
    KC_KANA                 = 0x15, //!< IME かなモード キー
    KC_HANGUL               = 0x15, //!< IME ハングル モード キー(KANAキーと同じ)
    KC_JUNJA                = 0x17, //!< IME Junja モード キー
    KC_FINAL                = 0x18, //!< IME Final モード キー
    KC_KANJI                = 0x19, //!< IME 漢字モード キー
    KC_HANJA                = 0x19, //!< IME Hanja モード キー (KANJIキーと同じ)
    KC_ESCAPE               = 0x1B, //!< Esc キー
    KC_IME_CONVERT          = 0x1C, //!< IME 変換キー
    KC_IME_NONCONVERT       = 0x1D, //!< IME 無変換キー
    KC_IME_ACCEPT           = 0x1E, //!< IME Accept キー
    KC_IME_MODECHANGE       = 0x1F, //!< IME モード変更キー
    KC_SPACE                = 0x20, //!< Space キー
    KC_PAGEUP               = 0x21, //!< PageUp キー
    KC_PAGEDOWN             = 0x22, //!< PageDown キー
    KC_END                  = 0x23, //!< End キー
    KC_HOME                 = 0x24, //!< Home キー
    KC_LEFT                 = 0x25, //!< ← キー
    KC_UP                   = 0x26, //!< ↑ キー
    KC_RIGHT                = 0x27, //!< → キー
    KC_DOWN                 = 0x28, //!< ↓ キー
    KC_SELECT               = 0x29, //!< Select キー
    KC_PRINT                = 0x2A, //!< Print キー
    KC_EXECUTE              = 0x2B, //!< Execute キー
    KC_PRINTSCREEN          = 0x2C, //!< PrintScreen キー
    KC_INSERT               = 0x2D, //!< Insert キー
    KC_DELETE               = 0x2E, //!< Delele キー
    KC_HELP                 = 0x2F, //!< Help キー
    KC_0                    = 0x30, //!< 0 キー
    KC_1                    = 0x31, //!< 1 キー
    KC_2                    = 0x32, //!< 2 キー
    KC_3                    = 0x33, //!< 3 キー
    KC_4                    = 0x34, //!< 4 キー
    KC_5                    = 0x35, //!< 5 キー
    KC_6                    = 0x36, //!< 6 キー
    KC_7                    = 0x37, //!< 7 キー
    KC_8                    = 0x38, //!< 8 キー
    KC_9                    = 0x39, //!< 9 キー
    KC_A                    = 0x41, //!< A キー
    KC_B                    = 0x42, //!< B キー
    KC_C                    = 0x43, //!< C キー
    KC_D                    = 0x44, //!< D キー
    KC_E                    = 0x45, //!< E キー
    KC_F                    = 0x46, //!< F キー
    KC_G                    = 0x47, //!< G キー
    KC_H                    = 0x48, //!< H キー
    KC_I                    = 0x49, //!< I キー
    KC_J                    = 0x4A, //!< J キー
    KC_K                    = 0x4B, //!< K キー
    KC_L                    = 0x4C, //!< L キー
    KC_M                    = 0x4D, //!< M キー
    KC_N                    = 0x4E, //!< N キー
    KC_O                    = 0x4F, //!< O キー
    KC_P                    = 0x50, //!< P キー
    KC_Q                    = 0x51, //!< Q キー
    KC_R                    = 0x52, //!< R キー
    KC_S                    = 0x53, //!< S キー
    KC_T                    = 0x54, //!< T キー
    KC_U                    = 0x55, //!< U キー
    KC_V                    = 0x56, //!< V キー
    KC_W                    = 0x57, //!< W キー
    KC_X                    = 0x58, //!< X キー
    KC_Y                    = 0x59, //!< Y キー
    KC_Z                    = 0x5A, //!< Z キー
    KC_LWIN                 = 0x5B, //!< 左の Windows ロゴ キー
    KC_RWIN                 = 0x5C, //!< 右の Windows ロゴ キー
    KC_APPS                 = 0x5D, //!< アプリケーション キー
    KC_SLEEP                = 0x5F, //!< Sleep キー
    KC_NUM_0                = 0x60, //!< 数値パッドの 0 キー
    KC_NUM_1                = 0x61, //!< 数値パッドの 1 キー
    KC_NUM_2                = 0x62, //!< 数値パッドの 2 キー
    KC_NUM_3                = 0x63, //!< 数値パッドの 3 キー
    KC_NUM_4                = 0x64, //!< 数値パッドの 4 キー
    KC_NUM_5                = 0x65, //!< 数値パッドの 5 キー
    KC_NUM_6                = 0x66, //!< 数値パッドの 6 キー
    KC_NUM_7                = 0x67, //!< 数値パッドの 7 キー
    KC_NUM_8                = 0x68, //!< 数値パッドの 8 キー
    KC_NUM_9                = 0x69, //!< 数値パッドの 9 キー
    KC_NUM_MULT             = 0x6A, //!< 数値パッドの * キー
    KC_NUM_PLUS             = 0x6B, //!< 数値パッドの + キー
    KC_NUM_COMMA            = 0x6C, //!< 数値パッドの , キー
    KC_NUM_MINUS            = 0x6D, //!< 数値パッドの - キー
    KC_NUM_PERIOD           = 0x6E, //!< 数値パッドの . キー
    KC_NUM_SLASH            = 0x6F, //!< 数値パッドの / キー
    KC_F1                   = 0x70, //!< F1 キー
    KC_F2                   = 0x71, //!< F2 キー
    KC_F3                   = 0x72, //!< F3 キー
    KC_F4                   = 0x73, //!< F4 キー
    KC_F5                   = 0x74, //!< F5 キー
    KC_F6                   = 0x75, //!< F6 キー
    KC_F7                   = 0x76, //!< F7 キー
    KC_F8                   = 0x77, //!< F8 キー
    KC_F9                   = 0x78, //!< F9 キー
    KC_F10                  = 0x79, //!< F10 キー
    KC_F11                  = 0x7A, //!< F11 キー
    KC_F12                  = 0x7B, //!< F12 キー
    KC_F13                  = 0x7C, //!< F13 キー
    KC_F14                  = 0x7D, //!< F14 キー
    KC_F15                  = 0x7E, //!< F15 キー
    KC_F16                  = 0x7F, //!< F16 キー
    KC_F17                  = 0x80, //!< F17 キー
    KC_F18                  = 0x81, //!< F18 キー
    KC_F19                  = 0x82, //!< F19 キー
    KC_F20                  = 0x83, //!< F20 キー
    KC_F21                  = 0x84, //!< F21 キー
    KC_F22                  = 0x85, //!< F22 キー
    KC_F23                  = 0x86, //!< F23 キー
    KC_F24                  = 0x87, //!< F24 キー
    KC_NUM_LOCK             = 0x90, //!< Num Lock キー
    KC_SCROLL_LOCK          = 0x91, //!< Scroll Lock キー
    KC_LSHIFT               = 0xA0, //!< 左 Shift キー
    KC_RSHIFT               = 0xA1, //!< 右 Shift キー
    KC_LCONTROL             = 0xA2, //!< 左 Ctrl キー
    KC_RCONTROL             = 0xA3, //!< 右 Ctrl キー
    KC_LALT                 = 0xA4, //!< 左 Alt キー
    KC_RALT                 = 0xA5, //!< 右 Alt キー
    KC_OEM_SEMICOLON        = 0xBA, //!< US キーボードのセミコロン キー
    KC_OEM_PLUS             = 0xBB, //!< US キーボードの + キー
    KC_OEM_COMMA            = 0xBC, //!< US キーボードの . キー
    KC_OEM_MINUS            = 0xBD, //!< US キーボードの - キー
    KC_OEM_PERIOD           = 0xBE, //!< US キーボードの . キー
    KC_OEM_QUESTION         = 0xBF, //!< US キーボードの ? キー
    KC_OEM_TILDE            = 0xC0, //!< US キーボードの ~ キー
    KC_OEM_OPEN_BRACKETS    = 0xDB, //!< US キーボードの [ キー
    KC_OEM_PIPE             = 0xDC, //!< US キーボードの | キー
    KC_OEM_CLOSE_BRACKETS   = 0xDD, //!< US キーボードの ] キー
    KC_OEM_QUOTE            = 0xDE, //!< US キーボードの " キー
    KC_OEM_8                = 0xDF, //!< US キーボードの 8 キー
    KC_OEM_BACKSLASH        = 0xE2  //!< US キーボードの \ キー
};


//---------------------------------------------------------------------------
//! @brief McsHID 関数の返り値に使用される列挙子です。
//---------------------------------------------------------------------------
enum MCSHID_Result
{
    //! 関数は成功しました。
    MCSHID_ERROR_SUCCESS        = 0,
    //! 通信エラーが発生しました。
    MCSHID_ERROR_COMERROR,
    //! Mcs サーバと接続されていません。
    MCSHID_ERROR_NOTCONNECT,

//  MCSHID_ERROR_XXXX           = 100,

    //! 列挙子の最後です。
    MCSHID_ERROR_LAST = MCSHID_ERROR_NOTCONNECT
};

//---------------------------------------------------------------------------
//! @brief キーイベント発生時の修飾キーを表す列挙子です。
//---------------------------------------------------------------------------
enum KEYMODIFIERTYPE
{
    //! Shift キーです。
    KEYMODIFIERTYPE_SHIFT   = 0x01,
    //! Ctrl キーです。
    KEYMODIFIERTYPE_CONTROL = 0x02,
    //! Alt キーです。
    KEYMODIFIERTYPE_ALT     = 0x04
};

//---------------------------------------------------------------------------
//! @brief キーの属性を表す列挙子です。
//---------------------------------------------------------------------------
enum KEYATTRIBUTETYPE
{
    //!数値パッド上の数字（0 - 9）キーです。
    KEYATTRIBUTETYPE_NUMPAD = 0x01
};

//---------------------------------------------------------------------------
//! @brief キーのカテゴリを表す列挙子です。
//---------------------------------------------------------------------------
enum KEYCATEGORYTYPE
{
    //! ファンクションキーです。
    KEYCATEGORYTYPE_FUNCTION,
    //! システムキーです。
    KEYCATEGORYTYPE_SYSTEM,
    //! アルファベットキーです。
    KEYCATEGORYTYPE_ALPHABET,
    //! 数字キーです。
    KEYCATEGORYTYPE_NUMERIC,
    //! 記号キーです。
    KEYCATEGORYTYPE_SYMBOL,

    //! 列挙子の総数です。
    KEYCATEGORYTYPE_Max
};


//---------------------------------------------------------------------------
//! @brief マウスボタンの列挙子です。
//---------------------------------------------------------------------------
enum MOUSEBUTTONTYPE
{
    //! 左ボタンです。
    MOUSEBUTTONTYPE_LEFT    = 0x01,
    //! 右ボタンです。
    MOUSEBUTTONTYPE_RIGHT   = 0x02,
    //! 中ボタンです。
    MOUSEBUTTONTYPE_MIDDLE  = 0x04,
    //! X ボタン 1 です。
    MOUSEBUTTONTYPE_X1      = 0x08,
    //! X ボタン 2 です。
    MOUSEBUTTONTYPE_X2      = 0x10
};

/* ========================================================================
    Type Definitions
   ======================================================================== */

//---------------------------------------------------------------------------
//! @brief キーボードのキーイベントについての情報を保持するクラスです。
//!
//! @details
//! :category 入力デバイス
//!
//! InputEventHandler::KeyDownEvent 、または InputEventHandler::KeyUpEvent の
//! イベントが発生したときのキーデータの詳細を提供します。
//!
//! @sa McsHID_RegisterEventHandler
//! @sa McsHID_Polling
//! @sa InputEventHandler
//---------------------------------------------------------------------------
class KeyEventArg
{
public:
    //! @name キー
    //@{

    //---------------------------------------------------------------------------
    //! @brief      修飾キーの状態を取得します。
    //!
    //! @return     修飾キーの状態を返します。
    //!
    //! @details
    //! Ctrl キー、Shift キー、および Alt キーが押されていたときに該当するビットが立ちます。
    //! ビットの詳細は mcs::KEYMODIFIERTYPE を参照してください。
    //---------------------------------------------------------------------------
    u8      GetModifier() const     { return m_Modifier; }

    //---------------------------------------------------------------------------
    //! @brief      キーの属性を取得します。
    //!
    //! @return     キーの属性を返します。
    //!
    //! @details
    //! 数値パッド上のキーが押されていた場合にビットが立ちます。
    //! ビットの詳細は mcs::KEYATTRIBUTETYPE を参照してください。
    //---------------------------------------------------------------------------
    u8      GetAttribute() const    { return m_Attribute; }

    //---------------------------------------------------------------------------
    //! @brief      キーのカテゴリを取得します。
    //!
    //! @return     キーのカテゴリを返します。
    //!
    //! @details
    //! キーのカテゴリ(数字、アルファベットなど)を取得します。
    //! 値の詳細は mcs::KEYCATEGORYTYPE を参照してください。
    //---------------------------------------------------------------------------
    u8      GetCategory() const     { return m_Category; }

    //---------------------------------------------------------------------------
    //! @brief      キーコードを取得します。
    //!
    //! @return     キーコードを返します。Windows の仮想キーと同じ値が入ってきます。
    //!
    //! @details
    //! mcs::Keycode を参照してください。
    //!
    //! @sa mcs::KEYCATEGORYTYPE
    //---------------------------------------------------------------------------
    u8      GetKeyCode() const   { return m_KeyCode; }

    //@}

private:
    u8      m_Modifier;
    u8      m_Attribute;
    u8      m_Category;
    u8      m_KeyCode;

    u8      m_Reserved[4];
};

//---------------------------------------------------------------------------
//! @brief 文字イベントについての情報を保持するクラスです。
//!
//! @details
//! :category 入力デバイス
//!
//! InputEventHandler::CharEvent のイベントが発生したときの文字データの詳細を提供します。
//!
//! @sa McsHID_RegisterEventHandler
//! @sa McsHID_Polling
//! @sa InputEventHandler
//---------------------------------------------------------------------------
class CharEventArg
{
public:
    //! @name 文字
    //@{

    //---------------------------------------------------------------------------
    //! @brief      押下されたキーより生成された文字を取得します。
    //!
    //! @return     押下されたキーより生成された文字を返します。
    //!
    //! @details
    //! (文字コードは UTF-16)
    //---------------------------------------------------------------------------
    u16     GetKeyChar() const      { return internal::NetToHost(m_KeyChar); }

    //@}

private:
    u16     m_KeyChar;

    u8      reserved[6];
};

//---------------------------------------------------------------------------
//! @brief マウスイベントについての情報を保持するクラスです。
//!
//! @details
//! :category 入力デバイス
//!
//! InputEventHandler::MouseMoveEvent などのマウスイベントが発生したときの
//! マウスの情報を提供します。
//!
//! @sa McsHID_RegisterEventHandler
//! @sa McsHID_Polling
//! @sa InputEventHandler
//---------------------------------------------------------------------------
class MouseEventArg
{
public:
    //! @name マウス
    //@{

    //---------------------------------------------------------------------------
    //! @brief      押下されたボタンの情報を取得します。
    //!
    //! @return     押下されたボタンの情報(mcs::MOUSEBUTTONTYPEの論理和)を返します。
    //!
    //! @sa mcs::MOUSEBUTTONTYPE
    //---------------------------------------------------------------------------
    u8      GetButton() const       { return m_Button; }

    //---------------------------------------------------------------------------
    //! @brief      マウスが移動したときの X 方向の移動量を取得します。
    //!
    //! @return     X 方向の移動量を返します。
    //---------------------------------------------------------------------------
    s16     GetMoveX() const        { return internal::NetToHost(m_MoveX); }

    //---------------------------------------------------------------------------
    //! @brief      マウスが移動したときの Y 方向の移動量を取得します。
    //!
    //! @return     Y 方向の移動量を返します。
    s16     GetMoveY() const        { return internal::NetToHost(m_MoveY); }

    //---------------------------------------------------------------------------
    //! @brief      ホイールが回転したときの回転した量を取得します。
    //!
    //! @return     ホイールの回転した量を返します。
    //---------------------------------------------------------------------------
    s16     GetWheel() const        { return internal::NetToHost(m_Wheel); }

    //@}

private:
    u8      m_Button;
    u8      m_Reserved;

    s16     m_MoveX;
    s16     m_MoveY;
    s16     m_Wheel;
};

//---------------------------------------------------------------------------
//! @brief キャプチャイベントについての情報を保持するクラスです。
//!
//! @details
//! :category 入力デバイス
//!
//! InputEventHandler::CaptureStartEvent などのキャプチャベントが発生したときの
//! イベントの情報を提供します。
//!
//! @sa McsHID_RegisterEventHandler
//! @sa McsHID_Polling
//! @sa InputEventHandler
//---------------------------------------------------------------------------
class CaptureEventArg
{
private:
    u8  reserved[8];
};

//---------------------------------------------------------------------------
//! @brief イベント処理を行うためのインターフェースを実装するクラスです。
//!
//! @details
//! :category 入力デバイス
//!
//! InputEventHandler クラスは、入力イベントに応じて呼び出されるメンバ関数を定義しています。
//! 入力イベントを処理するには、 InputEventHandler クラスの派生クラスを作成し、
//! 処理したいイベントに応じたメンバ関数をオーバーライドします。
//!
//! InputEventHandler クラスの各メンバの実装は何も行いません。
//!
//! 派生したクラスのオブジェクトを McsHID_RegisterEventHandler() で登録後、
//! McsHID_Polling() を呼び出すと、イベントに応じて派生クラスのメンバ関数が呼び出されます。
//!
//! @sa McsHID_RegisterEventHandler
//! @sa McsHID_Polling
//---------------------------------------------------------------------------
class InputEventHandler
{
public:
    //! @name キーボード
    //@{

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~InputEventHandler() {}

    //---------------------------------------------------------------------------
    //! @brief      キーボードのキーが押されたときに呼び出されます。
    //!
    //! @param[in]  arg     イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        KeyDownEvent(const KeyEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      キーボードのキーが離されたときに呼び出されます。
    //!
    //! @param[in]  arg     イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        KeyUpEvent(const KeyEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      文字が入力されたときに呼び出されます。
    //!
    //! @param[in]  arg     イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        CharEvent(const CharEventArg& arg);

//    //@}

//    //! @name マウス
//    //@{

    //---------------------------------------------------------------------------
    //! @brief      マウスが移動したときに呼び出されます。
    //!
    //! @param[in]  arg     イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        MouseMoveEvent(const MouseEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      マウスボタンが押されたときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        MouseDownEvent(const MouseEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      マウスボタンが離されたときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        MouseUpEvent(const MouseEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      マウスボタンがダブルクリックされたときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        MouseDoubleClickEvent(const MouseEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      マウスホイールが回転したときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        MouseWheelEvent(const MouseEventArg& arg);

    //---------------------------------------------------------------------------
    //! @brief      ホスト上でキャプチャが開始されたときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        CaptureStartEvent(const CaptureEventArg& /*arg*/) {}

    //---------------------------------------------------------------------------
    //! @brief      ホスト上でキャプチャが停止されたときに呼び出されます。
    //!
    //! @param[in]  arg    イベントの詳細を格納したオブジェクトです。
    //---------------------------------------------------------------------------
    virtual void        CaptureStopEvent(const CaptureEventArg& /*arg*/) {}

    //@}
};

namespace internal
{

struct InputDeviceWork
{
    InputEventHandler*  pInputEventHandler;
};

struct InputEventChunk
{
    u8              eventType;
    u8              reserved[3];

    union
    {
        KeyEventArg     keyEventArg;
        CharEventArg    charEventArg;
        MouseEventArg   mouseEventArg;
        CaptureEventArg captureEventArg;
    };
};

struct InputRequestChunk
{
    u32             request;
    s32             bodySize;
};

}   // namespace internal

//! @brief 受信バッファに必要な最低サイズです。
//!
//! McsHID_RegisterBuffer() で設定するバッファはこの定義以上のサイズでなければなりません。
const int   MCSHID_RECEIVEBUFFER_MINSIZE = sizeof(internal::InputDeviceWork) + sizeof(internal::InputEventChunk);

//@}

/* ========================================================================
    Function Prototype
   ======================================================================== */

#if defined(NW_MCS_ENABLE)

    //! @name 入力デバイス・初期化
    //@{

    //---------------------------------------------------------------------------
    //! @brief      InputDevice APIの初期化をおこないます。
    //!
    //! @details
    //! この関数を呼び出す前に、Mcs_Initialize() を呼び出して
    //! Mcs ライブラリを初期化しておく必要があります。
    //!
    //! この関数は何回よばれても問題ありませんが、McsHID_Initialize 自体は
    //! スレッドセーフではないので注意が必要です。
    //!
    //! @sa Mcs_Initialize
    //! @sa McsHID_IsInitialized
    //---------------------------------------------------------------------------
    void            McsHID_Initialize();

    //---------------------------------------------------------------------------
    //! @brief      InputDevice APIの終了処理をおこないます。
    //!
    //! @sa Mcs_Finalize
    //! @sa McsHID_IsInitialized
    //---------------------------------------------------------------------------
    void            McsHID_Finalize();

    //---------------------------------------------------------------------------
    //! @brief      InputDevice APIが初期化済みかどうかを取得します。
    //!
    //! @return     McsHID_Initialize() と McsHID_RegisterBuffer() が完了していれば true、
    //!             そうでなければ false を返します。
    //---------------------------------------------------------------------------
    bool            McsHID_IsInitialized();

    //---------------------------------------------------------------------------
    //! @brief      入力イベントデータの受信用のバッファを登録します。
    //!
    //! @param[in]  buf         登録する入力イベント受信バッファ。
    //! @param[in]  bufSize     登録する入力イベント受信バッファのサイズ。
    //!
    //! @details
    //! 指定したバッファに内部で使用するための情報エリアを確保します。
    //! mcs::MCSHID_RECEIVEBUFFER_MINSIZE 以上のサイズのバッファを割り当てるようにしてください。
    //!
    //! 受信用バッファが入力イベントデータで一杯になり、新規に受信したデータを格納するだけの
    //! 空き容量が無い場合は、その受信データは捨てられます。
    //! 従って、通信で使用するデータ量に合わせてバッファサイズを十分な大きさに
    //! 設定する必要があります。
    //!
    //! @sa McsHID_UnregisterBuffer
    //---------------------------------------------------------------------------
    void            McsHID_RegisterBuffer(void* buf, u32 bufSize);

    //---------------------------------------------------------------------------
    //! @brief      入力イベントデータの受信用のバッファの登録を解除します。
    //!
    //! @details
    //! McsHID_RegisterBuffer() で登録した受信用バッファを解除します。
    //!
    //! @sa McsHID_RegisterBuffer
    //---------------------------------------------------------------------------
    void            McsHID_UnregisterBuffer();

    //---------------------------------------------------------------------------
    //! @brief      データ受信用に登録されたバッファを取得します。
    //!
    //! @return     McsHID_RegisterBuffer で登録されたバッファです。
    //!             登録されたバッファが存在しない場合は NULL を返します。
    //---------------------------------------------------------------------------
    void*           McsHID_GetRegisteredBuffer();

    //---------------------------------------------------------------------------
    //! @brief      イベントを処理する InputEventHandler オブジェクトを登録します。
    //!
    //! @param[in]   pInputEventHandler   登録する InputEventHandler オブジェクトへのポインタ。
    //!
    //! @details
    //! あらかじめ、 McsHID_RegisterBuffer() でバッファを登録しておく必要があります。
    //!
    //! McsHID_Polling() を呼び出したときに、
    //! 受信した入力イベントに応じて引数 pInputEventHandler で指定したインスタンスの
    //! 各メソッドを呼び出します。
    //!
    //! 引数 pInputEventHandler で指定したインスタンスは InputEventHandler クラスを継承し、
    //! 必要なイベントに応じたメソッドをオーバーライドします。
    //!
    //! InputEventHandler クラスの既定の実装は何も行いません。
    //!
    //! @sa InputEventHandler
    //! @sa McsHID_RegisterBuffer
    //! @sa McsHID_Polling
    //---------------------------------------------------------------------------
    void            McsHID_RegisterEventHandler(InputEventHandler* pInputEventHandler);

    //@}

    //! @name 入力デバイス・制御
    //@{

    //---------------------------------------------------------------------------
    //! @brief      イベントのディスパッチを行います。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::MCSHID_Result)を返します。
    //!
    //! @details
    //! 受信した入力イベントを処理し、イベントに応じて登録されている
    //! コールバック関数を呼び出します。
    //!
    //! メインループ内で定期的に呼び出される必要があります。
    //!
    //! Mcs_Polling() でデータを受信した後に呼び出します。
    //!
    //! @sa Mcs_Polling
    //! @sa McsHID_Polling
    //---------------------------------------------------------------------------
    u32             McsHID_ProcessEvent();

    //---------------------------------------------------------------------------
    //! @brief      データの受信とイベントのディスパッチを行います。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::MCSHID_Result)を返します。
    //!
    //! @details
    //! まず Mcs_Polling()を呼び出し、
    //! 成功した場合には McsHID_ProcessEvent() を呼び出します。
    //!
    //! @sa Mcs_Polling
    //! @sa McsHID_ProcessEvent
    //---------------------------------------------------------------------------
    u32             McsHID_Polling();

    //---------------------------------------------------------------------------
    //! @brief      入力デバイスのキャプチャを終了します。
    //!
    //! @return     関数が成功した場合0を返します。
    //!             失敗した場合エラーコード(mcs::MCSHID_Result)を返します。
    //---------------------------------------------------------------------------
    u32             McsHID_StopCapture();

    //@}

#else   // #if defined(NW_MCS_ENABLE)

    inline void     McsHID_Initialize()                                                      {}

    inline void     McsHID_Finalize()                                                        {}

    inline bool     McsHID_IsInitialized() { return false; }

    inline void     McsHID_RegisterBuffer(void* /* buf*/, u32 /* bufSize */)                 {}

    inline void     McsHID_UnregisterBuffer()                                                {}

    inline void*    McsHID_GetRegisteredBuffer()                                             { return NULL; }

    inline void     McsHID_RegisterEventHandler(InputEventHandler* /* pInputEventHandler */) {}

    inline u32      McsHID_ProcessEvent()                                                    { return MCSHID_ERROR_COMERROR; }

    inline u32      McsHID_Polling()                                                         { return MCSHID_ERROR_COMERROR; }

    inline u32      McsHID_StopCapture()                                                     { return MCSHID_ERROR_COMERROR; }

#endif  // #if defined(NW_MCS_ENABLE)

}   // namespace mcs
}   // namespace nw

/* NW_MCS_INPUTDEVICE_H_ */
#endif
