﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_VECTOR4_H_
#define NW_MATH_VECTOR4_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct VEC4;

//---------------------------------------------------------------------------
//    VEC4用の関数
//---------------------------------------------------------------------------

namespace internal { namespace standard {

    VEC4* VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale);
    VEC4* VEC4Lerp(VEC4* pOut, const VEC4* p1, const VEC4* p2, f32 t);
    f32 VEC4Dot(const VEC4* p1, const VEC4* p2);
    f32 VEC4LenSq(const VEC4* p);
    f32 VEC4DistSq(const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    VEC4* VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale);
    VEC4* VEC4Lerp(VEC4* pOut, const VEC4* p1, const VEC4* p2, f32 t);
    f32 VEC4Dot(const VEC4* p1, const VEC4* p2);
    f32 VEC4LenSq(const VEC4* p);
    f32 VEC4DistSq(const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2);
    VEC4* VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

NW_MATH_INLINE bool VEC4IsZero(const VEC4* p);
NW_MATH_INLINE bool VEC4IsZeroWOne(const VEC4* p);
NW_MATH_INLINE VEC4* VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2);
NW_MATH_INLINE VEC4* VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2);
NW_MATH_INLINE VEC4* VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale);
NW_MATH_INLINE VEC4* VEC4Lerp(VEC4* pOut, const VEC4* p1, const VEC4* p2, f32 t);
NW_MATH_INLINE f32   VEC4Dot(const VEC4* p1, const VEC4* p2);
NW_MATH_INLINE f32   VEC4LenSq(const VEC4* p);
NW_MATH_INLINE f32   VEC4Len(const VEC4* p);
NW_MATH_INLINE VEC4* VEC4Normalize(VEC4* pOut, const VEC4* p);
NW_MATH_INLINE VEC4* VEC4FastNormalize(VEC4* pOut, const VEC4* p);
NW_MATH_INLINE VEC4* VEC4SafeNormalize(VEC4* pOut, const VEC4* p, const VEC4& alt);
NW_MATH_INLINE VEC4* VEC4FastSafeNormalize(VEC4* pOut, const VEC4* p, const VEC4& alt);
NW_MATH_INLINE f32   VEC4DistSq(const VEC4* p1, const VEC4* p2);
NW_MATH_INLINE VEC4* VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2);
NW_MATH_INLINE VEC4* VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2);

//========================================================================
//        クラスの定義
//========================================================================
//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        4次のベクトルの構造体です
//---------------------------------------------------------------------------
struct VEC4_
{
    f32 x;
    f32 y;
    f32 z;
    f32 w;
};


//---------------------------------------------------------------------------
//!   @brief        4次のベクトルクラスです。
//---------------------------------------------------------------------------
struct VEC4 : public VEC4_
{
public:
    static const int DIMENSION = 4; //!< 次元数です。

    //! @brief ゼロベクトルです。
    //!
    //! @return ゼロベクトルを返します。
    static const VEC4& Zero()
    {
        static const VEC4 zero(0.0f, 0.0f, 0.0f, 0.0f);

        return zero;
    }

    //!< @brief 同次座標でのゼロベクトルです。
    static const VEC4& ZeroWOne()
    {
        static const VEC4 zero(0.0f, 0.0f, 0.0f, 1.0f);

        return zero;
    }

    typedef VEC4 self_type; //!< 自分の型です
    typedef f32  value_type; //!< 要素の型です。
public:

    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    VEC4() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit VEC4(const f32* p) { x = p[0]; y = p[1]; z = p[2]; w = p[3]; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] v メンバ変数の初期化に用いる VEC4 です。
    explicit VEC4(const VEC4_& v) { x = v.x; y = v.y; z = v.z; w = v.w; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] fx メンバ変数 x の初期化に用いる浮動小数点値です。
    //! @param[in] fy メンバ変数 y の初期化に用いる浮動小数点値です。
    //! @param[in] fz メンバ変数 z の初期化に用いる浮動小数点値です。
    //! @param[in] fw メンバ変数 w の初期化に用いる浮動小数点値です。
    VEC4(f32 fx, f32 fy, f32 fz, f32 fw) { x = fx; y = fy; z = fz; w = fw; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] v メンバ変数の初期化に用いる VEC3 です。メンバ変数 w には 0.0f が設定されます。
    explicit VEC4(const VEC3& v) { x = v.x; y = v.y; z = v.z; w = 0.0f; }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return &x; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return &x; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return &x; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return &x; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC4));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC4));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //@}

    //----------------------------------------
    //! @name 演算
    //@{
    self_type& operator += (const self_type& rhs) { (void)VEC4Add(this, this, &rhs); return *this; }
    self_type& operator -= (const self_type& rhs) { (void)VEC4Sub(this, this, &rhs); return *this; }
    self_type& operator *= (f32 f) { (void)VEC4Scale(this, this, f); return *this; }
    self_type& operator /= (f32 f) { (void)VEC4Scale(this, this, 1/f); return *this; }

    self_type operator + () const { return *this; }
    self_type operator - () const { return self_type(-x, -y, -z, -w); }

    self_type operator + (const self_type& rhs) const { VEC4 tmp; (void)VEC4Add(&tmp, this, &rhs); return tmp; }
    self_type operator - (const self_type& rhs) const { VEC4 tmp; (void)VEC4Sub(&tmp, this, &rhs); return tmp; }
    self_type operator * (f32 f) const { VEC4 tmp; (void)VEC4Scale(&tmp, this, f); return tmp; }
    self_type operator / (f32 f) const { f32 r = 1.f / f; return operator*(r); }

    //! @brief 2つのベクトル間の線形補間を計算し設定します。
    //!
    //! @param[in] lhs 線形補間の始点となるベクトルです。
    //! @param[in] rhs 線形補間の終点となるベクトルです。
    //! @param[in] t   線形補間のパラメータ。0.0 であれば lhs が 1.0 であれば rhs が結果となります。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& Lerp(const VEC4& lhs, const VEC4& rhs, f32 t)
    {
        return *VEC4Lerp(this, &lhs, &rhs, t);
    }

    //! @brief 指定したベクトルとの内積を計算します。
    //!
    //! @param[in] vec 内積を行うベクトルです。
    //! @return 計算結果です。
    //!
    f32 Dot(const VEC4& vec) const
    {
        return VEC4Dot(this, &vec);
    }

    //! @brief ベクトルの長さの2乗を計算します。
    //!
    //! @return 計算結果です。
    f32 LengthSquare() const { return VEC4LenSq(this); }

    //! @brief ベクトルの長さを計算します。
    //!
    //! @return 計算結果です。
    f32 Length() const { return VEC4Len(this); }

    //! @brief ベクトルを正規化します。
    //!
    //! @return 処理結果として、オブジェクト自身への参照を返します。
    self_type& Normalize()
    {
        return *VEC4Normalize(this, this);
    }

    //! @brief ベクトルを正規化して設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetNormalize(const VEC4& src)
    {
        return *VEC4Normalize(this, &src);
    }

    //! @brief ベクトルを正規化します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 処理結果として、オブジェクト自身への参照を返します。
    self_type& SafeNormalize(const VEC4& alt)
    {
        return *VEC4SafeNormalize(this, this, alt);
    }

    //! @brief ベクトルを正規化して設定します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetSafeNormalize(const VEC4& src, const VEC4& alt)
    {
        return *VEC4SafeNormalize(this, &src, alt);
    }

    //! @brief 指定したベクトルとの距離の二乗を計算します。
    //!
    //! @param[in] vec 距離を計算するベクトルです。
    //! @return 計算結果です。
    f32 DistanceSquare(const VEC4& vec)
    {
        return VEC4DistSq(this, &vec);
    }

    //! @brief 2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを設定します。
    //!
    //! @param[in] lhs 最小値を計算する左辺ベクトルです。
    //! @param[in] rhs 最小値を計算する右辺ベクトルです。
    //! @return 計算結果として、オブジェクト自身への参照を返します。
    self_type& SetMaximize(const VEC4& lhs, const VEC4& rhs)
    {
        return *VEC4Maximize(this, &lhs, &rhs);
    }

    //! @brief ベクトルの外積を計算し設定します。
    //!
    //! @param[in] lhs 外積を計算する左辺ベクトルです。
    //! @param[in] rhs 外積を計算する右辺ベクトルです。
    //! @return 計算結果として、オブジェクト自身への参照を返します。
    self_type& SetMinimize(const VEC4& lhs, const VEC4& rhs)
    {
        return *VEC4Minimize(this, &lhs, &rhs);
    }


    //! @brief        ベクトルを行列で変換した値を設定します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransform(const MTX34& pM, const VEC4& src);

    //! @brief        ベクトルを行列で変換します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& Transform(const MTX34& pM) { return this->SetTransform(pM, *this); }

    //! @brief        ベクトルを行列で変換した値を設定します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransform(const MTX44& pM, const VEC4& src);

    //! @brief        ベクトルを行列で変換します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& Transform(const MTX44& pM) { return this->SetTransform(pM, *this); }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 値を個別に設定します。
    //!
    //! @param[in] fx メンバ変数 x に設定される浮動小数点値です。
    //! @param[in] fy メンバ変数 y に設定される浮動小数点値です。
    //! @param[in] fz メンバ変数 z に設定される浮動小数点値です。
    //! @param[in] fw メンバ変数 w に設定される浮動小数点値です。
    void Set(f32 fx, f32 fy, f32 fz, f32 fw) { x = fx; y = fy; z = fz; w = fw; }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC4 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return x == rhs.x && y == rhs.y && z == rhs.z && w == rhs.w; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC2 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return x != rhs.x || y != rhs.y || z != rhs.z || w != rhs.w; }

    //! @brief ゼロベクトルかどうかを判定します。
    //!
    //! @return ゼロベクトルであれば true を返します。
    bool IsZero() const { return VEC4IsZero(this); }

    //! @brief 同次座標のゼロベクトルかどうかを判定します。
    //!
    //! @return 同次座標のゼロベクトルであれば true を返します。
    bool IsZeroWOne() const { return VEC4IsZeroWOne(this); }
    //@}

    //! @brief ベクトルの内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      ベクトルの名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct VEC4 Vector4;  //!< @briefprivate

//---------------------------------------------------------------------------
//! @briefprivate
//! @details VEC4 と浮動小数点数の積を計算します。
//! @param[in] f 積に用いる浮動小数点数です。
//! @param[in] rhs 積に用いる VEC4 です。
//! @return 計算結果の VEC4 を返します。
//---------------------------------------------------------------------------
inline VEC4
operator * (f32 f, const VEC4& rhs) { VEC4 tmp; (void)VEC4Scale(&tmp, &rhs, f); return tmp; }


}  // namespace math
}  // namespace nw


#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Vector4.ipp>
#endif

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline bool  VEC4IsZero(const VEC4& v){ return VEC4IsZero( &v ); }
inline bool  VEC4IsZeroWOne(const VEC4& v){ return VEC4IsZeroWOne( &v ); }
inline VEC4* VEC4Add(VEC4* pOut, const VEC4& v1, const VEC4& v2) { return VEC4Add( pOut, &v1, &v2 ); }
inline VEC4* VEC4Sub(VEC4* pOut, const VEC4& v1, const VEC4& v2) { return VEC4Sub( pOut, &v1, &v2 ); }
inline VEC4* VEC4Scale(VEC4* pOut, const VEC4& v, f32 scale) { return VEC4Scale( pOut, &v, scale); }
inline VEC4* VEC4Lerp(VEC4* pOut, const VEC4& v1, const VEC4& v2, f32 t) { return VEC4Lerp( pOut, &v1, &v2, t ); }
inline f32   VEC4Dot(const VEC4& v1, const VEC4& v2) { return VEC4Dot( &v1, &v2 ); }
inline f32   VEC4LenSq(const VEC4& v) { return VEC4LenSq( &v ); }
inline f32   VEC4Len(const VEC4& v) { return VEC4Len( &v ); }
inline VEC4* VEC4Normalize(VEC4* pOut, const VEC4& v) { return VEC4Normalize( pOut, &v ); }
inline VEC4* VEC4FastNormalize(VEC4* pOut, const VEC4& v) { return VEC4FastNormalize( pOut, &v ); }
inline VEC4* VEC4SafeNormalize(VEC4* pOut, const VEC4& v, const VEC4& alt) { return VEC4SafeNormalize( pOut, &v, alt ); }
inline VEC4* VEC4FastSafeNormalize(VEC4* pOut, const VEC4& v, const VEC4& alt) { return VEC4FastSafeNormalize( pOut, &v, alt ); }
inline f32   VEC4DistSq(const VEC4& v1, const VEC4& v2) { return VEC4DistSq( &v1, &v2 ); }
inline VEC4* VEC4Maximize(VEC4* pOut, const VEC4& v1, const VEC4& v2) { return VEC4Maximize( pOut, &v1, &v2 ); }
inline VEC4* VEC4Minimize(VEC4* pOut, const VEC4& v1, const VEC4& v2) { return VEC4Minimize( pOut, &v1, &v2 ); }


}  // namespace math
}  // namespace nw


#endif
