﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_VECTOR3_H_
#define NW_MATH_VECTOR3_H_

#include <cstring>
#include <nw/math/math_Config.h>
#include <nw/math/math_Constant.h>


namespace nw {
namespace math {

struct VEC3;
struct MTX33;
struct MTX34;
struct MTX44;

namespace internal { namespace standard {

    VEC3* VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
    VEC3* VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
    f32 VEC3SquareDist(const VEC3* p1, const VEC3* p2);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    VEC3* VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
    VEC3* VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
    f32 VEC3SquareDist(const VEC3* p1, const VEC3* p2);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

// (x, y, z, 0)として変換
NW_MATH_INLINE bool VEC3IsZero(const VEC3* p);
NW_MATH_INLINE VEC3* VEC3Maximize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
NW_MATH_INLINE VEC3* VEC3Minimize(VEC3* pOut, const VEC3* p1, const VEC3* p2);
NW_MATH_INLINE VEC3* VEC3Cross(VEC3* pOut, const VEC3* p1, const VEC3* p2);
NW_MATH_INLINE VEC3* VEC3Normalize(VEC3* pOut, const VEC3* p);
NW_MATH_INLINE VEC3* VEC3FastNormalize(VEC3* pOut, const VEC3* p);
NW_MATH_INLINE VEC3* VEC3SafeNormalize(VEC3* pOut, const VEC3* p, const VEC3& alt);
NW_MATH_INLINE VEC3* VEC3FastSafeNormalize(VEC3* pOut, const VEC3* p, const VEC3& alt);
NW_MATH_INLINE f32   VEC3SquareDist(const VEC3* p1, const VEC3* p2);

inline VEC3* VEC3Add(VEC3* pOut, const VEC3* p1, const VEC3* p2);
inline VEC3* VEC3Sub(VEC3* pOut, const VEC3* p1, const VEC3* p2);
inline VEC3* VEC3Mult(VEC3* pOut, const VEC3* p1, const VEC3* p2);
inline VEC3* VEC3Scale(VEC3* pOut, const VEC3* p, f32 scale);
inline VEC3* VEC3Lerp(VEC3* pOut, const VEC3* p1, const VEC3* p2, f32 t);
inline f32   VEC3Dot(const VEC3* p1, const VEC3* p2);
inline f32   VEC3Len(const VEC3* p);
inline f32   VEC3SquareLen(const VEC3* p);
inline f32   VEC3Dist(const VEC3* p1, const VEC3* p2);

//========================================================================
//        クラスの定義
//========================================================================
//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        3次のベクトルの構造体です
//---------------------------------------------------------------------------
struct VEC3_
{
    f32 x;
    f32 y;
    f32 z;
};

//---------------------------------------------------------------------------
//!   @brief        3次のベクトルクラスです。
//---------------------------------------------------------------------------
struct VEC3 : public VEC3_
{
public:
    static const int DIMENSION = 3; //!< 次元数です。

    //!< @brief ゼロベクトルです。
    static const VEC3& Zero()
    {
        static const VEC3 zero(0.0f, 0.0f, 0.0f);

        return zero;
    }

    typedef VEC3 self_type; //!< 自分の型です
    typedef f32  value_type; //!< 要素の型です。
public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    VEC3() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit VEC3(const f32* p) { x = p[0]; y = p[1]; z = p[2]; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] v メンバ変数の初期化に用いる VEC3 です。
    explicit VEC3(const VEC3_& v) { x = v.x; y = v.y; z = v.z; }
    //! @brief コンストラクタです。
    //!
    //! @param[in] fx メンバ変数 x の初期化に用いる浮動小数点値です。
    //! @param[in] fy メンバ変数 y の初期化に用いる浮動小数点値です。
    //! @param[in] fz メンバ変数 z の初期化に用いる浮動小数点値です。
    VEC3(f32 fx, f32 fy, f32 fz) { x = fx; y = fy; z = fz; }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{
    //! @brief f32 型へのキャストです。
    operator f32*() { return &x; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return &x; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return &x; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return &x; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC3));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(typename ut::PeelPointer<ToPtr>::Type) <= sizeof(VEC3));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //@}

    //----------------------------------------
    //! @name 演算
    //@{

    self_type& operator += (const self_type& rhs) { (void)VEC3Add(this, this, &rhs); return *this; }
    self_type& operator -= (const self_type& rhs) { (void)VEC3Sub(this, this, &rhs); return *this; }
    self_type& operator *= (const self_type& rhs) { (void)VEC3Mult(this, this, &rhs); return *this; }
    self_type& operator *= (f32 f) { (void)VEC3Scale(this, this, f); return *this; }
    self_type& operator /= (f32 f) { return operator*=(1.f / f); }

    self_type operator + () const { return *this; }
    self_type operator - () const { return self_type(-x, -y, -z); }

    // ２項演算子は戻り値最適化が行われるかどうかで最適な実装が異なる
    self_type operator + (const self_type& rhs) const { VEC3 tmp; (void)VEC3Add(&tmp, this, &rhs); return tmp; }
    self_type operator - (const self_type& rhs) const { VEC3 tmp; (void)VEC3Sub(&tmp, this, &rhs); return tmp; }
    self_type operator * (f32 f) const { VEC3 tmp; (void)VEC3Scale(&tmp, this, f); return tmp; }
    self_type operator / (f32 f) const { f32 r = 1.f / f; return operator*(r); }

    //! @brief 2つのベクトル間の線形補間を計算し設定します。
    //!
    //! @param[in] lhs 線形補間の始点となるベクトルです。
    //! @param[in] rhs 線形補間の終点となるベクトルです。
    //! @param[in] t   線形補間のパラメータ。0.0 であれば lhs が 1.0 であれば rhs が結果となります。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& Lerp(const VEC3& lhs, const VEC3& rhs, f32 t)
    {
        return *VEC3Lerp(this, &lhs, &rhs, t);
    }

    //! @brief 指定したベクトルとの内積を計算します。
    //!
    //! @param[in] vec 内積を行うベクトルです。
    //! @return 計算結果です。
    //!
    f32 Dot(const VEC3& vec) const
    {
        return VEC3Dot(this, &vec);
    }

    //! @brief ベクトルの長さの2乗を計算します。
    //!
    //! @return 計算結果です。
    f32 LengthSquare() const { return VEC3SquareLen(this); }

    //! @brief ベクトルの長さを計算します。
    //!
    //! @return 計算結果です。
    f32 Length() const { return VEC3Len(this); }

    //! @brief ベクトルを正規化します。
    //!
    //! @return 処理結果として、オブジェクト自身への参照を返します。
    self_type& Normalize()
    {
        return *VEC3Normalize(this, this);
    }

    //! @brief ベクトルを正規化して設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetNormalize(const VEC3& src)
    {
        return *VEC3Normalize(this, &src);
    }

    //! @brief ベクトルを正規化します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 処理結果として、正規化に成功した場合はオブジェクト自身への参照を、失敗した場合は指定されたベクトルを返します。
    self_type& SafeNormalize(const VEC3& alt)
    {
        return *VEC3SafeNormalize(this, this, alt);
    }

    //! @brief ベクトルを正規化して設定します。
    //!        正規化に失敗した場合は指定されたベクトルを設定します。
    //!
    //! @param[in]    src   正規化する対象のベクトルです。
    //! @param[in]    alt   正規化に失敗した場合に設定するベクトルです。
    //! @return 設定結果として、正規化に成功した場合はオブジェクト自身への参照を、失敗した場合は指定した参照を返します。
    self_type& SetSafeNormalize(const VEC3& src, const VEC3& alt)
    {
        return *VEC3SafeNormalize(this, &src, alt);
    }

    //! @brief 指定したベクトルとの距離の二乗を計算します。
    //!
    //! @param[in] vec 距離を計算するベクトルです。
    //! @return 計算結果です。
    f32 DistanceSquare(const VEC3& vec)
    {
        return VEC3SquareDist(this, &vec);
    }

    //! @brief 指定したベクトルとの距離を計算します。
    //!
    //! @param[in] vec 距離を計算するベクトルです。
    //! @return 計算結果です。
    f32 Distance(const VEC3& vec)
    {
        return VEC3Dist(this, &vec);
    }

    //! @brief 2つのベクトルのそれぞれの成分の最大値から構成されるベクトルを設定します。
    //!
    //! @param[in] lhs 最大値を計算する左辺ベクトルです。
    //! @param[in] rhs 最大値を計算する右辺ベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetMaximize(const VEC3& lhs, const VEC3& rhs)
    {
        return *VEC3Maximize(this, &lhs, &rhs);
    }

    //! @brief 2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを設定します。
    //!
    //! @param[in] lhs 最小値を計算する左辺ベクトルです。
    //! @param[in] rhs 最小値を計算する右辺ベクトルです。
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    self_type& SetMinimize(const VEC3& lhs, const VEC3& rhs)
    {
        return *VEC3Minimize(this, &lhs, &rhs);
    }

    //! @brief this を左辺値としてベクトルの外積を計算します。
    //!
    //! @param[in] rhs 外積を計算する右辺ベクトルです。
    //! @return 計算結果として、オブジェクト自身への参照を返します。
    self_type& Cross(const VEC3& rhs)
    {
        return *VEC3Cross(this, this, &rhs);
    }

    //! @brief ベクトルの外積を計算し設定します。
    //!
    //! @param[in] lhs 外積を計算する左辺ベクトルです。
    //! @param[in] rhs 外積を計算する右辺ベクトルです。
    //! @return 計算結果として、オブジェクト自身への参照を返します。
    self_type& SetCross(const VEC3& lhs, const VEC3& rhs)
    {
        return *VEC3Cross(this, &lhs, &rhs);
    }

    //! @brief        ベクトルを行列で変換した値を設定します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransform(const MTX33& pM, const VEC3& src);

    //! @brief        ベクトルを行列で変換します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& Transform(const MTX33& pM) { return this->SetTransform(pM, *this); }

    //! @brief        ベクトルを行列で変換した値を設定します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransform(const MTX34& pM, const VEC3& src);

    //! @brief        ベクトルを行列で変換します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& Transform(const MTX34& pM) { return this->SetTransform(pM, *this); }

    //! @brief        ベクトルを行列で変換した値を設定します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransform(const MTX44& pM, const VEC3& src);

    //! @brief        ベクトルを行列で変換します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& Transform(const MTX44& pM) { return this->SetTransform(pM, *this); }

    //! @brief        ベクトルを行列で変換した値を設定します。ベクトルの 4 要素目を 0 として計算します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @param[in]    src   変換対象となるベクトル。
    //! @return       設定結果として、オブジェクト自身への参照を返します。
    NW_MATH_INLINE self_type& SetTransformNormal(const MTX34& pM, const VEC3& src);

    //! @brief        ベクトルを行列で変換します。ベクトルの 4 要素目を 0 として計算します。
    //!
    //! @param[in]    pM    変換行列へのポインタ。
    //! @return       変換結果として、オブジェクト自身への参照を返します。
    self_type& TransformNormal(const MTX34& pM) { return this->SetTransform(pM, *this); }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 値を個別に設定します。
    //!
    //! @param[in] fx メンバ変数 x に設定される浮動小数点値です。
    //! @param[in] fy メンバ変数 y に設定される浮動小数点値です。
    //! @param[in] fz メンバ変数 z に設定される浮動小数点値です。
    void Set(f32 fx, f32 fy, f32 fz) { x = fx; y = fy; z = fz; }

    //! @brief 値を設定します。
    //!
    //! @param[in] value 値の設定に用いる VEC3 です。
    void Set(const self_type& value) { x = value.x; y = value.y; z = value.z; }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{
    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC3 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return x == rhs.x && y == rhs.y && z == rhs.z; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる VEC2 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return x != rhs.x || y != rhs.y || z != rhs.z; }

    //! @brief ゼロベクトルかどうかを判定します。
    //!
    //! @return ゼロベクトルであれば true を返します。
    bool IsZero() const { return VEC3IsZero(this); }

    //@}

    //! @brief ベクトルの内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      ベクトルの名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct VEC3 Vector3;  //!< @briefprivate

//---------------------------------------------------------------------------
//    VEC3用の関数
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_INLINE VEC3*
VEC3Add(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x + p2->x;
    pOut->y = p1->y + p2->y;
    pOut->z = p1->z + p2->z;
    return pOut;
}

NW_INLINE VEC3*
VEC3Sub(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x - p2->x;
    pOut->y = p1->y - p2->y;
    pOut->z = p1->z - p2->z;
    return pOut;
}

NW_INLINE VEC3*
VEC3Mult(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x * p2->x;
    pOut->y = p1->y * p2->y;
    pOut->z = p1->z * p2->z;
    return pOut;
}

NW_INLINE VEC3*
VEC3Scale(VEC3* pOut, const VEC3* p, f32 scale)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->x = scale * p->x;
    pOut->y = scale * p->y;
    pOut->z = scale * p->z;
    return pOut;
}

NW_INLINE VEC3*
VEC3Lerp(VEC3* pOut, const VEC3* p1, const VEC3* p2, f32 t)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->x = p1->x + t * (p2->x - p1->x);
    pOut->y = p1->y + t * (p2->y - p1->y);
    pOut->z = p1->z + t * (p2->z - p1->z);
    return pOut;
}

NW_INLINE f32
VEC3Dot(const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    return p1->x * p2->x + p1->y * p2->y + p1->z * p2->z;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_INLINE VEC3*
VEC3Add(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->x) = __PS_ADD(tof32x2(p1->x), tof32x2(p2->x));

    pOut->z = p1->z + p2->z;

    return pOut;
}

NW_INLINE VEC3*
VEC3Sub(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->x) = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));

    pOut->z = p1->z - p2->z;

    return pOut;
}

NW_INLINE VEC3*
VEC3Mult(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->x) = __PS_MUL(tof32x2(p1->x), tof32x2(p2->x));

    pOut->z = p1->z * p2->z;

    return pOut;
}

NW_INLINE VEC3*
VEC3Scale(VEC3* pOut, const VEC3* p, f32 scale)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    tof32x2(pOut->x) = __PS_MULS0F(tof32x2(p->x), scale);

    pOut->z = scale * p->z;

    return pOut;
}

NW_INLINE VEC3*
VEC3Lerp(VEC3* pOut, const VEC3* p1, const VEC3* p2, f32 t)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    f32x2 tt = __PS_FDUP(t);
    f32x2 f0 = tof32x2(p1->x);
    f0 = __PS_NMSUB(f0, tt, f0);
    tof32x2(pOut->x) = __PS_MADD(tof32x2(p2->x), tt, f0);

    pOut->z = p1->z + t * (p2->z - p1->z);

    return pOut;
}

NW_INLINE f32
VEC3Dot(const VEC3* p1, const VEC3* p2)
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    f32x2 f0;
    f0 = __PS_MUL(tof32x2(p1->x), tof32x2(p2->x));
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0] + p1->z * p2->z;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ベクトル
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルの和を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC3*
VEC3Add(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3Add(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの差を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC3*
VEC3Sub(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3Sub(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの積を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC3*
VEC3Mult(VEC3* pOut, const VEC3* p1, const VEC3* p2)
{
    return NW_MATH_IMPL_NS::VEC3Mult(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルのスカラー倍を計算します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p      左辺値へのポインタ
//! @param[in]    scale  掛ける数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC3*
VEC3Scale(VEC3* pOut, const VEC3* p, f32 scale)
{
    return NW_MATH_IMPL_NS::VEC3Scale(pOut, p, scale);
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の線形補間を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    線形補間の始点となるベクトルへのポインタ
//! @param[in]    p2    線形補間の終点となるベクトルへのポインタ
//! @param[in]    t     線形補間のパラメータ。0.0 であれば p1 が 1.0 であれば p2 が結果となります。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
inline VEC3*
VEC3Lerp(VEC3* pOut, const VEC3* p1, const VEC3* p2, f32 t)
{
    return NW_MATH_IMPL_NS::VEC3Lerp(pOut, p1, p2, t);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの内積を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の内積を返します。
//---------------------------------------------------------------------------
inline f32
VEC3Dot(const VEC3* p1, const VEC3* p2)
{
    return internal::standard::VEC3Dot(p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの長さの2乗を計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さの2乗を返します。
//---------------------------------------------------------------------------
inline f32
VEC3SquareLen(const VEC3* p)
{
    return p->x * p->x + p->y * p->y + p->z * p->z;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの長さを計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さを返します。
//---------------------------------------------------------------------------
inline f32
VEC3Len(const VEC3* p)
{
    NW_ASSERT_NOT_NULL( p );

    return ::std::sqrtf( VEC3SquareLen( p ) );
}


//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の距離を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の距離を返します。
//---------------------------------------------------------------------------
inline f32
VEC3Dist( const VEC3* p1, const VEC3* p2 )
{
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    return ::std::sqrtf( VEC3SquareDist( p1, p2 ) );
}


//---------------------------------------------------------------------------
//! @briefprivate
//! @details VEC3 と浮動小数点数の積を計算します。
//! @param[in] f 積に用いる浮動小数点数です。
//! @param[in] rhs 積に用いる VEC3 です。
//! @return 計算結果の VEC3 を返します。
//---------------------------------------------------------------------------
inline VEC3
operator * (f32 f, const VEC3& rhs) { VEC3 tmp; (void)VEC3Scale(&tmp, &rhs, f); return tmp; }

//@}

}  // namespace math
}  // namespace nw



#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Vector3.ipp>
#endif

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline bool VEC3IsZero(const VEC3& v){ return VEC3IsZero( &v ); }
inline VEC3* VEC3Maximize(VEC3* pOut, const VEC3& v1, const VEC3& v2) { return VEC3Maximize( pOut, &v1, &v2 ); }
inline VEC3* VEC3Minimize(VEC3* pOut, const VEC3& v1, const VEC3& v2) { return VEC3Minimize( pOut, &v1, &v2 ); }
inline VEC3* VEC3Cross(VEC3* pOut, const VEC3& v1, const VEC3& v2) { return VEC3Cross( pOut, &v1, &v2 ); }
inline VEC3* VEC3Normalize(VEC3* pOut, const VEC3& v) { return VEC3Normalize( pOut, &v ); }
inline VEC3* VEC3FastNormalize(VEC3* pOut, const VEC3& v) { return VEC3FastNormalize( pOut, &v ); }
inline VEC3* VEC3SafeNormalize(VEC3* pOut, const VEC3& v, const VEC3& alt) { return VEC3SafeNormalize( pOut, &v, alt ); }
inline VEC3* VEC3FastSafeNormalize(VEC3* pOut, const VEC3& v, const VEC3& alt) { return VEC3FastSafeNormalize( pOut, &v, alt ); }
inline f32   VEC3SquareDist(const VEC3& v1, const VEC3& v2) { return VEC3SquareDist( &v1, &v2 ); }

inline VEC3* VEC3Add(VEC3* pOut, const VEC3& v1, const VEC3& v2) { return VEC3Add( pOut, &v1, &v2 ); }
inline VEC3* VEC3Sub(VEC3* pOut, const VEC3& v1, const VEC3& v2) { return VEC3Sub( pOut, &v1, &v2 ); }
inline VEC3* VEC3Scale(VEC3* pOut, const VEC3& v, f32 scale) { return VEC3Scale( pOut, &v, scale ); }
inline VEC3* VEC3Lerp(VEC3* pOut, const VEC3& v1, const VEC3& v2, f32 t) { return VEC3Lerp( pOut, &v1, &v2, t ); }
inline f32   VEC3Dot(const VEC3& v1, const VEC3& v2) { return VEC3Dot( &v1, &v2 ); }
inline f32   VEC3Len(const VEC3& v) { return VEC3Len( &v ); }
inline f32   VEC3SquareLen(const VEC3& v) { return VEC3SquareLen( &v ); }
inline f32   VEC3Dist(const VEC3& v1, const VEC3& v2) { return VEC3Dist( &v1, &v2 ); }

}  // namespace math
}  // namespace nw


#endif
