﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_TYPES_H_
#define NW_MATH_TYPES_H_

#include <nw/types.h>
#include <nw/ut/ut_TypeTraits.h>
#include <nw/math/math_Config.h>
#include <nw/math/math_Triangular.h>
#include <nw/math/math_Vector2.h>
#include <nw/math/math_Vector3.h>
#include <nw/math/math_Vector4.h>
#include <nw/math/math_Quaternion.h>
#include <nw/math/math_Matrix22.h>
#include <nw/math/math_Matrix23.h>
#include <nw/math/math_Matrix33.h>
#include <nw/math/math_Matrix34.h>
#include <nw/math/math_Matrix44.h>
#include <nw/math/math_Matrix43.h>
#include <nw/math/math_Transform.h>

#include <cstring>

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//! @brief        サイズチェック付きの reinterpret_cast です。
//!
//! @param[in]    ptr     キャストするポインタです。
//!
//! @return       キャスト後のポインタです。
//---------------------------------------------------------------------------
template <typename ToPtr, typename FromPtr>
ToPtr Cast(FromPtr ptr)
{
    NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(ut::PeelPointer<FromPtr>::Type));
    NW_STATIC_ASSERT(ut::IsPointer<FromPtr>::value);
    NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

    return reinterpret_cast<ToPtr>( ptr );
}


//========================================================================
//        クラスの定義
//========================================================================
struct VEC2;
struct VEC3;
struct VEC4;
struct QUAT;

namespace internal { namespace standard {

    VEC3* VEC3TransformArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, s32 count);
    VEC4* VEC3TransformArray(VEC4* pOut, const MTX44* pM, const VEC3* pV, u32 count);
    VEC3* VEC3TransformNormal(VEC3* pOut, const MTX34* pM, const VEC3* pV);
    VEC3* VEC3TransformNormalArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, u32 count);
    VEC3* VEC3TransformCoord(VEC3* pOut, const MTX44* pM, const VEC3* pV);
    VEC3* VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count);
    VEC4* VEC4Transform(VEC4* pOut, const MTX44* pM, const VEC4* pV);
    VEC4* VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 count);
    VEC4* VEC4Transform(VEC4* pOut, const MTX34* pM, const VEC4* pV);
    VEC4* VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 count);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    VEC3* VEC3TransformArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, s32 count);
    VEC4* VEC3TransformArray(VEC4* pOut, const MTX44* pM, const VEC3* pV, u32 count);
    VEC3* VEC3TransformNormal(VEC3* pOut, const MTX34* pM, const VEC3* pV);
    VEC3* VEC3TransformNormalArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, u32 count);
    VEC3* VEC3TransformCoord(VEC3* pOut, const MTX44* pM, const VEC3* pV);
    VEC3* VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count);
    VEC4* VEC4Transform(VEC4* pOut, const MTX44* pM, const VEC4* pV);
    VEC4* VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 count);
    VEC4* VEC4Transform(VEC4* pOut, const MTX34* pM, const VEC4* pV);
    VEC4* VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 count);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//---------------------------------------------------------------------------
//    MatrixUtility
//---------------------------------------------------------------------------

NW_MATH_INLINE MTX33* MTX34ToMTX33(MTX33* pOut, const MTX34* pM);
NW_MATH_INLINE MTX34* MTX33ToMTX34(MTX34* pOut, const MTX33* pM);

NW_MATH_INLINE MTX34* MTX43Transpose(MTX34* pOut, const MTX43* p);
NW_MATH_INLINE MTX43* MTX34Transpose(MTX43* pOut, const MTX34* p);

}  // namespace math
}  // namespace nw


#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Types.ipp>
#endif

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline MTX34* MTX43Transpose(MTX34* pOut, const MTX43& m) { return MTX43Transpose( pOut, &m ); }
inline MTX43* MTX34Transpose(MTX43* pOut, const MTX34& m) { return MTX34Transpose( pOut, &m ); }


}  // namespace math
}  // namespace nw

#endif


