﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_TRIANGULAR_H_
#define NW_MATH_TRIANGULAR_H_

#include <nw/math/math_Config.h>
#include <nw/math/math_Constant.h>
#include <nw/math/math_Arithmetic.h>

#include <cmath>


namespace nw {
namespace math {
namespace internal {

    struct SinCosSample
    {
        f32 sin_val;    // sin の値
        f32 sin_delta;  // 次の sin_val との差
        f32 cos_val;    // cos の値
        f32 cos_delta;  // 次の cos_val との差
    };

    extern const SinCosSample gSinCosTbl[256 + 1];

    u32 AtanIdx_(f32 x);

} // namespace internal
} // namespace math
} // namespace nw


namespace nw {
namespace math {

//========================================================================
//        sin/cos/tan
//========================================================================
//----------------------------------------
//! @name    三角関数
//@{

#define NW_MATH_RAD_TO_IDX(rad)  (static_cast<u32>(static_cast<s64>((rad) * (::nw::math::I_HALF_ROUND_IDX / ::nw::math::F_PI))))
#define NW_MATH_DEG_TO_IDX(deg)  (static_cast<u32>(static_cast<s64>((deg) * (::nw::math::I_HALF_ROUND_IDX / 180.0f))))
#define NW_MATH_DEG_TO_RAD(deg)  ((deg) * (::nw::math::F_PI / 180.0f))
#define NW_MATH_RAD_TO_DEG(rad)  ((rad) * (180.0f / ::nw::math::F_PI))
#define NW_MATH_IDX_TO_RAD(idx)  ((idx) * (::nw::math::F_PI / ::nw::math::I_HALF_ROUND_IDX))
#define NW_MATH_IDX_TO_DEG(idx)  ((idx) * (180.0f / ::nw::math::I_HALF_ROUND_IDX))


//---------------------------------------------------------------------------
//! @brief        Radian から Degree への変換です。
//!
//! @param[in]    rad    変換する Radian 値です。
//!
//! @return       Degree 値を返します。
//---------------------------------------------------------------------------
inline f32
RadToDeg(f32 rad)
{
    return rad * (180.0f / F_PI);
}


//---------------------------------------------------------------------------
//! @brief        Degree から Radian への変換です。
//!
//! @param[in]    deg    変換する Degree 値です。
//!
//! @return       Radian 値を返します。
//---------------------------------------------------------------------------
inline f32
DegToRad(f32 deg)
{
    return deg * (F_PI / 180.0f);
}

//---------------------------------------------------------------------------
//! @brief        正弦の値を求めます
//!
//! @param[in]    idx  1 円周を 0x100000000 とする角度
//!
//! @return       idx に対する正弦の値を返します。
//---------------------------------------------------------------------------
inline f32
SinIdx(u32 idx)
{
    u32 index = (idx >> 24) & 0xff;
    u32 rest = idx & 0xffffff;

    return internal::gSinCosTbl[index].sin_val + internal::gSinCosTbl[index].sin_delta * rest / 0x1000000;
}

//---------------------------------------------------------------------------
//! @brief        余弦の値を求めます
//!
//! @param[in]    idx  1 円周を 0x100000000 とする角度
//!
//! @return       idx に対する余弦の値を返します。
//---------------------------------------------------------------------------
inline f32
CosIdx(u32 idx)
{
    u32 index = (idx >> 24) & 0xff;
    u32 rest = idx & 0xffffff;

    return internal::gSinCosTbl[index].cos_val + internal::gSinCosTbl[index].cos_delta * rest / 0x1000000;
}

//---------------------------------------------------------------------------
//! @brief        正弦と余弦の値を求めます
//!
//! @param[out]   pSin  正弦の値を格納するバッファへのポインタ
//! @param[out]   pCos  余弦の値を格納するバッファへのポインタ
//! @param[in]    idx  1 円周を 0x100000000 とする角度
//!
//! @return       なし
//---------------------------------------------------------------------------
inline void
SinCosIdx(f32* pSin, f32* pCos, u32 idx)
{
    u32 index = (idx >> 24) & 0xff;
    float rest = static_cast<float>(idx & 0xffffff) / 0x1000000;
    const internal::SinCosSample* table = &internal::gSinCosTbl[index];

    *pSin = table->sin_val + table->sin_delta * rest;
    *pCos = table->cos_val + table->cos_delta * rest;
}

//---------------------------------------------------------------------------
//! @brief        正接の値を求めます
//!
//! @param[in]    idx  1 円周を 0x100000000 とする角度
//!
//! @return       idx に対する正接の値を返します。
//---------------------------------------------------------------------------
inline f32
TanIdx(u32 idx)
{
    u32 index = (idx >> 24) & 0xff;
    float rest = static_cast<float>(idx & 0xffffff) / 0x1000000;
    const internal::SinCosSample* table = &internal::gSinCosTbl[index];

    return (table->sin_val + table->sin_delta * rest) / (table->cos_val + table->cos_delta * rest);
}



//---------------------------------------------------------------------------
//! @brief        正弦の値を求めます。
//!
//! @param[in]    rad  ラジアン単位の角度
//!
//! @return       rad に対する正弦の値を返します。
//---------------------------------------------------------------------------
inline f32 SinRad(f32 rad)  { return SinIdx(NW_MATH_RAD_TO_IDX(rad)); }

//---------------------------------------------------------------------------
//! @brief        余弦の値を求めます。
//!
//! @param[in]    rad  ラジアン単位の角度
//!
//! @return       rad に対する余弦の値を返します。
//---------------------------------------------------------------------------
inline f32 CosRad(f32 rad)  { return CosIdx(NW_MATH_RAD_TO_IDX(rad)); }

//---------------------------------------------------------------------------
//! @brief        正弦と余弦の値を求めます。
//!
//! @param[out]   pSin 正弦の値を格納するバッファへのポインタ
//! @param[out]   pCos 余弦の値を格納するバッファへのポインタ
//! @param[in]    rad  ラジアン単位の角度
//!
//! @return       なし
//---------------------------------------------------------------------------
inline void SinCosRad(f32* pSin, f32* pCos, f32 rad) { SinCosIdx(pSin, pCos, NW_MATH_RAD_TO_IDX(rad)); }

//---------------------------------------------------------------------------
//! @brief        正接の値を求めます。
//!
//! @param[in]    rad  ラジアン単位の角度
//!
//! @return       rad に対する正接の値を返します。
//---------------------------------------------------------------------------
inline f32 TanRad(f32 rad)  { return TanIdx(NW_MATH_RAD_TO_IDX(rad)); }



//---------------------------------------------------------------------------
//! @brief        正弦の値を求めます。
//!
//! @param[in]    deg  1 円周を 360.0 とする角度
//!
//! @return       deg に対する正弦の値を返します。
//---------------------------------------------------------------------------
inline f32 SinDeg(f32 deg)  { return SinIdx(NW_MATH_DEG_TO_IDX(deg)); }

//---------------------------------------------------------------------------
//! @brief        余弦の値を求めます。
//!
//! @param[in]    deg  1 円周を 360.0 とする角度
//!
//! @return       deg に対する余弦の値を返します。
//---------------------------------------------------------------------------
inline f32 CosDeg(f32 deg)  { return CosIdx(NW_MATH_DEG_TO_IDX(deg)); }

//---------------------------------------------------------------------------
//! @brief        正弦と余弦の値を求めます。
//!
//! @param[out]   pSin 正弦の値を格納するバッファへのポインタ
//! @param[out]   pCos 余弦の値を格納するバッファへのポインタ
//! @param[in]    deg  1 円周を 360.0 とする角度
//!
//! @return       なし
//---------------------------------------------------------------------------
inline void SinCosDeg(f32* pSin, f32* pCos, f32 deg) { SinCosIdx(pSin, pCos, NW_MATH_DEG_TO_IDX(deg)); }

//---------------------------------------------------------------------------
//! @brief        正接の値を求めます。
//!
//! @param[in]    deg  1 円周を 360.0 とする角度
//!
//! @return       deg に対する正接の値を返します。
//---------------------------------------------------------------------------
inline f32 TanDeg(f32 deg)  { return TanIdx(NW_MATH_DEG_TO_IDX(deg)); }

//---------------------------------------------------------------------------
//! @brief        逆余弦の値を求めます。（標準関数版）
//!
//! @param[in]    x  逆余弦を求める値
//!
//! @return       x に対する逆余弦の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 NW_fAcos(f32 x) { return ::std::acosf(x); }

//---------------------------------------------------------------------------
//! @brief        逆正弦の値を求めます。（標準関数版）
//!
//! @param[in]    x  逆正弦を求める値
//!
//! @return       x に対する逆正弦の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 NW_fAsin(f32 x) { return ::std::asinf(x); }

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。（標準関数版）
//!
//! @param[in]    x  逆正接を求める値
//!
//! @return       x に対する逆正接の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 NW_fAtan(f32 x) { return ::std::atanf(x); }

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。（標準関数版）
//!
//! @param[in]    y  正接の値の分子
//! @param[in]    x  正接の値の分母
//!
//! @return       y/x の逆正接の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 NW_fAtan2(f32 y, f32 x) { return ::std::atan2f(y, x); }



//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    x  逆正接を求める値
//!
//! @return       1 円周を 0x100000000 とする角度で逆正接の値を返します。
//---------------------------------------------------------------------------
inline u32
AtanIdx(f32 x)
{
    /*
        アークタンジェントについては、

        atanIdx(t) = 0x40000000 - atanIdx(1 / t)   という式が成り立つ。

        また、グラフが原点に対して点対称となるため、

        atanIdx( - t) = - atanIdx(t)   である。

        アークタンジェントは、0から1の範囲では急激な変化がなく、
        テーブル引きでも高い精度が得られるため、この式を利用して、0から1の
        範囲外では逆数を取る等の方法で0から1の範囲に変換して求める。
     */
    if ( x >= 0.f )
    {
        if (x > 1.f )
        {
            return 0x40000000 - internal::AtanIdx_(1.f / x);
        }
        else
        {
            return internal::AtanIdx_(x);
        }
    }
    else
    {
        if ( x < -1.f )
        {
            return 0xc0000000 + internal::AtanIdx_(-1.f / x);
        }
        else
        {
            return - static_cast<int>(internal::AtanIdx_(-x));
        }
    }
}

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    y  正接の値の分子
//! @param[in]    x  正接の値の分母
//!
//! @return       y/x の逆正接の値を 0x100000000 を 1 円周とする単位で返します
//---------------------------------------------------------------------------
inline u32
Atan2Idx(f32 y, f32 x)
{
    if ( x == 0.f && y == 0.f )
    {
        return 0;
    }

    if ( x >= 0.f )
    {
        if ( y >= 0.f )
        {
            if ( x >= y )
            {
                return internal::AtanIdx_(y / x);
            }
            else
            {
                return 0x40000000 - internal::AtanIdx_(x / y);
            }
        }
        else
        {
            if ( x >= -y )
            {
                return - static_cast<int>(internal::AtanIdx_(-y / x));
            }
            else
            {
                return 0xc0000000 + internal::AtanIdx_(x / -y);
            }
        }
    }
    else
    {
        if ( y >= 0.f )
        {
            if ( -x >= y )
            {
                return 0x80000000 - internal::AtanIdx_(y / -x);
            }
            else
            {
                return 0x40000000 + internal::AtanIdx_(-x / y);
            }
        }
        else
        {
            if ( x <= y )
            {
                return 0x80000000 + internal::AtanIdx_(y / x);
            }
            else
            {
                return 0xc0000000 - internal::AtanIdx_(x / y);
            }
        }
    }
}

//---------------------------------------------------------------------------
//! @brief        逆正弦の値を求めます。
//!
//! @param[in]    x  逆正弦を求める値
//!
//! @return       1 円周を 0x100000000 とする角度で逆正弦の値を返します。
//---------------------------------------------------------------------------
inline u32
AsinIdx(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AsinIdx: Input is out of the domain.");

    const f32 F_SQRT0_5 = 0.707106781186548f;   // = sqrt(0.5)

    // return atanIdx(x / sqrt(1.f - x*x)); に等しい処理、展開することで割り算が一回減る
    if ( x >= 0.f )
    {
        /*
            -1 <= x && x <= 1の範囲においては、x が増えるにつれて x / sqrt(1.f - x*x)
            は単純に増加するため、atanIdxにおいてtの値で分岐している箇所は、

            t = x / sqrt(1.f - x*x)

            の方程式をxについて解くことにより対応する数値を求めることができる。
            式変形により

            x = sqrt( t^2 / 1 + t^2 )

            となるため、t = 0のときのxの値は0、t = 1のときのxの値はsqrt(0.5)で
            あることがわかる。
        */
        if (x > F_SQRT0_5 )
        {
            return 0x40000000 - internal::AtanIdx_(FSqrt(1.f - x*x) / x);
        }
        else
        {
            return internal::AtanIdx_(x / FSqrt(1.f - x*x));
        }
    }
    else
    {
        if ( x < -F_SQRT0_5 )
        {
            return 0xc0000000 + internal::AtanIdx_(-FSqrt(1.f - x*x) / x);
        }
        else
        {
            return - static_cast<int>(internal::AtanIdx_(-x / FSqrt(1.f - x*x)));
        }
    }
}

//---------------------------------------------------------------------------
//! @brief        逆余弦の値を求めます。
//!
//! @param[in]    x  逆余弦を求める値
//!
//! @return       1 円周を 0x100000000 とする角度で逆余弦の値を返します。
//---------------------------------------------------------------------------
inline u32
AcosIdx(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AcosIdx: Input is out of the domain.");

    const f32 F_SQRT0_5 = 0.707106781186548f;   // = sqrt(0.5)

    // return 0x40000000 - asinIdx(x); に等しい処理、展開することで割り算が一回減る
    if ( x >= 0.f )
    {
        if (x > F_SQRT0_5 )
        {
            return internal::AtanIdx_(FSqrt(1.f - x*x) / x);
        }
        else
        {
            return 0x40000000 - internal::AtanIdx_(x / FSqrt(1.f - x*x));
        }
    }
    else
    {
        if ( x < -F_SQRT0_5 )
        {
            return 0x80000000 - internal::AtanIdx_(-FSqrt(1.f - x*x) / x);
        }
        else
        {
            return 0x40000000 + internal::AtanIdx_(-x / FSqrt(1.f - x*x));
        }
    }
}



//---------------------------------------------------------------------------
//! @brief        逆正弦の値を求めます。
//!
//! @param[in]    x  逆正弦を求める値
//!
//! @return       x に対する逆正弦の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 AsinRad(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AsinRad: Input is out of the domain.");
    return NW_MATH_IDX_TO_RAD(AsinIdx(x));
}

//---------------------------------------------------------------------------
//! @brief        逆余弦の値を求めます。
//!
//! @param[in]    x  逆余弦を求める値
//!
//! @return       x に対する逆余弦の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 AcosRad(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AcosRad: Input is out of the domain.");
    return NW_MATH_IDX_TO_RAD(AcosIdx(x));
}

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    x  逆正接を求める値
//!
//! @return       x に対する逆正接の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 AtanRad(f32 x)  { return NW_MATH_IDX_TO_RAD(AtanIdx(x)); }

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    y  正接の値の分子
//! @param[in]    x  正接の値の分母
//!
//! @return       y/x の逆正接の値をラジアン単位で返します
//---------------------------------------------------------------------------
inline f32 Atan2Rad(f32 y, f32 x)  { return NW_MATH_IDX_TO_RAD(Atan2Idx(y, x)); }



//---------------------------------------------------------------------------
//! @brief        逆正弦の値を求めます。
//!
//! @param[in]    x  逆正弦を求める値
//!
//! @return       x に対する逆正弦の値を 360.0 を 1 円周とする単位で返します
//---------------------------------------------------------------------------
inline f32 AsinDeg(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AsinDeg: Input is out of the domain.");
    return NW_MATH_IDX_TO_DEG(AsinIdx(x));
}

//---------------------------------------------------------------------------
//! @brief        逆余弦の値を求めます。
//!
//! @param[in]    x  逆余弦を求める値
//!
//! @return       x に対する逆余弦の値を 360.0 を 1 円周とする単位で返します
//---------------------------------------------------------------------------
inline f32 AcosDeg(f32 x)
{
    NW_MATH_WARNING(x <= 1.f && x >= -1.f, "AcosDeg: Input is out of the domain.");
    return NW_MATH_IDX_TO_DEG(AcosIdx(x));
}

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    x  逆正接を求める値
//!
//! @return       x に対する逆正接の値を 360.0 を 1 円周とする単位で返します
//---------------------------------------------------------------------------
inline f32 AtanDeg(f32 x)  { return NW_MATH_IDX_TO_DEG(AtanIdx(x)); }

//---------------------------------------------------------------------------
//! @brief        逆正接の値を求めます。
//!
//! @param[in]    y  正接の値の分子
//! @param[in]    x  正接の値の分母
//!
//! @return       y/x の逆正接の値を 360.0 を 1 円周とする単位で返します
//---------------------------------------------------------------------------
inline f32 Atan2Deg(f32 y, f32 x)  { return NW_MATH_IDX_TO_DEG(Atan2Idx(y, x)); }

//@}

} // namespace math
} // namespace nw


/* NW_MATH_TRIANGULAR_H_ */
#endif
