﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_QUATERNION_H_
#define NW_MATH_QUATERNION_H_

#include <cstring>
#include <nw/math/math_Config.h>

// MEMO: SDK で QUOT* を define している箇所があり、インクルードの順番によって
//       ビルドが出来なくなることがあるため、回避策として明示的に SDK のヘッダを
//       ここでインクルードしています。
#if defined(NW_PLATFORM_CAFE)
    #include <cafe/mtx/mtxVec.h>
#endif


namespace nw {
namespace math {

//---------------------------------------------------------------------------
//    QUAT用の関数
//---------------------------------------------------------------------------
struct QUAT;

namespace internal { namespace standard {

    QUAT* QUATAdd(QUAT* pOut, const QUAT* q1, const QUAT* q2);
    QUAT* QUATSub(QUAT *pOut,  const QUAT *q1, const QUAT *q2);
    QUAT* QUATMult(QUAT* pOut, const QUAT* q1, const QUAT* q2);
    f32 QUATDot(const QUAT* q1, const QUAT* q2);
    QUAT* QUATScale(QUAT* pOut, const QUAT* q, f32 scale);
    QUAT* QUATNormalize(QUAT* pOut, const QUAT* q);
    QUAT* QUATInverse(QUAT* pOut, const QUAT* q);
    QUAT* QUATExp(QUAT* pOut, const QUAT* q);
    QUAT* QUATLogN(QUAT* pOut, const QUAT* q);
    QUAT* QUATLerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
    QUAT* QUATSlerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
    QUAT* QUATMakeClosest(QUAT* pOut, const QUAT *q, const QUAT *qto);
    VEC3* VEC3CalcRPY(VEC3* pOut, const QUAT *q);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    QUAT* QUATAdd(QUAT* pOut, const QUAT* q1, const QUAT* q2);
    QUAT* QUATSub(QUAT *pOut,  const QUAT *q1, const QUAT *q2);
    QUAT* QUATMult(QUAT* pOut, const QUAT* q1, const QUAT* q2);
    f32 QUATDot(const QUAT* q1, const QUAT* q2);
    QUAT* QUATScale(QUAT* pOut, const QUAT* q, f32 scale);
    QUAT* QUATNormalize(QUAT* pOut, const QUAT* q);
    QUAT* QUATInverse(QUAT* pOut, const QUAT* q);
    QUAT* QUATExp(QUAT* pOut, const QUAT* q);
    QUAT* QUATLogN(QUAT* pOut, const QUAT* q);
    QUAT* QUATLerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
    QUAT* QUATSlerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
    QUAT* QUATMakeClosest(QUAT* pOut, const QUAT *q, const QUAT *qto);
    VEC3* VEC3CalcRPY(VEC3* pOut, const QUAT *q);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

NW_MATH_INLINE QUAT* QUATAdd(QUAT* pOut, const QUAT* q1, const QUAT* q2);
NW_MATH_INLINE QUAT* QUATSub(QUAT* pOut, const QUAT* q1, const QUAT* q2);
NW_MATH_INLINE QUAT* QUATDivide(QUAT* pOut, const QUAT* q1, const QUAT* q2);
NW_MATH_INLINE QUAT* QUATMult(QUAT* pOut, const QUAT* q1, const QUAT* q2);
NW_MATH_INLINE f32   QUATDot(const QUAT* q1, const QUAT* q2);
NW_MATH_INLINE QUAT* QUATInverse(QUAT* pOut, const QUAT* q);
NW_MATH_INLINE QUAT* QUATScale(QUAT* pOut, const QUAT* q, f32 scale);
NW_MATH_INLINE QUAT* QUATNormalize(QUAT* pOut, const QUAT* q);
NW_MATH_INLINE QUAT* QUATExp(QUAT* pOut, const QUAT* q);
NW_MATH_INLINE QUAT* QUATLogN(QUAT* pOut, const QUAT* q);
NW_MATH_INLINE QUAT* QUATLerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
NW_MATH_INLINE QUAT* QUATSlerp(QUAT* pOut, const QUAT* q1, const QUAT* q2, f32 t);
NW_MATH_INLINE QUAT* MTX34ToQUAT(QUAT* pOut, const MTX34* pMtx);
NW_MATH_INLINE QUAT* QUATSquad(QUAT* pOut, const QUAT* p, const QUAT* a, const QUAT* b, const QUAT* q, f32 t);

NW_MATH_INLINE QUAT* QUATMakeClosest( QUAT* pOut, const QUAT *q, const QUAT *qto );
NW_MATH_INLINE QUAT* QUATRotAxisRad( QUAT* pOut, const VEC3 *axis, f32 rad );
NW_MATH_INLINE QUAT* QUATMakeVectorRotation( QUAT *pOut, const VEC3* pFrom, const VEC3* pTo );

//---------------------------------------------------------------------------
//    QUAT
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        クォータニオンの構造体です
//---------------------------------------------------------------------------
struct QUAT_
{
    f32 x;
    f32 y;
    f32 z;
    f32 w;
};

//---------------------------------------------------------------------------
//! @brief        クォータニオンクラスです
//---------------------------------------------------------------------------
struct QUAT : public QUAT_
{
public:
    typedef QUAT self_type;
    typedef f32  value_type;
public:
    QUAT() {}
    explicit QUAT(const f32* p) { x = p[0]; y = p[1]; z = p[2]; w = p[3]; }
    explicit QUAT(const QUAT_& rhs) { x = rhs.x; y = rhs.y; z = rhs.z; w = rhs.w; }
    QUAT(f32 fx, f32 fy, f32 fz, f32 fw) { x = fx; y = fy; z = fz; w = fw; }

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return &x; }
    //! @brief f32 型へのキャストです。
    operator const f32*() const { return &x; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return &x; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return &x; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(QUAT));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(QUAT));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //@}

#if 0
    // 関数名がDolphinSDKと同じになるのでキャスト演算子は見合わせる
    operator Quaternion*() { return (Quaternion*)&x; }
    operator const Quaternion*() const { return (const Quaternion*)&x; }
#endif

    self_type& operator += (const self_type& rhs) { (void)QUATAdd(this, this, &rhs); return *this; }
    self_type& operator -= (const self_type& rhs) { (void)QUATSub(this, this, &rhs); return *this; }
    self_type& operator *= (f32 f) { (void)QUATScale(this, this, f); return *this; }
    self_type& operator /= (f32 f) { return operator*=(1.f / f); }

    self_type operator + () const { return *this; }
    self_type operator - () const { return self_type(-x, -y, -z, -w); }

    self_type operator + (const self_type& rhs) const { QUAT tmp; (void)QUATAdd(&tmp, this, &rhs); return tmp; }
    self_type operator - (const self_type& rhs) const { QUAT tmp; (void)QUATSub(&tmp, this, &rhs); return tmp; }
    self_type operator * (f32 f) const { QUAT tmp; (void)QUATScale(&tmp, this, f); return tmp; }
    self_type operator / (f32 f) const { return operator*(1.f / f); }

    bool operator == (const self_type& rhs) const { return x == rhs.x && y == rhs.y && z == rhs.z && w == rhs.w; }
    bool operator != (const self_type& rhs) const { return x != rhs.x || y != rhs.y || z != rhs.z || w != rhs.w; }

    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct QUAT Quaternion;  //!< @briefprivate

//---------------------------------------------------------------------------
//! @briefprivate
//! @details QUAT と浮動小数点数の積を計算します。
//! @param[in] f 積に用いる浮動小数点数です。
//! @param[in] rhs 積に用いる QUAT です。
//! @return 計算結果の QUAT を返します。
//---------------------------------------------------------------------------
inline QUAT
operator * (f32 f, const QUAT& rhs) { QUAT tmp; (void)QUATScale(&tmp, &rhs, f); return tmp; }

}  // namespace math
}  // namespace nw


#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Quaternion.ipp>
#endif

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード
inline QUAT* QUATAdd(QUAT* pOut, const QUAT& q1, const QUAT& q2) { return QUATAdd( pOut, &q1, &q2 ); }
inline QUAT* QUATSub(QUAT* pOut, const QUAT& q1, const QUAT& q2) { return QUATSub( pOut, &q1, &q2 ); }
inline QUAT* QUATDivide(QUAT* pOut, const QUAT& q1, const QUAT& q2) { return QUATDivide( pOut, &q1, &q2 ); }
inline QUAT* QUATMult(QUAT* pOut, const QUAT& q1, const QUAT& q2) { return QUATMult( pOut, &q1, &q2 ); }
inline f32   QUATDot(const QUAT& q1, const QUAT& q2) { return QUATDot( &q1, &q2 ); }
inline QUAT* QUATInverse(QUAT* pOut, const QUAT& q) { return QUATInverse( pOut, &q ); }
inline QUAT* QUATScale(QUAT* pOut, const QUAT& q, f32 scale) { return QUATScale( pOut, &q, scale ); }
inline QUAT* QUATNormalize(QUAT* pOut, const QUAT& q) { return QUATNormalize( pOut, &q ); }
inline QUAT* QUATExp(QUAT* pOut, const QUAT& q) { return QUATExp( pOut, &q ); }
inline QUAT* QUATLogN(QUAT* pOut, const QUAT& q) { return QUATLogN( pOut, &q ); }
inline QUAT* QUATLerp(QUAT* pOut, const QUAT& q1, const QUAT& q2, f32 t) { return QUATLerp( pOut, &q1, &q2, t ); }
inline QUAT* QUATSlerp(QUAT* pOut, const QUAT& q1, const QUAT& q2, f32 t) { return QUATSlerp( pOut, &q1, &q2, t ); }
inline QUAT* MTX34ToQUAT(QUAT* pOut, const MTX34& mtx) { return MTX34ToQUAT( pOut, &mtx ); }
inline QUAT* QUATSquad(QUAT* pOut, const QUAT& p, const QUAT& a, const QUAT& b, const QUAT& q, f32 t) { return QUATSquad( pOut, &p, &a, &b, &q, t ); }

inline QUAT* QUATMakeClosest( QUAT*  pOut, const QUAT& q, const QUAT& qto ) { return QUATMakeClosest( pOut, &q, &qto ); }
inline QUAT* QUATRotAxisRad( QUAT* pOut, const VEC3& axis, f32 rad ) { return QUATRotAxisRad( pOut, &axis, rad ); }
inline QUAT* QUATMakeVectorRotation( QUAT *pOut, const VEC3& from, const VEC3& to ) { return QUATMakeVectorRotation( pOut, &from, &to ); }

}  // namespace math
}  // namespace nw


#endif
