﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_MATRIX23_H_
#define NW_MATH_MATRIX23_H_

#include <cstring>
#include <nw/math/math_Config.h>

namespace nw {
namespace math {

struct MTX23;
struct MTX22;

namespace internal { namespace standard {

    MTX23* MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Mult(MTX23* pOut, const MTX23* p, f32 f);
    MTX23* MTX23Mult(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Scale(MTX23* pOut, const MTX23* pM, const VEC2* pS);
    MTX23* MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2);

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

    MTX23* MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Mult(MTX23* pOut, const MTX23* p, f32 f);
    MTX23* MTX23Mult(MTX23* pOut, const MTX23* p1, const MTX23* p2);
    MTX23* MTX23Scale(MTX23* pOut, const MTX23* pM, const VEC2* pS);
    MTX23* MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2);

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//------------------------------------------------------------------------
//    MTX23用の関数
//------------------------------------------------------------------------
NW_MATH_INLINE MTX23* MTX23Copy(MTX23* pOut, const MTX23* p);
NW_MATH_INLINE MTX23* MTX23Zero(MTX23* pOut);
NW_MATH_INLINE MTX23* MTX23Identity(MTX23* pOut);
NW_MATH_INLINE bool   MTX23IsIdentity(const MTX23* p);
NW_MATH_INLINE MTX23* MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2);
NW_MATH_INLINE MTX23* MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2);
NW_MATH_INLINE MTX23* MTX23Mult(MTX23* pOut, const MTX23* p, f32 f);
NW_MATH_INLINE MTX23* MTX23Mult(MTX23* pOut, const MTX23* p1, const MTX23* p2);
NW_MATH_INLINE MTX23* MTX23Scale(MTX23* pOut, const MTX23* pM, const VEC2* pS);
NW_MATH_INLINE MTX23* MTX23Translate(MTX23* pOut, const MTX23* pM, const VEC2* pT);
NW_MATH_INLINE MTX23* MTX23RotIdx(MTX23* pOut, u32 idx);
NW_MATH_INLINE MTX23* MTX23RotCenterIdx(MTX23* pOut, const VEC2* pCenter, u32 idx);

NW_MATH_INLINE MTX23* MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2);
NW_MATH_INLINE MTX23* MTX22ToMTX23(MTX23* pOut, const MTX22* pM);

//---------------------------------------------------------------------------
//! @briefprivate
//!
//! @details        2行3列の行列の構造体です
//---------------------------------------------------------------------------
#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4201)
#endif
struct MTX23_
{
    struct BaseData
    {
        f32 _00, _01, _02;
        f32 _10, _11, _12;
    };

    union
    {
        struct
        {
            f32 _00, _01, _02;
            f32 _10, _11, _12;
        };
        BaseData f;
        f32 m[2][3];
        f32 a[6];
        VEC3_ v[2];
    };
};
#if defined(_WIN32)
#pragma warning(pop)
#endif
//---------------------------------------------------------------------------
//!   @brief        2行3列の行列クラスです。
//---------------------------------------------------------------------------
struct MTX23 : public MTX23_
{
public:
    static const int ROW_COUNT = 2; //!< 行数です。
    static const int COLUMN_COUNT = 3; //!< 列数です。

    //! @brief 単位行列です。
    //!
    //! @return 単位行列を返します。
    static const MTX23& Identity()
    {
        static const MTX23 identity(
            1.0f, 0.0f, 0.0f,
            0.0f, 1.0f, 0.0f);

        return identity;
    }

    typedef MTX23 self_type; //!< 自分の型です。
    typedef f32   value_type; //!< 要素の型です。
public:
    //----------------------------------------
    //! @name 作成
    //@{

    //! @brief コンストラクタです。
    MTX23() {}
    //! @brief コンストラクタです。
    //!
    //! @param[in] p メンバ変数の初期化に用いる浮動小数点配列の先頭です。
    explicit MTX23(const f32* p) { (void)MTX23Copy(this, reinterpret_cast<const MTX23*>(p)); }
    //! @brief コンストラクタです。
    //!
    //! @param[in] rhs メンバ変数の初期化に用いる MTX22 です。メンバ変数 _02 と _12 には 0.0f が入ります。
    explicit MTX23(const MTX22& rhs) { MTX22ToMTX23(this, &rhs); }
    //! @brief コンストラクタです。
    //!
    //! @param[in] x00 メンバ変数 _00 の初期化に用いる浮動小数点値です。
    //! @param[in] x01 メンバ変数 _01 の初期化に用いる浮動小数点値です。
    //! @param[in] x02 メンバ変数 _02 の初期化に用いる浮動小数点値です。
    //! @param[in] x10 メンバ変数 _10 の初期化に用いる浮動小数点値です。
    //! @param[in] x11 メンバ変数 _11 の初期化に用いる浮動小数点値です。
    //! @param[in] x12 メンバ変数 _12 の初期化に用いる浮動小数点値です。
    MTX23(f32 x00, f32 x01, f32 x02,
          f32 x10, f32 x11, f32 x12)
    {
        f._00 = x00; f._01 = x01; f._02 = x02;
        f._10 = x10; f._11 = x11; f._12 = x12;
    }

    //@}

    //----------------------------------------
    //! @name 変換
    //@{

    //! @brief f32 型へのキャストです。
    operator f32*() { return this->a; }

    //! @brief f32 型へのキャストです。
    operator const f32*() const { return this->a; }

    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    f32* ToF32() { return this->a; }
    //! @brief f32 型へのキャストです。
    //!
    //! @return キャスト結果です。
    const f32* ToF32() const { return this->a; }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX23));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    //! @brief メモリフォーマットが同じ型へのキャストです。
    //!
    //! @return キャスト結果です。
    template <typename ToPtr>
    ToPtr Cast() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelPointer<ToPtr>::Type) <= sizeof(MTX23));
        NW_STATIC_ASSERT(ut::IsPointer<ToPtr>::value);

        return reinterpret_cast<ToPtr>( this );
    }

    template <typename ToRef>
    ToRef CastToRef()
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX23));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<typename ut::PeelReference<ToRef>::Type*>( this );
    }

    template <typename ToRef>
    ToRef CastToRef() const
    {
        NW_STATIC_ASSERT(sizeof(ut::PeelReference<ToRef>::Type) <= sizeof(MTX23));
        NW_STATIC_ASSERT(ut::IsReference<ToRef>::value);

        return *reinterpret_cast<const typename ut::PeelReference<ToRef>::Type*>( this );
    }

    //! @brief メモリフォーマットが同じ型からのキャストです。
    //!
    //! @param[in] fromPtr キャスト元となるポインタです。
    //! @return キャスト結果です。
    template <typename FromPtr>
    static MTX23* CastFrom(FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX23));
        return reinterpret_cast<MTX23*>( fromPtr );
    }

    template <typename FromPtr>
    static const MTX23* CastFrom(const FromPtr* fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX23));
        return reinterpret_cast<const MTX23*>( fromPtr );
    }

    template <typename FromPtr>
    static MTX23& CastFrom(FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX23));
        return *reinterpret_cast<MTX23*>( &fromPtr );
    }

    template <typename FromPtr>
    static const MTX23& CastFrom(const FromPtr& fromPtr)
    {
        NW_STATIC_ASSERT(sizeof(FromPtr) >= sizeof(MTX23));
        return *reinterpret_cast<const MTX23*>( &fromPtr );
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    VEC3& GetRow(int index)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<VEC3*>(&this->v[index]);
    }

    //! @brief VEC3 型として行を取得します。
    //!
    //! @param[in] index 取得する行のインデックスです。
    //! @return 行の取得結果です。
    const VEC3& GetRow(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, ROW_COUNT);
        return *reinterpret_cast<const VEC3*>(&this->v[index]);
    }

    //! @brief VEC2 型として列を取得します。
    //!
    //! @param[in] index 取得する列のインデックスです。
    //! @return 列の取得結果です。
    VEC2 GetColumn(int index) const
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        VEC2 column;
        column.x = this->m[0][index];
        column.y = this->m[1][index];
        return column;
    }

    //! @brief VEC2 型で列を設定します。
    //!
    //! @param[in] index 設定する列のインデックスです。
    //! @param[in] column 設定する VEC2 です。
    void SetColumn(int index, const VEC2& column)
    {
        NW_MATH_MINMAXLT_ASSERT(index, 0, COLUMN_COUNT);
        this->m[0][index] = column.x;
        this->m[1][index] = column.y;
    }
    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief 単位行列に設定します。
    //!
    //! @return 設定した行列です。
    self_type& SetIdentity() { return *MTX23Identity(this); }

    //@}

    //----------------------------------------
    //! @name 演算
    //@{

    self_type& operator += (const self_type& rhs) { return *MTX23Add(this, this, &rhs); }
    self_type& operator -= (const self_type& rhs) { return *MTX23Sub(this, this, &rhs); }

    self_type& operator *= (f32 value)
    {
        return *MTX23Mult(this, this, value);
    }
    self_type& operator /= (f32 value) { return operator*=(1.f / value); }

    self_type operator + () const { return *this; }
    self_type operator - () const
    {
        return MTX23(-f._00, -f._01, -f._02,
                     -f._10, -f._11, -f._12);
    }

    self_type operator + (const self_type& rhs) const { MTX23 tmp; return *MTX23Add(&tmp, this, &rhs); }
    self_type operator - (const self_type& rhs) const { MTX23 tmp; return *MTX23Sub(&tmp, this, &rhs); }

    self_type operator * (f32 value) const { MTX23 tmp; return *MTX23Mult(&tmp, this, value); }
    self_type operator / (f32 value) const { return *this * (1.f / value); }

    //@}

    //----------------------------------------
    //! @name 設定
    //@{

    //! @brief スケール変換行列に設定します。
    //!
    //! @param[in] matrix 元となる行列です。
    //! @param[in] scale スケール値です。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetScale(const MTX23& matrix, const VEC2& scale)
    {
        return *MTX23Scale(this, &matrix, &scale);
    }

    //! @brief 平行移動行列に設定します。
    //!
    //! @param[in] matrix 元となる行列です。
    //! @param[in] translate 平行移動値です。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetTranslate(const MTX23& matrix, const VEC2& translate)
    {
        return *MTX23Translate(this, &matrix, &translate);
    }

    //! @brief XYZ 順の回転行列に設定します。
    //!
    //! @param[in] rotate 回転値です。単位はラジアンです。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetRotate(f32 rotate)
    {
        return *MTX23RotIdx(this, NW_MATH_RAD_TO_IDX(rotate));
    }

    //! @brief 指定した軸周りの回転行列に設定します。
    //!
    //! @param[in] center 基準となる座標です。
    //! @param[in] rotate 回転させる角度です。
    //!
    //! @return 設定結果として、オブジェクト自身への参照を返します。
    //!
    self_type& SetRotate(const VEC2& center, f32 rotate)
    {
        return *MTX23RotCenterIdx(this, &center, NW_MATH_RAD_TO_IDX(rotate));
    }

    //@}

    //----------------------------------------
    //! @name 比較
    //@{

    //! @brief 同値であれば true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX23 です。
    //! @return 比較結果です。
    bool operator == (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX23)) == 0; }

    //! @brief 同値でなければ true を返します。
    //!
    //! @param[in] rhs 比較対象となる MTX23 です。
    //! @return 比較結果です。
    bool operator != (const self_type& rhs) const { return ::std::memcmp(this, &rhs, sizeof(MTX23)) != 0; }

    //! @brief 単位行列かどうかを判定します。
    //!
    //! @return 単位行列であれば true を返します。
    bool IsIdentity() const { return MTX23IsIdentity(this); }
    //@}

    //! @brief 行列の内容をデバッグ出力に書き出します。
    //!
    //! @param[in] bNewline  true なら出力の最後に改行を出力します。
    //! @param[in] name      行列の名前を指定します。NULL指定可能です。
    void Report(bool bNewline = true, const char* name = NULL) const;
};

typedef struct MTX23 Matrix23; //!< @briefprivate

}  // namespace math
}  // namespace nw

namespace nw {
namespace math {

//-- const 引数を参照にしたオーバーロード

inline MTX23* MTX23Copy(MTX23* pOut, const MTX23& m) { return MTX23Copy( pOut, &m ); }
inline bool   MTX23IsIdentity(const MTX23& m) { return MTX23IsIdentity( &m ); }
inline MTX23* MTX23Add(MTX23* pOut, const MTX23& m1, const MTX23& m2) { return MTX23Add( pOut, &m1, &m2 ); }
inline MTX23* MTX23Sub(MTX23* pOut, const MTX23& m1, const MTX23& m2) { return MTX23Sub( pOut, &m1, &m2 ); }
inline MTX23* MTX23Mult(MTX23* pOut, const MTX23& m, f32 f) { return MTX23Mult( pOut, &m, f ); }
inline MTX23* MTX23Mult(MTX23* pOut, const MTX23& m1, const MTX23& m2) { return MTX23Mult( pOut, &m1, &m2); }
inline MTX23* MTX23Scale(MTX23* pOut, const MTX23& m, const VEC2& vS) { return MTX23Scale( pOut, &m, &vS); }
inline MTX23* MTX23Translate(MTX23* pOut, const MTX23& m, const VEC2& vT) { return MTX23Translate( pOut, &m, &vT); }
inline MTX23* MTX23RotCenterIdx(MTX23* pOut, const VEC2& vCenter, u32 idx) { return MTX23RotCenterIdx( pOut, &vCenter, idx ); }

inline MTX23* MTX23MAdd(MTX23* pOut, f32 t, const MTX23& m1, const MTX23& m2) { return MTX23MAdd( pOut, t, &m1, &m2 ); }
inline MTX23* MTX22ToMTX23(MTX23* pOut, const MTX22& m) { return MTX22ToMTX23( pOut, &m ); }

}  // namespace math
}  // namespace nw

#if defined(NW_MATH_AS_INLINE)
#include <nw/math/inline/math_Matrix23.ipp>
#endif

#endif
