﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_MATH_GEOMETRY_H_
#define NW_MATH_GEOMETRY_H_

#include "nw/math/math_Config.h"
#include "nw/math/math_Types.h"

namespace nw { namespace math {

// 線や面等のジオメトリオブジェクトを定義しておく

// 名前予約
struct LINE2;
struct RAY2;
struct SEGMENT2;
struct CIRCLE2;
struct TRIANGLE2;

struct LINE3;
struct RAY3;
struct SEGMENT3;
struct SPHERE;
struct TRIANGLE3;
struct PLANE;
struct CYLINDER;
struct CONE;
struct AABB;
struct CAPSULE;

//! @brief 交差判定の結果を表す列挙型です。
enum IntersectionResult
{
    INTERSECTION_NONE = 0,      //!< 交差しない場合の返り値です。
    INTERSECTION_1 = 1,         //!< 交差する場合の返り値です。
    INTERSECTION_2 = 2,         //!< 交点が 2 つ存在する場合の返り値です。

    INTERSECTION_LINE3_ON_PLANE = 2,    //!< 線が平面上にある場合の返り値です。
    INTERSECTION_RAY3_ON_PLANE = INTERSECTION_LINE3_ON_PLANE,
    INTERSECTION_SEGMENT3_ON_PLANE = INTERSECTION_LINE3_ON_PLANE,

    INTERSECTION_OUTSIDE = 0,   //!< AABB がフラスタムの外にある場合の返り値です。
    INTERSECTION_INSIDE = 1,    //!< AABB が完全にフラスタムの内部にある場合の返り値です。
    INTERSECTION_INTERSECT = 2  //!< AABB の一部がフラスタム内部にある場合の返り値です。
};

//========================================================================
//        クラス定義
//========================================================================


//---------------------------------------------------------------------------
//! @brief    直線を表す構造体です。
//!
//!    L: P + td
//!    パラメトリック形式の線(3D)である。Pが起点でdが線の方向です。
//!    使用時にdは正規化されていることが前提となっています。
//---------------------------------------------------------------------------
struct LINE3
{
public:
    typedef LINE3 self_type;    //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ LINE3() {}

    /* ctor */ explicit LINE3(const f32* p, bool isNormalized = false)
        : P(p), d(p + 3)
    {
        if (!isNormalized) Normalize();
    }

    LINE3(const VEC3& Pt, const VEC3& dir, bool isNormalized = false)
        : P(Pt), d(dir)
    {
        if (!isNormalized) Normalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &P.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &P.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる LINE3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return P == rhs.P && d == rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる LINE3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return P != rhs.P || d != rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief         線分を、線分の始点を始点とする線に変換します。
    //!
    //! @param[in]     S 変換のための始点をもつ線分です。
    //---------------------------------------------------------------------------
    void Set(const SEGMENT3* S);

    //---------------------------------------------------------------------------
    //! @brief         直線のベクトルを正規化します。
    //---------------------------------------------------------------------------
    void Normalize() { (void)VEC3Normalize(&d, &d); }
public:
    VEC3 P; //!< 直線の起点です。
    VEC3 d; //!< @brief 直線のベクトルです。
            //!< @details 直接代入する場合は正規化しておくこと。
};

//---------------------------------------------------------------------------
//! @brief    Pを起点とするレイを表す構造体です。
//!
//!    R: P + td (ただし、t >= 0)
//!    起点からdの方向だけに向かう線である。Pが起点でdが線の方向です。
//!    使用時にdは正規化されていることが前提となっています。
//---------------------------------------------------------------------------
struct RAY3
{
public:
    typedef RAY3 self_type; //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    RAY3() {}

    RAY3(const f32* p, bool isNormalized = false)
        : P(p), d(p + 3)
    {
        if (!isNormalized) Normalize();
    }

    RAY3(const VEC3& Pt, const VEC3& dir, bool isNormalized = false)
        : P(Pt), d(dir)
    {
        if (!isNormalized) Normalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &P.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &P.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる RAY3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return P == rhs.P && d == rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる RAY3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return P != rhs.P || d != rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief         レイのベクトルを正規化します。
    //---------------------------------------------------------------------------
    void Normalize() { (void)VEC3Normalize(&d, &d); }
public:
    VEC3 P; //!< レイの起点です。
    VEC3 d; //!< @brief レイのベクトルです。
            //!< @details 直接代入する場合は正規化しておくこと
};


//---------------------------------------------------------------------------
//! @brief    線分を表す構造体です。
//!
//!    S: (1 - t)P0 + tP1 (ただし、0 <= t <= 1)
//!    P0とP1を繋ぐ線分です。
//---------------------------------------------------------------------------
struct SEGMENT3
{
public:
    typedef SEGMENT3 self_type; //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    SEGMENT3() {}

    /* ctor */ explicit SEGMENT3(const f32* p) : P0(p), P1(p + 3) {}

    SEGMENT3(const VEC3& pt0, const VEC3& pt1) : P0(pt0), P1(pt1) {}

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &P0.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &P0.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる SEGMENT3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return P0 == rhs.P0 && P1 == rhs.P1; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる SEGMENT3 です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return P0 != rhs.P0 || P1 == rhs.P1; }
public:
    VEC3 P0; //!< 端点０です。
    VEC3 P1; //!< 端点１です。
};

//---------------------------------------------------------------------------
inline void
LINE3::Set(const SEGMENT3* S)
{
    P = S->P0;
    VEC3Sub(&d, &S->P1, &S->P0);
    Normalize();
}


//---------------------------------------------------------------------------
//! @brief    球を表す構造体です。
//!
//!    S: |P - C| <= r
//!    Cを中心の位置、rを半径とする球です。
//---------------------------------------------------------------------------
struct SPHERE
{
public:
    typedef SPHERE self_type;   //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ SPHERE() {}

    /* ctor */ explicit SPHERE(const f32* p) : C(p), r(*(p + 3)) {}

    /* ctor */ SPHERE(const VEC3& center, f32 radius) : C(center), r(radius) {}

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &C.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &C.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる SPHERE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return C == rhs.C && r == rhs.r; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる SPHERE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return C != rhs.C || r != rhs.r; }

    //---------------------------------------------------------------------------
    //! @brief 点の集合を包含する球をセットします。
    //!
    //! @param[in] arrayPoint 設定に用いる点の集合です。
    //! @param[in] numPoints 集合に含まれる点の数です。
    //---------------------------------------------------------------------------
    void Set(const VEC3* arrayPoint, unsigned int numPoints);
public:
    VEC3 C; //!< 中心座標です。
    f32  r; //!< 半径です。
};


//---------------------------------------------------------------------------
//! @brief    平面を表す構造体です。
//!
//! @details  (N.x)x + (N.y)y + (N.z)z + d = 0
//!
//!    平面の陰関数表現をパラメータとして持っています。
//!    法線ベクトルNは正規化されている必要があります。
//---------------------------------------------------------------------------
struct PLANE
{
public:
    typedef PLANE self_type;    //!< :private
public:

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    PLANE() {}

    PLANE(const f32* p, bool isNormalized = false)
        : N(p), d(*(p + 3))
    {
        if (!isNormalized) Normalize();
    }

    PLANE(f32 A, f32 B, f32 C, f32 D, bool isNormalized = false)
        : N(A, B, C), d(D)
    {
        if (!isNormalized) Normalize();
    }

    PLANE(const VEC3& P0, const VEC3& P1, const VEC3& P2)
    {
        Set(&P0, &P1, &P2);
    }

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &N.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &N.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる PLANE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return N == rhs.N && d == rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる PLANE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return N != rhs.N || d != rhs.d; }

    //---------------------------------------------------------------------------
    //! @brief     点が平面のどちら側にあるかテストする関数です。
    //!
    //! @param[in] P テストの対象となる点です。
    //!
    //! @return    P が平面と重なる場合は 0 を、平面の法線方向にある場合は正、そうでない場合は負の値を返します。
    //---------------------------------------------------------------------------
    f32 Test(const VEC3& P) const
    {
        return d + VEC3Dot(&N, &P);
    }

    //---------------------------------------------------------------------------
    //! @brief        法線を正規化します。
    //---------------------------------------------------------------------------
    void Normalize()
    {
        f32 r = FrSqrt(VEC3SquareLen(&N));
        (void)VEC3Scale(&N, &N, r);
        d *= r;
    }

    //---------------------------------------------------------------------------
    //! @brief        3点を含む平面を設定します。
    //!
    //! @param[in]    P0      点０です。
    //! @param[in]    P1      点１です。
    //! @param[in]    P2      点２です。
    //---------------------------------------------------------------------------
    void Set(const VEC3* P0, const VEC3* P1, const VEC3* P2);
public:
    VEC3 N; //!< @brief 面の法線です。
            //!< @details 直接代入する場合は正規化しておくこと。
    f32 d;  //!< 原点からの距離です。
};


//---------------------------------------------------------------------------
//! @brief    カプセルを表す構造体です。
//!
//!    線分と線分に対する距離でカプセルが定義されます。
//---------------------------------------------------------------------------
struct CAPSULE
{
public:
    typedef CAPSULE self_type;  //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ CAPSULE() {}

    /* ctor */ explicit CAPSULE(const f32* p) : S(p), r(*(p + 6)) {}

    /* ctor */ CAPSULE(const SEGMENT3& S_, f32 r_) : S(S_), r(r_) {}

    /* ctor */ CAPSULE(const VEC3& P0, const VEC3& P1, f32 r_) : S(P0, P1), r(r_) {}

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &S.P0.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &S.P0.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる CAPSULE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return S == rhs.S && r == rhs.r; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる CAPSULE です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return S != rhs.S || r != rhs.r; }

public:
    SEGMENT3 S; //!< カプセルの線分です。
    f32 r;      //!< カプセルの半径です。
};


//---------------------------------------------------------------------------
//! @brief    AABB を表す構造体です。
//!
//! @details
//!    xyz軸に沿ったボックスです。2つの点PminとPmaxで定義されます。
//!    Pminのそれぞれの座標はPmaxの座標よりも小さい必要があります。
//---------------------------------------------------------------------------
struct AABB
{
public:
    typedef AABB self_type; //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    AABB() {}

    AABB(const f32* p, bool isNormalized = false)
        : Pmin(p), Pmax(p + 3)
    {
        if (!isNormalized) Normalize();
    }

    AABB(const VEC3& min, const VEC3& max, bool isNormalized = false)
        : Pmin(min), Pmax(max)
    {
        if (!isNormalized) Normalize();
    }

    //---------------------------------------------------------------------------
    //! @brief        f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator f32*() { return &Pmin.x; }

    //---------------------------------------------------------------------------
    //! @brief        const f32ポインタへのキャスト演算子です。
    //---------------------------------------------------------------------------
    operator const f32*() const { return &Pmin.x; }

    //---------------------------------------------------------------------------
    //! @brief        比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる AABB です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator==(const self_type& rhs) const { return Pmin == rhs.Pmin && Pmax == rhs.Pmax; }

    //---------------------------------------------------------------------------
    //! @brief        否定の比較演算子です。
    //!
    //! @param[in]    rhs 比較対象となる AABB です。
    //!
    //! @return       比較結果です。
    //---------------------------------------------------------------------------
    bool operator!=(const self_type& rhs) const { return Pmin != rhs.Pmin || Pmax != rhs.Pmax; }

    //---------------------------------------------------------------------------
    //! @brief        点の集合から AABB を計算します。
    //!
    //! @param[in]    arrayPoint 点の集合です。
    //! @param[in]    numPoints  点の数です。
    //---------------------------------------------------------------------------
    void Set(const VEC3* arrayPoint, unsigned int numPoints);

    //---------------------------------------------------------------------------
    //! @brief        AABB の変換結果から AABB を計算します。
    //!
    //! @param[in]    box 元の AABB です。
    //! @param[in]    M   変換行列です。
    //---------------------------------------------------------------------------
    void Set(const AABB* box, const MTX34* M);

    //---------------------------------------------------------------------------
    //! @brief        Pmin と Pmax の大小関係を正規化します。
    //---------------------------------------------------------------------------
    void Normalize();

public:
    VEC3 Pmin;  //!< AABB の最小点です。
    VEC3 Pmax;  //!< AABB の最大点です。
};




//---------------------------------------------------------------------------
//! @brief    円柱を表す構造体です。
//!
//!    半径rで原点からZ軸にhだけの高さを持つ円柱と解釈されます。
//!    lineやplaneとの交差判定は座標変換してから行う必要があります。
//---------------------------------------------------------------------------
struct CYLINDER
{
public:
    typedef CYLINDER self_type; //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ CYLINDER() {}

    /* ctor */ explicit CYLINDER(const f32* p)
        : r(*p), h(*(p + 1)) {}

    /* ctor */ CYLINDER(f32 radius, f32 height)
        : r(radius), h(height) {}

public:
    f32 r;  //!< 半径です。
    f32 h;  //!< 高さです。
};


//---------------------------------------------------------------------------
//! @brief    円錐を表す構造体です。
//!
//!    半径rで原点からZ軸にhだけの高さを持つ円錐と解釈されます。
//!    lineやplaneとの交差判定は座標変換してから行う必要があります。
//---------------------------------------------------------------------------
struct CONE
{
public:
    typedef CONE self_type; //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ CONE() {}

    /* ctor */ explicit CONE(const f32* p)
        : r(*p), h(*(p + 1)) {}

    /* ctor */ CONE(f32 radius, f32 height)
        : r(radius), h(height) {}
public:
    f32 r;  //!< 半径です。
    f32 h;  //!< 高さです。
};



//---------------------------------------------------------------------------
//! @brief    フラスタムを表す構造体です。
//!
//!    主にカリングに使用するためのフラスタムの定義です。
//---------------------------------------------------------------------------
class FRUSTUM
{
public:
    typedef FRUSTUM self_type;  //!< :private
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FRUSTUM() {}

    FRUSTUM(f32 fovy, f32 aspect, f32 n, f32 f, const MTX34& camera)
    {
        Set(fovy, aspect, n, f, camera);
    }

    FRUSTUM(f32 top, f32 bottom, f32 left, f32 right, f32 n, f32 f, const MTX34& camera)
    {
        Set(top, bottom, left, right, n, f, camera);
    }
private:
    MTX34 cam;

    /* ビュー座標系用のデータメンバ */
    PLANE leftPlane;
    PLANE rightPlane;
    PLANE topPlane;
    PLANE bottomPlane;
    f32   znear;
    f32   zfar;

    /* ワールド座標系用のデータメンバ */
    AABB  box;       // フラスタムを包含するAABBで粗い判定に用いる
    PLANE planes[6]; // left, right, near, far, up, downの順(Optimized View Frustum Culling Algorithmより)

public:
    //---------------------------------------------------------------------------
    //! @brief        フラスタムを設定します。
    //!
    //! @param[in]    fovy    垂直方向画角です。
    //! @param[in]    aspect  アスペクト比です。
    //! @param[in]    n       ニアクリップの距離です。
    //! @param[in]    f       ファークリップの距離です。
    //! @param[in]    camera  カメラ行列です。
    //---------------------------------------------------------------------------
    void Set(f32 fovy, f32 aspect, f32 n, f32 f, const MTX34& camera);

    //---------------------------------------------------------------------------
    //! @brief        フラスタムを設定します。
    //!
    //! @param[in]    top     ニアクリップ面の top 座標です。
    //! @param[in]    bottom  ニアクリップ面の bottom 座標です。
    //! @param[in]    left    ニアクリップ面の left 座標です。
    //! @param[in]    right   ニアクリップ面の right 座標です。
    //! @param[in]    n       ニアクリップの距離です。
    //! @param[in]    f       ファークリップの距離です。
    //! @param[in]    camera  カメラ行列です。
    //---------------------------------------------------------------------------
    void Set(f32 top, f32 bottom, f32 left, f32 right, f32 n, f32 f, const MTX34& camera);

public:
    //---------------------------------------------------------------------------
    //! @brief 交差判定関数です。
    //! @details SPHEREはワールド座標系
    //!
    //! @param[in] S 交差判定の対象となる球です。
    //!
    //! @return 球の一部または全部がフラスタム内にあれば true を返します。
    //---------------------------------------------------------------------------
    bool IntersectSphere(const SPHERE* S) const;

    //---------------------------------------------------------------------------
    //! @brief 交差判定関数です。
    //! @details AABBはワールド座標系
    //!
    //! @param[in] B 交差判定の対象となる AABB です。
    //!
    //! @return AABB の一部または全部がフラスタム内にあれば true を返します。
    //---------------------------------------------------------------------------
    bool IntersectAABB(const AABB* B) const;

    //---------------------------------------------------------------------------
    //! @brief 交差判定関数です。
    //! @details AABBはワールド座標系
    //!
    //! @param[in] B 交差判定の対象となる AABB です。
    //!
    //! @return 交差判定の結果を返します。
    //---------------------------------------------------------------------------
    IntersectionResult IntersectAABB_Ex(const AABB* B) const;
};


//
// 3Dジオメトリ間の距離
//

// 点

//---------------------------------------------------------------------------
//! @brief        点と線の距離の 2 乗を求めます。
//!
//! @param[in]    P             点
//! @param[in]    L             線
//! @param[in]    t             線上に存在する P に最も近い点を、L->P + t * L->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       点から線への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToLine3(const VEC3* P, const LINE3* L, f32* t);

//---------------------------------------------------------------------------
//! @brief        点と射線の距離の 2 乗を求めます。
//!
//! @param[in]    P             点
//! @param[in]    R             射線
//! @param[in]    t             射線上に存在する P に最も近い点を、R->P + t * R->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       点から射線への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToRay3(const VEC3* P, const RAY3* R, f32* t);

//---------------------------------------------------------------------------
//! @brief        点と線分の距離の 2 乗を求めます。
//!
//! @param[in]    P             点
//! @param[in]    S             線分
//! @param[in]    t             線分上に存在する P に最も近い点を、P->P0 + t * (P->P1 - P->P0) / | P->P1 - P->P0 |
//!                             としたときの t が格納されます。NULL を指定することも可能です。
//!
//! @return       点から線分への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToSegment3(const VEC3* P, const SEGMENT3* S, f32* t);

//---------------------------------------------------------------------------
//! @brief        点から平面への距離の 2 乗を求めます。
//!
//! @param[in]    P             点
//! @param[in]    J             平面
//! @param[in]    Q             点に最も近い平面上の点が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       点から平面への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToPlane(const VEC3* P, const PLANE* J, VEC3* Q);

//---------------------------------------------------------------------------
//! @brief        球と平面の距離の 2 乗を求めます。
//!
//! @param[in]    S             球
//! @param[in]    J             平面
//!
//! @return       球から平面への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqSphereToPlane(const SPHERE* S, const PLANE* J);

//---------------------------------------------------------------------------
//! @brief        点から連続する線分の距離の 2 乗を求めます。
//!
//!               nw::math::DistSqPoint3ToSegment3 を繰り返し使用するより高速です。
//!
//! @param[in]    P             点
//! @param[in]    vertices      連続する線分を構成する点の配列
//! @param[in]    nVertices     vertices の要素数
//!
//! @return       点から連続する線分の距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToPolyline3(const VEC3* P, const VEC3* vertices, unsigned int nVertices);

//---------------------------------------------------------------------------
//! @brief        点から AABB への距離の 2 乗を求めます。
//!
//! @param[in]    P             点
//! @param[in]    B             AABB
//! @param[in]    q             点に最も近い AABB 上の点が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       点から AABB への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqPoint3ToAABB(const VEC3* P, const AABB* B, VEC3* q);


// 線・レイ・線分

//---------------------------------------------------------------------------
//! @brief        線と線の距離の 2 乗を求めます。
//!
//! @param[in]    L0            線0
//! @param[in]    L1            線1
//! @param[in]    s             線 L0 上に存在する L1 に最も近い点を、L0->P + s * L0->d としたときの s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             線 L1 上に存在する L0 に最も近い点を、L1->P + t * L1->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線から線への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqLine3ToLine3(const LINE3* L0, const LINE3* L1, f32* s, f32* t);

//---------------------------------------------------------------------------
//! @brief        線分から線分の距離の 2 乗を求めます。
//!
//! @param[in]    S0            線分0
//! @param[in]    S1            線分1
//! @param[in]    s             線分 S0 上に存在する S1 に最も近い点の、S0->P0 からの距離 s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             線分 S1 上に存在する S0 に最も近い点の、S1->P0 からの距離 t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線分から線分への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqSegment3ToSegment3(const SEGMENT3* S0, const SEGMENT3* S1, f32* s, f32* t);

//---------------------------------------------------------------------------
//! @brief        線と射線の距離の 2 乗を求めます。
//!
//! @param[in]    L             線
//! @param[in]    R             射線
//! @param[in]    s             線 L 上に存在する R に最も近い点を、L->P + s * R->d としたときの s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             射線 R 上に存在する L に最も近い点を、R->P + t * R->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線から射線への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqLine3ToRay3(const LINE3* L, const RAY3* R, f32* s, f32* t);

//---------------------------------------------------------------------------
//! @brief        線と線分の距離の 2 乗を求めます。
//!
//! @param[in]    L0            線
//! @param[in]    S             線分
//! @param[in]    s             線 L0 上に存在する S に最も近い点を、L0->P + s * L0->d としたときの s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             線分 S 上に存在する L0 に最も近い点の S->P0 からの距離 t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線から線分への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqLine3ToSegment3(const LINE3* L0, const SEGMENT3* S, f32* s, f32* t);

//---------------------------------------------------------------------------
//! @brief        線と線分の距離の 2 乗を求めます。
//!
//! @param[in]    R0            射線0
//! @param[in]    R1            射線1
//! @param[in]    s             射線 R0 上に存在する R1 に最も近い点を、R0->P + s * R0->d としたときの s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             射線 R1 上に存在する R0 に最も近い点を、R1->P + t * R1->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線から線分への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqRay3ToRay3(const RAY3* R0, const RAY3* R1, f32* s, f32* t);

//---------------------------------------------------------------------------
//! @brief        線と線分の距離の 2 乗を求めます。
//!
//! @param[in]    R0            射線
//! @param[in]    S             線分
//! @param[in]    s             射線 R0 上に存在する S に最も近い点を、R0->P + s * R0->d としたときの s が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t             線分 S 上に存在する R0 に最も近い点の、S->P0 からの距離t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       線から線分への距離の 2 乗を返します。
//---------------------------------------------------------------------------
f32 DistSqRay3ToSegment3(const RAY3* R0, const SEGMENT3* S, f32* s, f32* t);



//
// 3Dジオメトリ間の交差
//

//---------------------------------------------------------------------------
//! @brief        直線と平面が交差するかどうかを判定します。
//!
//! @param[in]    L             線
//! @param[in]    J             平面
//! @param[in]    t             交点を、L0->P + t * L0->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    I             交点の座標が格納されます。NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionLine3Plane(const LINE3* L, const PLANE* J, f32* t, VEC3* I);

//---------------------------------------------------------------------------
//! @brief        直線と平面が交差するかどうかを判定します。
//!
//! @param[in]    R             射線
//! @param[in]    J             平面
//! @param[in]    t             交点を、R->P + t * R->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    I             交点の座標が格納されます。NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionRay3Plane(const RAY3* R, const PLANE* J, f32* t, VEC3* I);

//---------------------------------------------------------------------------
//! @brief        線分と平面が交差するかどうかを判定します。
//!
//! @param[in]    S             線分
//! @param[in]    J             平面
//! @param[in]    t             S->P0 から交点へ線分沿いに進んだ場合の距離が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    I             交点の座標が格納されます。NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionSegment3Plane(const SEGMENT3* S, const PLANE* J, f32* t, VEC3* I);

//---------------------------------------------------------------------------
//! @brief        直線と球が交差するかどうかを判定します。
//!
//!               交点が 2 つ存在する場合は t0 < t1 が保証されます。
//!
//! @param[in]    L             線
//! @param[in]    sphere        球
//! @param[in]    t0            交点の 1 つを、L->P + t0 * L->d としたときの t0 が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t1            交点の 1 つを、L->P + t1 * L->d としたときの t1 が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionLine3Sphere(const LINE3* L, const SPHERE* sphere, f32* t0, f32* t1);

//---------------------------------------------------------------------------
//! @brief        射線と球が交差するかどうかを判定します。
//!
//!               交点が 2 つ存在する場合は t0 < t1 が保証されます。
//!
//! @param[in]    R             射線
//! @param[in]    sphere        球
//! @param[in]    t0            交点の 1 つを、R->P + t0 * R->d としたときの t0 が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t1            交点の 1 つを、R->P + t1 * R->d としたときの t1 が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionRay3Sphere(const RAY3* R, const SPHERE* sphere, f32* t0, f32* t1);

//---------------------------------------------------------------------------
//! @brief        射線と球が交差するかどうかを判定します。
//!
//! @param[in]    R             射線
//! @param[in]    sphere        球
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionRay3Sphere(const RAY3* R, const SPHERE* sphere);

//---------------------------------------------------------------------------
//! @brief        線分と球が交差するかどうかを判定します。
//!
//!               交点が 2 つ存在する場合は t0 < t1 が保証されます。
//!
//! @param[in]    S             線分
//! @param[in]    sphere        球
//! @param[in]    t0            交点の 1 つの S->P0 からの線分沿いの距離が格納されます。
//!                             NULL を指定することも可能です。
//! @param[in]    t1            交点の 1 つの S->P0 からの線分沿いの距離が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
IntersectionResult IntersectionSegment3Sphere(const SEGMENT3* S, const SPHERE* sphere, f32* t0, f32* t1);

//---------------------------------------------------------------------------
//! @brief        射線と AABB が交差するかどうかを判定します。
//!
//!               交差した点を求めるためのパラメータを取得することも可能です。
//!
//! @param[in]    R             射線
//! @param[in]    box           AABB
//! @param[in]    t             射線の起点に最も近い交点を、R->P + t * R->d としたときの t が格納されます。
//!                             NULL を指定することも可能です。
//!
//! @return       射線と AABB が交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionRay3AABB(const RAY3* R, const AABB* box, f32* t);

//---------------------------------------------------------------------------
//! @brief        2 つの AABB の領域が重なりあうかどうかを返します。
//!
//! @param[in]    a             AABB 0
//! @param[in]    b             AABB 1
//!
//! @return       2 つの AABB が重なりあうかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionAABB(const AABB* a, const AABB* b);

//---------------------------------------------------------------------------
//! @brief        球と AABB が交差するかどうかを返します。
//!
//! @param[in]    sphere        球
//! @param[in]    aabb          AABB
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionSphereAABB(const SPHERE* sphere, const AABB* aabb);

//---------------------------------------------------------------------------
//! @brief        球と球が交差するかどうかを返します。
//!
//! @param[in]    s0            球0
//! @param[in]    s1            球1
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionSphere(const SPHERE* s0, const SPHERE* s1);

//---------------------------------------------------------------------------
//! @brief        平面と AABB が交差するかどうかを返します。
//!
//! @param[in]    J             平面
//! @param[in]    B             AABB
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionPlaneAABB(const PLANE* J, const AABB* B);

//---------------------------------------------------------------------------
//! @brief        カプセルとカプセルが交差するかどうかを返します。
//!
//! @param[in]    C0            カプセル0
//! @param[in]    C1            カプセル1
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionCapsule(const CAPSULE* C0, const CAPSULE* C1);

//---------------------------------------------------------------------------
//! @brief        射線とカプセルが交差するかどうかを返します。
//!
//! @param[in]    R             射線
//! @param[in]    C             カプセル
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionRay3Capsule(const RAY3* R, const CAPSULE* C);

//---------------------------------------------------------------------------
//! @brief        線とカプセルが交差するかどうかを返します。
//!
//! @param[in]    L             線
//! @param[in]    C             カプセル
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionLine3Capsule(const LINE3* L, const CAPSULE* C);

//---------------------------------------------------------------------------
//! @brief        平面とカプセルが交差するかどうかを返します。
//!
//! @param[in]    J             平面
//! @param[in]    C             カプセル
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
bool               IntersectionPlaneCapsule(const PLANE* J, const CAPSULE* C);

//---------------------------------------------------------------------------
//! @brief        球とフラスタムが交差するかどうかを返します。
//!
//!               球の中心はワールド座標系の座標である必要があります。
//!
//! @param[in]    S             球
//! @param[in]    F             フラスタム
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
inline bool
IntersectionSphereFrustum(const SPHERE* S, const FRUSTUM* F)
{
    return F->IntersectSphere(S);
}

//---------------------------------------------------------------------------
//! @brief        AABB とフラスタムが交差するかどうかを返します。
//!
//! @param[in]    B             AABB
//! @param[in]    F             フラスタム
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
inline bool
IntersectionAABBFrustum(const AABB* B, const FRUSTUM* F)
{
    return F->IntersectAABB(B);
}

//---------------------------------------------------------------------------
//! @brief        AABB とフラスタムが交差するかどうかを返します。
//!
//! @param[in]    B             AABB
//! @param[in]    F             フラスタム
//!
//! @return       交差したかどうかを返します。
//---------------------------------------------------------------------------
inline IntersectionResult
IntersectionAABBFrustumEx(const AABB* B, const FRUSTUM* F)
{
    return F->IntersectAABB_Ex(B);
}


//
// 3Dジオメトリのマージ
//

//---------------------------------------------------------------------------
//! @brief        2 つの球を包含する球を設定します。
//!
//!               s2 は s0 か s1 と同一でも構いません。
//!
//! @param[in]    s2            s0 と s1 を包含する球
//! @param[in]    s0            球0
//! @param[in]    s1            球1
//!
//! @return       s2 を返します。
//---------------------------------------------------------------------------
SPHERE* MergeSphere(SPHERE* s2, const SPHERE* s0, const SPHERE* s1);

//---------------------------------------------------------------------------
//! @brief        2 つの AABB を包含する AABB を設定します。
//!
//!               a2 は a0 か a1 と同一でも構いません。
//!
//! @param[in]    a2            a0 と a1 を包含する AABB
//! @param[in]    a0            AABB 0
//! @param[in]    a1            AABB 1
//!
//! @return       a2 を返します。
//---------------------------------------------------------------------------
AABB* MergeAABB(AABB* a2, const AABB* a0, const AABB* a1);


}}  // nw::math

#endif

