﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//        VEC4
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_MATH_INLINE VEC4*
VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = p1->x + p2->x;
    pOut->y = p1->y + p2->y;
    pOut->z = p1->z + p2->z;
    pOut->w = p1->w + p2->w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = p1->x - p2->x;
    pOut->y = p1->y - p2->y;
    pOut->z = p1->z - p2->z;
    pOut->w = p1->w - p2->w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    pOut->x = scale * p->x;
    pOut->y = scale * p->y;
    pOut->z = scale * p->z;
    pOut->w = scale * p->w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Lerp(VEC4* pOut, const VEC4* __restrict p1, const VEC4* __restrict p2, f32 t)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    // (1-t)*p1 + t*p2
    pOut->x = p1->x + t * (p2->x - p1->x);
    pOut->y = p1->y + t * (p2->y - p1->y);
    pOut->z = p1->z + t * (p2->z - p1->z);
    pOut->w = p1->w + t * (p2->w - p1->w);

    return pOut;
}

NW_MATH_INLINE f32
VEC4Dot(const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    return p1->x * p2->x + p1->y * p2->y + p1->z * p2->z + p1->w * p2->w;
}

NW_MATH_INLINE f32
VEC4LenSq(const VEC4* __restrict p)
{
    NW_ASSERT_NOT_NULL(p);

    return p->x * p->x + p->y * p->y + p->z * p->z + p->w * p->w;
}

NW_MATH_INLINE f32
VEC4DistSq(const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    VEC4 tmp;
    return standard::VEC4LenSq(standard::VEC4Sub(&tmp, p1, p2));
}

NW_MATH_INLINE VEC4*
VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x > p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y > p2->y) ? p1->y : p2->y;
    pOut->z = (p1->z > p2->z) ? p1->z : p2->z;
    pOut->w = (p1->w > p2->w) ? p1->w : p2->w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    pOut->x = (p1->x < p2->x) ? p1->x : p2->x;
    pOut->y = (p1->y < p2->y) ? p1->y : p2->y;
    pOut->z = (p1->z < p2->z) ? p1->z : p2->z;
    pOut->w = (p1->w < p2->w) ? p1->w : p2->w;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC4*
VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    tof32x2(pOut->x) = __PS_ADD(tof32x2(p1->x), tof32x2(p2->x));
    tof32x2(pOut->z) = __PS_ADD(tof32x2(p1->z), tof32x2(p2->z));

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    tof32x2(pOut->x) = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));
    tof32x2(pOut->z) = __PS_SUB(tof32x2(p1->z), tof32x2(p2->z));

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32x2 xy = __PS_MULS0F(tof32x2(p->x), scale);
    f32x2 zw = __PS_MULS0F(tof32x2(p->z), scale);

    tof32x2(pOut->x) = xy;
    tof32x2(pOut->z) = zw;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Lerp(VEC4* pOut, const VEC4* __restrict p1, const VEC4* __restrict p2, f32 t)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    // (1-t)*p1 + t*p2

    f32x2 tt = __PS_FDUP(t);

    f32x2 f0 = tof32x2(p1->x);
    f0 = __PS_NMSUB(f0, tt, f0);
    f0 = __PS_MADD(tof32x2(p2->x), tt, f0);

    f32x2 f1 = tof32x2(p1->z);
    f1 = __PS_NMSUB(f1, tt, f1);
    f1 = __PS_MADD(tof32x2(p2->z), tt, f1);

    tof32x2(pOut->x) = f0;
    tof32x2(pOut->z) = f1;

    return pOut;
}

NW_MATH_INLINE f32
VEC4Dot(const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0;
    f0 = __PS_MUL(tof32x2(p1->x), tof32x2(p2->x));
    f0 = __PS_MADD(tof32x2(p1->z), tof32x2(p2->z), f0);
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

NW_MATH_INLINE f32
VEC4LenSq(const VEC4* __restrict p)
{
    NW_ASSERT_NOT_NULL(p);

    f32x2 f0, f1;
    f1 = tof32x2(p->x);
    f0 = __PS_MUL(f1, f1);
    f1 = tof32x2(p->z);
    f0 = __PS_MADD(f1, f1, f0);
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

NW_MATH_INLINE f32
VEC4DistSq(const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0, f1;
    f1 = __PS_SUB(tof32x2(p1->x), tof32x2(p2->x));
    f0 = __PS_MUL(f1, f1);
    f1 = __PS_SUB(tof32x2(p1->z), tof32x2(p2->z));
    f0 = __PS_MADD(f1, f1, f0);
    f0 = __PS_SUM0(f0, f0, f0);

    return f0[0];
}

NW_MATH_INLINE VEC4*
VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0, f1;

    f0 = tof32x2(p1->x);
    f1 = tof32x2(p2->x);
    tof32x2(pOut->x) = __PS_SEL(__PS_SUB(f0, f1), f0, f1);

    f0 = tof32x2(p1->z);
    f1 = tof32x2(p2->z);
    tof32x2(pOut->z) = __PS_SEL(__PS_SUB(f0, f1), f0, f1);

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p1);
    NW_ASSERT_NOT_NULL(p2);

    f32x2 f0, f1;

    f0 = tof32x2(p1->x);
    f1 = tof32x2(p2->x);
    tof32x2(pOut->x) = __PS_SEL(__PS_SUB(f1, f0), f0, f1);

    f0 = tof32x2(p1->z);
    f1 = tof32x2(p2->z);
    tof32x2(pOut->z) = __PS_SEL(__PS_SUB(f1, f0), f0, f1);

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ベクトル
//@{

//---------------------------------------------------------------------------
//! @brief        ベクトルがゼロベクトルかどうか判定します。
//!
//! @param[in]    p    判定対象のベクトルへのポインタ。
//!
//! @return       p がゼロベクトルであれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
VEC4IsZero(const VEC4* p)
{
    return p->x == 0.f && p->y == 0.f && p->z == 0.f && p->w == 0.f;
}

//---------------------------------------------------------------------------
//! @brief        同次座標でベクトルがゼロベクトルかどうか判定します。
//!
//! @param[in]    p    判定対象のベクトルへのポインタ。
//!
//! @return       p がゼロベクトルであれば true そうでなければ false を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
VEC4IsZeroWOne(const VEC4* p)
{
    return p->x == 0.f && p->y == 0.f && p->z == 0.f && p->w == 1.f;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの和を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Add(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4Add(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの差を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Sub(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4Sub(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルのスカラー倍を計算します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p      左辺値へのポインタ
//! @param[in]    scale  掛ける数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Scale(VEC4* pOut, const VEC4* p, f32 scale)
{
    return NW_MATH_IMPL_NS::VEC4Scale(pOut, p, scale);
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の線形補間を計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    線形補間の始点となるベクトルへのポインタ
//! @param[in]    p2    線形補間の終点となるベクトルへのポインタ
//! @param[in]    t     線形補間のパラメータ。0.0 であれば p1 が 1.0 であれば p2 が結果となります。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Lerp(VEC4* pOut, const VEC4* __restrict p1, const VEC4* __restrict p2, f32 t)
{
    return NW_MATH_IMPL_NS::VEC4Lerp(pOut, p1, p2, t);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの内積を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の内積を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE f32
VEC4Dot(const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4Dot(p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの長さの2乗を計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さの2乗を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE f32
VEC4LenSq(const VEC4* __restrict p)
{
    return NW_MATH_IMPL_NS::VEC4LenSq(p);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの長さを計算します。
//!
//! @param[in]    p  対象のベクトルへのポインタ。
//!
//! @return       p の長さを返します。
//---------------------------------------------------------------------------
f32 VEC4Len(const VEC4* p)
{
    return FSqrt(VEC4LenSq(p));

}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Normalize(VEC4* pOut, const VEC4* p)
{
    (void)VEC4Scale(pOut, p, FrSqrt(VEC4LenSq(p)));

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               VEC4Normalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4FastNormalize(VEC4* pOut, const VEC4* p)
{
    (void)VEC4Scale(pOut, p, FrFastSqrt(VEC4LenSq(p)));

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4SafeNormalize(VEC4* pOut, const VEC4* p, const VEC4& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = VEC4LenSq(p);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    (void)VEC4Scale(pOut, p, FrSqrt(mag));

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ベクトルを正規化します。
//!               正規化に失敗した場合は指定されたベクトルを設定します。
//!               VEC4SafeNormalize より高速ですが、求められる計算結果の精度に差があります。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じベクトルを指していても構いません。
//! @param[in]    p     対象のベクトルへのポインタ
//! @param[in]    alt   正規化に失敗した場合に設定するベクトル
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4FastSafeNormalize(VEC4* pOut, const VEC4* p, const VEC4& alt)
{
    NW_ASSERT_NOT_NULL(pOut);
    NW_ASSERT_NOT_NULL(p);

    f32 mag = VEC4LenSq(p);

    if (mag == 0 || mag == F_INF || isnan(mag))
    {
        *pOut = alt;

        return pOut;
    }

    (void)VEC4Scale(pOut, p, FrFastSqrt(mag));

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        2つのベクトル間の距離の2乗を計算します。
//!
//! @param[in]    p1    左辺値へのポインタ
//! @param[in]    p2    右辺値へのポインタ
//!
//! @return       p1 と p2 の距離の2乗を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE f32
VEC4DistSq(const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4DistSq(p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最大値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Maximize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4Maximize(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        2つのベクトルのそれぞれの成分の最小値から構成されるベクトルを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p1, p2 と同じベクトルを指していても構いません。
//! @param[in]    p1    対象のベクトル1へのポインタ。
//! @param[in]    p2    対象のベクトル2へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Minimize(VEC4* pOut, const VEC4* p1, const VEC4* p2)
{
    return NW_MATH_IMPL_NS::VEC4Minimize(pOut, p1, p2);
}

//@}

}  // namespace math
}  // namespace nw
