﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace math {

namespace internal { namespace standard {

NW_MATH_INLINE VEC3*
VEC3TransformArray(VEC3* pOut, const MTX34* __restrict pM, const VEC3* __restrict pV, s32 count)
{
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT( count > 1 );

    VEC3 vTmp;
    VEC3* __restrict pOutBase = pOut;
    const f32 (*const m)[4] = pM->m;

    for (s32 i = 0; i < count; ++i)
    {
        // Vec has a 4th implicit 'w' coordinate of 1
        vTmp.x = m[0][0] * pV->x + m[0][1] * pV->y + m[0][2] * pV->z + m[0][3];
        vTmp.y = m[1][0] * pV->x + m[1][1] * pV->y + m[1][2] * pV->z + m[1][3];
        vTmp.z = m[2][0] * pV->x + m[2][1] * pV->y + m[2][2] * pV->z + m[2][3];

        // copy back
        pOut->x = vTmp.x;
        pOut->y = vTmp.y;
        pOut->z = vTmp.z;

        pV++;
        pOut++;
    }

    return pOutBase;
}

NW_MATH_INLINE VEC4*
VEC3TransformArray(VEC4* pOut, const MTX44* __restrict pM, const VEC3* __restrict pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );
    NW_ASSERT( count > 1 );

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        (void)standard::VEC3Transform(pDst, pM, pV);
        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformNormal(VEC3* pOut, const MTX34* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC3 tmp;
    tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z;
    tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z;
    tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z;

    pOut->x = tmp.x;
    pOut->y = tmp.y;
    pOut->z = tmp.z;

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformNormalArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    u32 i;
    VEC3 tmp;
    VEC3* pDst = pOut;

    for (i = 0; i < count; i++)
    {
        // Vec has a 4th implicit 'w' coordinate of 1
        tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z;
        tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z;
        tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z;

        // copy back
        pDst->x = tmp.x;
        pDst->y = tmp.y;
        pDst->z = tmp.z;

        pV++;
        pDst++;
    }

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformCoord(VEC3* pOut, const MTX44* pM, const VEC3* pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC3 tmp;
    f32 rw;

    tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z + pM->f._03;
    tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z + pM->f._13;
    tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z + pM->f._23;
       rw = pM->f._30 * pV->x + pM->f._31 * pV->y + pM->f._32 * pV->z + pM->f._33;
    rw = 1.f / rw;

    pOut->x = rw * tmp.x;
    pOut->y = rw * tmp.y;
    pOut->z = rw * tmp.z;

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC3* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        (void)standard::VEC3TransformCoord(pDst, pM, pV);
        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX44* __restrict pM, const VEC4* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC4 tmp;
    tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z + pM->f._03 * pV->w;
    tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z + pM->f._13 * pV->w;
    tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z + pM->f._23 * pV->w;
    tmp.w = pM->f._30 * pV->x + pM->f._31 * pV->y + pM->f._32 * pV->z + pM->f._33 * pV->w;

    pOut->x = tmp.x;
    pOut->y = tmp.y;
    pOut->z = tmp.z;
    pOut->w = tmp.w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        (void)standard::VEC4Transform(pDst, pM, pV);
        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX34* __restrict pM, const VEC4* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC4 tmp;
    tmp.x = pM->f._00 * pV->x + pM->f._01 * pV->y + pM->f._02 * pV->z + pM->f._03 * pV->w;
    tmp.y = pM->f._10 * pV->x + pM->f._11 * pV->y + pM->f._12 * pV->z + pM->f._13 * pV->w;
    tmp.z = pM->f._20 * pV->x + pM->f._21 * pV->y + pM->f._22 * pV->z + pM->f._23 * pV->w;

    pOut->x = tmp.x;
    pOut->y = tmp.y;
    pOut->z = tmp.z;
    pOut->w = pV->w;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        (void)standard::VEC4Transform(pDst, pM, pV);
        ++pDst;
        ++pV;
    }

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE VEC3*
VEC3TransformArray(VEC3* pOut, const MTX34* __restrict pM, const VEC3* __restrict pV, s32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );
    NW_ASSERT( count > 1 );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);

    f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    f32x2 m02m12 = __PS_MERGE00(m02m03, m12m13);
    f32x2 m03m13 = __PS_MERGE11(m02m03, m12m13);

    f32x2 m20m30 = __PSQ_LX(pM, offsetof(MTX34, f._20), 1, 0);
    f32x2 m21m31 = __PSQ_LX(pM, offsetof(MTX34, f._21), 1, 0);
    f32x2 m22m32 = __PSQ_LX(pM, offsetof(MTX34, f._22), 1, 0);
    f32x2 m23m33 = __PSQ_LX(pM, offsetof(MTX34, f._23), 1, 0);

    VEC3* pDst = pOut;
    for (s32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z 1 }
        f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
        f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

        // fp0 = [ m00 m01 m02 m03 ] . T[ x y z 1 ]
        //       [ m10 m11 m12 m03 ] .
        fp0 = __PS_MADDS0(m02m12, zw, m03m13);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m00m10, xy, fp0);

        // fp1 = [ m20 m21 m22, m23 ] . T[ x y z 1 ]
        //       [ m30 m31 m32, m33 ]
        fp1 = __PS_MADDS0(m22m32, zw, m23m33);
        fp1 = __PS_MADDS1(m21m31, xy, fp1);
        fp1 = __PS_MADDS0(m20m30, xy, fp1);

        tof32x2(pDst->x) = fp0;
        pDst->z = fp1[0];

        pV++;
        pDst++;
    }

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC3TransformArray(VEC4* pOut, const MTX44* __restrict pM, const VEC3* __restrict pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );
    NW_ASSERT( count > 1 );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);

    f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    f32x2 m02m12 = __PS_MERGE00(m02m03, m12m13);
    f32x2 m03m13 = __PS_MERGE11(m02m03, m12m13);

    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, f._20), 0, 0);
    f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, f._30), 0, 0);
    f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, f._22), 0, 0);
    f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, f._32), 0, 0);

    f32x2 m20m30 = __PS_MERGE00(m20m21, m30m31);
    f32x2 m21m31 = __PS_MERGE11(m20m21, m30m31);
    f32x2 m22m32 = __PS_MERGE00(m22m23, m32m33);
    f32x2 m23m33 = __PS_MERGE11(m22m23, m32m33);

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z 1 }
        f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
        f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

        // fp0 = [ m00 m01 m02, m03 ] . T[ x y z 1 ]
        //       [ m10 m11 m12, m13 ]
        fp0 = __PS_MADDS0(m02m12, zw, m03m13);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m00m10, xy, fp0);

        // fp1 = [ m20 m21 m22, m23 ] . T[ x y z 1 ]
        //       [ m30 m31 m32, m33 ]
        fp1 = __PS_MADDS0(m22m32, zw, m23m33);
        fp1 = __PS_MADDS1(m21m31, xy, fp1);
        fp1 = __PS_MADDS0(m20m30, xy, fp1);

        tof32x2(pDst->x) = fp0;
        tof32x2(pDst->z) = fp1;

        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformNormal(VEC3* pOut, const MTX34* __restrict pM, const VEC3* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX34, m[0][0]), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX34, m[0][2]), 1, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX34, m[1][0]), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX34, m[1][2]), 1, 0);
    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX34, m[2][0]), 0, 0);
    f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX34, m[2][2]), 1, 0);

    f32x2 fp0, fp1, fp2;

    // { xy zw } = { x y z 1 }
    f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
    f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

    // fp0 = [ m00 m01 m02 0 ] . T[ x y z 1 ]
    fp0 = __PS_MUL(m00m01, xy);
    fp0 = __PS_SUM0(fp0, fp0, fp0);
    fp0 = __PS_MADD(m02m03, zw, fp0);

    pOut->x = fp0[0];

    // fp1 = [ m10 m11 m12 0 ] . T[ x y z 1 ]
    fp1 = __PS_MUL(m10m11, xy);
    fp1 = __PS_SUM0(fp1, fp1, fp1);
    fp1 = __PS_MADD(m12m13, zw, fp1);

    pOut->y = fp1[0];

    // fp2 = [ m20 m21 m22 0 ] . T[ x y z 1 ]
    fp2 = __PS_MUL(m20m21, xy);
    fp2 = __PS_SUM0(fp2, fp2, fp2);
    fp2 = __PS_MADD(m22m23, zw, fp2);

    pOut->z = fp2[0];

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformNormalArray(VEC3* pOut, const MTX34* __restrict pM, const VEC3* __restrict pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );
    NW_ASSERT( count > 1 );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX34, f._00), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX34, f._10), 0, 0);
    f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    f32x2 m02m12 = { pM->f._02, pM->f._12 };
    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX34, f._20), 0, 0);
    f32x2 m22    = __PSQ_LX(pM, offsetof(MTX34, f._22), 1, 0);

    VEC3* pDst = pOut;
    for (s32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z 1 }
        f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);
        f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);

        // fp0 = [ m00 m01 m02 m03 ] . T[ x y z 0 ]
        //       [ m10 m11 m12 m13 ] .
        fp0 = __PS_MULS0(m02m12, zw);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m00m10, xy, fp0);

        // fp1 = [ m20 m21 m22 m23 ] . T[ x y z 0 ]
        fp1 = __PS_MUL(m20m21, xy);
        fp1 = __PS_SUM0(fp1, fp1, fp1);
        fp1 = __PS_MADDS0(m22, zw, fp1);

        tof32x2(pDst->x) = fp0;
        pDst->z = fp1[0];

        pV++;
        pDst++;
    }

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformCoord(VEC3* pOut, const MTX44* __restrict pM, const VEC3* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    // { xy zw } = { x y z 1 }
    const f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
    const f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

    f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, f._30), 0, 0);
    f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, f._32), 0, 0);

    f32x2 fp0, fp1, fp2, fp3;

    // fp3 = [ m30 m31 m32, m33 ] . T[ x y z 1 ]
    fp3 = __PS_MUL(m30m31, xy);
    fp3 = __PS_MADD(m32m33, zw, fp3);
    fp3 = __PS_SUM0(fp3, fp3, fp3);

    // rw[1] = 1 / w
    const f32x2 rw = __PS_DIV(zw, __PS_MERGE00(fp3, fp3));

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);
    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, f._20), 0, 0);
    f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, f._22), 0, 0);

    // fp0 = [ m00 m01 m02, m03 ] . T[ x y z 1 ]
    //       [ m10 m11 m12, m13 ]
    fp0 = __PS_MUL(m00m01, xy);
    fp1 = __PS_MUL(m10m11, xy);
    fp0 = __PS_MADD(m02m03, zw, fp0);
    fp1 = __PS_MADD(m12m13, zw, fp1);
    fp0 = __PS_SUM0(fp0, fp0, fp0);
    fp0 = __PS_SUM1(fp1, fp0, fp1);
    fp0 = __PS_MULS1(fp0, rw);

    // fp2 = [ m20 m21 m22, m23 ] . T[ x y z 1 ]
    fp2 = __PS_MUL(m20m21, xy);
    fp2 = __PS_MADD(m22m23, zw, fp2);
    fp2 = __PS_SUM0(fp2, fp2, fp2);
    fp2 = __PS_MULS1(fp2, rw);

    tof32x2(pOut->x) = fp0;
    pOut->z = fp2[0];

    return pOut;
}

NW_MATH_INLINE VEC3*
VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    const f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    const f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    const f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    const f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);
    const f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, f._20), 0, 0);
    const f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, f._22), 0, 0);
    const f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, f._30), 0, 0);
    const f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, f._32), 0, 0);

    const f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    const f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    const f32x2 m02m12 = __PS_MERGE00(m02m03, m12m13);
    const f32x2 m03m13 = __PS_MERGE11(m02m03, m12m13);
    const f32x2 m20m30 = __PS_MERGE00(m20m21, m30m31);
    const f32x2 m21m31 = __PS_MERGE11(m20m21, m30m31);
    const f32x2 m22m32 = __PS_MERGE00(m22m23, m32m33);
    const f32x2 m23m33 = __PS_MERGE11(m22m23, m32m33);

    VEC3* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z 1 }
        const f32x2 xy = __PSQ_LX(pV, offsetof(VEC3, x), 0, 0);
        const f32x2 zw = __PSQ_LX(pV, offsetof(VEC3, z), 1, 0);

        // fp0 = [ m00 m01 m02, m03 ] . T[ x y z 1 ]
        //       [ m10 m11 m12, m13 ]
        fp0 = __PS_MADDS0(m02m12, zw, m03m13);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m00m10, xy, fp0);

        // fp1 = [ m20 m21 m22, m23 ] . T[ x y z 1 ]
        //       [ m30 m31 m32, m33 ]
        fp1 = __PS_MADDS0(m22m32, zw, m23m33);
        fp1 = __PS_MADDS1(m21m31, xy, fp1);
        fp1 = __PS_MADDS0(m20m30, xy, fp1);

        // rw[1] = 1 / w
        const f32x2 rw = __PS_DIV(zw, __PS_MERGE11(fp1, fp1));

        fp0 = __PS_MULS1(fp0, rw);
        fp1 = __PS_MULS1(fp1, rw);

        tof32x2(pDst->x) = fp0;
        pDst->z = fp1[0];

        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX44* __restrict pM, const VEC4* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    const f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, m[0][0]), 0, 0);
    const f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, m[0][2]), 0, 0);
    const f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, m[1][0]), 0, 0);
    const f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, m[1][2]), 0, 0);
    const f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, m[2][0]), 0, 0);
    const f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, m[2][2]), 0, 0);
    const f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, m[3][0]), 0, 0);
    const f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, m[3][2]), 0, 0);

    f32x2 fp0, fp1, fp2, fp3;

    // { xy zw } = { x y z w }
    const f32x2 xy = __PSQ_LX(pV, offsetof(VEC4, x), 0, 0);
    const f32x2 zw = __PSQ_LX(pV, offsetof(VEC4, z), 0, 0);

    // fp1 = [ m00 m01 m02, m03 ] . T[ x y z w ]
    //       [ m10 m11 m12, m13 ]
    fp0 = __PS_MUL(m00m01, xy);
    fp1 = __PS_MUL(m10m11, xy);
    fp0 = __PS_MADD(m02m03, zw, fp0);
    fp1 = __PS_MADD(m12m13, zw, fp1);
    fp0 = __PS_SUM0(fp0, fp0, fp0);
    fp1 = __PS_SUM1(fp1, fp0, fp1);

    // fp3 = [ m20 m21 m22, m23 ] . T[ x y z w ]
    //       [ m30 m31 m32, m33 ]
    fp2 = __PS_MUL(m20m21, xy);
    fp3 = __PS_MUL(m30m31, xy);
    fp2 = __PS_MADD(m22m23, zw, fp2);
    fp3 = __PS_MADD(m32m33, zw, fp3);
    fp2 = __PS_SUM0(fp2, fp2, fp2);
    fp3 = __PS_SUM1(fp3, fp2, fp3);

    tof32x2(pOut->x) = fp1;
    tof32x2(pOut->z) = fp3;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);

    f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    f32x2 m02m12 = __PS_MERGE00(m02m03, m12m13);
    f32x2 m03m13 = __PS_MERGE11(m02m03, m12m13);

    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, f._20), 0, 0);
    f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, f._22), 0, 0);
    f32x2 m30m31 = __PSQ_LX(pM, offsetof(MTX44, f._30), 0, 0);
    f32x2 m32m33 = __PSQ_LX(pM, offsetof(MTX44, f._32), 0, 0);

    f32x2 m20m30 = __PS_MERGE00(m20m21, m30m31);
    f32x2 m21m31 = __PS_MERGE11(m20m21, m30m31);
    f32x2 m22m32 = __PS_MERGE00(m22m23, m32m33);
    f32x2 m23m33 = __PS_MERGE11(m22m23, m32m33);

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z w }
        f32x2 xy = __PSQ_LX(pV, offsetof(VEC4, x), 0, 0);
        f32x2 zw = __PSQ_LX(pV, offsetof(VEC4, z), 0, 0);

        // fp0 = [ m00 m01 m02, m03 ] . T[ x y z w ]
        //       [ m10 m11 m12, m13 ]
        fp0 = __PS_MULS0(m00m10, xy);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m02m12, zw, fp0);
        fp0 = __PS_MADDS1(m03m13, zw, fp0);

        // fp1 = [ m20 m21 m22, m23 ] . T[ x y z w ]
        //       [ m30 m31 m32, m33 ]
        fp1 = __PS_MULS0(m20m30, xy);
        fp1 = __PS_MADDS1(m21m31, xy, fp1);
        fp1 = __PS_MADDS0(m22m32, zw, fp1);
        fp1 = __PS_MADDS1(m23m33, zw, fp1);

        tof32x2(pDst->x) = fp0;
        tof32x2(pDst->z) = fp1;

        ++pDst;
        ++pV;
    }

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX34* __restrict pM, const VEC4* __restrict pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    const f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, m[0][0]), 0, 0);
    const f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, m[0][2]), 0, 0);
    const f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, m[1][0]), 0, 0);
    const f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, m[1][2]), 0, 0);
    const f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, m[2][0]), 0, 0);
    const f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, m[2][2]), 0, 0);

    f32x2 fp0, fp1, fp2;

    // { xy zw } = { x y z w }
    const f32x2 xy = __PSQ_LX(pV, offsetof(VEC4, x), 0, 0);
    const f32x2 zw = __PSQ_LX(pV, offsetof(VEC4, z), 0, 0);

    // fp1 = [ m00 m01 m02 m03 ] . T[ x y z w ]
    //       [ m10 m11 m12 m13 ]
    fp0 = __PS_MUL(m00m01, xy);
    fp0 = __PS_MADD(m02m03, zw, fp0);
    fp0 = __PS_SUM0(fp0, fp0, fp0);

    fp1 = __PS_MUL(m10m11, xy);
    fp1 = __PS_MADD(m12m13, zw, fp1);
    fp1 = __PS_SUM1(fp1, fp0, fp1);

    // fp2 = [ m20 m21 m22 m23 ] . T[ x y z w ]
    //       [   0   0   0    1 ]
    fp2 = __PS_MUL(m20m21, xy);
    fp2 = __PS_MADD(m22m23, zw, fp2);
    fp2 = __PS_SUM0(fp2, zw, fp2);

    tof32x2(pOut->x) = fp1;
    tof32x2(pOut->z) = fp2;

    return pOut;
}

NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 count)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    f32x2 m00m01 = __PSQ_LX(pM, offsetof(MTX44, f._00), 0, 0);
    f32x2 m02m03 = __PSQ_LX(pM, offsetof(MTX44, f._02), 0, 0);
    f32x2 m10m11 = __PSQ_LX(pM, offsetof(MTX44, f._10), 0, 0);
    f32x2 m12m13 = __PSQ_LX(pM, offsetof(MTX44, f._12), 0, 0);

    f32x2 m20m21 = __PSQ_LX(pM, offsetof(MTX44, f._20), 0, 0);
    f32x2 m22m23 = __PSQ_LX(pM, offsetof(MTX44, f._22), 0, 0);

    f32x2 m00m10 = __PS_MERGE00(m00m01, m10m11);
    f32x2 m01m11 = __PS_MERGE11(m00m01, m10m11);
    f32x2 m02m12 = __PS_MERGE00(m02m03, m12m13);
    f32x2 m03m13 = __PS_MERGE11(m02m03, m12m13);

    VEC4* pDst = pOut;
    for (u32 i = 0; i < count; ++i)
    {
        f32x2 fp0, fp1;

        // { xy zw } = { x y z w }
        f32x2 xy = __PSQ_LX(pV, offsetof(VEC4, x), 0, 0);
        f32x2 zw = __PSQ_LX(pV, offsetof(VEC4, z), 0, 0);

        // fp0 = [ m00 m01 m02, m03 ] . T[ x y z w ]
        //       [ m10 m11 m12, m13 ]
        // fp1 = [ m20 m21 m22, m23 ] . T[ x y z w ]
        //       [   0   0   0    1 ]
        fp0 = __PS_MULS0(m00m10, xy);
        fp0 = __PS_MADDS1(m01m11, xy, fp0);
        fp0 = __PS_MADDS0(m02m12, zw, fp0);
        fp0 = __PS_MADDS1(m03m13, zw, fp0);

        fp1 = __PS_MUL(m20m21, xy);
        fp1 = __PS_MADD(m22m23, zw, fp1);
        fp1 = __PS_SUM0(fp1, zw, fp1);

        tof32x2(pDst->x) = fp0;
        tof32x2(pDst->z) = fp1;

        ++pDst;
        ++pV;
    }

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    ユーティリティ
//@{

//---------------------------------------------------------------------------
//! @brief        2x3行列を2x2行列にコピーします。
//!
//! @param[out]   pOut    コピー先の行列へのポインタです。
//! @param[in]    pM      コピー元の行列へのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX22*
MTX23ToMTX22(MTX22* pOut, const MTX23* pM)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );

    pOut->f._00 = pM->f._00; pOut->f._01 = pM->f._01;
    pOut->f._10 = pM->f._10; pOut->f._11 = pM->f._11;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        3x4行列を3x3行列にコピーします。
//!
//! @param[out]   pOut  コピー先の行列へのポインタ。
//! @param[in]    pM    コピー元の行列へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE  MTX33*
MTX34ToMTX33(MTX33* pOut, const MTX34* pM)
{
    pOut->f._00 = pM->f._00; pOut->f._01 = pM->f._01; pOut->f._02 = pM->f._02;
    pOut->f._10 = pM->f._10; pOut->f._11 = pM->f._11; pOut->f._12 = pM->f._12;
    pOut->f._20 = pM->f._20; pOut->f._21 = pM->f._21; pOut->f._22 = pM->f._22;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        3x3行列を3x4行列にコピーします。
//!
//! @param[out]   pOut  コピー先の行列へのポインタ。
//! @param[in]    pM    コピー元の行列へのポインタ
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX34*
MTX33ToMTX34(MTX34* pOut, const MTX33* pM)
{
    pOut->f._00 = pM->f._00; pOut->f._01 = pM->f._01; pOut->f._02 = pM->f._02;
    pOut->f._10 = pM->f._10; pOut->f._11 = pM->f._11; pOut->f._12 = pM->f._12;
    pOut->f._20 = pM->f._20; pOut->f._21 = pM->f._21; pOut->f._22 = pM->f._22;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        回転行列を元にクォータニオンを作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//! @param[in]    pMtx  クォータニオンの元になる回転行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE QUAT*
MTX34ToQUAT(QUAT* pOut, const MTX34* pMtx)
{
    f32 tr, s;
    s32 i, j, k;
    s32 nxt[3] = {1, 2, 0};
    f32 q[3];

    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pMtx );

    const f32 (*const m)[4] = pMtx->m;

    tr = m[0][0] + m[1][1] + m[2][2];
    if ( tr > 0.0f )
    {
        s = (f32)::std::sqrtf(tr + 1.0f);
        pOut->w = s * 0.5f;
        s = 0.5f / s;
        pOut->x = (m[2][1] - m[1][2]) * s;
        pOut->y = (m[0][2] - m[2][0]) * s;
        pOut->z = (m[1][0] - m[0][1]) * s;
    }
    else
    {
        i = 0;
        if (m[1][1] > m[0][0]) i = 1;
        if (m[2][2] > m[i][i]) i = 2;
        j = nxt[i];
        k = nxt[j];
        s = (f32)::std::sqrtf( (m[i][i] - (m[j][j] + m[k][k])) + 1.0f );
        q[i] = s * 0.5f;

        if (s!=0.0f)
            s = 0.5f / s;

        pOut->w = (m[k][j] - m[j][k]) * s;
        q[j] = (m[i][j] + m[j][i]) * s;
        q[k] = (m[i][k] + m[k][i]) * s;

        pOut->x = q[0];
        pOut->y = q[1];
        pOut->z = q[2];
    }

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列の転置行列を作成します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。p と同じ行列を指していても構いません。
//! @param[in]    p     転置する行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX43*
MTX34Transpose(MTX43* pOut, const MTX34* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->f._00 = p->f._00;
    pOut->f._01 = p->f._10;
    pOut->f._02 = p->f._20;

    pOut->f._10 = p->f._01;
    pOut->f._11 = p->f._11;
    pOut->f._12 = p->f._21;

    pOut->f._20 = p->f._02;
    pOut->f._21 = p->f._12;
    pOut->f._22 = p->f._22;

    pOut->f._30 = p->f._03;
    pOut->f._31 = p->f._13;
    pOut->f._32 = p->f._23;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        4x3行列の転置を、3x4行列として取得します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。
//! @param[in]    p     元となる行列へのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX34*
MTX43Transpose(MTX34* pOut, const MTX43* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->f._00 = p->f._00;
    pOut->f._01 = p->f._10;
    pOut->f._02 = p->f._20;
    pOut->f._03 = p->f._30;

    pOut->f._10 = p->f._01;
    pOut->f._11 = p->f._11;
    pOut->f._12 = p->f._21;
    pOut->f._13 = p->f._31;

    pOut->f._20 = p->f._02;
    pOut->f._21 = p->f._12;
    pOut->f._22 = p->f._22;
    pOut->f._23 = p->f._32;

    return pOut;
}



//---------------------------------------------------------------------------
//! @brief        2次元ベクトルを行列で変換します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。 pV と同じベクトルを指しても構いません。
//! @param[in]    pM      変換行列へのポインタです。
//! @param[in]    pV      元となるベクトルへのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC2*
VEC2Transform(VEC2* pOut, const MTX23* pM, const VEC2* pV)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pV );

    VEC2 tmp;
    const VEC2* pVec;
    if (pOut == pV)
    {
        tmp.x = pV->x;
        tmp.y = pV->y;
        pVec = &tmp;
    }
    else
    {
        pVec = pV;
    }

    pOut->x = pM->f._00 * pVec->x + pM->f._01 * pVec->y + pM->f._02;
    pOut->y = pM->f._10 * pVec->x + pM->f._11 * pVec->y + pM->f._12;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。それぞれのベクトルの 4 要素目を 1 として計算します。
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM     変換行列へのポインタ。
//! @param[in]    pV     元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3TransformArray(VEC3* pOut, const MTX34* __restrict pM, const VEC3* __restrict pV, s32 count)
{
    return NW_MATH_IMPL_NS::VEC3TransformArray(pOut, pM, pV, count);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。それぞれのベクトルの 4 要素目を 1 として計算します。
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM     変換行列へのポインタ。
//! @param[in]    pV     元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC3TransformArray(VEC4* pOut, const MTX44* __restrict pM, const VEC3* __restrict pV, u32 count)
{
    return NW_MATH_IMPL_NS::VEC3TransformArray(pOut, pM, pV, count);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。ベクトルの 4 要素目を 0 として計算します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3TransformNormal(VEC3* pOut, const MTX34* pM, const VEC3* pV)
{
    return NW_MATH_IMPL_NS::VEC3TransformNormal(pOut, pM, pV);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。それぞれのベクトルの 4 要素目を 0 として計算します。
//!
//! @param[out]   pOut  計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3TransformNormalArray(VEC3* pOut, const MTX34* pM, const VEC3* pV, u32 count)
{
    return NW_MATH_IMPL_NS::VEC3TransformNormalArray(pOut, pM, pV, count);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。ベクトルの 4 要素目を 1 として計算し、第1～3要素を第4要素で割った3次元ベクトルを pOut に格納します
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3TransformCoord(VEC3* pOut, const MTX44* pM, const VEC3* pV)
{
    return NW_MATH_IMPL_NS::VEC3TransformCoord(pOut, pM, pV);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。それぞれのベクトルの 4 要素目を 1 として計算し、第1～3要素を第4要素で割った3次元ベクトルを pOut に格納します
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM     変換行列へのポインタ。
//! @param[in]    pV     元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC3*
VEC3TransformCoordArray(VEC3* pOut, const MTX44* pM, const VEC3* pV, u32 count)
{
    return NW_MATH_IMPL_NS::VEC3TransformCoordArray(pOut, pM, pV, count);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX44* __restrict pM, const VEC4* __restrict pV)
{
    return NW_MATH_IMPL_NS::VEC4Transform(pOut, pM, pV);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM     変換行列へのポインタ。
//! @param[in]    pV     元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX44* pM, const VEC4* pV, u32 count)
{
    return NW_MATH_IMPL_NS::VEC4TransformArray(pOut, pM, pV, count);
}


//---------------------------------------------------------------------------
//! @brief        ベクトルを行列で変換します。
//!
//! @param[out]   pOut  計算結果を受け取るバッファへのポインタ。pV と同じベクトルを指していても構いません。
//! @param[in]    pM    変換行列へのポインタ。
//! @param[in]    pV    元となるベクトルへのポインタ。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4Transform(VEC4* pOut, const MTX34* __restrict pM, const VEC4* __restrict pV)
{
    return NW_MATH_IMPL_NS::VEC4Transform(pOut, pM, pV);
}

//---------------------------------------------------------------------------
//! @brief        ベクトルの配列を行列で変換します。
//!
//! @param[out]   pOut   計算結果を受け取る配列の先頭へのポインタ。
//! @param[in]    pM     変換行列へのポインタ。
//! @param[in]    pV     元となるベクトル配列の先頭へのポインタ。
//! @param[in]    count  変換するベクトル配列の要素数
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE VEC4*
VEC4TransformArray(VEC4* pOut, const MTX34* pM, const VEC4* pV, u32 count)
{
    return NW_MATH_IMPL_NS::VEC4TransformArray(pOut, pM, pV, count);
}

//@}

}  // namespace math
}  // namespace nw


