﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo/HAL Laboratory, Inc. All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

namespace nw {
namespace math {

//---------------------------------------------------------------------------
//        MTX23
//---------------------------------------------------------------------------

namespace internal { namespace standard {

NW_MATH_INLINE MTX23*
MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 + p2->f._00;
    pOut->f._01 = p1->f._01 + p2->f._01;
    pOut->f._02 = p1->f._02 + p2->f._02;

    pOut->f._10 = p1->f._10 + p2->f._10;
    pOut->f._11 = p1->f._11 + p2->f._11;
    pOut->f._12 = p1->f._12 + p2->f._12;

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = p1->f._00 - p2->f._00;
    pOut->f._01 = p1->f._01 - p2->f._01;
    pOut->f._02 = p1->f._02 - p2->f._02;

    pOut->f._10 = p1->f._10 - p2->f._10;
    pOut->f._11 = p1->f._11 - p2->f._11;
    pOut->f._12 = p1->f._12 - p2->f._12;

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    pOut->f._00 = p->f._00 * f;
    pOut->f._01 = p->f._01 * f;
    pOut->f._02 = p->f._02 * f;

    pOut->f._10 = p->f._10 * f;
    pOut->f._11 = p->f._11 * f;
    pOut->f._12 = p->f._12 * f;

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* __restrict p1, const MTX23* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    MTX23  tmp;
    MTX23* __restrict pMtx;

    if ( (pOut == p1) || (pOut == p2) )
    {
        pMtx = &tmp;
    }
    else
    {
        pMtx = pOut;
    }

    pMtx->f._00 = p1->f._00 * p2->f._00 + p1->f._01 * p2->f._10;
    pMtx->f._01 = p1->f._00 * p2->f._01 + p1->f._01 * p2->f._11;
    pMtx->f._02 = p1->f._00 * p2->f._02 + p1->f._01 * p2->f._12 + p1->f._02;

    pMtx->f._10 = p1->f._10 * p2->f._00 + p1->f._11 * p2->f._10;
    pMtx->f._11 = p1->f._10 * p2->f._01 + p1->f._11 * p2->f._11;
    pMtx->f._12 = p1->f._10 * p2->f._02 + p1->f._11 * p2->f._12 + p1->f._12;

    if (pMtx == &tmp)
    {
        MTX23Copy(pOut, &tmp);
    }

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Scale(MTX23* pOut, const MTX23* __restrict pM, const VEC2* __restrict pS)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pS );

    pOut->f._00 = pM->f._00 * pS->x;
    pOut->f._10 = pM->f._10 * pS->x;

    pOut->f._01 = pM->f._01 * pS->y;
    pOut->f._11 = pM->f._11 * pS->y;

    if (pOut != pM)
    {
        pOut->f._02 = pM->f._02;
        pOut->f._12 = pM->f._12;
    }

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    pOut->f._00 = t * p1->f._00 + p2->f._00;
    pOut->f._01 = t * p1->f._01 + p2->f._01;
    pOut->f._02 = t * p1->f._02 + p2->f._02;

    pOut->f._10 = t * p1->f._10 + p2->f._10;
    pOut->f._11 = t * p1->f._11 + p2->f._11;
    pOut->f._12 = t * p1->f._12 + p2->f._12;

    return pOut;
}

}} // namespace internal::standard

#if defined(NW_MATH_ENABLE_INTRINSICS)

namespace internal { namespace intrinsics {

NW_MATH_INLINE MTX23*
MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_ADD(tof32x2(p1->a[0]), tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_ADD(tof32x2(p1->a[2]), tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_ADD(tof32x2(p1->a[4]), tof32x2(p2->a[4]));

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_SUB(tof32x2(p1->a[0]), tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_SUB(tof32x2(p1->a[2]), tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_SUB(tof32x2(p1->a[4]), tof32x2(p2->a[4]));

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* p, f32 f)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    tof32x2(pOut->a[0]) = __PS_MULS0F(tof32x2(p->a[0]), f);
    tof32x2(pOut->a[2]) = __PS_MULS0F(tof32x2(p->a[2]), f);
    tof32x2(pOut->a[4]) = __PS_MULS0F(tof32x2(p->a[4]), f);

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* __restrict p1, const MTX23* __restrict p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    // d = a . b;
    //   = { a00 a01  a02 } . { b00 b01  b02 }
    //     { a10 a11  a12 }   { b10 b11  b12 }
    //                        {   0   0    1 }
    //
    // { f0 f4 } = { b00 b01  b02 }
    // { f1 f5 }   { b10 b11  b12 }

    const f32x2 f0 = __PSQ_LX(p2, offsetof(MTX23, f._00), 0, 0);
    const f32x2 f1 = __PSQ_LX(p2, offsetof(MTX23, f._10), 0, 0);
    const f32x2 f4 = __PS_FDUP(p2->f._02);
    const f32x2 f5 = __PS_FDUP(p2->f._12);

    f32x2 f8, f9, f10;

    // { d00 d01 d02 } = { f9 f10 } := { a00 a01 a02 } . { f0 f4 }
    //                                                   { f1 f5 }
    //                                                   {  0  1 }

    f8 = __PSQ_LX(p1, offsetof(MTX23, f._00), 0, 0); // { a00 a01 }
    f9 = __PS_MULS0(f0, f8);
    f10 = __PS_MULS0(f4, f8);
    f9 = __PS_MADDS1(f1, f8, f9);
    f10 = __PS_MADDS1(f5, f8, f10);
    f8 = __PSQ_LX(p1, offsetof(MTX23, f._02), 0, 0); // { a02 a10 }
    f10 = __PS_ADD(f8, f10);
    __PSQ_STX(pOut, offsetof(MTX23, f._00), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX23, f._02), f10, 1, 0);

    //                                                   { f0 f4 }
    // { d10 d11 d12 } = { f9 f10 } := { a10 a11 a12 } . { f1 f5 }
    //                                                   {  0  1 }

    f9 = __PS_MULS1(f0, f8);
    f10 = __PS_MULS1(f4, f8);
    f8 = __PSQ_LX(p1, offsetof(MTX23, f._11), 0, 0); // { a11 a12 }
    f9 = __PS_MADDS0(f1, f8, f9);
    f10 = __PS_MADDS0(f5, f8, f10);
    f10 = __PS_SUM0(f10, f10, f8);
    __PSQ_STX(pOut, offsetof(MTX23, f._10), f9, 0, 0);
    __PSQ_STX(pOut, offsetof(MTX23, f._12), f10, 1, 0);

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23Scale(MTX23* pOut, const MTX23* __restrict pM, const VEC2* __restrict pS)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );
    NW_ASSERT_NOT_NULL( pS );

    // スケール行列を右からかけるバージョン
    //
    // d = m . Mscale(s)
    //
    // d = { m00 m01 m02 } . { sx 0  0 }
    //     { m10 m11 m12 }   { 0  sy 0 }
    //                       { 0  0  1 }
    //
    //   = { sx*m00 sy*m01 m02 }
    //     { sx*m10 sy*m11 m12 }

    // xy = { sx, sy }
    f32x2 xy = tof32x2(pS->x);

    tof32x2(pOut->f._00) = __PS_MUL(tof32x2(pM->f._00), xy);
    tof32x2(pOut->f._10) = __PS_MUL(tof32x2(pM->f._10), xy);

    if (pOut != pM)
    {
        pOut->f._02 = pM->f._02;
        pOut->f._12 = pM->f._12;
    }

    return pOut;
}

NW_MATH_INLINE MTX23*
MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p1 );
    NW_ASSERT_NOT_NULL( p2 );

    tof32x2(pOut->a[0]) = __PS_MADDS0F(tof32x2(p1->a[0]), t, tof32x2(p2->a[0]));
    tof32x2(pOut->a[2]) = __PS_MADDS0F(tof32x2(p1->a[2]), t, tof32x2(p2->a[2]));
    tof32x2(pOut->a[4]) = __PS_MADDS0F(tof32x2(p1->a[4]), t, tof32x2(p2->a[4]));

    return pOut;
}

}} // namespace internal::intrinsics

#endif // NW_MATH_ENABLE_INTRINSICS

//----------------------------------------
//! @name    行列
//@{

//---------------------------------------------------------------------------
//! @brief        2x3行列をコピーします。
//!
//! @param[out]   pOut    コピー先の行列のポインタです。
//! @param[in]    p       コピー元の行列のポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Copy(MTX23* pOut, const MTX23* p)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( p );

    if (pOut != p)
    {
        *pOut = *p;
    }

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        ゼロ行列を作成します。
//!
//! @param[out]   pOut    ゼロ行列を格納するバッファへのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Zero(MTX23* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    pOut->f._00 = pOut->f._01 = pOut->f._02 =
    pOut->f._10 = pOut->f._11 = pOut->f._12 = 0.f;

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        2x3の単位行列を作成します。
//!
//! @param[out]   pOut    単位行列を格納するバッファへのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Identity(MTX23* pOut)
{
    NW_ASSERT_NOT_NULL( pOut );

    MTX23Copy(pOut, MTX23::Identity());

    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        単位行列であるかどうかを判定します。
//!
//! @param[in]    p    判定対象の行列です。
//!
//! @return       p が単位行列であれば、true を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE bool
MTX23IsIdentity(const MTX23* p)
{
    NW_ASSERT_NOT_NULL( p );

    return p->f._00 == 1.f && p->f._01 == 0.f && p->f._02 == 0.f &&
           p->f._10 == 0.f && p->f._11 == 1.f && p->f._12 == 0.f;
}

//---------------------------------------------------------------------------
//! @brief        2x3 行列を加算します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。p1, p2 と同じ行列を指しても構いません。
//! @param[in]    p1      元の行列へのポインタです。
//! @param[in]    p2      足す行列へのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Add(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    return NW_MATH_IMPL_NS::MTX23Add(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        2x3 行列の減算です。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。p1, p2 と同じ行列を指しても構いません。
//! @param[in]    p1      元の行列へのポインタです。
//! @param[in]    p2      減算する行列へのポインタです。
//!
//! @return
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Sub(MTX23* pOut, const MTX23* p1, const MTX23* p2)
{
    return NW_MATH_IMPL_NS::MTX23Sub(pOut, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        2x3行列のスカラー積を計算します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。p と同じ行列を指していても構いません。
//! @param[in]    p       元の行列へのポインタです。
//! @param[in]    f       掛ける数です。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* p, f32 f)
{
    return NW_MATH_IMPL_NS::MTX23Mult(pOut, p, f);
}

//---------------------------------------------------------------------------
//! @brief        2x3 行列の積を計算します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。 p1, p2 と同じ行列を指していても構いません。
//! @param[in]    p1      左辺値へのポインタです。
//! @param[in]    p2      右辺値へのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Mult(MTX23* pOut, const MTX23* __restrict p1, const MTX23* __restrict p2)
{
    return NW_MATH_IMPL_NS::MTX23Mult(pOut, p1, p2);
}


//---------------------------------------------------------------------------
//! @brief        2x3 行列にスケール変換を適用します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。pMと同じ行列を指していても構いません。
//! @param[in]    pM      元となる行列へのポインタです。
//! @param[in]    pS      それぞれの軸方向のスケール値が格納されたベクトルのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Scale(MTX23* pOut, const MTX23* __restrict pM, const VEC2* __restrict pS)
{
    return NW_MATH_IMPL_NS::MTX23Scale(pOut, pM, pS);
}

//---------------------------------------------------------------------------
//! @brief        2x3 行列に平行移動を適用します。移動行列を右から掛けます。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。pMと同じ行列を指していても構いません。
//! @param[in]    pM      元となる行列へのポインタです。
//! @param[in]    pT      それぞれの軸方向の移動量が格納されたベクトルのポインタです。
//!
//! @return       pOUt を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23Translate(MTX23* pOut, const MTX23* pM, const VEC2* pT)
{
    if (pOut != pM)
    {
        (void)MTX23Copy(pOut, pM);
    }

    VEC2 tmp;
    VEC2Transform(&tmp, pM, pT);

    pOut->f._02 = tmp.x;
    pOut->f._12 = tmp.y;
    return pOut;
}

//---------------------------------------------------------------------------
//! @brief        回転行列を作成します。
//!
//! @param[out]   pOut   計算結果を受け取るバッファへのポインタです。
//! @param[in]    idx    1円周を 0x100000000 とする単位での角度です。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23RotIdx(MTX23* pOut, u32 idx)
{
    NW_ASSERT_NOT_NULL( pOut );

    f32 sin, cos;

    SinCosIdx(&sin, &cos, idx);

    pOut->f._00 = pOut->f._11 = cos;
    pOut->f._01 = sin;
    pOut->f._10 = -sin;
    pOut->f._02 = pOut->f._12 = 0.f;

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        回転の中心を指定した回転行列を作成します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。
//! @param[in]    pCenter 回転の中心座標です。
//! @param[in]    idx    1円周を 0x100000000 とする単位での角度です。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23RotCenterIdx(MTX23* pOut, const VEC2* pCenter, u32 idx)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pCenter );

    MTX23 trans;
    trans.f._00 = trans.f._11 = 1.f;
    trans.f._10 = trans.f._01 = 0.f;
    trans.f._02 = pCenter->x;
    trans.f._12 = pCenter->y;

    f32 sin, cos;

    SinCosIdx(&sin, &cos, idx);

    (void)MTX23RotIdx(pOut, idx);

    MTX23Mult(pOut, &trans, pOut);

    trans.f._02 = -trans.f._02;
    trans.f._12 = -trans.f._12;

    MTX23Mult(pOut, pOut, &trans);

    return pOut;
}


//---------------------------------------------------------------------------
//! @brief        行列を実数倍して、別の行列を足します。
//!
//! @param[out]   pOut    計算結果を受け取るバッファへのポインタです。p1, p2 と同じ行列を指していても構いません。
//! @param[in]    t       掛ける数です。
//! @param[in]    p1      元の行列へのポインタです。
//! @param[in]    p2      加算する行列へのポインタです。
//!
//! @return
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX23MAdd(MTX23* pOut, f32 t, const MTX23* p1, const MTX23* p2)
{
    return NW_MATH_IMPL_NS::MTX23MAdd(pOut, t, p1, p2);
}

//---------------------------------------------------------------------------
//! @brief        2x2 行列を 2x3行列にコピーします。
//!
//! @param[out]   pOut    コピー先の行列へのポインタです。
//! @param[in]    pM      コピー元の行列へのポインタです。
//!
//! @return       pOut を返します。
//---------------------------------------------------------------------------
NW_MATH_INLINE MTX23*
MTX22ToMTX23(MTX23* pOut, const MTX22* pM)
{
    NW_ASSERT_NOT_NULL( pOut );
    NW_ASSERT_NOT_NULL( pM );

    pOut->f._00 = pM->f._00;
    pOut->f._01 = pM->f._01;
    pOut->f._10 = pM->f._10;
    pOut->f._11 = pM->f._11;

    pOut->f._02 = pOut->f._12 = 0.f;
    return pOut;
}

//@}

}  // namespace math
}  // namespace nw
