﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_RESSKELETALANIM_H_
#define NW_G3D_RES_RESSKELETALANIM_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/res/g3d_ResCommon.h>
#include <nw/g3d/res/g3d_ResDictionary.h>
#include <nw/g3d/res/g3d_ResAnimCurve.h>
#include <nw/g3d/res/g3d_ResSkeleton.h>
#include <nw/g3d/res/g3d_Binding.h>

namespace nw { namespace g3d { namespace res {

//! @brief ボーンアニメーションの結果を格納する構造体です。
struct BoneAnimResult
{
    // NOTE: この定義順を変更する場合は BinSkeletalAnim の修正が必要。
    bit32 flag;
    Vec3 scale;
    Vec3 translate;
    u32 reserved;
    union
    {
        Vec3 euler; // euler 角
        Quat quat;
    };
};

//! @brief ボーンアニメーションブレンドの結果を格納する構造体です。
struct BoneAnimBlendResult
{
    bit32 flag;
    Vec3 scale;
    Vec3 translate;
    float weight;

    union
    {
        Vec3 axes[2]; // 3x3 回転行列の0行目と1行目。2行目は外積により求められる。
        Quat rotate;
    };
};

//--------------------------------------------------------------------------------------------------

//! @brief ボーンアニメーションの構造体です。
struct ResBoneAnimData
{
    bit32 flag;
    BinString ofsName;
    u8 beginRotate; //!< 最初の Rotate カーブのインデックス
    u8 beginTranslate; //!< 最初の Translate カーブのインデックス
    u8 numCurve;
    u8 beginBaseTranslate; //!< Translate の初期値へのオフセット
    s32 beginCurve; // SkeletalAnim 内での先頭 Curve のインデックス
    Offset ofsCurveArray;
    Offset ofsBaseValueArray; //!< 初期値が存在する場合は SRT 順に格納される
};

//! @brief ボーンアニメーションのリソースです。
class ResBoneAnim : private ResBoneAnimData
{
    NW_G3D_RES_COMMON(ResBoneAnim);

public:
    //! @brief ボーンアニメーションに関するフラグです。
    enum Flag
    {
        // フラグが立っている場合は初期値が存在する
        // カーブによって上書きされないものはコンスタントとなる

        //! @brief Scale の初期値が存在します。
        BASE_SCALE                  = 0x1 << 3,

        //! @brief Rotate の初期値が存在します。
        BASE_ROTATE                 = 0x1 << 4,

        //! @brief Translate の初期値が存在します。
        BASE_TRANSLATE              = 0x1 << 5,

        //! @briefprivate
        BASE_MASK                   = BASE_SCALE | BASE_ROTATE | BASE_TRANSLATE,

        //! @brief ScaleX のカーブが存在します。
        CURVE_SCALE_X               = 0x1 << 6,

        //! @brief ScaleY のカーブが存在します。
        CURVE_SCALE_Y               = 0x1 << 7,

        //! @brief ScaleZ のカーブが存在します。
        CURVE_SCALE_Z               = 0x1 << 8,

        //! @brief RotateX のカーブが存在します。
        CURVE_ROTATE_X              = 0x1 << 9,

        //! @brief RotateY のカーブが存在します。
        CURVE_ROTATE_Y              = 0x1 << 10,

        //! @brief RotateZ のカーブが存在します。
        CURVE_ROTATE_Z              = 0x1 << 11,

        //! @brief RotateW のカーブが存在します。
        CURVE_ROTATE_W              = 0x1 << 12,

        //! @brief TranslateX のカーブが存在します。
        CURVE_TRANSLATE_X           = 0x1 << 13,

        //! @brief TranslateY のカーブが存在します。
        CURVE_TRANSLATE_Y           = 0x1 << 14,

        //! @brief TranslateZ のカーブが存在します。
        CURVE_TRANSLATE_Z           = 0x1 << 15,

        //! @briefprivate
        CURVE_SCALE_MASK            = CURVE_SCALE_X | CURVE_SCALE_Y | CURVE_SCALE_Z,

        //! @briefprivate
        CURVE_ROTATE_MASK           = CURVE_ROTATE_X | CURVE_ROTATE_Y | CURVE_ROTATE_Z | CURVE_ROTATE_W,

        //! @briefprivate
        CURVE_TRANSLATE_MASK        = CURVE_TRANSLATE_X | CURVE_TRANSLATE_Y | CURVE_TRANSLATE_Z,

        //! @briefprivate
        CURVE_MASK                  = CURVE_SCALE_MASK | CURVE_ROTATE_MASK | CURVE_TRANSLATE_MASK,

        // 以下のフラグは LocalMtx の flag にコピーされる

        //! @brief MayaSSC で計算します。
        SEGMENT_SCALE_COMPENSATE    = ResBone::SEGMENT_SCALE_COMPENSATE,

        //! @brief Scale が等方です。
        SCALE_UNIFORM               = ResBone::SCALE_UNIFORM,

        //! @brief Scale の体積が 1 です。
        SCALE_VOLUME_ONE            = ResBone::SCALE_VOLUME_ONE,

        //! @brief Rotate が 0 です。
        ROTATE_ZERO                 = ResBone::ROTATE_ZERO,

        //! @brief Translate が 0 です。
        TRANSLATE_ZERO              = ResBone::TRANSLATE_ZERO,

        //! @brief Scale が 1 です。
        SCALE_ONE                   = ResBone::SCALE_ONE,

        //! @brief Rotate, Translate が 0 です。
        ROTTRANS_ZERO               = ResBone::ROTTRANS_ZERO,

        //! @brief トランスフォームが単位行列です。
        IDENTITY                    = ResBone::IDENTITY,

        //! @briefprivate
        TRANSFORM_MASK              = ResBone::TRANSFORM_MASK
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief ResBone を用いて BoneAnimResult の初期化を行います。
    void Init(BoneAnimResult* pResult, const ResBone* bone) const;

    //! @brief Init() 関数の別名関数です。
    void Initialize(BoneAnimResult* pResult, const ResBone* bone) const
    {
        Init(pResult, bone);
    }

    //@}

    //----------------------------------------
    //! @name 評価
    //@{

    //! @brief ボーンアニメーションを評価します。
    void Eval(BoneAnimResult* pResult, float frame) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(BoneAnimResult* pResult, float frame) const
    {
        Eval(pResult, frame);
    }

    //! @brief ボーンアニメーションを評価します。
    void Eval(BoneAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const;

    //! @brief Eval() 関数の別名関数です。
    void Evaluate(BoneAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
    {
        Eval(pResult, frame, pFrameCache);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResAnimCurve, Curve, GetName())

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief スケルトンのアニメーションの構造体です。
struct ResSkeletalAnimData
{
    BinaryBlockHeader blockHeader;
    BinString ofsName;
    BinString ofsPath;

    bit32 flag;
    s32 numFrame;
    u16 numBoneAnim;
    u16 numUserData;
    s32 numCurve;
    u32 bakedSize;

    Offset ofsBoneAnimArray;
    Offset ofsBindSkeleton; //!< PreBind でバインドされたスケルトン
    Offset ofsBindIndexArray; //!< 各アニメーションからターゲットへのインデックス配列
    Offset ofsUserDataDic;
};

//! @brief スケルトンのアニメーション
class ResSkeletalAnim : private ResSkeletalAnimData
{
    NW_G3D_RES_COMMON(ResSkeletalAnim);

public:
    enum Signature { SIGNATURE = NW_G3D_MAKE_U8X4_AS_U32('F', 'S', 'K', 'A') };

    //! @brief スケルタルアニメーションに関するフラグです。
    enum Flag
    {
        //! @brief カーブがベイク済みです。
        CURVE_BAKED         = AnimFlag::CURVE_BAKED,

        //! @brief ループを行います。
        PLAYPOLICY_LOOP     = AnimFlag::PLAYPOLICY_LOOP,

        // ScaleMode
        //! @briefprivate
        SCALE_SHIFT         = ResSkeleton::SCALE_SHIFT,

        //! @brief スケール計算をしません。
        SCALE_NONE          = ResSkeleton::SCALE_NONE,

        //! @brief 標準的な方法でスケールの計算をします。
        SCALE_STD           = ResSkeleton::SCALE_STD,

        //! @brief Maya 方式でスケールの計算をします。
        SCALE_MAYA          = ResSkeleton::SCALE_MAYA,

        //! @brief Softimage 方式でスケールの計算をします。
        SCALE_SOFTIMAGE     = ResSkeleton::SCALE_SOFTIMAGE,

        //! @briefprivate
        SCALE_MASK          = ResSkeleton::SCALE_MASK,

        // RotateMode
        //! @briefprivate
        ROT_SHIFT           = ResSkeleton::ROT_SHIFT,

        //! @brief Euler 角方式で回転を計算します。
        ROT_EULER_XYZ       = ResSkeleton::ROT_EULER_XYZ,

        //! @brief Quaternion 方式で回転を計算します。
        ROT_QUAT            = ResSkeleton::ROT_QUAT,

        //! @briefprivate
        ROT_MASK            = ResSkeleton::ROT_MASK,

        //! @briefprivate
        TRANSFORM_MASK      = ResSkeleton::TRANSFORM_MASK
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief 事前に ResSkeleton と関連付けを行います。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult PreBind(const ResSkeleton* pSkeleton);

    //! @brief 事前に ResSkeleton と関連付けが成功するか、チェックのみを行います。
    BindResult BindCheck(const ResSkeleton* target) const;

    //! @brief スケルタルアニメーションをコマ化します。
    bool BakeCurve(void* pBuffer, size_t bufferSize);

    //! @brief スケルタルアニメーションをカーブに戻します。
    void* ResetCurve();

    //! @brief スケルタルアニメーションをリセットします。
    void Reset();

    //@}

    //----------------------------------------
    //! @name 取得/破棄
    //@{

    //! @brief ループするかどうかを取得します。
    bool IsLooped() const { return 0 != (ref().flag & PLAYPOLICY_LOOP); }

    //! @brief カーブがベイクされているかどうかを取得します。
    bool IsCurveBaked() const { return 0 != (ref().flag & CURVE_BAKED); }

    //! @brief スケールの計算方式を取得します。
    int GetScaleMode() const { return ref().flag & SCALE_MASK; }

    //! @brief 回転の計算方式を取得します。
    int GetRotateMode() const { return ref().flag & ROT_MASK; }

    //! @brief カーブ数を取得します。
    int GetCurveCount() const { return ref().numCurve; }

    //! @brief フレーム数を取得します。
    int GetFrameCount() const { return ref().numFrame; }

    //! @brief コマ化に必要なバッファサイズを取得します。
    size_t GetBakedSize() const { return ref().bakedSize; }

    NW_G3D_RES_FIELD_STRING_DECL(Name)
    NW_G3D_RES_FIELD_STRING_DECL(Path)
    NW_G3D_RES_FIELD_CLASS_DECL(ResSkeleton, BindSkeleton)
    NW_G3D_RES_FIELD_CLASS_ARRAY_DECL_DETAIL(ResBoneAnim, BoneAnim, GetName())

    //@}

    //----------------------------------------
    //! @name ユーザデータ
    //@{

    NW_G3D_RES_FIELD_DIC_DECL_DETAIL(ResUserData, UserData, GetName())

    //@}
};

}}} // namespace nw::g3d::res

#endif // NW_G3D_RES_RESSKELETALANIM_H_
