﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_RES_BINDING_H_
#define NW_G3D_RES_BINDING_H_

#include <nw/g3d/g3d_config.h>

namespace nw { namespace g3d { namespace res {

//! @brief バインド対象です。
enum BindTarget
{
    //! @brief テクスチャを対象にします。
    BIND_TEXTURE                    = 0x1 << 0,

    //! @brief 関数を対象にします。
    BIND_FUNCTION                   = 0x1 << 0
};

//! @brief バインドの結果を表すクラスです。
//!
//! 最大16個の結果を表すことができます。
//! 1つの結果を表現できれば十分な場合は mask に MASK_ALL を用いることができます。
class BindResult
{
public:
    //! @brief 関連付けの成功も失敗もしていない初期状態を表す結果です。
    BindResult() : flag(0) {}

    //! @brief 関連付けの成功を表す結果です。
    static BindResult Bound(bit32 mask = MASK_ALL)
    {
        NW_G3D_ASSERT(mask <= MASK_ALL);
        return BindResult(mask << SHIFT_SUCCESS);
    }

    //! @brief 関連付けの失敗を表す結果です。
    static BindResult NotBound(bit32 mask = MASK_ALL)
    {
        NW_G3D_ASSERT(mask <= MASK_ALL);
        return BindResult(mask << SHIFT_FAILURE);
    }

    //! @brief mask が指定するタイプの関連付け状態を調べます。
    //!
    //! 累積結果の中で1つでもバインドに失敗していた場合は false を返します。
    bool IsComplete(bit32 mask = MASK_ALL)
    {
        NW_G3D_ASSERT(mask <= MASK_ALL);
        return 0 == ((flag >> SHIFT_FAILURE) & mask);
    }


    //! @brief mask が指定するタイプの関連付け状態を調べます。
    //!
    //! 累積結果の中で1つでもバインドに成功していた場合は false を返します。
    bool IsMissed(bit32 mask = MASK_ALL)
    {
        NW_G3D_ASSERT(mask <= MASK_ALL);
        return 0 == ((flag >> SHIFT_SUCCESS) & mask);
    }

    //! @brief 結果を累積します。
    const BindResult operator|(BindResult rhs) const { return BindResult(*this) |= rhs; }

    //! @brief 結果を累積します。
    BindResult& operator|=(BindResult rhs)
    {
        flag |= rhs.flag;
        return *this;
    }

private:
    enum
    {
        SHIFT_FAILURE   = 0,
        SHIFT_SUCCESS   = 16,
        MASK_ALL        = 0xFFFF
    };

    explicit BindResult(bit32 flag) : flag(flag) {}

    bit32 flag;
};

}}} // namespace nw::g3d

#endif // NW_G3D_RES_BINDING_H_
