﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_ASSERT_H_
#define NW_G3D_ASSERT_H_

#include <stdio.h>
#include <stdlib.h>

#if NW_G3D_IS_HOST_WIN
#include <malloc.h>
#include <crtdbg.h>
#elif defined(ANDROID)
#include <android/log.h>
#elif defined(__APPLE__)
#include <stdio.h>
#elif NW_G3D_IS_HOST_CAFE
#include <cafe/os.h>
#endif

#ifndef NW_G3D_ASSERTMSG
#if NW_G3D_IS_DEBUG || NW_G3D_IS_DEVELOP
#define NW_G3D_ASSERTMSG(exp, ...)                                                                 \
    (void) ((exp) ||                                                                               \
        (nw::g3d::DebugPrint("\t%s(%d)\n", NW_G3D_CODE_POSITION_NAME, __LINE__),                   \
         nw::g3d::DebugPrint("NW: Failed assertion. (" #exp ")\n"),                                \
         nw::g3d::DebugPrint(__VA_ARGS__),                                                         \
         NW_G3D_DEBUG_BREAK(), ::fflush(stdout), ::fflush(stderr), ::abort(), 0))
#else
#define NW_G3D_ASSERTMSG(exp, ...) ((void)0)
#endif
#endif // NW_G3D_ASSERTMSG

//--------------------------------------------------------------------------------------------------
//  Assert
//
#define NW_G3D_ASSERT(exp) \
    NW_G3D_ASSERTMSG(exp, "")

#define NW_G3D_ASSERT_NOT_NULL(exp) \
    NW_G3D_ASSERT_NOT_NULL_DETAIL(exp, "")

#define NW_G3D_ASSERT_INDEX_BOUNDS(index, size)                                                    \
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(index, size, "")

#define NW_G3D_ASSERT_ALIGNMENT(value, alignment)                                                  \
    NW_G3D_ASSERT_ALIGNMENT_DETAIL(value, alignment, "")

#define NW_G3D_ASSERT_ADDR_ALIGNMENT(addr, alignment)                                              \
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(addr, alignment, "")


//--------------------------------------------------------------------------------------------------
//  メッセージ付き Assert
//
#if NW_G3D_IS_DEBUG || NW_G3D_IS_DEVELOP
#define NW_G3D_ASSERT_NOT_NULL_DETAIL(exp, ...)                                                    \
    NW_G3D_ASSERTMSG(exp, "NW: Pointer must not be NULL. \n" __VA_ARGS__)

#define NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(index, size, ...)                                        \
    do                                                                                             \
    {                                                                                              \
        if(static_cast<uint>(index) > static_cast<uint>(size))                                     \
        {                                                                                          \
            nw::g3d::DebugPrint(                                                                   \
                    "NW: " #index " is out of bounds. (%d)\n"                                      \
                    "0 <= " #index " < %d is not satisfied.\n",                                    \
                    static_cast<int>(index), static_cast<int>(size)),                              \
            NW_G3D_ASSERTMSG(0, __VA_ARGS__);                                                      \
        }                                                                                          \
    } while (NW_G3D_STATIC_CONDITION(0))

#define NW_G3D_ASSERT_ALIGNMENT_DETAIL(value, alignment, ...)                                      \
    do                                                                                             \
    {                                                                                              \
        if(static_cast<uint>((value) & (alignment - 1)) != 0)                                      \
        {                                                                                          \
            nw::g3d::DebugPrint(                                                                   \
                    "NW: Failed alignment. \n"                                                     \
                    "value: 0x%x\n"                                                                \
                    "alignment: %d\n",                                                             \
                    value, static_cast<int>(alignment)),                                           \
            NW_G3D_ASSERTMSG(0, __VA_ARGS__);                                                      \
        }                                                                                          \
    } while (NW_G3D_STATIC_CONDITION(0))

#define NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(addr, alignment, ...)                                   \
    do                                                                                              \
    {                                                                                               \
        if((reinterpret_cast<uintptr_t>(addr) & (alignment - 1)) != 0)                              \
        {                                                                                           \
            nw::g3d::DebugPrint(                                                                    \
                    "NW: Failed alignment. \n"                                                      \
                    "ptr: 0x%x\n"                                                                   \
                    "alignment: %d\n",                                                              \
                    addr, static_cast<int>(alignment)),                                             \
            NW_G3D_ASSERTMSG(0, __VA_ARGS__);                                                       \
        }                                                                                           \
    } while (NW_G3D_STATIC_CONDITION(0))

#else
#define NW_G3D_ASSERT_NOT_NULL_DETAIL(exp, ...)                     (void)0
#define NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(index, size, ...)         (void)0
#define NW_G3D_ASSERT_ALIGNMENT_DETAIL(value, alignment, ...)       (void)0
#define NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(addr, alignment, ...)   (void)0
#endif

//--------------------------------------------------------------------------------------------------

#define NW_G3D_NOT_IMPLEMENTED() \
    NW_G3D_ASSERTMSG(0, "NW: Not implemented yet.\n")

//--------------------------------------------------------------------------------------------------

#define NW_G3D_NOT_SUPPORTED() \
    NW_G3D_ASSERTMSG(0, "NW: Not supported.\n")

//--------------------------------------------------------------------------------------------------

#if defined(_DEBUG) || defined(NW_DEBUG) || defined(NW_DEVELOP)
#define NW_G3D_WARNING(exp, ...)                                                                   \
    (void) ((exp) || (nw::g3d::DebugPrint("NW: "), nw::g3d::DebugPrint(__VA_ARGS__),               \
    nw::g3d::DebugPrint("\t%s(%d)\n", NW_G3D_CODE_POSITION_NAME, __LINE__), 0))
#else
#define NW_G3D_WARNING(exp, ... ) ((void)0)
#endif

namespace nw { namespace g3d {

void DebugPrint(const char* format, ...);

}} // namespace nw::g3d

#if NW_G3D_IS_HOST_WIN
#define NW_G3D_DEBUG_BREAK() _CrtDbgBreak()
#elif NW_G3D_IS_HOST_CAFE
#define NW_G3D_DEBUG_BREAK() OSHalt(NULL)
#elif defined(ANDROID) || defined(__APPLE__)
#define NW_G3D_DEBUG_BREAK() ((void)0)
#else
#define NW_G3D_DEBUG_BREAK()
#endif

#endif // NW_G3D_ASSERT_H_
