﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_TEXPATTERNANIMOBJ_H_
#define NW_G3D_TEXPATTERNANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResTexPatternAnim.h>

namespace nw { namespace g3d {

class ModelObj;
class MaterialObj;

//! @brief テクスチャパターンアニメーションインスタンスです。
class TexPatternAnimObj : public ModelAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    TexPatternAnimObj() : ModelAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    virtual BindResult Bind(const ResModel* pModel);
    virtual BindResult Bind(const ModelObj* pModel);
    virtual void BindFast(const ResModel* pModel);

    //! @brief 指定したインデックスのマテリアルにフラグを設定します。
    void SetBindFlag(int materialIndex, BindFlag flag)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        SetBindFlagImpl(materialIndex, flag);
    }

    //! @brief 指定したインデックスのマテリアルのフラグを取得します。
    BindFlag GetBindFlag(int materialIndex)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return GetBindFlagImpl(materialIndex);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    virtual void ApplyTo(ModelObj* pModelObj) const;
    //! @brief Calc()の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResTexPatternAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するテクスチャパターンアニメーションを差し替えます。
    //!
    void SetResource(ResTexPatternAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource()
    {
        m_pRes = NULL;
        SetTargetUnbound();
    }

    //! @brief アニメーション対象に対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResModel* pModel) const
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return pModel->GetMaterialCount() <= GetBindTable().GetSize();
    }

    //! @brief アニメーションリソースに対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResTexPatternAnim* pRes) const
    {
        NW_G3D_ASSERT_NOT_NULL(pRes);
        return pRes->GetMatAnimCount() <= m_MaxMatAnim && pRes->GetPatAnimCount() <= m_MaxSubBind
            && pRes->GetTextureRefCount() <= m_MaxTexture;
    }

    //! @brief アニメーションがもつテクスチャの数を取得します。
    int GetTextureCount() const { return m_MaxTexture; }

    //! @brief テクスチャを取得します。
    //!
    //! インデックスはアニメーションがもつテクスチャの数の範囲内とする必要があります。
    //!
    ResTexture* GetResTexture(int idxTexture) { return m_ppTextureArray[idxTexture]; }

    //! @brief テクスチャを取得します。
    //!
    //! インデックスはアニメーションがもつテクスチャの数の範囲内とする必要があります。
    //!
    const ResTexture* GetResTexture(int idxTexture) const
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(idxTexture, GetTextureCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        return m_ppTextureArray[idxTexture];
    }

    //! @brief テクスチャを設定します。
    //!
    //! アニメーションが参照するテクスチャを差し替えることができます。
    //!
    //! インデクスはアニメーションがもつテクスチャの数の範囲内とする必要があります。
    //!
    void SetResTexture(int idxTexture, ResTexture* pTexture)
    {
        NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(idxTexture, GetTextureCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pTexture, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        m_ppTextureArray[idxTexture] = pTexture;
    }

    //! @brief テクスチャをリソースの状態に戻します。
    void ClearTexture()
    {
        for (int idxTex = 0, numTex = m_pRes->GetTextureRefCount(); idxTex < numTex; ++idxTex)
        {
            m_ppTextureArray[idxTex] = m_pRes->GetTextureRef(idxTex)->Get();
        }
    }

    //@}

protected:

    //! @briefprivate マテリアルアニメーション取得します。
    //!
    //!
    ResTexPatternMatAnim* GetMatAnim(int animIndex)
    {
        return ResTexPatternMatAnim::ResCast(&m_pMatAnimArray[animIndex]);
    }

    //! @briefprivate マテリアルアニメーション取得します。
    //!
    //!
    const ResTexPatternMatAnim* GetMatAnim(int animIndex) const
    {
        return ResTexPatternMatAnim::ResCast(&m_pMatAnimArray[animIndex]);
    }

    //! @briefprivate マテリアルごとに関連付けを行う内部関数です。
    //!
    //!
    BindResult SubBind(const ResTexPatternMatAnim* pMatAnim, const ResMaterial* pMaterial);

    //! @briefprivate マテリアルごとに関連付けを行う内部関数です。
    //!
    //!
    BindResult SubBindFast(const ResTexPatternMatAnim* pMatAnim);

    //! @briefprivate マテリアルごとに適用を行う内部関数です。
    //!
    //!
    void ApplyTo(MaterialObj* pMaterialObj, int animIndex) const;

private:
    class Sizer;
    ResTexPatternAnim* m_pRes;
    ResTexPatternMatAnimData* m_pMatAnimArray;
    s32 m_MaxMatAnim;
    s32 m_MaxSubBind;
    s32 m_MaxTexture;
    s8* m_pSubBindIndexArray;
    ResTexture** m_ppTextureArray;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(TexPatternAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate TexPatternAnimObj のサイズを計算するためのクラスです。
//!
//!
class TexPatternAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief TexPatternAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        BIND_TABLE,
        SUBBIND_TABLE,
        FRAMECACHE_ARRAY,
        TEXTURE_TABLE,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief TexPatternAnimObj::Init() に渡して初期化を行うパラメータです。
class TexPatternAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResTexPatternAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxMatCount()、SetMaxMatAnimCount()、SetMaxPatAnimCount()、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumMat = m_NumMatAnim = m_NumPatAnim = m_NumCurve = m_NumTexture = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        m_NumMat = std::max(m_NumMat, pResModel->GetMaterialCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResTexPatternAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResTexPatternAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumMatAnim = std::max(m_NumMatAnim, pResAnim->GetMatAnimCount());
        m_NumPatAnim = std::max(m_NumPatAnim, pResAnim->GetPatAnimCount());
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_NumTexture = std::max(m_NumTexture, pResAnim->GetTextureRefCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のマテリアル数を指定します。
    void SetMaxMatCount(int matCount)
    {
        NW_G3D_ASSERT(matCount >= 0);
        m_NumMat = matCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアル数を取得します。
    int GetMaxMatCount() const { return m_NumMat; }

    //! @brief 使用できる最大のマテリアルアニメーション数を指定します。
    void SetMaxMatAnimCount(int matAnimCount)
    {
        NW_G3D_ASSERT(matAnimCount >= 0);
        m_NumMatAnim = matAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアルアニメーション数を取得します。
    int GetMaxMatAnimCount() const { return m_NumMatAnim; }

    //! @brief 使用できる最大のパターンアニメーション数を指定します。
    void SetMaxPatAnimCount(int patAnimCount)
    {
        NW_G3D_ASSERT(patAnimCount >= 0);
        m_NumPatAnim = patAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のパターンアニメーション数を取得します。
    int GetMaxPatAnimCount() const { return m_NumPatAnim; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 使用できる最大のテクスチャ数を指定します。
    void SetMaxTextureCount(int textureCount)
    {
        NW_G3D_ASSERT(textureCount >= 0);
        m_NumTexture = textureCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のテクスチャ数を取得します。
    int GetMaxTextureCount() const { return m_NumTexture; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const
    {
        return m_NumMat >= 0 && m_NumMatAnim >= 0 && m_NumPatAnim >= 0 && m_NumCurve >= 0;
    }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumMat;
    int m_NumMatAnim;
    int m_NumPatAnim;
    int m_NumCurve;
    int m_NumTexture;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief TexPatternAnimObj の構築を行うクラスです。
class TexPatternAnimObj::Builder : public TexPatternAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResTexPatternAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxMatCount()、SetMaxMatAnimCount()、SetMaxPatAnimCount()、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief TexPatternAnimObj を構築します。
    //!
    //! @param[in] pTexPatternAnimObj TexPatternAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(TexPatternAnimObj* pTexPatternAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pTexPatternAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief TexPatternAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief TexPatternAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  TexPatternAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief TexPatternAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_TEXPATTERNANIMOBJ_H_
