﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SKELETON_H_
#define NW_G3D_SKELETON_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GfxObject.h>
#include <nw/g3d/res/g3d_ResSkeleton.h>
#include <nw/g3d/g3d_Sizer.h>

namespace nw { namespace g3d {

//! @brief ボーンのローカル行列を表す構造体です。
struct LocalMtxData
{
    bit32 flag;
    Vec3 scale;
    Mtx34 mtxRT;
};

//! @brief ボーンのローカル行列を表すクラスです。
//!
//! flag が0の場合は scale と mtxRT をそのまま使用して計算されます。
//! さらに次のビットが立っている場合は、ライブラリがそれぞれのビットに応じた
//! 特殊計算や最適化計算を行うことがあります。
//!
//! ResBone::SEGMENT_SCALE_COMPENSATE <br>
//! ResBone::SCALE_UNIFORM            <br>
//! ResBone::SCALE_VOLUME_ONE         <br>
//! ResBone::ROTATE_ZERO              <br>
//! ResBone::TRANSLATE_ZERO           <br>
//!
class LocalMtx : public LocalMtxData
{
public:
    //! @brief コンストラクタです。
    LocalMtx() : LocalMtxData() { flag = 0; }

    //! @brief 内容の編集後に呼び出す関数です。
    //!
    //! この関数を呼び出すことで最適化フラグがリセットされます。
    //!
    void EndEdit() { flag &= ~ResBone::IDENTITY; }

    //! @brief LocalMtxData を LocalMtx に変換します。
    static LocalMtx& DownCast(LocalMtxData& data) { return static_cast<LocalMtx&>(data); }

    //! @brief LocalMtxData を LocalMtx に変換します。
    static const LocalMtx& DownCast(const LocalMtxData& data)
    {
        return static_cast<const LocalMtx&>(data);
    }

    //! @brief LocalMtxData を LocalMtx に変換します。
    static LocalMtx* DownCast(LocalMtxData* data) { return static_cast<LocalMtx*>(data); }

    //! @brief LocalMtxData を LocalMtx に変換します。
    static const LocalMtx* DownCast(const LocalMtxData* data)
    {
        return static_cast<const LocalMtx*>(data);
    }
};

//! @brief ワールドマトリクスを操作するためのクラスです。
class WorldMtxManip
{
public:
    //! @brief コンストラクタです。
    WorldMtxManip(Mtx34* pMtx, Vec3* pScale, bit32* pFlag)
        : m_pMtx(pMtx), m_pScale(pScale), m_pFlag(pFlag)
    {
        NW_G3D_ASSERT_NOT_NULL(pMtx);
        NW_G3D_ASSERT_NOT_NULL(pFlag);
        // ScaleMode が SCALE_NONE の場合 pScale は NULL
    }

    //! @brief スケールを取得します。
    const Vec3* GetScale() const { return m_pScale; }

    //! @brief ワールド行列を取得します。
    const Mtx34* GetMtx() const { return m_pMtx; }

    //! @brief 編集するためにワールド行列を取得します。
    //!
    //! 最適化フラグをリセットします。
    //!
    Mtx34* EditMtx()
    {
        *m_pFlag &= ~(ResBone::ROTTRANS_ZERO | ResBone::HI_ROTTRANS_ZERO);
        return m_pMtx;
    }

    //! @brief res::ResBone::Flag を取得します。
    bit32 GetFlag() const { return *m_pFlag; }

private:
    Mtx34* m_pMtx;
    Vec3* m_pScale;
    bit32* m_pFlag;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(WorldMtxManip);
};

//! @brief ボーンのワールド行列更新後に呼び出されるコールバックの基底クラスです。
class ICalcWorldCallback
{
public:
    enum { INVALID_BONE = 0xFFFF };

    //! @brief デストラクタです。
    virtual ~ICalcWorldCallback() {}

    //! @brief Exec() 用いる設定です。
    class CallbackArg
    {
    public:
        //! @brief  コンストラクタです。
        CallbackArg(const int& boneIndex, int callbackBone)
            : m_BoneIndex(boneIndex), m_CallbackBone(callbackBone)
        {
        }

        //! @brief 計算に用いるボーンインデクスを取得します。
        int GetBoneIndex() const { return m_BoneIndex; }

        //! @brief コールバックを呼び出すボーンインデクスを取得します。
        int GetCallbackBoneIndex() const { return m_CallbackBone; }

        //! @brief コールバックを呼び出すボーンインデクスを設定します。
        //!
        //! コールバック計算内部で対象のボーンインデクスを変更することで
        //! 複数のボーンに対してコールバックを呼び出すことができます。
        //!
        void SetCallbackBoneIndex(int boneIndex) { m_CallbackBone = boneIndex; }

    private:
        const int& m_BoneIndex;
        int m_CallbackBone;

        NW_G3D_DISALLOW_COPY_AND_ASSIGN(CallbackArg);
    };

    //! @brief コールバックを呼び出します。
    virtual void Exec(CallbackArg& arg, WorldMtxManip& manip) = 0;
};

//! @brief ICalcWorldCallbackの別名です。
typedef ICalcWorldCallback ICalculateWorldCallback;

//--------------------------------------------------------------------------------------------------

//! @brief スケルトンインスタンスです。
class SkeletonObj
{
public:
    //! @brief スケルトンの状態を表すフラグです。
    enum Flag
    {
        //! @briefprivate UniformBlock が構築済みであるかどうかを表すフラグです。
        //!
        BLOCK_BUFFER_VALID  = 0x1 << 0,

        //! @briefprivate UniformBlock のエンディアンスワップを行うかどうかを表すフラグです。
        //!
        BLOCK_BUFFER_SWAP   = 0x1 << 1,

        //! @briefprivate ローカル行列の一部がキャッシュに乗っているかどうかを表すフラグです。
        //!
        CACHE_LOCAL_MTX     = 0x1 << 2,

        //! @briefprivate 古いローカル行列の一部がキャッシュに乗っているかどうかを表すフラグです。
        //!
        INVALID_LOCAL_MTX   = 0x1 << 3,

        //! @briefprivate
        MASK_LOCAL_MTX      = CACHE_LOCAL_MTX | INVALID_LOCAL_MTX,

        //! @briefprivate ワールド行列の一部がキャッシュに乗っているかどうかを表すフラグです。
        //!
        CACHE_WORLD_MTX     = 0x1 << 4,

        //! @briefprivate 古いワールド行列の一部がキャッシュに乗っているかどうかを表すフラグです。
        //!
        INVALID_WORLD_MTX   = 0x1 << 5,

        //! @briefprivate
        MASK_WORLD_MTX      = CACHE_WORLD_MTX | INVALID_WORLD_MTX,

        //! @briefprivate 行列のスケーリング方法のビットへのシフトを表すフラグです。
        //!
        SCALE_SHIFT         = ResSkeleton::SCALE_SHIFT,

        //! @brief スケール計算を行わないことを表すフラグです。
        SCALE_NONE          = ResSkeleton::SCALE_NONE,

        //! @brief 通常の方式でスケール計算を行うことを表すフラグです。
        SCALE_STD           = ResSkeleton::SCALE_STD,

        //! @brief Maya 方式でスケール計算を行うことを表すフラグです。
        SCALE_MAYA          = ResSkeleton::SCALE_MAYA,

        //! @brief Softimage 方式でスケール計算を行うことを表すフラグです。
        SCALE_SOFTIMAGE     = ResSkeleton::SCALE_SOFTIMAGE,

        //! @briefprivate 行列のスケーリング方法のビットのマスクを表すフラグです。
        //!
        SCALE_MASK          = ResSkeleton::SCALE_MASK,

        //! @briefprivate 行列の回転方法のビットへのシフトを表すフラグです。
        //!
        ROT_SHIFT           = ResSkeleton::ROT_SHIFT,

        //! @brief Euler 角方式で回転計算を行うことを表すフラグです。
        ROT_EULER_XYZ       = ResSkeleton::ROT_EULER_XYZ,

        //! @brief Quaternion 方式で回転計算を行うことを表すフラグです。
        ROT_QUAT            = ResSkeleton::ROT_QUAT,

        //! @briefprivate 行列の回転方法のビットのマスクを表すフラグです。
        //!
        ROT_MASK            = ResSkeleton::ROT_MASK
    };

    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = LL_CACHE_FETCH_SIZE,

        //! @brief SetupBlockBuffer() で渡すバッファの必要アライメントサイズです。
        BLOCK_BUFFER_ALIGNMENT  = GX2_UNIFORM_BLOCK_ALIGNMENT
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    SkeletonObj() : m_pRes(NULL), m_pUserPtr(NULL), m_pBufferPtr(NULL), m_pBlockBuffer(NULL) {}

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock のサイズを計算します。
    size_t CalcBlockBufferSize() const;

    //! @brief CalcBlockBufferSize() の別名関数です。
    size_t CalculateBlockBufferSize() const
    {
        return CalcBlockBufferSize();
    }

    //! @brief UniformBlock を構築します。
    bool SetupBlockBuffer(void* pBuffer, size_t bufferSize);

    //! @brief UniformBlock を破棄します。
    void CleanupBlockBuffer();

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief ローカル行列をリソースに基づいて初期化します。
    //!
    //! Init() 内で呼ばれるため、意図的に結果を初期化したい場合以外は呼ぶ必要はありません。
    //!
    void ClearLocalMtx();

    //! @brief ローカル行列からワールド行列を計算します。
    void CalcWorldMtx(const Mtx34& baseMtx);

    //! @brief CalcWorldMtx() の別名関数です。
    void CalculateWorldMtx(const Mtx34& baseMtx)
    {
        CalcWorldMtx(baseMtx);
    }

    //! @brief ビルボード行列を計算します。
    void CalcBillboardMtx(Mtx34* pMtxArray, const Mtx34& cameraMtx, int boneIndex,
        bool combineWorld = false) const;

    //! @brief CalcBillboardMtx() の別名関数です。
    void CalculateBillboardMtx(Mtx34* pMtxArray, const Mtx34& cameraMtx, int boneIndex,
        bool combineWorld = false) const
    {
        CalcBillboardMtx(pMtxArray, cameraMtx, boneIndex, combineWorld);
    }

    //! @brief ワールド行列から行列パレットを計算します。
    //!
    //! GPU から参照されるバッファを書き換えるため、
    //! 前回の描画で GPU が参照し終わった後に呼ぶ必要があります。
    //! 構築時のオプションでダブルバッファ化するか、
    //! フレームバッファのコピーアウト中などに呼ぶことを想定しています。
    //!
    void CalcMtxBlock(int bufferIndex = 0);

    //! @brief CalcMtxBlock() の別名関数です。
    void CalculateSkeleton(int bufferIndex = 0)
    {
        CalcMtxBlock(bufferIndex);
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    ResSkeleton* GetResource() { return m_pRes; }

    //! @brief リソースを取得します。
    const ResSkeleton* GetResource() const { return m_pRes; }

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief SetupBlockBuffer() 時に渡されたバッファを取得します。
    void* GetBlockBufferPtr() { return m_pBlockBuffer; }

    //! @brief UniformBlock が構築済みであるかどうかを取得します。
    bool IsBlockBufferValid() const { return (m_Flag & BLOCK_BUFFER_VALID) != 0; }

    //! @brief 行列のスケーリング方法を取得します。
    int GetScaleMode() const { return m_Flag & SCALE_MASK; }

    //! @brief 行列の回転方法を取得します。
    int GetRotateMode() const { return m_Flag & ROT_MASK; }

    //! @brief ローカル行列の配列を取得します。
    LocalMtx* GetLocalMtxArray()
    {
        m_Flag |= CACHE_LOCAL_MTX;
        return LocalMtx::DownCast(m_pLocalMtxArray);
    }

    //! @brief ローカル行列の配列を取得します。
    const LocalMtx* GetLocalMtxArray() const
    {
        m_Flag |= CACHE_LOCAL_MTX;
        return LocalMtx::DownCast(m_pLocalMtxArray);
    }

    //! @brief ロックドキャッシュ上のローカル行列の配列を取得します。ロックドキャッシュ上に無い場合は NULL が返ります。
    const LocalMtx* GetLocalMtxArrayLC() const
    {
        if( m_pLocalMtxArray != m_pMemLocalMtxArray )
        {
            return static_cast<const LocalMtx*>( LocalMtx::DownCast( m_pLocalMtxArray ) );
        }
        return NULL;
    }

    //! @brief ワールド行列の配列を取得します。
    Mtx34* GetWorldMtxArray() {
        m_Flag |= CACHE_WORLD_MTX;
        return m_pWorldMtxArray;
    }

    //! @brief ワールド行列の配列を取得します。
    const Mtx34* GetWorldMtxArray() const
    {
        m_Flag |= CACHE_WORLD_MTX;
        return m_pWorldMtxArray;
    }

    //! @brief ロックドキャッシュ上のワールド行列の配列を取得します。ロックドキャッシュ上に無い場合は NULL が返ります。
    const Mtx34* GetWorldMtxArrayLC() const
    {
        if( m_pWorldMtxArray != m_pMemWorldMtxArray )
        {
            return static_cast<const Mtx34*>( m_pWorldMtxArray );
        }
        return NULL;
    }

    //! @brief 行列パレットを取得します。
    GfxBuffer& GetMtxBlock() { return m_MtxBlock; }

    //! @brief 行列パレットを取得します。
    const GfxBuffer& GetMtxBlock() const { return m_MtxBlock; }

    //! @brief ロックドキャッシュ上の行列パレットを取得します。ロックドキャッシュ上に無い場合は NULL が返ります。
    const Mtx34* GetMtxBlockLC() const { return static_cast<Mtx34*>( m_pLCMtxBlock ); }

    //! @brief UniformBlock をバッファリングする数を設定します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief UniformBlock のエンディアンスワップを行うように設定します。
    void EnableBlockSwap() { m_Flag |= BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行わないように設定します。
    void DisableBlockSwap() { m_Flag &= ~BLOCK_BUFFER_SWAP; }

    //! @brief UniformBlock のエンディアンスワップを行うかどうかを取得します。
    bool IsBlockSwapEnabled() const { return (m_Flag & BLOCK_BUFFER_SWAP) != 0; }

    //! @brief ユーザポインタを設定します。
    void SetUserPtr(void* pUserPtr) { m_pUserPtr = pUserPtr; }

    //! @brief ユーザポインタを取得します。
    void* GetUserPtr() { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    const void* GetUserPtr() const { return m_pUserPtr; }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    T* GetUserPtr() { return static_cast<T*>(m_pUserPtr); }

    //! @brief ユーザポインタを取得します。
    template <typename T>
    const T* GetUserPtr() const { return static_cast<const T*>(m_pUserPtr); }

    //@}

    //----------------------------------------
    //! @name ボーン
    //@{

    //! @brief ボーン数を取得します。
    int GetBoneCount() const { return m_NumBone; }

    //! @brief インデクス引きでボーンを取得します。
    //!
    //! インデクスはモデルが持つボーン数の範囲内とする必要があります。
    //!
    ResBone* GetBone(int boneIndex)
    {
        return ResBone::ResCast(&m_pBoneArray[boneIndex]);
    }

    //! @brief インデクス引きでボーンを取得します。
    //!
    //! インデクスはモデルが持つボーン数の範囲内とする必要があります。
    //!
    const ResBone* GetBone(int boneIndex) const
    {
        return ResBone::ResCast(&m_pBoneArray[boneIndex]);
    }

    //! @brief 名前引きでボーンを取得します。
    //!
    //! 指定した名前のボーンが存在しない場合は NULL を返します。
    //!
    ResBone* GetBone(const char* name) { return m_pRes->GetBone(name); }

    //! @brief 名前引きでボーンを取得します。
    //!
    //! 指定した名前のボーンが存在しない場合は NULL を返します。
    //!
    const ResBone* GetBone(const char* name) const { return m_pRes->GetBone(name); }

    //! @brief インデクスからボーンの名前を取得します。
    //!
    //! インデクスはスケルトンが持つボーン数の範囲内とする必要があります。
    //!
    const char* GetBoneName(int boneIndex) const { return m_pRes->GetBoneName(boneIndex); }

    //! @brief ボーンの名前からインデクスを取得します。
    //!
    //! 指定した名前のボーンが存在しない場合は -1 を返します。
    //!
    int GetBoneIndex(const char* name) const { return m_pRes->GetBoneIndex(name); }

    //@}

    //----------------------------------------
    //! @name コールバック
    //@{

    //! @brief ボーンのワールド行列の計算後に呼び出されるコードバックを設定します。
    void SetCalcWorldCallback(ICalcWorldCallback* pCallback) { m_pCallback = pCallback; }

    //! @brief SetCalcWorldCallback() の別名関数です。
    void SetCalculateWorldCallback(ICalculateWorldCallback* pCallback)
    {
        SetCalcWorldCallback(pCallback);
    }

    //! @brief ボーンのワールド行列の計算後に呼び出されるコールバックを取得します。
    ICalcWorldCallback* GetCalcWorldCallback() { return m_pCallback; }

    //! @brief GetCalcWorldCallback() の別名関数です。
    ICalculateWorldCallback* GetCalculateWorldCallback()
    {
        return GetCalcWorldCallback();
    }

    //! @brief ボーンのワールド行列の計算後に呼び出されるコールバックを取得します。
    const ICalcWorldCallback* GetCalcWorldCallback() const { return m_pCallback; }

    //! @brief GetCalcWorldCallback() の別名関数です。
    const ICalculateWorldCallback* GetCalculateWorldCallback() const
    {
        return GetCalcWorldCallback();
    }

    //! @brief コールバック呼び出しを行うボーンのインデクスを設定します。
    //!
    //! コールバック内で ICalcWorldCallback::SetCallbackBoneIndex() によって
    //! 対象のボーンインデクスを変更することで、複数のボーンに対してコールバックを
    //! 実行することができます。
    //!
    void SetCallbackBoneIndex(int boneIndex)
    {
        m_CallbackBone = (boneIndex >= 0 && boneIndex < m_NumBone) ?
            static_cast<u16>(boneIndex) : ICalcWorldCallback::INVALID_BONE;
    }

    //! @brief コールバック呼び出しを行うボーンのインデクスを取得します。
    int GetCallbackBoneIndex() const
    {
        return m_CallbackBone >= ICalcWorldCallback::INVALID_BONE ? -1 : m_CallbackBone;
    }

    //@}

    //----------------------------------------
    //! @name ロックドキャッシュ
    //@{

    //! @brief ローカル行列を割り当てるのに必要な LockedCache のサイズを計算します。
    size_t CalcLocalMtxLCSize() const
    {
        return Align(sizeof(LocalMtx) * m_NumBone, LL_CACHE_FETCH_SIZE);
    }

    //! @brief ローカル行列を LockedCache に割り当てます。
    //!
    //! load に true を指定した場合はメモリの内容を LockedCache に読み込みます。
    //! メモリの内容を読み込む際はキャッシュがフラッシュされている必要があります。
    //!
    size_t LCMountLocalMtx(void* pLC, size_t size, bool load);

    //! @brief LockedCache に割り当てたローカル行列を元に戻します。
    //!
    //! store に true を指定した場合は LockedCache の内容をメモリに書き戻します。
    //! メモリに書き戻す際はキャッシュがフラッシュされている必要があります。
    //!
    void LCUnmountLocalMtx(bool store);

    //! @briefprivate
    bool IsLocalMtxOnCache() const { return !!(m_Flag & MASK_LOCAL_MTX); }

    //! @brief ワールド行列を割り当てるのに必要な LockedCache のサイズを計算します。
    size_t CalcWorldMtxLCSize() const
    {
        return Align(sizeof(Mtx34) * m_NumBone, LL_CACHE_FETCH_SIZE);
    }

    //! @brief ワールド行列を LockedCache に割り当てます。
    //!
    //! load に true を指定した場合はメモリの内容を LockedCache に読み込みます。
    //! メモリの内容を読み込む際はキャッシュがフラッシュされている必要があります。
    //!
    size_t LCMountWorldMtx(void* pLC, size_t size, bool load);

    //! @brief LockedCache に割り当てたワールド行列を元に戻します。
    //!
    //! store に true を指定した場合は LockedCache の内容をメモリに書き戻します。
    //! メモリに書き戻す際はキャッシュがフラッシュされている必要があります。
    //!
    void LCUnmountWorldMtx(bool store);

    //! @briefprivate
    bool IsWorldMtxOnCache() const { return !!(m_Flag & MASK_WORLD_MTX); }

    //! @brief 行列パレットを割り当てるのに必要な LockedCache のサイズを計算します。
    size_t CalcMtxBlockLCSize() const
    {
        return Align(sizeof(Mtx34) * m_pRes->GetMtxCount(), LL_CACHE_FETCH_SIZE);
    }

    //! @brief 行列パレットを LockedCache に割り当てます。
    //!
    //! load に true を指定した場合はメモリの内容を LockedCache に読み込みます。
    //! メモリの内容を読み込む際はキャッシュがフラッシュされている必要があります。
    //!
    size_t LCMountMtxBlock(void* pLC, size_t size, bool load, int bufferIndex = 0);

    //! @brief LockedCache に割り当てた行列パレットを元に戻します。
    //!
    //! store に true を指定した場合は LockedCache の内容をメモリに書き戻します。
    //! メモリに書き戻す際はキャッシュがフラッシュされている必要があります。
    //!
    void LCUnmountMtxBlock(bool store, int bufferIndex = 0);

    //@}

protected:
    class Impl;

    //! @briefprivate ワールド行列計算を行う内部関数です。
    //!
    template<typename CalcType, bool useCallback>
    void CalcWorldImpl(const Mtx34& baseMtx);

private:
    class Sizer;

    ResSkeleton* m_pRes;

    mutable bit16 m_Flag;
    u8 m_NumBuffering;
    u8 reserved;

    ResBoneData* m_pBoneArray;
    LocalMtxData* m_pLocalMtxArray;
    Mtx34* m_pWorldMtxArray;
    Vec3* m_pScaleArray; // Softimage の Hierarchical Scaling で使用

    GfxBuffer m_MtxBlock; // 頂点と法線の行列用ブロック

    u16 m_NumBone;
    u16 m_CallbackBone;
    ICalcWorldCallback* m_pCallback;

    void* m_pMemLocalMtxArray;
    void* m_pMemWorldMtxArray;
    void* m_pLCMtxBlock;
    void* m_pUserPtr;
    void* m_pBufferPtr;
    void* m_pBlockBuffer;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(SkeletonObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate SkeletonObj のサイズを計算するためのクラスです。
//!
class SkeletonObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief SkeletonObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        WORLD_MTX,
        LOCAL_MTX,
        SCALE,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletonObj::Init() に渡して初期化を行うパラメータです。
class SkeletonObj::InitArg
{
public:
    //! @brief コンストラクタです。
    explicit InitArg(ResSkeleton* resource)
        : m_pRes(resource)
        , m_NumBuffering(1)
    {
        NW_G3D_ASSERT_NOT_NULL(resource);
    }

    //! @brief UniformBlock をバッファリングする数を設定します。
    void BufferingCount(int count) { m_NumBuffering = count; m_Sizer.Invalidate(); }

    //! @brief UniformBlock をバッファリングする数を取得します。
    int GetBufferingCount() const { return m_NumBuffering; }

    //! @brief リソースを取得します。
    ResSkeleton* GetResource() const { return m_pRes; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    ResSkeleton* m_pRes;
    int m_NumBuffering;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletonObj の構築を行うクラスです。
class SkeletonObj::Builder : public SkeletonObj::InitArg
{
public:
    //! @brief コンストラクタです。
    Builder(ResSkeleton* resource) : InitArg(resource), m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief SkeletonObj を構築します。
    //!
    //! @param[in] pSkeletonObj SkeletonObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(SkeletonObj* pSkeletonObj, void* pBuffer, size_t bufferSize) const
    {
        return pSkeletonObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief SkeletonObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief SkeletonObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  SkeletonObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief SkeletonObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

}} // namespace nw::g3d

#endif // NW_G3D_SKELETON_H_
