﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SKELETALANIMOBJ_H_
#define NW_G3D_SKELETALANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResModel.h>
#include <nw/g3d/res/g3d_ResSkeletalAnim.h>

namespace nw { namespace g3d {

class SkeletonObj;
class ModelObj;

//! @brief スケルタルアニメーションインスタンスです。
class SkeletalAnimObj : public ModelAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = LL_CACHE_FETCH_SIZE,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    SkeletalAnimObj() : ModelAnimObj(), m_pRes(NULL), m_pRetargetingPreCalcArray(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    virtual BindResult Bind(const ResModel* pModel);
    virtual BindResult Bind(const ModelObj* pModelObj);
    virtual void BindFast(const ResModel* pModel);

    //! @brief アニメーション対象に関連付けます。
    BindResult Bind(const ResSkeleton* pSkeleton);

    //! @brief アニメーション対象に関連付けます。
    BindResult Bind(const SkeletonObj* pSkeletonObj);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    BindResult Bind(const ResModel* pModel, const ResModel* pHostModel);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    BindResult Bind(const ModelObj* pModelObj, const ModelObj* pHostModelObj);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    BindResult Bind(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeleton);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    BindResult Bind(const SkeletonObj* pSkeletonObj, const SkeletonObj* pHostSkeletonObj);

    //! @brief アニメーション対象に関連付けます。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    void BindFast(const ResSkeleton* pSkeleton);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult BindFast(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeletonObj);

    //! @brief アニメーション対象に関連付け、リターゲッティング情報を生成します。
    //!
    //! リソース同士を PreBind によって事前に関連付けることにより、
    //! BindFast 時には名前引きを行わない比較的高速な関連付けを行います。
    //!
    BindResult BindFast(const ResModel* pModel, const ResModel* pHostModel);

    //! @brief 指定したインデックスのボーンにフラグを設定します。
    void SetBindFlag(int boneIndex, BindFlag flag)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        SetBindFlagImpl(boneIndex, flag);
    }

    //! @brief スケルトン構造を辿って指定したインデックスのボーン以下のボーンにフラグを設定します。
    void SetBindFlag(const ResSkeleton* pSkeleton, int boneIndex, BindFlag flag);

    //! @brief 指定したインデックスのボーンのフラグを取得します。
    BindFlag GetBindFlag(int boneIndex)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return GetBindFlagImpl(boneIndex);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult();
    virtual void Calc();
    virtual void ApplyTo(ModelObj* pModelObj) const;

    //! @brief ResSkeleton を用いて初期化を行います。
    void ClearResult(const ResSkeleton* pSkeleton);

    //! @brief アニメーション結果を対象に適用します。
    void ApplyTo(SkeletonObj* pSkeletonObj) const;

    //! @brief Calc()の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResSkeletalAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するスケルタルアニメーションを差し替えます。
    //!
    void SetResource(ResSkeletalAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource()
    {
        m_pRes = NULL;
        SetTargetUnbound();
    }

    //! @brief アニメーションの計算結果を取得します。
    BoneAnimResult* GetResultArray()
    {
        m_Flag |= CACHE_RESULT;
        return static_cast<BoneAnimResult*>(GetResultBuffer());
    }

    //! @brief アニメーションの計算結果を取得します。
    const BoneAnimResult* GetResultArray() const
    {
        m_Flag |= CACHE_RESULT;
        return static_cast<const BoneAnimResult*>(GetResultBuffer());
    }

    //! @brief ロックドキャッシュ上のアニメーションの計算結果を取得します。ロックドキャッシュ上に無い場合は NULL が返ります。
    const BoneAnimResult* GetResultArrayLC() const
    {
        const BoneAnimResult* pMtx = static_cast<const BoneAnimResult*>( GetResultBuffer() );
        if( pMtx != m_pMemResultBuffer )
        {
            return pMtx;
        }
        return NULL;
    }

    //! @brief アニメーション対象に対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResSkeleton* pSkeleton) const
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return pSkeleton->GetBoneCount() <= GetBindTable().GetSize();
    }

    //! @brief アニメーションリソースに対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResSkeletalAnim* pRes) const
    {
        NW_G3D_ASSERT_NOT_NULL(pRes);
        return pRes->GetBoneAnimCount() <= m_MaxBoneAnim;
    }

    //@}

    //----------------------------------------
    //! @name ロックドキャッシュ
    //@{

    //! @brief 割り当てるのに必要な LockedCache のサイズを計算します。
    size_t CalcLCSize() const
    {
        return Align(sizeof(BoneAnimResult) * GetBindTable().GetAnimCount(), LL_CACHE_FETCH_SIZE);
    }

    //! @brief 結果のバッファを LockedCache に割り当てます。
    //!
    //! load に true を指定した場合はメモリの内容を LockedCache に読み込みます。
    //! メモリの内容を読み込む際はキャッシュがフラッシュされている必要があります。
    //!
    size_t LCMount(void* pLC, size_t size, bool load);

    //! @brief LockedCache に割り当てた結果のバッファを元に戻します。
    //!
    //! store に true を指定した場合は LockedCache の内容をメモリに書き戻します。
    //! メモリに書き戻す際はキャッシュがフラッシュされている必要があります。
    //!
    void LCUnmount(bool store);

    //! @briefprivate
    bool IsResultOnCache() const { return !!(m_Flag & MASK_RESULT); }

    //@}

protected:
    //! @briefprivate アニメーション計算の状態を表すフラグです。
    //!
    enum Flag
    {
        CACHE_RESULT        = 0x1 << 3,
        INVALID_RESULT      = 0x1 << 4,
        MASK_RESULT         = CACHE_RESULT | INVALID_RESULT,
        RETARGETING_ENABLED = 0x1 << 5
    };

    //! @briefprivate ボーンアニメーションを取得します。
    //!
    ResBoneAnim* GetBoneAnim(int animIndex)
    {
        return ResBoneAnim::ResCast(&m_pBoneAnimArray[animIndex]);
    }

    //! @briefprivate ボーンアニメーションを取得します。
    //!
    const ResBoneAnim* GetBoneAnim(int animIndex) const
    {
        return ResBoneAnim::ResCast(&m_pBoneAnimArray[animIndex]);
    }

    //! @briefprivate アニメーションの関連付けを行う内部関数です。
    //!
    BindResult BindImpl(const ResSkeleton* pSkeleton);

    //! @briefprivate アニメーションカーブの評価を行う内部関数です。
    //!
    template <typename ConvRot, bool retargeting>
    void CalcImpl();

    //! @briefprivate アニメーション結果を対象に適用する内部関数です。
    //!
    template <typename ConvRot>
    void ApplyToImpl(SkeletonObj* pSkeletonObj) const;

    //! @briefprivate リターゲッティングの事前計算を行う内部用の初期化関数です。
    //!
    BindResult InitRetargeting(const ResSkeleton* pTargetSkeleton, const ResSkeleton* pHostSkeleton);

private:
    class Sizer;
    class Impl;
    ResSkeletalAnim* m_pRes;
    ResBoneAnimData* m_pBoneAnimArray;
    s32 m_MaxBoneAnim;
    mutable bit32 m_Flag; // const であってもキャッシュの状態を管理するため mutable にする。
    void* m_pMemResultBuffer; // LCUnmount() で復帰するためにポインタを記憶しておく。
    Quat* m_pRetargetingPreCalcArray; // リターゲッティングの事前計算バッファ。

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(SkeletalAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate SkeletalAnimObj のサイズを計算するためのクラスです。
//!
class SkeletalAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief SkeletalAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        BIND_TABLE,
        FRAMECACHE_ARRAY,
        RETARGETING_PRECALC_BUFFER,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletalAnimObj::Init() に渡して初期化を行うパラメータです。
class SkeletalAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResSkeletalAnim とすべての ResSkeleton に対して
    //! Reserve() するか、SetMaxBoneCount()、SetMaxBoneAnimCount()、SetMaxCurveCount() で
    //! バッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumBone = m_NumBoneAnim = m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
        m_RetargetingEnabled = false;
    }

    //! @brief 対象となる ResSkeleton を使用するのに必要なパラメータを設定します。
    void Reserve(const ResSkeleton* pResSkeleton)
    {
        NW_G3D_ASSERT_NOT_NULL(pResSkeleton);
        m_NumBone = std::max(m_NumBone, pResSkeleton->GetBoneCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        Reserve(pResModel->GetSkeleton());
    }

    //! @brief 対象となる ResSkeletalAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResSkeletalAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumBoneAnim = std::max(m_NumBoneAnim, pResAnim->GetBoneAnimCount());
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のボーン数を指定します。
    void SetMaxBoneCount(int boneCount)
    {
        NW_G3D_ASSERT(boneCount >= 0);
        m_NumBone = boneCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のボーン数を取得します。
    int GetMaxBoneCount() const
    {
        NW_G3D_ASSERT(m_NumBone >= 0);
        return m_NumBone;
    }

    //! @brief 使用できる最大のボーンアニメーション数を指定します。
    void SetMaxBoneAnimCount(int boneAnimCount)
    {
        NW_G3D_ASSERT(boneAnimCount >= 0);
        m_NumBoneAnim = boneAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のボーンアニメーション数を取得します。
    int GetMaxBoneAnimCount() const { return m_NumBoneAnim; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief アニメーションのリターゲッティングを有効にします。
    void EnableRetargeting() { m_RetargetingEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーションのリターゲッティングを無効にします。
    void DisableRetargeting() { m_RetargetingEnabled = false; m_Sizer.Invalidate(); }

    //! @brief EnableRetargeting() の別名関数です。
    void SetRetargetingEnabled()
    {
        EnableRetargeting();
    }

    //! @brief DisableRetargeting() の別名関数です。
    void SetRetargetingDisabled()
    {
        DisableRetargeting();
    }

    //! @brief アニメーションのリターゲッティングが有効かどうかを取得します。
    bool IsRetargetingEnabled() const { return m_RetargetingEnabled; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const { return m_NumBone >= 0 && m_NumBoneAnim >= 0 && m_NumCurve >= 0; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }


private:
    int m_NumBone;
    int m_NumBoneAnim;
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    bool m_RetargetingEnabled;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletalAnimObj の構築を行うクラスです。
class SkeletalAnimObj::Builder : public SkeletalAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべてのスケルタルアニメリソースとすべてのスケルトンに対して
    //! Reserve() するか、SetMaxBoneCount()、SetMaxBoneAnimCount()、SetMaxCurveCount() で
    //! バッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief SkeletalAnimObj を構築します。
    //!
    //! @param[in] pSkeletalAnimObj SkeletalAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(SkeletalAnimObj* pSkeletalAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pSkeletalAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief SkeletalAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief SkeletalAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  SkeletalAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief SkeletalAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

//! @brief アニメーションブレンディング時に呼び出されるコールバックの基底クラスです。
class ICalcBlendWeightCallback
{
public:
    enum { INVALID_BONE = 0xFFFF };

    //! @brief デストラクタです。
    virtual ~ICalcBlendWeightCallback() {}

    //! @brief Exec() 用いる設定です。
    class CallbackArg
    {
    public:
        //! @brief  コンストラクタです。
        CallbackArg(const int& boneIndex, float weight)
            : m_BoneIndex(boneIndex), m_BlendWeight(weight)
        {
        }

        //! @brief 計算に用いるボーンインデクスを取得します。
        int GetBoneIndex() const
        {
            return m_BoneIndex;
        }

        //! @brief ブレンドウェイトをセットします。
        void SetBlendWeight(float weight)
        {
            m_BlendWeight = weight;
        }

        //! @brief ブレンドウェイトを取得します。
        float GetBlendWeight() const
        {
            return m_BlendWeight;
        }

    private:
        int m_BoneIndex;
        float m_BlendWeight;

        NW_G3D_DISALLOW_COPY_AND_ASSIGN(CallbackArg);
    };

    //! @brief コールバックを呼び出します。
    virtual void Exec(CallbackArg& arg) = 0;
};

typedef ICalcBlendWeightCallback ICalculateBlendWeightCallback;

//--------------------------------------------------------------------------------------------------

//! @brief スケルタルアニメーションのブレンドを行うクラスです。
class SkeletalAnimBlender
{
public:
    //! @brief ブレンド方式です。
    enum BlendMode
    {
        //! @brief 線形補間です。
        BLEND_INTERPOLATE,

        //! @brief 加算合成です。
        BLEND_ADD
    };

    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = LL_CACHE_FETCH_SIZE,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    SkeletalAnimBlender() : m_pBufferPtr(NULL), m_pCallback(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    //! @brief ブレンドの前処理を行います。
    void ClearResult();

    //! @brief SkeletalAnimObj::Calc() 後にブレンド計算を行います。
    //!
    //! 既に計算済みの場合は SkeletalAnimObj::Calc() はスキップされます。
    //! SkeletalAnimObj に SKIP_APPLY フラグが指定されているボーンはブレンドされません。
    //!
    void Blend(SkeletalAnimObj* pAnimObj, float weight);

    //! @brief ブレンド結果を対象に適用します。
    //!
    //! 累積ウェイトが 0 ボーンは適用処理を行いません。
    //! 累積ウェイトが 1 でないボーンは正規化してから適用します。
    //!
    void ApplyTo(SkeletonObj* pSkeletonObj) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief Init() 時に渡されたバッファを取得します。
    void* GetBufferPtr() { return m_pBufferPtr; }

    //! @brief 書き込み先のボーン数を設定します。
    //!
    //! GetMaxBoneCount() で取得できる最大ボーン数以下である必要があります。
    //!
    void SetBoneCount(int count)
    {
        NW_G3D_ASSERT(count <= m_MaxBone);
        m_NumBone = static_cast<u16>(count);
    }

    //! @brief 書き込み先のボーン数を取得します。
    int GetBoneCount() const { return m_NumBone; }

    //! @brief ブレンド計算を行う最大ボーン数を取得します。
    int GetMaxBoneCount() const { return m_MaxBone; }

    //! @brief ブレンド方式を設定します。
    void SetBlendMode(BlendMode mode) { m_Flag = (m_Flag & ~BLEND_MASK) | mode; }

    //! @brief ブレンド方式を取得します。
    BlendMode GetBlendMode() const { return BlendMode(m_Flag & BLEND_MASK); }

    //! @brief 球面線形補間を有効にします。
    void EnableSlerp() { m_Flag |= USE_SLERP; }

    //! @brief 球面線形補間を無効にします。
    void DisableSlerp() { m_Flag &= ~USE_SLERP; }

    //! @brief 球面線形補間が有効であるかどうかを取得します。
    bool IsSlerpEnabled() const { return 0 != (m_Flag & USE_SLERP); }

    //! @brief ブレンド結果を取得します。
    BoneAnimBlendResult* GetResultArray()
    {
        m_Flag |= CACHE_RESULT;
        return static_cast<BoneAnimBlendResult*>(m_pResultBuffer);
    }

    //! @brief ブレンド結果を取得します。
    const BoneAnimBlendResult* GetResultArray() const
    {
        m_Flag |= CACHE_RESULT;
        return static_cast<const BoneAnimBlendResult*>(m_pResultBuffer);
    }

    //! @brief ブレンド結果を取得します。
    BoneAnimBlendResult* GetResultArrayMutable() const
    {
        m_Flag |= CACHE_RESULT;
        return static_cast<BoneAnimBlendResult*>(m_pResultBuffer);
    }

    //! @brief ロックドキャッシュ上のブレンド結果を取得します。ロックドキャッシュ上に無い場合は NULL が返ります。
    const BoneAnimBlendResult* GetLocalMtxArrayLC() const
    {
        if( m_pResultBuffer != m_pMemResultBuffer )
        {
            return static_cast<const BoneAnimBlendResult*>( m_pResultBuffer );
        }
        return NULL;
    }

    //! @brief ブレンド時に呼び出されるコールバックを設定します。
    void SetCalcBlendWeightCallback(ICalcBlendWeightCallback* pCallback)
    {
        m_pCallback = pCallback;
    }

    //! @brief SetCalcBlendWeightCallback() の別名関数です。
    void SetCalculateBlendWeightCallback(ICalculateBlendWeightCallback* pCallback)
    {
        SetCalcBlendWeightCallback(pCallback);
    }

    //! @brief ブレンド時に呼び出されるコールバックを取得します。
    ICalcBlendWeightCallback* GetCalcBlendWeightCallback()
    {
        return m_pCallback;
    }

    //! @brief GetCalcBlendWeightCallback() の別名関数です。
    ICalculateBlendWeightCallback* GetCalculateBlendWeightCallback()
    {
        return GetCalcBlendWeightCallback();
    }

    //! @brief ブレンド時に呼び出されるコールバックを取得します。
    const ICalcBlendWeightCallback* GetCalcBlendWeightCallback() const
    {
        return m_pCallback;
    }

    //! @brief GetCalcBlendWeightCallback() の別名関数です。
    const ICalculateBlendWeightCallback* GetCalculateBlendWeightCallback() const
    {
        return GetCalcBlendWeightCallback();
    }

    //@}

    //----------------------------------------
    //! @name ロックドキャッシュ
    //@{

    //! @brief 割り当てるのに必要な LockedCache のサイズを計算します。
    size_t CalcLCSize() const
    {
        return Align(sizeof(BoneAnimBlendResult) * m_NumBone, LL_CACHE_FETCH_SIZE);
    }

    //! @brief 結果のバッファを LockedCache に割り当てます。
    //!
    //! load に true を指定した場合はメモリの内容を LockedCache に読み込みます。
    //! メモリの内容を読み込む際はキャッシュがフラッシュされている必要があります。
    //!
    size_t LCMount(void* pLC, size_t size, bool load);

    //! @brief LockedCache に割り当てた結果のバッファを元に戻します。
    //!
    //! store に true を指定した場合は LockedCache の内容をメモリに書き戻します。
    //! メモリに書き戻す際はキャッシュがフラッシュされている必要があります。
    //!
    void LCUnmount(bool store);

    //! @briefprivate
    bool IsResultOnCache() const { return !!(m_Flag & MASK_RESULT); }

    //@}

protected:
    class Impl;

    //! @briefprivate ブレンド計算の状態を表すフラグです。
    //!
    //!
    enum Flag
    {
        BLEND_MASK      = 0x1,
        USE_SLERP       = 0x1 << 1,
        NORMALIZED      = 0x1 << 2,
        CACHE_RESULT    = 0x1 << 3,
        INVALID_RESULT  = 0x1 << 4,
        MASK_RESULT     = CACHE_RESULT | INVALID_RESULT,
    };

    //! @briefprivate ブレンドを行う内部関数です。
    //!
    //!
    template <typename ConvRot, bool useCallback>
    void BlendImpl(SkeletalAnimObj* pAnimObj, float weight);

    //! @briefprivate ブレンド結果を適用する内部関数です。
    //!
    //!
    template <typename ConvRot, BlendMode mode>
    void ApplyToImpl(SkeletonObj* pSkeletonObj) const;

private:
    class Sizer;
    void* m_pResultBuffer;
    u16 m_MaxBone; //!< 初期化時に指定されてバッファサイズの計算に使用されたボーン数
    u16 m_NumBone; //!< 書き込み対象のボーンインデックスの最大値+1
    mutable bit32 m_Flag; // ブレンド結果の正規化をキャッシュするため mutable にする。
    void* m_pMemResultBuffer;
    void* m_pBufferPtr;
    ICalcBlendWeightCallback* m_pCallback;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(SkeletalAnimBlender);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate SkeletalAnimBlender のサイズを計算するためのクラスです。
//!
class SkeletalAnimBlender::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief SkeletalAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletalAnimBlender::Init() に渡して初期化を行うパラメータです。
class SkeletalAnimBlender::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResSkeleton に対してReserve() するか、
    //! SetMaxBoneCount()、でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear() { m_NumBone = -1; }

    //! @brief 対象となる ResSkeleton を使用するのに必要なパラメータを設定します。
    void Reserve(const ResSkeleton* pResSkeleton)
    {
        NW_G3D_ASSERT_NOT_NULL(pResSkeleton);
        m_NumBone = std::max(m_NumBone, pResSkeleton->GetBoneCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        Reserve(pResModel->GetSkeleton());
    }

    //! @brief 使用できる最大のボーン数を指定します。
    void SetMaxBoneCount(int boneCount)
    {
        NW_G3D_ASSERT(boneCount >= 0);
        m_NumBone = boneCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のボーン数を取得します。
    int GetMaxBoneCount() const { return m_NumBone; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const { return m_NumBone >= 0; }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumBone;
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief SkeletalAnimBlender の構築を行うクラスです。
class SkeletalAnimBlender::Builder : public SkeletalAnimBlender::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResSkeleton に対してReserve() するか、
    //! SetMaxBoneCount()、でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief SkeletalAnimBlender を構築します。
    //!
    //! @param[in] pSkeletalAnimBlender SkeletalAnimBlender へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(SkeletalAnimBlender* pSkeletalAnimBlender, void* pBuffer, size_t bufferSize) const
    {
        return pSkeletalAnimBlender->Init(*this, pBuffer, bufferSize);
    }

    //! @brief SkeletalAnimBlender 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief SkeletalAnimBlender 構築に必要なメモリサイズを取得します。
    //!
    //! @return  SkeletalAnimBlender 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief SkeletalAnimBlender 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};

//--------------------------------------------------------------------------------------------------

// 回転成分計算用ユーティリティ

//! @brief Euler 角表現から軸回転へ変換するクラスです。
class EulerToAxes
{
public:
    //! @brief Euler 角表現から変換します。
    static void Convert(Vec3* pAxes, const BoneAnimResult* pResult);
};

//! @brief Quaternion 表現から軸回転へ変換するクラスです。
class QuatToAxes
{
public:
    //! @brief Quaternion から変換します。
    static void Convert(Vec3* pAxes, const BoneAnimResult* pResult);
};

//! @brief 軸回転から回転行列へ変換するクラスです。
class AxesToMtx
{
public:

    //! @brief BoneAnimBlendResult から変換します。
    static void Convert(Mtx34* pMtx, const BoneAnimBlendResult* pResult);
};

//! @brief Euler 角表現から回転行列へ変換するクラスです。
class EulerToMtx
{
public:
    //! @brief BoneAnimResult から変換します。
    static void Convert(Mtx34* pMtx, const BoneAnimResult* pResult)
    {
        pMtx->SetR(pResult->euler);
    }
};

//! @brief Quaternion から回転行列へ変換するクラスです。
class QuatToMtx
{
public:

    //! @brief BoneAnimResult から変換します。
    static void Convert(Mtx34* pMtx, const BoneAnimResult* pResult)
    {
        pMtx->SetR(pResult->quat);
    }

    //! @brief BoneAnimBlendResult から変換します。
    static void Convert(Mtx34* pMtx, const BoneAnimBlendResult* pResult)
    {
        pMtx->SetR(pResult->rotate);
    }
};

//--------------------------------------------------------------------------------------------------

namespace detail
{

// TODO: 閾値を検討
NW_G3D_INLINE
bool LengthSqEqualsZero(float length)
{
    return length == 0.0f;
}

}

NW_G3D_INLINE
void EulerToAxes::Convert(Vec3* pAxes, const BoneAnimResult* pResult)
{
    const Vec3& euler = pResult->euler;
    Vec3& axisX = pAxes[0];
    Vec3& axisY = pAxes[1];

#if defined( __ghs__ )
    f32x2 sincosx = Math::SinCos(euler.x);
    f32x2 sincosy = Math::SinCos(euler.y);
    f32x2 sincosz = Math::SinCos(euler.z);

    f32x2 cossinx = __PS_MERGE10(sincosx, sincosx);

    f32x2 sxsz_cxcz = __PS_MUL(sincosx, sincosz);
    f32x2 cxsz_sxcz = __PS_MUL(cossinx, sincosz);
    f32x2 cxcz_sxsz = __PS_MERGE10(sxsz_cxcz, sxsz_cxcz);
    f32x2 sxcz_cxsz = __PS_MERGE10(cxsz_sxcz, cxsz_sxcz);

    f32x2 sy_sy = __PS_MERGE00(sincosy, sincosy);
    f32x2 cysz_cycz = __PS_MULS1(sincosz, sincosy);

    f32x2 xz_yy = __PS_MADD(cxcz_sxsz, sy_sy, sxsz_cxcz);
    f32x2 xy_yz = __PS_MSUB(sxcz_cxsz, sy_sy, cxsz_sxcz);

    axisX.x = cysz_cycz[1];
    axisY.x = cysz_cycz[0];

    reinterpret_cast<f32x2&>(axisX.a[1]) = __PS_MERGE00(xy_yz, xz_yy);
    reinterpret_cast<f32x2&>(axisY.a[1]) = __PS_MERGE11(xz_yy, xy_yz);
#else
    float sx, sy, sz, cx, cy, cz;
    Math::SinCos(&sx, &cx, euler.x);
    Math::SinCos(&sy, &cy, euler.y);
    Math::SinCos(&sz, &cz, euler.z);

    float cxcz = cx * cz;
    float sxsy = sx * sy;
    float cxsz = cx * sz;

    axisX.x = cy * cz;
    axisY.x = cy * sz;

    axisX.y = sxsy * cz - cxsz;
    axisY.y = sxsy * sz + cxcz;

    axisX.z = cxcz * sy + sx * sz;
    axisY.z = cxsz * sy - sx * cz;
#endif
}

NW_G3D_INLINE
void QuatToAxes::Convert(Vec3* pAxes, const BoneAnimResult* pResult)
{
    const Quat& quat = pResult->quat;
    Vec3& axisX = pAxes[0];
    Vec3& axisY = pAxes[1];

    float yy2 = 2 * quat.y * quat.y;
    float zz2 = 2 * quat.z * quat.z;
    float xx2 = 2 * quat.x * quat.x;
    float xy2 = 2 * quat.x * quat.y;
    float xz2 = 2 * quat.x * quat.z;
    float yz2 = 2 * quat.y * quat.z;
    float wz2 = 2 * quat.w * quat.z;
    float wx2 = 2 * quat.w * quat.x;
    float wy2 = 2 * quat.w * quat.y;

    axisX.x = 1.0f - yy2 - zz2;
    axisX.y = xy2 - wz2;
    axisX.z = xz2 + wy2;

    axisY.x = xy2 + wz2;
    axisY.y = 1.0f - xx2 - zz2;
    axisY.z = yz2 - wx2;
}

NW_G3D_INLINE
void AxesToMtx::Convert(Mtx34* pMtx, const BoneAnimBlendResult* pResult)
{
    const Vec3& axisX = pResult->axes[0];
    const Vec3& axisY = pResult->axes[1];
    Vec3 axisZ;

    axisZ.Cross(axisX, axisY);
    float lengthSqY = Vec3::LengthSq(axisY);
    float lengthSqZ = Vec3::LengthSq(axisZ);
    if (nw::g3d::detail::LengthSqEqualsZero(lengthSqY) || nw::g3d::detail::LengthSqEqualsZero(lengthSqZ))
    {
        // 縮退していた場合は何も書き込まない。
        NW_G3D_WARNING(false, "Failed to normalize matrix.\n");
        return;
    }

#if defined( __ghs__ )
    f32x2 lengthSqYZ = { lengthSqY, lengthSqZ };
    f32x2 rsqrtYZ = Math::RSqrt(lengthSqYZ);
    float rsqrtY = rsqrtYZ[0], rsqrtZ = rsqrtYZ[1];
#else
    float rsqrtY = Math::RSqrt(lengthSqY);
    float rsqrtZ = Math::RSqrt(lengthSqZ);
#endif

    Vec3& unitX = *Vec3::Cast(pMtx->v[0].a);
    Vec3& unitY = *Vec3::Cast(pMtx->v[1].a);
    Vec3& unitZ = *Vec3::Cast(pMtx->v[2].a);
    unitY.Mul(axisY, rsqrtY);
    unitZ.Mul(axisZ, rsqrtZ);
    unitX.Cross(unitY, unitZ);
}

}} // namespace nw::g3d

#endif // NW_G3D_SKELETALANIMOBJ_H_
