﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_SHADERPARAMANIMOBJ_H_
#define NW_G3D_SHADERPARAMANIMOBJ_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/g3d_AnimObj.h>
#include <nw/g3d/res/g3d_ResShaderParamAnim.h>

namespace nw { namespace g3d {

class ModelObj;
class MaterialObj;

//! @brief シェーダパラメータアニメーションインスタンスです。
class ShaderParamAnimObj : public ModelAnimObj
{
public:
    class Builder;
    class InitArg;

    //! @brief インスタンスの構築時に渡すバッファの必要アライメントサイズです。
    enum Alignment
    {
        //! @brief Init() で渡すバッファの必要アライメントサイズです。
        BUFFER_ALIGNMENT        = 4,
    };

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @brief コンストラクタです。
    //!
    //! 実際の構築処理は Init() で行います。
    //!
    ShaderParamAnimObj() : ModelAnimObj(), m_pRes(NULL) {}

    //! @brief インスタンスの初期化を行います。
    bool Init(const InitArg& arg, void* pBuffer, size_t bufferSize);

    //! @brief 計算に必要なバッファサイズを計算します。
    static size_t CalcBufferSize(const InitArg& arg);

    //@}

    //----------------------------------------
    //! @name 関連付け
    //@{

    virtual BindResult Bind(const ResModel* pModel);
    virtual BindResult Bind(const ModelObj* pModel);
    virtual void BindFast(const ResModel* pModel);

    //! @brief 指定したインデックスのマテリアルにフラグを設定します。
    void SetBindFlag(int materialIndex, BindFlag flag)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        SetBindFlagImpl(materialIndex, flag);
    }

    //! @brief 指定したインデックスのマテリアルのフラグを取得します。
    BindFlag GetBindFlag(int materialIndex)
    {
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return GetBindFlagImpl(materialIndex);
    }

    //@}

    //----------------------------------------
    //! @name 更新
    //@{

    virtual void ClearResult() {} // 定数の結果をバッファに持たないので何もしない。
    virtual void Calc();
    virtual void ApplyTo(ModelObj* pModelObj) const;
    //! @brief Calc()の別名関数です。
    void Calculate()
    {
        Calc();
    }

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief リソースを取得します。
    const ResShaderParamAnim* GetResource() const { return m_pRes; };

    //! @brief リソースを設定します。
    //!
    //! 計算するシェーダパラメータアニメーションを差し替えます。
    //!
    void SetResource(ResShaderParamAnim* pRes);

    //! @brief リソースを取り外します。
    void ResetResource()
    {
        m_pRes = NULL;
        SetTargetUnbound();
    }

    //! @brief アニメーション対象に対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResModel* pModel) const
    {
        NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
        NW_G3D_ASSERT_NOT_NULL(m_pRes);
        return pModel->GetMaterialCount() <= GetBindTable().GetSize();
    }

    //! @brief アニメーションリソースに対して十分なバッファが存在するかどうかを取得します。
    bool IsAcceptable(const ResShaderParamAnim* pRes) const
    {
        NW_G3D_ASSERT_NOT_NULL(pRes);
        return pRes->GetMatAnimCount() <= m_MaxMatAnim && pRes->GetParamAnimCount() <= m_MaxSubBind;
    }

    //@}

protected:

    //! @briefprivate マテリアルアニメーション取得します。
    //!
    ResShaderParamMatAnim* GetMatAnim(int animIndex)
    {
        return ResShaderParamMatAnim::ResCast(&m_pMatAnimArray[animIndex]);
    }

    //! @briefprivate マテリアルアニメーション取得します。
    //!
    const ResShaderParamMatAnim* GetMatAnim(int animIndex) const
    {
        return ResShaderParamMatAnim::ResCast(&m_pMatAnimArray[animIndex]);
    }

    //! @briefprivate マテリアルごとに関連付けを行う内部関数です。
    //!
    BindResult SubBind(const ResShaderParamMatAnim* pMatAnim, const ResMaterial* pMaterial);

    //! @briefprivate マテリアルごとに関連付けを行う内部関数です。
    //!
    BindResult SubBindFast(const ResShaderParamMatAnim* pMatAnim);

    //! @briefprivate マテリアルごとに適用を行う内部関数です。
    //!
    void ApplyTo(MaterialObj* pMaterialObj, int animIndex) const;

private:
    class Sizer;
    ResShaderParamAnim* m_pRes;
    ResShaderParamMatAnimData* m_pMatAnimArray;
    s32 m_MaxMatAnim;
    s32 m_MaxSubBind;
    u16* m_pSubBindIndexArray;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(ShaderParamAnimObj);
};

//--------------------------------------------------------------------------------------------------

//! @briefprivate ShaderParamAnimObj のサイズを計算するためのクラスです。
//!
class ShaderParamAnimObj::Sizer : public nw::g3d::Sizer
{
public:
    //! @brief コンストラクタです。
    Sizer() : nw::g3d::Sizer() {}

    //! @brief ShaderParamAnimObj::InitArg に基づいてサイズを計算します。
    void Calc(const InitArg& arg);

    enum
    {
        RESULT_BUFFER,
        BIND_TABLE,
        SUBBIND_TABLE,
        FRAMECACHE_ARRAY,
        NUM_CHUNK
    };

    Chunk chunk[NUM_CHUNK];
};

//--------------------------------------------------------------------------------------------------

//! @brief ShaderParamAnimObj::Init() に渡して初期化を行うパラメータです。
class ShaderParamAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! インスタンスで使用するすべての ResShaderParamAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxMatCount()、SetMaxMatAnimCount()、SetMaxParamAnimCount()、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    InitArg() { Clear(); }

    //! @brief パラメータを初期化します。
    void Clear()
    {
        m_NumMat = m_NumMatAnim = m_NumParamAnim = m_NumCurve = -1;
        m_ContextEnabled = true;
        m_ContextAvailable = false;
    }

    //! @brief 対象となる ResModel を使用するのに必要なパラメータを設定します。
    void Reserve(const ResModel* pResModel)
    {
        NW_G3D_ASSERT_NOT_NULL(pResModel);
        m_NumMat = std::max(m_NumMat, pResModel->GetMaterialCount());
        m_Sizer.Invalidate();
    }

    //! @brief 対象となる ResShaderParamAnim を使用するのに必要なパラメータを設定します。
    void Reserve(const ResShaderParamAnim* pResAnim)
    {
        NW_G3D_ASSERT_NOT_NULL(pResAnim);
        m_NumMatAnim = std::max(m_NumMatAnim, pResAnim->GetMatAnimCount());
        m_NumParamAnim = std::max(m_NumParamAnim, pResAnim->GetParamAnimCount());
        m_NumCurve = std::max(m_NumCurve, pResAnim->GetCurveCount());
        m_ContextAvailable |= !pResAnim->IsCurveBaked();
        m_Sizer.Invalidate();
    }

    //! @brief アニメーション高速化のためのコンテクストを有効にします。
    void EnableContext() { m_ContextEnabled = true; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストを無効にします。
    void DisableContext() { m_ContextEnabled = false; m_Sizer.Invalidate(); }

    //! @brief アニメーション高速化のためのコンテクストが有効かどうかを取得します。
    bool IsContextEnabled() const { return m_ContextAvailable && m_ContextEnabled; }

    //! @brief 使用できる最大のマテリアル数を指定します。
    void SetMaxMatCount(int matCount)
    {
        NW_G3D_ASSERT(matCount >= 0);
        m_NumMat = matCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアル数を取得します。
    int GetMaxMatCount() const { return m_NumMat; }

    //! @brief 使用できる最大のマテリアルアニメーション数を指定します。
    void SetMaxMatAnimCount(int matAnimCount)
    {
        NW_G3D_ASSERT(matAnimCount >= 0);
        m_NumMatAnim = matAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のマテリアルアニメーション数を取得します。
    int GetMaxMatAnimCount() const { return m_NumMatAnim; }

    //! @brief 使用できる最大のパラメータアニメーション数を指定します。
    void SetMaxParamAnimCount(int paramAnimCount)
    {
        NW_G3D_ASSERT(paramAnimCount >= 0);
        m_NumParamAnim = paramAnimCount;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のパラメータアニメーション数を取得します。
    int GetMaxParamAnimCount() const { return m_NumParamAnim; }

    //! @brief 使用できる最大のカーブ数を指定します。
    void SetMaxCurveCount(int curveCount)
    {
        NW_G3D_ASSERT(curveCount >= 0);
        m_NumCurve = curveCount;
        m_ContextAvailable = true;
        m_Sizer.Invalidate();
    }

    //! @brief 使用できる最大のカーブ数を取得します。
    int GetMaxCurveCount() const { return m_NumCurve; }

    //! @brief 必要なパラメータが設定されているかどうかを取得します。
    bool IsValid() const
    {
        return m_NumMat >= 0 && m_NumMatAnim >= 0 && m_NumParamAnim >= 0 && m_NumCurve >= 0;
    }

    //! @brief サイズ計算用のオブジェクトを取得します。
    Sizer& GetSizer() const { return m_Sizer; }

private:
    int m_NumMat;
    int m_NumMatAnim;
    int m_NumParamAnim;
    int m_NumCurve;
    bool m_ContextEnabled;
    bool m_ContextAvailable; // リソースがベイクされていないか、カーブ数を手動指定した場合に true
    mutable Sizer m_Sizer; // キャッシュするために mutable にする
};

//--------------------------------------------------------------------------------------------------

//! @brief ShaderParamAnimObj の構築を行うクラスです。
class ShaderParamAnimObj::Builder : public ShaderParamAnimObj::InitArg
{
public:
    //! @brief コンストラクタです。
    //!
    //! @details
    //! インスタンスで使用するすべての ResShaderParamAnim とすべての ResModel に対して
    //! Reserve() するか、SetMaxMatCount()、SetMaxMatAnimCount()、SetMaxParamAnimCount()、
    //! SetMaxCurveCount() でバッファサイズの計算に必要なパラメータを指定する必要があります。
    //!
    Builder() : m_Size(0), m_IsCalculated(false)
    {
    }

    //! @brief ShaderParamAnimObj を構築します。
    //!
    //! @param[in] pShaderParamAnimObj ShaderParamAnimObj へのポインタ
    //! @param[in] pBuffer バッファへのポインタ
    //! @param[in] bufferSize バッファのサイズ
    //!
    //! @pre
    //! - CalculateMemorySize() を呼び、メモリサイズが計算済みである
    //! - bufferSize >= GetWorkMemorySize() で返すサイズ
    //!
    bool Build(ShaderParamAnimObj* pShaderParamAnimObj, void* pBuffer, size_t bufferSize) const
    {
        return pShaderParamAnimObj->Init(*this, pBuffer, bufferSize);
    }

    //! @brief ShaderParamAnimObj 構築に必要なメモリサイズを計算します。
    void CalculateMemorySize()
    {
        m_Size = CalcBufferSize(*this);
        m_IsCalculated = true;
    }

    //! @brief ShaderParamAnimObj 構築に必要なメモリサイズを取得します。
    //!
    //! @return  ShaderParamAnimObj 構築に必要なメモリサイズを返します。
    //!
    size_t GetWorkMemorySize() const
    {
        return m_Size;
    }

    //! @brief ShaderParamAnimObj 構築に必要なメモリサイズが計算済みかを取得します。
    //!
    //! @return 計算済みの場合は true、未計算の場合は false を返します。
    //!
    bool IsMemoryCalculated() const
    {
        return m_IsCalculated;
    }

private:
    size_t m_Size;
    bool m_IsCalculated;
};


}} // namespace nw::g3d

#endif // NW_G3D_SHADERPARAMANIMOBJ_H_
