﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GFXSTATE_H_
#define NW_G3D_FND_GFXSTATE_H_

#include <nw/g3d/g3d_config.h>
#include <nw/g3d/fnd/g3d_GX2Struct.h>
#include <cafe/gx2/gx2RenderStateReg.h>

// TODO: デフォルト値の検討

#define NW_G3D_GFX_STATE(class_name)                                                               \
public:                                                                                            \
    /* ---------------------------------------- */                                                 \
    /*! @name 共通 */                                                                              \
    /*@{ */                                                                                        \
    /*! @briefprivate */                                                                       \
    typedef class_name self_type;                                                                  \
    /*! @briefprivate */                                                                       \
    typedef class_name##_t base_type;                                                              \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static self_type* DownCast(base_type* ptr) { return static_cast<self_type*>(ptr); }            \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static const self_type* DownCast(const base_type* ptr)                                         \
        { return static_cast<const self_type*>(ptr); }                                             \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static self_type& DownCast(base_type& ref) { return static_cast<self_type&>(ref); }            \
    /*! @brief class_name##_t から class_name に変換します。 */                                    \
    static const self_type& DownCast(const base_type& ref)                                         \
        { return static_cast<const self_type&>(ref); }                                             \
    /*! @brief コンストラクタです。 SetDefault() を呼び出します。 */                               \
    class_name() { SetDefault(); }                                                                 \
    /*@} */                                                                                        \


namespace nw { namespace g3d { namespace fnd {

//! @brief ポリゴンの構造体です。
struct GfxPolygonCtrl_t
{
    GX2PolygonControlReg gx2PolygonControl;
};

//! @brief ポリゴンです。
class GfxPolygonCtrl : public GfxPolygonCtrl_t
{
    NW_G3D_GFX_STATE(GfxPolygonCtrl);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ポリゴンを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 表面カリングを設定します。
    void SetCullFront(GX2Boolean cull);

    //! @brief 裏面カリングを設定します。
    void SetCullBack(GX2Boolean cull);

    //! @brief 表面ポリゴンの頂点順を設定します。
    void SetFrontFace(GX2FrontFaceMode mode);

    //! @brief 表面カリングを取得します。
    GX2Boolean GetCullFront() const;

    //! @brief 裏面カリングを取得します。
    GX2Boolean GetCullBack() const;

    //! @brief 表面ポリゴンの頂点順を取得します。
    GX2FrontFaceMode GetFrontFace() const;

    //! @brief ポリゴンモードを有効にするかどうかを設定します。
    void SetPolygonModeEnable(GX2Boolean enable);

    //! @brief 表面ポリゴンのモードを設定します。
    void SetPolygonModeFront(GX2PolygonMode mode);

    //! @brief 裏面ポリゴンのモードを設定します。
    void SetPolygonModeBack(GX2PolygonMode mode);

    //! @brief ポリゴンモードを有効にするかどうかを取得します。
    GX2Boolean GetPolygonModeEnable() const;

    //! @brief 表面ポリゴンのモードを取得します。
    GX2PolygonMode GetPolygonModeFront() const;

    //! @brief 裏面ポリゴンのモードを取得します。
    GX2PolygonMode GetPolygonModeBack() const;

    //! @brief 表面ポリゴンのポリゴンオフセットを有効にするかどうかを設定します。
    void SetPolygonOffsetFrontEnable(GX2Boolean enable);

    //! @brief 裏面ポリゴンのポリゴンオフセットを有効にするかどうかを設定します。
    void SetPolygonOffsetBackEnable(GX2Boolean enable);

    //! @brief ポイントやラインへのポリゴンオフセットを有効にするかどうかを設定します。
    void SetPointLineOffsetEnable(GX2Boolean enable);

    //! @brief 表面ポリゴンのポリゴンオフセットを有効にするかどうかを取得します。
    GX2Boolean GetPolygonOffsetFrontEnable() const;

    //! @brief 裏面ポリゴンのポリゴンオフセットを有効にするかどうかを取得します。
    GX2Boolean GetPolygonOffsetBackEnable() const;

    //! @brief ポイントやラインへのポリゴンオフセットを有効にするかどうかを取得します。
    GX2Boolean GetPointLineOffsetEnable() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief 深度テストの構造体です。
struct GfxDepthCtrl_t
{
    GX2DepthStencilControlReg gx2DepthStencilControl;
};

//! @brief 深度テストです。
class GfxDepthCtrl : public GfxDepthCtrl_t
{
    NW_G3D_GFX_STATE(GfxDepthCtrl);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief 深度テストを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 深度テストが有効かどうかを設定します。
    void SetDepthTestEnable(GX2Boolean enable);

    //! @brief 深度値の書き込みが有効かどうかを設定します。
    void SetDepthWriteEnable(GX2Boolean enable);

    //! @brief 深度の比較関数を設定します。
    void SetDepthFunc(GX2CompareFunction func);

    //! @brief ステンシルテストが有効かどうかを設定します。
    void SetStencilTestEnable(GX2Boolean enable);

    //! @brief 裏面ステンシルテストが有効かどうかを設定します。
    void SetBackStencilEnable(GX2Boolean enable);

    //! @brief 深度テストが有効かどうかを取得します。
    GX2Boolean GetDepthTestEnable() const;

    //! @brief 深度値の書き込みが有効かどうかを取得します。
    GX2Boolean GetDepthWriteEnable() const;

    //! @brief 深度の比較関数を取得します。
    GX2CompareFunction GetDepthFunc() const;

    //! @brief ステンシルテストが有効かどうかを取得します。
    GX2Boolean GetStencilTestEnable() const;

    //! @brief 裏面ステンシルテストが有効かどうかを取得します。
    GX2Boolean GetBackStencilEnable() const;

    //! @brief 表面ステンシルの比較関数を設定します。
    void SetFrontStencilFunc(GX2CompareFunction func);

    //! @brief 表面ステンシルがステンシル、深度テストを通過した場合のオペレーションを設定します。
    void SetFrontStencilZPass(GX2StencilFunction func);

    //! @brief 表面ステンシルが深度テストを失敗した場合のオペレーションを設定します。
    void SetFrontStencilZFail(GX2StencilFunction func);

    //! @brief 表面ステンシルがステンシルテストを失敗した場合のオペレーションを設定します。
    void SetFrontStencilFail(GX2StencilFunction func);

    //! @brief 表面ステンシルの比較関数を取得します。
    GX2CompareFunction GetFrontStencilFunc() const;

    //! @brief 表面ステンシルがステンシル、深度テストを通過した場合のオペレーションを取得します。
    GX2StencilFunction GetFrontStencilZPass() const;

    //! @brief 表面ステンシルが深度テストを失敗した場合のオペレーションを取得します。
    GX2StencilFunction GetFrontStencilZFail() const;

    //! @brief 表面ステンシルがステンシルテストを失敗した場合のオペレーションを取得します。
    GX2StencilFunction GetFrontStencilFail() const;

    //! @brief 裏面ステンシルの比較関数を設定します。
    void SetBackStencilFunc(GX2CompareFunction func);

    //! @brief 裏面ステンシルがステンシル、深度テストを通過した場合のオペレーションを設定します。
    void SetBackStencilZPass(GX2StencilFunction func);

    //! @brief 裏面ステンシルが深度テストを失敗した場合のオペレーションを設定します。
    void SetBackStencilZFail(GX2StencilFunction func);

    //! @brief 裏面ステンシルがステンシルテストを失敗した場合のオペレーションを設定します。
    void SetBackStencilFail(GX2StencilFunction func);

    //! @brief 裏面ステンシルの比較関数を取得します。
    GX2CompareFunction GetBackStencilFunc() const;

    //! @brief 裏面ステンシルがステンシル、深度テストを通過した場合のオペレーションを取得します。
    GX2StencilFunction GetBackStencilZPass() const;

    //! @brief 裏面ステンシルが深度テストを失敗した場合のオペレーションを取得します。
    GX2StencilFunction GetBackStencilZFail() const;

    //! @brief 裏面ステンシルがステンシルテストを失敗した場合のオペレーションを取得します。
    GX2StencilFunction GetBackStencilFail() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ステンシルマスクの構造体です。
struct GfxStencilMask_t
{
    GX2StencilMaskReg gx2StencilMask;
};

//! @brief ステンシルテストマスクです。
class GfxStencilMask : public GfxStencilMask_t
{
    NW_G3D_GFX_STATE(GfxStencilMask);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ステンシルマスクを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 表面の参照マスクを設定します。
    void SetValueMaskFront(u8 mask);

    //! @brief 表面の書き込みマスクを設定します。
    void SetWriteMaskFront(u8 mask);

    //! @brief 表面のステンシルの参照値を設定します。
    void SetRefValueFront(u8 value);

    //! @brief 表面の参照マスクを取得します。
    u8 GetValueMaskFront() const;

    //! @brief 表面の書き込みマスクを取得します。
    u8 GetWriteMaskFront() const;

    //! @brief 表面のステンシルの参照値を取得します。
    u8 GetRefValueFront() const;

    //! @brief 裏面の参照マスクを設定します。
    void SetValueMaskBack(u8 mask);

    //! @brief 裏面の書き込みマスクを設定します。
    void SetWriteMaskBack(u8 mask);

    //! @brief 裏面のステンシルの参照値を設定します。
    void SetRefValueBack(u8 value);

    //! @brief 裏面の参照マスクを取得します。
    u8 GetValueMaskBack() const;

    //! @brief 裏面の書き込みマスクを取得します。
    u8 GetWriteMaskBack() const;

    //! @brief 裏面のステンシルの参照値を取得します。
    u8 GetRefValueBack() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief アルファテストの構造体です。
struct GfxAlphaTest_t
{
    GX2AlphaTestReg gx2AlphaTest;
};

//! @brief アルファテストです。
class GfxAlphaTest : public GfxAlphaTest_t
{
    NW_G3D_GFX_STATE(GfxAlphaTest);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief アルファテストを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief アルファテストを有効にするかどうかを設定します。
    void SetAlphaTestEnable(GX2Boolean enable);

    //! @brief アルファの比較関数を設定します。
    void SetAlphaFunc(GX2CompareFunction func);

    //! @brief アルファ参照値を設定します。
    void SetRefValue(f32 value);

    //! @brief アルファテストを有効にするかどうかを取得します。
    GX2Boolean GetAlphaTestEnable() const;

    //! @brief アルファの比較関数を取得します。
    GX2CompareFunction GetAlphaFunc() const;

    //! @brief アルファ参照値を取得します。
    f32 GetRefValue() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief カラーコントロールの構造体です。
struct GfxColorCtrl_t
{
    GX2ColorControlReg gx2ColorControl;
};

//! @brief カラーコントロールです。
class GfxColorCtrl : public GfxColorCtrl_t
{
    NW_G3D_GFX_STATE(GfxColorCtrl);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief カラーコントロールを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 論理演算を設定します。
    void SetLogicOp(GX2LogicOp lop);

    //! @brief レンダーターゲットごとにブレンドの有効マスクを設定します。
    void SetBlendEnableMask(u8 mask);

    //! @brief レンダリングターゲット 0 の出力をすべてのレンダリングターゲットに送信するかどうかを設定します。
    void SetMultiWriteEnable(GX2Boolean enable);

    //! @brief カラーバッファへの書き込みを有効にするかどうかを設定します。
    void SetColorBufferEnable(GX2Boolean enable);

    //! @brief 論理演算を取得します。
    GX2LogicOp GetLogicOp() const;

    //! @brief ブレンドの有効マスクを取得します。
    u8 GetBlendEnableMask() const;

    //! @brief レンダリングターゲット 0 の出力をすべてのレンダリングターゲットに送信するかどうかを取得します。
    GX2Boolean GetMultiWriteEnable() const;

    //! @brief カラーバッファへの書き込みを有効にするかどうかを取得します。
    GX2Boolean GetColorBufferEnable() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ブレンドコントロールの構造体です。
struct GfxBlendCtrl_t
{
    GX2BlendControlReg gx2BlendControl;
};

//! @brief ブレンドコントロールです。
class GfxBlendCtrl : public GfxBlendCtrl_t
{
    NW_G3D_GFX_STATE(GfxBlendCtrl);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ブレンドコントロールを設定します。
    void Load() const;

    //! @brief ブレンドコントロールを設定します。
    void Load(GX2RenderTarget target) const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief ソースカラーのブレンド関数を設定します。
    void SetColorSrcBlend(GX2BlendFunction func);

    //! @brief デスティネーションカラーのブレンド関数を設定します。
    void SetColorDstBlend(GX2BlendFunction func);

    //! @brief カラーの重ね合わせ方法を設定します。
    void SetColorCombine(GX2BlendCombine combine);

    //! @brief アルファブレンドをセパレート方式で行うかどうかを設定します。
    void SetSeparateAlphaBlend(GX2Boolean enable);

    //! @brief ソースアルファのブレンド関数を設定します。
    void SetAlphaSrcBlend(GX2BlendFunction func);

    //! @brief デスティネーションアルファのブレンド関数を設定します。
    void SetAlphaDstBlend(GX2BlendFunction func);

    //! @brief アルファの重ね合わせ方法を設定します。
    void SetAlphaCombine(GX2BlendCombine combine);

    //! @brief ソースカラーのブレンド関数を取得します。
    GX2BlendFunction GetColorSrcBlend() const;

    //! @brief デスティネーションカラーのブレンド関数を取得します。
    GX2BlendFunction GetColorDstBlend() const;

    //! @brief カラーの重ね合わせ方法を取得します。
    GX2BlendCombine GetColorCombine() const;

    //! @brief アルファブレンドをセパレート方式で行うかどうかを取得します。
    GX2Boolean GetSeparateAlphaBlend() const;

    //! @brief ソースアルファのブレンド関数を取得します。
    GX2BlendFunction GetAlphaSrcBlend() const;

    //! @brief デスティネーションアルファのブレンド関数を取得します。
    GX2BlendFunction GetAlphaDstBlend() const;

    //! @brief アルファの重ね合わせ方法を取得します。
    GX2BlendCombine GetAlphaCombine() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ブレンドカラーの構造体です。
struct GfxBlendColor_t
{
   GX2BlendConstantColorReg gx2BlendConstantColor;
};

//! @brief ブレンドカラーです。
class GfxBlendColor : public GfxBlendColor_t
{
    NW_G3D_GFX_STATE(GfxBlendColor);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ブレンドカラーを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief ブレンドカラーを設定します。
    void SetConstantColor(const f32 color[4]);

    //! @brief ブレンドカラーを取得します。
    f32* GetConstantColor();

    //! @brief ブレンドカラーを取得します。
    const f32* GetConstantColor() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ポリゴンオフセットの構造体です。
struct GfxPolygonOffset_t
{
    GX2PolygonOffsetReg gx2PolygonOffset;
};

//! @brief ポリゴンオフセットです。
class GfxPolygonOffset : public GfxPolygonOffset_t
{
    NW_G3D_GFX_STATE(GfxPolygonOffset);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ポリゴンオフセットを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief ポリゴンの表面のオフセットスケールを設定します。
    void SetFrontScale(f32 scale);

    //! @brief ポリゴンの表面のオフセット値を設定します。
    void SetFrontOffset(f32 offset);

    //! @brief ポリゴンの裏面のオフセットスケールを設定します。
    void SetBackScale(f32 scale);

    //! @brief ポリゴンの裏面のオフセット値を設定します。
    void SetBackOffset(f32 offset);

    //! @brief オフセットの最大値または最小値を設定します。
    void SetClamp(f32 clamp);

    //! @brief ポリゴンの表面のオフセットスケールを取得します。
    f32 GetFrontScale() const;

    //! @brief ポリゴンの表面のオフセット値を取得します。
    f32 GetFrontOffset() const;

    //! @brief ポリゴンの裏面のオフセットスケールを取得します。
    f32 GetBackScale() const;

    //! @brief ポリゴンの裏面のオフセット値を取得します。
    f32 GetBackOffset() const;

    //! @brief オフセットの最大値または最小値を取得します。
    f32 GetClamp() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ポイントサイズの構造体です。
struct GfxPointSize_t
{
    GX2PointSizeReg gx2PointSize;
};

//! @brief ポイントサイズです。
class GfxPointSize : public GfxPointSize_t
{
    NW_G3D_GFX_STATE(GfxPointSize);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ポイントサイズを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 幅を設定します。
    void SetWidth(f32 width);

    //! @brief 高さを設定します。
    void SetHeight(f32 height);

    //! @brief 幅を取得します。
    f32 GetWidth() const;

    //! @brief 高さを取得します。
    f32 GetHeight() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ポイントサイズの制限の構造体です。
struct GfxPointLimits_t
{
    GX2PointLimitsReg gx2PointLimits;
};

//! @brief ポイントサイズの制限です。
class GfxPointLimits : public GfxPointLimits_t
{
    NW_G3D_GFX_STATE(GfxPointLimits);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ポイントサイズの制限を設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 最小サイズを設定します。
    void SetMinSize(f32 size);

    //! @brief 最大サイズを設定します。
    void SetMaxSize(f32 size);

    //! @brief 最小サイズを取得します。
    f32 GetMinSize() const;

    //! @brief 最大サイズを取得します。
    f32 GetMaxSize() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ライン幅の構造体です。
struct GfxLineWidth_t
{
    GX2LineWidthReg gx2LineWidth;
};

//! @brief ライン幅です。
class GfxLineWidth : public GfxLineWidth_t
{
    NW_G3D_GFX_STATE(GfxLineWidth);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ライン幅を設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief ライン幅を設定します。
    void SetWidth(f32 width);

    //! @brief ライン幅を取得します。
    f32 GetWidth() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief チャンネルマスクの構造体です。
struct GfxChannelMasks_t
{
    GX2TargetChannelMaskReg gx2TargetChannelMask;
};

//! @brief チャンネルマスクです。
class GfxChannelMasks : public GfxChannelMasks_t
{
    NW_G3D_GFX_STATE(GfxChannelMasks);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief チャンネルマスクを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief 指定したターゲットのチャンネルマスクを設定します。
    void SetChannelMask(GX2RenderTarget target, GX2ChannelMask mask);

    //! @brief 指定したターゲットのチャンネルマスクを取得します。
    GX2ChannelMask GetChannelMask(GX2RenderTarget target) const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief ビューポートの構造体です。
struct GfxViewport_t
{
    f32 reg[6];
};

//! @brief ビューポートです。
class GfxViewport : public GfxViewport_t
{
    NW_G3D_GFX_STATE(GfxViewport);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief ビューポートを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief ビューポートを設定します。
    //!
    //! GX2 は左上原点なのに対して OpenGL は左下原点です。変換はこの関数の外側で行ってください。
    //!
    void SetViewport(float originX, float originY, float width, float height,
        float nearZ, float farZ);

    //! @brief 原点の X 座標を取得します。
    float GetOriginX() const;

    //! @brief 原点の Y 座標を取得します。
    float GetOriginY() const;

    //! @brief 幅を取得します。
    float GetWidth() const;

    //! @brief 高さを取得します。
    float GetHeight() const;

    //! @brief Near 平面のデプスを取得します。
    float GetNearZ() const;

    //! @brief Far 平面のデプスを取得します。
    float GetFarZ() const;

    //@}
};

//--------------------------------------------------------------------------------------------------

//! @brief シザリングの構造体です。
struct GfxScissor_t
{
    u32 reg[4];
};

//! @brief シザリングです。
class GfxScissor : public GfxScissor_t
{
    NW_G3D_GFX_STATE(GfxScissor);

public:
    //----------------------------------------
    //! @name 描画
    //@{

    //! @brief シザリングを設定します。
    void Load() const;

    //@}

    //----------------------------------------
    //! @name 取得/設定
    //@{

    //! @brief GX2 オブジェクトをデフォルト状態に設定します。
    void SetDefault();

    //! @brief シザリングを設定します。
    //!
    //! GX2 は左上原点なのに対して OpenGL は左下原点です。変換はこの関数の外側で行ってください。
    //!
    void SetScissor(u32 originX, u32 originY, u32 width, u32 height);

    //! @brief 原点の X 座標を取得します。
    u32 GetOriginX() const;

    //! @brief 原点の Y 座標を取得します。
    u32 GetOriginY() const;

    //! @brief 幅を取得します。
    u32 GetWidth() const;

    //! @brief 高さを取得します。
    u32 GetHeight() const;

    //@}
};

}}} // namespace nw::g3d::fnd

#endif // NW_G3D_FND_GFXSTATE_H_
