﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_G3D_FND_GFXCONTEXT_H_
#define NW_G3D_FND_GFXCONTEXT_H_

#include <nw/g3d/g3d_config.h>

#include <cafe/gx2/gx2Enum.h>

#if defined(NW_G3D_IS_GL_ES)
#if defined(ANDROID)
#include <android_native_app_glue.h>
#endif
#endif

namespace nw { namespace g3d { namespace fnd {

//! @brief グラフィックスコンテクストです。
class GfxContext
{
public:

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @briefprivate
    GfxContext() {}

    //! @brief グラフィックスコンテクストを構築します。
    void Setup();

    //! @brief グラフィックスコンテクストを破棄します。
    void Cleanup();

    //! @brief 指定したレンダリングコンテクストでシステムのグラフィックスコンテクストを再初期化します。
    //!
    //! PC 版以外では何も行いません。
    //!
    static void Reinit(void* hRC);

    //@}

    //----------------------------------------
    //! @name 有効化/無効化
    //@{

    //! @brief グラフィックスコンテクストを有効化します。
    void Activate();

    //! @brief CPU 処理用にコンテクストを有効化します。
    //!
    //! PC 版以外では何も行いません。
    //!
    static void Prepare();

    //! @brief CPU 処理用にコンテクストを有効化します。
    //!
    //! 特定のドライバで GL の VertexArrayObject 及び QueryObject が
    //! 生成したコンテクスト以外のコンテクストで利用できない問題への暫定対処用です。
    //!
    void TempPrepare();

    //! @brief グラフィックスコンテクストを無効化します。
    static void Invalidate();

    //@}

#if NW_G3D_IS_GX2
    //! @briefprivate
    GX2ContextState gx2ContextState;
#elif NW_G3D_IS_GL
    //! @briefprivate
    void* hRC;
#endif

#if defined(NW_G3D_IS_GL_ES) && defined(ANDROID)
    static bool InitEGL(const struct android_app* app);
#endif

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN(GfxContext);
};

//--------------------------------------------------------------------------------------------------

class GfxManage
{
public:
    //! @brief グラフィックスコマンドをフラッシュし、コマンドの完了まで待ちます。
    static void DrawDone();

    //! @brief バッファリングされているグラフィックスコマンドを GPU に送ります。
    static void FlushCommands();
};

//--------------------------------------------------------------------------------------------------

//! @brief GPU キャッシュです。
class GPUCache
{
public:
    //! @brief GPU の読み込みキャッシュを破棄するグラフィックスコマンドを発行します。
    static void InvalidateAll();

    //! @brief GPU の書き込みキャッシュを吐き出すグラフィックスコマンドを発行します。
    static void FlushAll();
};

//--------------------------------------------------------------------------------------------------

//! @brief CPU キャッシュです。
class CPUCache
{
public:
    //! @brief データキャッシュをメモリに書き出してキャッシュ上から破棄します。
    //!
    //! 結果が確実に反映されている必要がある処理の前には Sync() を呼んでください。
    //!
    static void Flush(const void* addr, size_t size);

    //! @brief データキャッシュをメモリに書き出します。
    //!
    //! 結果が確実に反映されている必要がある処理の前には Sync() を呼んでください。
    //!
    static void Store(const void* addr, size_t size);

    //! @brief データキャッシュを破棄します。
    //!
    //! 指定するアドレスとサイズは CACHE_BLOCK_SIZE バイトアライメントが必要です。
    //!
    static void Invalidate(void* addr, size_t size);

    //! @brief データキャッシュがメモリに書き出されたことを保証します。
    static void Sync();

    //! @brief データキャッシュが有効かどうかを取得します。
    //!
    //! キャッシュ禁止領域であれば false を、それ以外の場合は true を返します。
    //!
    static bool IsValid(const void* addr, size_t size);

    //! @brief データキャッシュをゼロクリアします。
    //!
    //! 指定するアドレスとサイズは CACHE_BLOCK_SIZE バイトアライメントが必要です。
    //!
    static void FillZero(void* addr, size_t size);
};

//--------------------------------------------------------------------------------------------------

//! @brief ロックドキャッシュです。
class LockedCache
{
public:
    //! @brief 現在のコアで確保可能なロックドキャッシュのサイズを取得します。
    static size_t GetAllocatableSize();

    //! @brief 指定したサイズのロックドキャッシュを現在のコアから取得します。
    static void* Alloc(size_t size);

    //! @brief 指定したロックドキャッシュ領域を解放します。
    static void Free(void* ptr);

    //! @brief DMA を有効化します。
    static bool EnableDMA();

    //! @brief DMA を無効化します。
    static void DisableDMA();

    //! @brief DMA を有効かどうかを取得します。
    static bool IsDMAEnabled();

    //! @brief LCDMA を使用してメモリからロックドキャッシュにコピーを行います。
    static u32 LoadDMABlocks(void* pDstLC, const void* pSrcMem, u32 numBlocks);

    //! @brief LCDMA を使用してロックドキャッシュからメモリにコピーを行います。
    static u32 StoreDMABlocks(void* pDstMem, const void* pSrcLC, u32 numBlocks);

    //! @brief DMA キューのトランザクションが指定した数以下になるのを待ちます。
    static void WaitDMAQueue(u32 length);
};

//--------------------------------------------------------------------------------------------------

//! @brief GPU クロックです。
class GPUClock
{
public:
    typedef s64 Tick;

    //----------------------------------------
    //! @name 構築/破棄
    //@{

    //! @briefprivate
    GPUClock()
#if NW_G3D_IS_GL
        : handle(0)
#endif
    {
    }

    //! @brief GPU のタイムスタンプを構築します。
    void Setup();

    //! @brief GPU のタイムスタンプを破棄します。
    void Cleanup();

    //@}

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief GPU タイムスタンプを取得するグラフィックスコマンドを発行します。
    void Query();

    //! @brief タイムスタンプが書き込まれたかどうかを取得します。
    bool IsReady() const;

    //! @brief タイムスタンプの値を取得します。
    Tick GetTimeStamp() const;

    //! @brief 現在のタイムスタンプを取得します。
    static Tick Now();

    //! @brief 1秒あたりの Tick 数を取得します。
    static Tick GetFreq() { return s_Freq; }

    //@}

    //----------------------------------------
    //! @name 単位変換
    //@{

    //! @brief タイムスタンプを秒単位に変換します。
    static float ToSeconds(Tick ticks) { return static_cast<float>(ticks) / s_Freq; }

    //! @brief タイムスタンプをミリ秒単位に変換します。
    static float ToMilliSecs(Tick ticks) { return ToSeconds(ticks) * 1000; }

    //! @brief タイムスタンプをマイクロ秒単位に変換します。
    static float ToMicroSecs(Tick ticks) { return ToSeconds(ticks) * (1000 * 1000); }

    //! @brief タイムスタンプを 60fps 換算のパーセンテージに変換します。
    static float ToPercents(Tick ticks) { return ToSeconds(ticks) * (60 * 100); }

    //@}

    union
    {
        //! @brief GX2 が書き込む実タイムスタンプです。
        u64 timestamp;

        //! @brief GL のハンドルです。
        u32 handle;
    };

private:
    static Tick s_Freq;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(GPUClock);
};

//--------------------------------------------------------------------------------------------------

//! @brief CPU クロックです。
class CPUClock
{
public:
    typedef s64 Tick;

    //----------------------------------------
    //! @name 取得
    //@{

    //! @brief 現在のタイムスタンプを取得します。
    static Tick Now();

    //! @brief 1秒あたりの Tick 数を取得します。
    static Tick GetFreq() { return s_Freq.Get(); }

    //@}

    //----------------------------------------
    //! @name 単位変換
    //@{

    //! @brief タイムスタンプを秒単位に変換します。
    static float ToSeconds(Tick ticks) { return static_cast<float>(ticks) / s_Freq.Get(); }

    //! @brief タイムスタンプをミリ秒単位に変換します。
    static float ToMilliSecs(Tick ticks) { return ToSeconds(ticks) * 1000; }

    //! @brief タイムスタンプをマイクロ秒単位に変換します。
    static float ToMicroSecs(Tick ticks) { return ToSeconds(ticks) * (1000 * 1000); }

    //! @brief タイムスタンプを 60fps 換算のパーセンテージに変換します。
    static float ToPercents(Tick ticks) { return ToSeconds(ticks) * (60 * 100); }

    //@}

private:
    class Freq
    {
    public:
        Freq() { Init(); }

        void Init();
        Tick Get() const { return m_Value; }

    private:
        Tick m_Value;
    };
    static Freq s_Freq;

    NW_G3D_DISALLOW_COPY_AND_ASSIGN(CPUClock);
};

//--------------------------------------------------------------------------------------------------

#if NW_G3D_IS_GL

class GfxTexture;
class GfxColorBuffer;
class GfxDepthBuffer;

namespace detail {

//! @briefprivate
class SystemShader
{
public:
    enum Sampler
    {
        SAMPLER_TEX
    };

    enum Location
    {
        LOCATION_GAMMA,
        LOCATION_SLICE,

        NUM_LOCATION
    };

    //! @briefprivate
    u32 hShader;

    //! @briefprivate
    u32 texUnit;

    //! @briefprivate
    u32 locations[NUM_LOCATION];
};

//! @briefprivate
class SystemContext
{
public:
    //! @briefprivate
    SystemContext();

    //! @briefprivate
    ~SystemContext();

    //! @briefprivate
    void Setup();

    //! @briefprivate
    void Cleanup();

    //! @briefprivate
    void BindFrameBuffer(GfxTexture* pTexture, u32 mipLevel, u32 slice);

    //! @briefprivate
    void BindFrameBuffer(GfxColorBuffer* pColorBuffer, int layer = -1);

    //! @briefprivate
    void BindFrameBuffer(GfxDepthBuffer* pDepthBuffer, int layer = -1);

    //! @briefprivate
    void BindTexture(u32 handle, u32 unit, u32 mipLevel, GX2SurfaceDim dim = GX2_SURFACE_DIM_2D);

    //! @briefprivate
    void DrawScreenQuad();

    //! @briefprivate
    const SystemShader& SelectCopyShader(GX2SurfaceDim dim, bool swap) const;

    //! @briefprivate
    GfxContext ctx;

    //! @briefprivate
    u32 hFrameBuffer;

    //! @briefprivate
    u32 hSampler;

    //! @briefprivate
    SystemShader copyShader;

    //! @briefprivate
    SystemShader resolveShader;

    //! @briefprivate
    SystemShader copy3DShader;

    //! @briefprivate
    SystemShader copy2DArrayShader;

    //! @briefprivate
    SystemShader resolveArrayShader;

    //! @briefprivate
    SystemShader swapCopyShader;

    //! @briefprivate
    SystemShader swapResolveShader;

    //! @briefprivate
    SystemShader swapCopy3DShader;

    //! @briefprivate
    SystemShader swapCopy2DArrayShader;

    //! @briefprivate
    SystemShader swapResolveArrayShader;
};

//! @briefprivate
SystemContext& GetSystemContext();

} // namespace detail

#endif // NW_G3D_IS_GL

}}} // namespace nw::g3d::fnd

#endif // NW_G3D_FND_GFXCONTEXT_H_
