﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/g3d_config.h>
#include <cafe/gx2/gx2Enum.h>

#include <nw/g3d/edit/g3d_EditDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO


namespace nw { namespace g3d {

namespace edit {

namespace detail {

#if defined( _MSC_VER )
#pragma warning(push)
#pragma warning(disable:4201)
#endif

struct SetupRenderInfoData
{
    u32 type;
    BinString ofsName;
    Offset ofsChoice;
    Offset ofsDefault;
};

struct SetupRenderInfoChoiceData
{
    u32 numChoice;
    union
    {
        s32 iValue[1];
        f32 fValue[1];
        BinString ofsString[1];
    };
};

struct SetupRenderInfoDefaultData
{
    u32 numDefault;
    union
    {
        s32 iValue[1];
        f32 fValue[1];
        BinString ofsString[1];
    };
};

} // namespace detail

namespace detail {

//--------------------------------------------------------------------------------------------------
//  Assert
#define NW_G3D_EDIT_ASSERT(exp) \
    NW_G3D_EDIT_ASSERTMSG(exp, "")

#define NW_G3D_EDIT_ASSERTMSG(exp, ...) \
    NW_G3D_ASSERTMSG(exp, "NW:Edit: Failed assertion. " #exp "\n" __VA_ARGS__)

#define NW_G3D_EDIT_ASSERT_NOT_NULL(exp) \
    NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(exp, "")

#define NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(exp, ...) \
    NW_G3D_EDIT_ASSERTMSG(exp, "NW:Edit: Pointer must not be NULL. ("#exp")\n" __VA_ARGS__)

// TODO: NintendoSDKを使えるようになったらNN_UNEXPECCTED_DEFAULTに置き換える
#define NW_G3D_EDIT_UNEXPECTED_DEFAULT NW_G3D_ASSERTMSG(false, "NW:Edit: Unexpected default.\n")


#if defined(_DEBUG) || defined(NW_DEBUG)
#define NW_G3D_EDIT_PRINT( ... )                                                \
    nw::g3d::DebugPrint("NW:Edit: "), nw::g3d::DebugPrint(__VA_ARGS__)
#else
#define NW_G3D_EDIT_PRINT( ... ) ((void)0)
#endif

bool IsRuntimeDebugLogEnabled();

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
#define NN_DETAIL_G3D_EDIT_CONFIG_DEBUG_LOG_ENABLED (1)
#else
#define NN_DETAIL_G3D_EDIT_CONFIG_DEBUG_LOG_ENABLED (0)
#endif

#if NN_DETAIL_G3D_EDIT_CONFIG_DEBUG_LOG_ENABLED
#define NW_G3D_EDIT_DEBUG_PRINT( ... ) \
    if (IsRuntimeDebugLogEnabled()) { nw::g3d::DebugPrint("NW:Edit:(Debug) "), nw::g3d::DebugPrint(__VA_ARGS__); }
#else
#define NW_G3D_EDIT_DEBUG_PRINT( ... ) static_cast<void>(0)
#endif



#define NW_G3D_EDIT_WARNING_INDEX_BOUNDS(index, size)                           \
    NW_G3D_EDIT_WARNING(static_cast<uint>(index) < static_cast<uint>(size),     \
    #index " is out of bounds. (%d)\n"                                          \
    "0 <= " #index " < %d is not satisfied.\n",                                 \
    static_cast<int>(index), static_cast<int>(size))

#if defined(_DEBUG) || defined(NW_DEBUG)
#define NW_G3D_EDIT_WARNING(exp, ...)                                                       \
    (void) ((exp) || (nw::g3d::DebugPrint("NW:Edit: "), nw::g3d::DebugPrint(__VA_ARGS__),   \
    nw::g3d::DebugPrint("\t%s(%d)\n", NW_G3D_CODE_POSITION_NAME, __LINE__), 0))
#else
#define NW_G3D_EDIT_WARNING(exp, ... ) ((void)0)
#endif

// 実機側のチャンネル名です。
static const char* EDIT_CHANNEL_NAME = "NWG3D_EDIT";

// ネットワーク生存確認用のチャンネル名です。
static const char* EDIT_PING_CHANNEL_NAME = "NWG3D_EDIT_PING";

// PC版で使用するポート番号です。
static const u16 EDIT_DEFAULT_PORT_NUMBER = 61440;
static const u16 EDIT_PING_PORT_NUMBER = EDIT_DEFAULT_PORT_NUMBER + 1;

static const size_t NW_G3D_EDIT_FILENAME_MAX = 256;

static const size_t NW_G3D_EDIT_ALIGNMENT = 32;

struct PacketHeader
{
    u32 magic;      // 送信クライアントを識別するためのマジックコード

    union
    {
        u8 version[4];// パケットバージョン
        u32 verWord;
    };

    s32 dataSize;   // データサイズ
    u32 command;    // コマンド
};

struct Packet
{
    PacketHeader    header;
    s8              data[4];// データ（可変）
};

//--------------------------------------------------------------------------------------------------
// Ping

enum PingState
{
    PING_STATE_NORMAL = 0,
    PING_STATE_KEEPING = 1 << 0,
    PING_STATE_LOCK = 1 << 1,
    PING_STATE_UNLOCK = 1 << 2,
};

struct PingBlock
{
    u32 uniqueID;
    u32 codePage;
    u32 state;
    u32 padding;
};

struct PingPacket
{
    PacketHeader    header;
    PingBlock       block;
};

//--------------------------------------------------------------------------------------------------
// アタッチ
enum AttachKind
{
    ATTACH_MODEL = 0x00,
    DETACH_MODEL = 0x01,
    ATTACH_SHADER_ARCHIVE = 0x02,
    DETACH_SHADER_ARCHIVE = 0x03,
    ATTACH_KIND_ERROR = 0xFF
};

enum AttachFlag
{
    ATTACH_SHADER_ARCHIVE_IS_BINARY = 0x01
};

struct AttachBlock
{
    u32 attachedKey;
    u16 attachKind;
    u16 flag;
    s8 fileName[NW_G3D_EDIT_FILENAME_MAX];
    s8 attachFileName[NW_G3D_EDIT_FILENAME_MAX];
};

struct AttachPacket
{
    PacketHeader    header;
    AttachBlock     block;
};

struct FileLoadedBlock
{
    u32 fileDataKind;
    u32 toolKey;
    u32 resFileKey;
    u32 modelObjKey;
    u32 resModelKey;
    u32 newResFileKey;
    u32 padding[2];
};

struct FileLoadedPacket
{
    PacketHeader    header;
    FileLoadedBlock block;
};

struct MaterialShaderIndexData
{
    u32 numMaterial;
    s32 index[1];// データ（可変）
};

struct FileInfoData
{
    struct OffsetFileInfo
    {
        u32 ofsFileName;
        u32 fileSize;
        u16 fileDataKind;
        u16 padding;
        u32 fileAlignment;
    };
    OffsetFileInfo fileInfo[1];// numFile分 offset が並びます
};

struct MultiFileDataBlock
{
    u16 numFile;
    u16 loopCount;
    u32 padding[2];
    Offset ofsFileInfo;
};

struct ModelOptimizedShaderBlock
{
    u32 modelObjKey;
    Offset ofsMaterialShaderIndices;
    MultiFileDataBlock multiFile;
};

struct ModelOptimizedShaderPacket
{
    PacketHeader header;
    ModelOptimizedShaderBlock block;
};

//--------------------------------------------------------------------------------------------------
// ファイルデータ

enum FileDataKind
{
    FILEDATA_MODEL =                0x00,
    FILEDATA_TEXTURE =              0x01,

    FILEDATA_SHADER_PARAM_ANIM =    0x02,
    FILEDATA_MAT_COLOR_ANIM =       0x03,
    FILEDATA_TEXTURE_SRT_ANIM =     0x04,
    FILEDATA_TEXTURE_PATTERN_ANIM = 0x05,
    FILEDATA_SKELETAL_ANIM =        0x06,
    FILEDATA_BONE_VISIBILITY_ANIM = 0x07,
    FILEDATA_MAT_VISIBILITY_ANIM =  0x08,
    FILEDATA_SHAPE_ANIM =           0x09,
    FILEDATA_SCENE_ANIM =           0x0A,
    FILEDATA_MATERIAL_ANIM =        0x0B,

    FILEDATA_SHADER_DEFINE =        0x11,
    FILEDATA_SHADER_ARCHIVE =       0x12,
    FILEDATA_SHADER_PROGRAM =       0x13,

    FILEDATA_KIND_MAX,
    FILEDATA_KIND_ERROR =           0xFF
};

struct FileDataBlock
{
    u32             key;
    u32             resFileKey;
    u32             kind;
    u32             fileSize;
    u32             pathSize;
    s32             shadingModelIndex;
    s32             shaderProgramIndex;
    u32             fileAlignment;
    s8              fileName[NW_G3D_EDIT_FILENAME_MAX];
    s8              retargetingHostModelName[NW_G3D_EDIT_FILENAME_MAX];
};

struct FileDataPacket
{
    PacketHeader    header;
    FileDataBlock   block;
};

//--------------------------------------------------------------------------------------------------
// エディットデータ

enum EditValueKind
{
    BOOL_VALUE = 0x00,
    U32_VALUE,
    S32_VALUE,
    F32_VALUE,
    VECTOR4_VALUE,
    SAMPLER_VALUE,
    SAMPLER_TEXTURE_REF,
    SHADER_ARCHIVE_REF,
    EDIT_VALUE_KIND_MAX
};

enum EditPlayPolicyKind
{
    EDIT_PLAY_POLICY_AUTO = 0x00,
    EDIT_PLAY_POLICY_ONCE,
    EDIT_PLAY_POLICY_LOOP
};

struct EditValueInfoBlock
{
    u32 key;
    u16 valueKind;
    u16 editTargetKind;
    u32 length;
    u32 indexSize;
};

struct EditValuePacket
{
    PacketHeader        header;
    EditValueInfoBlock  block;
};

struct EditSimpleValue
{
    union
    {
        union
        {
            bit32                       flag;
            u32                         uValue;
            struct
            {
                u16                     uHigh;
                u16                     uLow;
            };
        };

        s32                             sValue;
        f32                             fValue;
        bool                            bValue;

        res::ResRenderState::Mode       renderStateMode;
        res::ResRenderState::BlendMode  renderStateBlendMode;

        GX2CompareFunction              depthTestFunc;
        GX2CompareFunction              alphaTestFunc;
        GX2BlendCombine                 blendCombine;
        GX2BlendFunction                colorBlendFunc;
        GX2BlendFunction                alphaBlendFunc;
        GX2LogicOp                      logicOp;
    };
};

struct EditValueBlock
{
    EditSimpleValue     value;
    s32                 index[1];// データ（可変）
};

struct EditVector4Value
{
    union
    {
        f32     fValue[4];
        s32     sValue[4];
        u32     uValue[4];
    };
};

struct EditVector4ValueBlock
{
    EditVector4Value    value;
    s32                 index[1];// データ（可変）
};

struct EditMatrix44Value
{
    enum { DIM_MAJOR = 4, DIM_MINOR = 4, DIM = DIM_MAJOR * DIM_MINOR };

    union
    {
        struct
        {
            float m00, m01, m02, m03;
            float m10, m11, m12, m13;
            float m20, m21, m22, m23;
            float m30, m31, m32, m33;
        };
        float a[DIM];
        float m[DIM_MAJOR][DIM_MINOR];
    };
};

struct EditSamplerValue
{
    s32 samplerIndex;

    union
    {
        GX2TexClamp         clamp;
        GX2TexBorderType    borderType;
        GX2TexXYFilterType  filter;
        GX2TexZFilterType   zFilter;
        GX2TexMipFilterType mipFilter;
        GX2TexAnisoRatio    ratio;
        f32                 lod;
        f32                 bias;
        GX2CompareFunction  func;
        u32                 uValue;
        f32                 fValue;
        s32                 sValue;
    };
};

struct EditSamplerValueBlock
{
    EditSamplerValue    value;
    s32                 index[1];// データ（可変）
};

struct EditSamplerTextureRef
{
    s32 samplerIndex;
    s8  textureName[NW_G3D_EDIT_FILENAME_MAX];
};

struct EditShaderArchiveRef
{
    s8  shaderArchiveName[NW_G3D_EDIT_FILENAME_MAX];
    s8  shadingModelName[NW_G3D_EDIT_FILENAME_MAX];
};

struct EditShaderParamVectorValue
{
    EditVector4Value    value;
    s8                  paramName[NW_G3D_EDIT_FILENAME_MAX];
};

struct EditShaderParamVectorValueBlock
{
    EditShaderParamVectorValue  value;
    s32                         index[1];// データ（可変）
};

struct EditShaderParamMatrixValue
{
    EditMatrix44Value   value;
    s8                  paramName[NW_G3D_EDIT_FILENAME_MAX];
};

struct EditShaderParamMatrixValueBlock
{
    EditShaderParamMatrixValue  value;
    s32                         index[1];// データ（可変）
};

//--------------------------------------------------------------------------------------------------
// シェーディングモデル変更通知用データ

struct EditShadingModelValueBlock
{
    s32                 index[1];// データ（可変）
};

//--------------------------------------------------------------------------------------------------
// アニメーションエディットデータ

struct BindAnimInfo
{
    u32 modelKey;
    u32 attachModelFlag;
    u32 animationKeySize;
};

struct BindAnimInfoBlock
{
    BindAnimInfo info;
    u32 animationKeys[1];
};

struct BindAnimPacket
{
    PacketHeader        header;
    BindAnimInfoBlock   block;
};

struct AnimCurveInfo
{
    u32 animationKey;
    u16 animationKind;
    u16 padding;
};

struct AnimCurveValue
{
    union
    {
        s32 index;
        s32 materialIndex;
        s32 vertexShapeIndex;
        s32 cameraIndex;
        s32 lightIndex;
        s32 fogIndex;
    };

    s32 curveIndex;
    u32 curveSize;
};

struct AnimCurveBlock
{
    AnimCurveInfo               info;
    AnimCurveValue              value;
    u32                         curveData[1];// ResAnimCurve のデータが入ります。
};

struct AnimCurvePacket
{
    PacketHeader    header;
    AnimCurveBlock  block;
};

struct AnimEditInfoBlock
{
    u32 animationKey;
    union
    {
        f32 fValue;
        u32 iValue;
    };
    u32 modelKey;
    u32 padding[1];
};

struct AnimEditInfoPacket
{
    PacketHeader        header;
    AnimEditInfoBlock   block;
};

//--------------------------------------------------------------------------------------------------
// アニメーションフレームコントロールデータ

struct FrameCtrlBlock
{
    f32 frame;
    f32 frameStep;
    union
    {
        u32 playPolicy;
        u32 modelKey;
    };
    f32 frameCount;
};

struct FrameCtrlPacket
{
    PacketHeader        header;
    FrameCtrlBlock      block;
};

//--------------------------------------------------------------------------------------------------
// 描画情報受信用データ

struct RenderInfoRecvBlock
{
    u32 modelKey;
    u16 numRenderInfo;
    u16 numMaterialIndex;

    Offset ofsRenderInfoArray;
    Offset ofsMaterialIndexArray;
};

struct RenderInfoRecvPacket
{
    PacketHeader        header;
    RenderInfoRecvBlock block;
};

//--------------------------------------------------------------------------------------------------
// 描画情報送信用データ

struct RenderInfoSendInfo
{
    u32 modelKey;
    s32 materialIndex;
    u32 labelInfoNum;
    u32 padding;
};

struct RenderInfoLabelInfo
{
    u32 labelOffset;
    u32 renderInfoType;
    u32 valueNum;
    u32 valueOffset;

    u32 padding[2];

    union
    {
        // renderInfoType が STRING 型の時に使用します。
        struct
        {
            u32 itemNum;
            u32 itemOffset;
        };
        // renderInfoType が INT 型の時に使用します。
        struct
        {
            s32 iMinValue;
            s32 iMaxValue;
        };
        // renderInfoType が FLOAT 型の時に使用します。
        struct
        {
            f32 fMinValue;
            f32 fMaxValue;
        };
    };
};

struct RenderInfoChoiceInfo
{
    u32 choiceOffset;
    u32 aliasOffset;
    u32 aliasSize;
    u32 padding;
};

struct RenderInfoSendPacket
{
    PacketHeader        header;
    RenderInfoSendInfo  info;
};

//--------------------------------------------------------------------------------------------------
// 描画情報選択編集用データ
struct RenderInfoEditInfo
{
    u32 type;

    u32 modelKey;
    s32 materialIndex;

    union
    {
        s32 slotIndex;
        s32 arraySize;
    };

    union
    {
        s32 itemIndex;
        s32 iValue;
        f32 fValue;
    };
    u32 labelOffset;
    u32 padding[2];
};

struct RenderInfoEditPacket
{
    PacketHeader        header;
    RenderInfoEditInfo  info;
};

//--------------------------------------------------------------------------------------------------
// 描画情報辞書更新用データ

struct RenderInfoUpdateBlock
{
    u32 modelKey;
    s32 materialIndex;
    u32 renderInfoDataSize;
    u32 padding;
    u32 renderInfoData[4];// ResRenderInfo の実データが入ります。
};

struct RenderInfoUpdatePacket
{
    PacketHeader            header;
    RenderInfoUpdateBlock   block;
};

//--------------------------------------------------------------------------------------------------
// ボーンバインド編集用データ

struct BondBindEditBlock
{
    u32 parentModelKey;
    u32 parentBoneIndex;
    u32 childModelKey;
    u16 isBind;
    u16 padding;
};

struct BondBindEditPacket
{
    PacketHeader        header;
    BondBindEditBlock   block;
};

//--------------------------------------------------------------------------------------------------
// 配置情報受信用データ

struct ModelLayoutRecvBlock
{
    u32 modelKey;
    u32 padding;
};

struct ModelLayoutRecvPacket
{
    PacketHeader            header;
    ModelLayoutRecvBlock    block;
};

//--------------------------------------------------------------------------------------------------
// 配置情報編集用データ

struct ModelLayoutEditBlock
{
    u32 isBind;
    u32 modelKey;
    u32 padding;

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } scale;

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } rotate;

    union
    {
        struct
        {
            f32 x, y, z;
        };
        f32 a[3];
    } translate;
};

struct ModelLayoutEditPacket
{
    PacketHeader            header;
    ModelLayoutEditBlock    block;
};

//--------------------------------------------------------------------------------------------------
// シェイプ Lod level 受信用データ

struct ShapeLodLevelEditBlock
{
    u32 modelKey;
    s32 lodLevel;
};

struct ShapeLodLevelEditPacket
{
    PacketHeader            header;
    ShapeLodLevelEditBlock  block;
};

//--------------------------------------------------------------------------------------------------
// 変更されたシェーダプログラム要求用データ

struct ShaderProgramSendInfo
{
    u32 shaderArchiveKey;
    s32 shadingModelIndex;
    s32 shaderProgramIndex;
    u32 optionInfoNum;
};

struct ShaderProgramOptionInfo
{
    u32 optionOffset;
    u32 choiceOffset;
};

struct ShaderProgramSendPacket
{
    PacketHeader            header;
    ShaderProgramSendInfo   info;
};

//--------------------------------------------------------------------------------------------------
// 3DEditorへ通知する選択要求用データ

struct PickupSendInfo
{
    u32 materialPickupNum;
    u32 padding[3];
};

struct MaterialPickupInfo
{
    u32 modelObjKey;
    s32 materialIndex;
};

struct PickupSendPacket
{
    PacketHeader            header;
    PickupSendInfo          info;
};

//--------------------------------------------------------------------------------------------------
// クライアントへのエラー通知用データ

// クライアントにランタイムエラーを通知するためのエラーコードです。
enum RuntimeErrorCode
{
    RUNTIME_ERROR_CODE_NO_ERROR = 0,
    RUNTIME_ERROR_CODE_INVALID_MATERIAL_COUNT = 1,
    RUNTIME_ERROR_CODE_ATTACH_CANCELED = 2,
    RUNTIME_ERROR_CODE_LOAD_FILE_FAILED = 3,
    RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY = 4,
    RUNTIME_ERROR_CODE_BIND_ANIM_FAILED = 5,
    RUNTIME_ERROR_CODE_RETARGET_HOST_MODEL_NOT_FOUND = 6,
    RUNTIME_ERROR_CODE_DUPLICATE_MODEL_OBJ_KEY = 7,
    RUNTIME_ERROR_CODE_INVALID_SHADER_DETECTED = 8,
    RUNTIME_ERROR_CODE_INVALID_MODEL_ATTACHED = 9,
    RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND = 10,
    RUNTIME_ERROR_CODE_OPEN_FILE_FAILED = 11,
    RUNTIME_ERROR_CODE_UNKNOWN_ERROR = -1,
};

struct RuntimeErrorNotificationInfo
{
    s32 runtimeErrorCode;
};

struct RuntimeErrorNotificationPacket
{
    PacketHeader            header;
    RuntimeErrorNotificationInfo info;
};

//--------------------------------------------------------------------------------------------------
// ユーザスクリプト用データ

// 今のところモデル以外に適用する公開インタフェースは存在しないが、
// 将来モデル以外にスクリプトを適用したい要望はありそうなので、パケットは汎用的に作っておく

//! @brief スクリプト適用対象の種類です。
enum UserScriptTargetKind
{
    USER_SCRIPT_TARGET_KIND_MODEL,                    //!< モデルです。
    USER_SCRIPT_TARGET_KIND_TEXTURE,                  //!< テクスチャです。
    USER_SCRIPT_TARGET_KIND_SKELTAL_ANIM,             //!< スケルタルアニメです。
    USER_SCRIPT_TARGET_KIND_SHADER_PARAM_ANIM,        //!< シェーダパラメータアニメです。
    USER_SCRIPT_TARGET_KIND_TEXTURE_PATTERN_ANIM,     //!< テクスチャパターンアニメです。
    USER_SCRIPT_TARGET_KIND_BONE_VISIBILITY_ANIM,     //!< ボーンビジビリティアニメです。
    USER_SCRIPT_TARGET_KIND_MATERIAL_VISIBILITY_ANIM, //!< マテリアルビジビリティアニメです。
    USER_SCRIPT_TARGET_KIND_SHAPE_ANIM,               //!< シェイプアニメです。
    USER_SCRIPT_TARGET_KIND_SCENE_ANIM,               //!< シーンアニメです。
    USER_SCRIPT_TARGET_KIND_SHADER,                   //!< シェーダです。
};

struct UserScriptBlock
{
    u32 targetModelObjKey;

    s32 scriptTextLength;
    s32 targetNameLength;
    s32 selectedMaterialCount;
    s32 selectedShapeCount;
    s32 selectedBoneCount;

    s32 targetKind; //!< UserScriptTargetKind

    // dynamicData は下記のようなバイト列になる
    // selectedBoneIndices[selectedBoneCount * sizeof(s32)],
    // selectedShapeIndices[selectedShapeCount * sizeof(s32)],
    // selectedMaterialIndices[selectedMaterialCount * sizeof(s32)]
    // scriptText[scriptTextLength + 1],
    // targetName[targetNameLength + 1],
    s8  dynamicData[1];
    s8  padding[3];
};

struct UserScriptPacket
{
    PacketHeader    header;
    UserScriptBlock block;
};

//--------------------------------------------------------------------------------------------------
// 引数用構造体

struct EditMaterialArg
{
    u32     key;
    u16     valueKind;
    u16     editTargetKind;
    void*   value;
    u32     indexSize;
    s32*    index;
};

struct BindAnimArg
{
    u32     modelKey;
    u32*    animationKeys;
    u32     animationKeySize;
};

struct EditAnimCurveArg
{
    u32     animationKey;
    u16     animationKind;
    void*   value;
    void*   curveData;
    u32     curveDataSize;
};

struct EditBoneArg
{
    u32     modelKey;
    u16     valueKind;
    u16     editTargetKind;
    void*   value;
    u32     indexSize;
    s32*    index;
};

//--------------------------------------------------------------------------------------------------
// コマンド定義

enum CommandCategory
{
    FILEDATA_CATEGORY =             0x00,
    SYSTEM_CATEGORY =               0x01,
    MATERIAL_CATEGORY =             0x02,
    ANIMATION_CATEGORY =            0x03,
    EDIT_CATEGORY =                 0x04,
    SHADER_CATEGORY =               0x05,
    BONE_CATEGORY =                 0x06,
    MODEL_CATEGORY =                0x07,
    MODEL_ANIMATION_CATEGORY =      0x08,
    SCENE_ANIMATION_CATEGORY =      0x09,
    PICK_CATEGORY =                 0x0A,
    OTHER_CATEGORY =                0x0B,
};

enum CommandCategoryFlag
{
    FILEDATA_CATEGORY_FLAG =            FILEDATA_CATEGORY << 8,
    SYSTEM_CATEGORY_FLAG =              SYSTEM_CATEGORY << 8,
    MATERIAL_CATEGORY_FLAG =            MATERIAL_CATEGORY << 8,
    ANIMATION_CATEGORY_FLAG =           ANIMATION_CATEGORY << 8,
    EDIT_CATEGORY_FLAG =                EDIT_CATEGORY << 8,
    SHADER_CATEGORY_FLAG =              SHADER_CATEGORY << 8,
    BONE_CATEGORY_FLAG =                BONE_CATEGORY << 8,
    MODEL_CATEGORY_FLAG =               MODEL_CATEGORY << 8,
    MODEL_ANIMATION_CATEGORY_FLAG =     MODEL_ANIMATION_CATEGORY << 8,
    SCENE_ANIMATION_CATEGORY_FLAG =     SCENE_ANIMATION_CATEGORY << 8,
    PICK_CATEGORY_FLAG =                PICK_CATEGORY << 8,
    OTHER_CATEGORY_FLAG =               OTHER_CATEGORY << 8,
};

enum CommandFlag
{
    FILEDATA_LOAD_FILE_COMMAND_FLAG =           0x01 | FILEDATA_CATEGORY_FLAG,
    FILEDATA_UNLOAD_FILE_COMMAND_FLAG =         0x02 | FILEDATA_CATEGORY_FLAG,
    FILEDATA_RELOAD_FILE_COMMAND_FLAG =         0x03 | FILEDATA_CATEGORY_FLAG,
    FILEDATA_UNLOAD_ALL_COMMAND_FLAG =          0x04 | FILEDATA_CATEGORY_FLAG,

    SYSTEM_PING_RECV_COMMAND_FLAG =             0x01 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_PING_SEND_COMMAND_FLAG =             0x02 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_BEGIN_FREEZE_COMMAND_FLAG =          0x03 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_END_FREEZE_COMMAND_FLAG =            0x04 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_PACKET_VERSION_ERROR_COMMAND_FLAG =  0x05 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_CODEPAGE_COMMAND_FLAG =              0x06 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG =  0x07 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_RUNTIME_LOG_COMMAND_FLAG =           0x08 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_PLAY_FRAME_CTRL_COMMAND_FLAG =       0x10 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_STOP_FRAME_CTRL_COMMAND_FLAG =       0x11 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_SEND_FRAME_COMMAND_FLAG =            0x12 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_SEND_FRAME_STEP_COMMAND_FLAG =       0x13 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_SEND_MODEL_NEXT_ANIM_COMMAND_FLAG =  0x14 | SYSTEM_CATEGORY_FLAG,
    SYSTEM_SEND_MODEL_PREV_ANIM_COMMAND_FLAG =  0x15 | SYSTEM_CATEGORY_FLAG,

    SYSTEM_RUNTIME_STATE_NORMAL_COMMAND_FLAG =  0x20 | SYSTEM_CATEGORY_FLAG,

    SYSTEM_RUNTIME_ERROR_COMMAND_FLAG =         0x30 | SYSTEM_CATEGORY_FLAG,

    EDIT_MATERIAL_COMMAND_FLAG =                0x00 | MATERIAL_CATEGORY_FLAG,

    EDIT_SEND_ATTACH_COMMAND_FLAG =             0x00 | EDIT_CATEGORY_FLAG,
    EDIT_RECV_ATTACH_COMMAND_FLAG =             0x01 | EDIT_CATEGORY_FLAG,
    EDIT_SEND_DETACH_COMMAND_FLAG =             0x02 | EDIT_CATEGORY_FLAG,
    EDIT_FILE_LOADED_COMMAND_FLAG =             0x03 | EDIT_CATEGORY_FLAG,
    EDIT_FILE_RELOADED_COMMAND_FLAG =           0x04 | EDIT_CATEGORY_FLAG,
    EDIT_SEND_RENDER_INFO_COMMAND_FLAG =        0x10 | EDIT_CATEGORY_FLAG,
    EDIT_RECV_RENDER_INFO_COMMAND_FLAG =        0x11 | EDIT_CATEGORY_FLAG,
    EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG = 0x12 | EDIT_CATEGORY_FLAG,
    EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG =  0x13 | EDIT_CATEGORY_FLAG,
    EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG =      0x14 | EDIT_CATEGORY_FLAG,
    EDIT_SEND_MODIFIED_SHADER_COMMAND_FLAG =    0x15 | EDIT_CATEGORY_FLAG,
    EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG =    0x16 | EDIT_CATEGORY_FLAG,
    EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG =     0x17 | EDIT_CATEGORY_FLAG,
    EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG =    0x18 | EDIT_CATEGORY_FLAG,

    ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG =    0x00 | ANIMATION_CATEGORY_FLAG,
    ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG =    0x01 | ANIMATION_CATEGORY_FLAG,
    ANIMATION_PLAY_POLICY_COMMAND_FLAG =        0x02 | ANIMATION_CATEGORY_FLAG,
    ANIMATION_FRAME_STEP_COMMAND_FLAG =         0x03 | ANIMATION_CATEGORY_FLAG,
    ANIMATION_FRAME_COUNT_COMMAND_FLAG =        0x04 | ANIMATION_CATEGORY_FLAG,

    EDIT_SHADER_COMMAND_FLAG =                  0x00 | SHADER_CATEGORY_FLAG,

    EDIT_BONE_COMMAND_FLAG =                    0x00 | BONE_CATEGORY_FLAG,

    EDIT_MODEL_BONE_BIND_COMMAND_FLAG =         0x00 | MODEL_CATEGORY_FLAG,
    EDIT_MODEL_LAYOUT_COMMAND_FLAG =            0x01 | MODEL_CATEGORY_FLAG,
    EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG =       0x02 | MODEL_CATEGORY_FLAG,
    EDIT_SEND_MODEL_LAYOUT_COMMAND_FLAG =       0x03 | MODEL_CATEGORY_FLAG,
    EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG =     0x04 | MODEL_CATEGORY_FLAG,
    EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG =   0x05 | MODEL_CATEGORY_FLAG,

    MODEL_ANIMATION_BIND_COMMAND_FLAG =         0x00 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_UNBIND_COMMAND_FLAG =       0x01 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG =   0x02 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_PLAY_COMMAND_FLAG =         0x03 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_STOP_COMMAND_FLAG =         0x04 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG = 0x05 | MODEL_ANIMATION_CATEGORY_FLAG,
    MODEL_ANIMATION_EDIT_MIRRORING_ENABLED_COMMAND_FLAG =   0x06 | MODEL_ANIMATION_CATEGORY_FLAG,

    SCENE_ANIMATION_BIND_COMMAND_FLAG =         0x00 | SCENE_ANIMATION_CATEGORY_FLAG,
    SCENE_ANIMATION_UNBIND_COMMAND_FLAG =       0x01 | SCENE_ANIMATION_CATEGORY_FLAG,
    SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG =   0x02 | SCENE_ANIMATION_CATEGORY_FLAG,

    PICK_TOOL_MATERIAL_COMMAND_FLAG =           0x00 | PICK_CATEGORY_FLAG,
    PICK_RUNTIME_MODEL_COMMAND_FLAG =           0x01 | PICK_CATEGORY_FLAG,
    PICK_RUNTIME_MATERIAL_COMMAND_FLAG =        0x02 | PICK_CATEGORY_FLAG,
    PICK_RUNTIME_BONE_COMMAND_FLAG =            0x03 | PICK_CATEGORY_FLAG,
    PICK_RUNTIME_SHAPE_COMMAND_FLAG =           0x04 | PICK_CATEGORY_FLAG,

    OTHER_EXECUTE_USER_SCRIPT_FLAG =           0x00 | OTHER_CATEGORY_FLAG,
};

static const u32 COMMAND_CATEGORY_FLAG_MASK = 0xFF00;

enum EditTargetKind
{
    EDIT_TARGET_MATERIAL_VISIBILITY =                       0x00 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_DISPLAYFACE =                      0x01 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_RENDER_STATE_MODE =                0x10 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_RENDER_STATE_BLEND_MODE =          0x11 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_DEPTHTEST_ENABLE =                 0x12 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_DEPTHTEST_WRITE_ENABLE =           0x13 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_DEPTHTEST_FUNC =                   0x14 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_ALPHATEST_ENABLE =                 0x15 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_ALPHATEST_FUNC =                   0x16 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_ALPHATEST_VALUE =                  0x17 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_COLOR_COMBINE =                    0x20 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_ALPHA_COMBINE =                    0x21 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_COLOR_SRC_BLEND =                  0x22 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_COLOR_DST_BLEND =                  0x23 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_ALPHA_SRC_BLEND =                  0x24 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_ALPHA_DST_BLEND =                  0x25 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_CONSTANT_COLOR =                   0x26 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_LOGIC_OP =                         0x27 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SAMPLER_NAME =                     0x40 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_TEXTURE_REF =              0x41 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_WRAP_U =                   0x42 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_WRAP_V =                   0x43 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_WRAP_W =                   0x44 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MAG_FILTER =               0x45 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MIN_FILTER =               0x46 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MIP_FILTER =               0x47 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MAX_ANISO  =               0x48 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MIN_LOD  =                 0x49 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_MAX_LOD  =                 0x4A | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SAMPLER_LOD_BIAS =                 0x4B | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL =                0x60 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL2 =               0x61 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL3 =               0x62 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_BOOL4 =               0x63 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_INT =                 0x64 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_INT2 =                0x65 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_INT3 =                0x66 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_INT4 =                0x67 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT =                0x68 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT2 =               0x69 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT3 =               0x6A | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_UINT4 =               0x6B | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT =               0x6C | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2 =              0x6D | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3 =              0x6E | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4 =              0x6F | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x2 =            0x70 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x3 =            0x71 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT2x4 =            0x72 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x2 =            0x73 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x3 =            0x74 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT3x4 =            0x75 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x2 =            0x76 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x3 =            0x77 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_FLOAT4x4 =            0x78 | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT3D =               0x79 | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_SRT2D =               0x7A | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_MAYA =         0x7B | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_3DSMAX =       0x7C | MATERIAL_CATEGORY_FLAG,
    EDIT_TARGET_MATERIAL_SHADER_PARAM_TEXSRT_SOFTIMAGE =    0x7D | MATERIAL_CATEGORY_FLAG,

    EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE =        0x00 | MODEL_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE =         0x01 | MODEL_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE =     0x02 | MODEL_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE =      0x03 | MODEL_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE =               0x04 | MODEL_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_MODEL_ANIMATION_MATERIAL_CURVE =            0x05 | MODEL_ANIMATION_CATEGORY_FLAG,

    EDIT_TARGET_UPDATE_SHADING_MODEL =                      0x00 | SHADER_CATEGORY_FLAG,

    EDIT_TARGET_BONE_VISIBILITY =                           0x00 | BONE_CATEGORY_FLAG,
    EDIT_TARGET_BONE_BILLBOARD =                            0x01 | BONE_CATEGORY_FLAG,

    EDIT_TARGET_SCENE_ANIMATION_CAMERA_CURVE =              0x00 | SCENE_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_SCENE_ANIMATION_LIGHT_CURVE =               0x01 | SCENE_ANIMATION_CATEGORY_FLAG,
    EDIT_TARGET_SCENE_ANIMATION_FOG_CURVE =                 0x02 | SCENE_ANIMATION_CATEGORY_FLAG,

    EDIT_TARGET_KIND_MAX = 0xFFFF
};

#if defined( _MSC_VER )
#pragma warning(pop)
#endif

} // namespace detail

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
