﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eftvw2_String.h>
#include <nw/eft/eftvw2_Function.h>
#include <nw/eft/eftcom_Guid.h>
#include <nw/eft/eftvw2_Data.h>

namespace nw     {
namespace eftvw2 {

//---------------------------------------------------
//! @brief　プレビュークラス破棄コールバック 型。
//---------------------------------------------------
typedef void (*DestroyPreviewCallback)( nw::eft2::Heap* eftHeap, Preview* preview );

//---------------------------------------------------------------------------
//! @brief  プレビュークラスです。
//---------------------------------------------------------------------------
class Preview
{
public:
    //---------------------------------------------------------------------------
    //! @brief  プレビュータイプ
    //---------------------------------------------------------------------------
    enum EftPreviewType
    {
        EFT_VWR_PRV_TOP         = 0,                //!< トッププレビュー
        EFT_VWR_PRV_MODEL       = 1,                //!< モデルプレビュー
        EFT_VWR_PRV_EFFECT      = 2,                //!< エフェクトプレビュー
        EFT_VWR_PRV_USR_MODEL   = 3,                //!< ユーザーモデルプレビュー
        EFT_VWR_PRV_MAX,                            //!< TBD
    };

    //---------------------------------------------------------------------------
    //! @brief  プレビューフラグ
    //---------------------------------------------------------------------------
    enum EftPreviewFlag
    {
        EFT_VWR_PREV_CONSTRAIN      = 0x1 << 1,     //!< コンストレインされている
        EFT_VWR_PREV_FUNCTION       = 0x1 << 2,     //!< ファンクションを利用しているか
    };

    //---------------------------------------------------------------------------
    //! @brief  マトリクス適用タイプ
    //---------------------------------------------------------------------------
    enum MatrixApplyType
    {
        EFT_VWR_MATRIX_APPLY_SRT      = 0,            //!< SRT追従
        EFT_VWR_MATRIX_APPLY_T        = 1,            //!< Tのみ追従
    };

    //----------------------------------------
    //! @name 初期化/終了 処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //! @param[in] type     TBD
    //---------------------------------------------------------------------------
    explicit Preview( EftPreviewType type );

    //---------------------------------------------------------------------------
    //! @brief  コンストラクタです。
    //---------------------------------------------------------------------------
    Preview();

    //---------------------------------------------------------------------------
    //! @brief  デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~Preview() {}

    //---------------------------------------------------------------------------
    //! @brief  初期化処理です。
    //! @param[in] heap     TBD
    //! @param[in] type     TBD
    //! @param[in] guid     TBD
    //---------------------------------------------------------------------------
    void Initialize( nw::eft2::Heap* heap, EftPreviewType type, eftcom::Guid guid );

    //---------------------------------------------------------------------------
    //! @brief  終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief  プレビューリソースを設定します。
    //! @param[in] resource     TBD
    //---------------------------------------------------------------------------
    void SetResPreview( ResPreview* resource )
    {
        mResPreview = *resource;
        mIsSetResPreview = true;
    }

    //---------------------------------------------------------------------------
    //! @brief  プレビューリソースを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    ResPreview* GetResPreview() { return &mResPreview; }

    //---------------------------------------------------------------------------
    //! @brief  リソースマトリクスを設定します。
    //! @param[in] matrix     TBD
    //---------------------------------------------------------------------------
    void SetResMatrix( nw::math::Matrix34& matrix )
    {
        mResMatrix = matrix;
    }

    //@}

    //----------------------------------------
    //! @name 情報取得
    //@{

    //---------------------------------------------------------------------------
    //! @brief  プレビューのタイプを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    EftPreviewType GetPreviewType() const { return mPreviewType; }

    //---------------------------------------------------------------------------
    //! @brief  guidを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    eftcom::Guid GetGuID() const { return mGuId; }
    eftcom::Guid GetGuid() const { return mGuId; }

    //---------------------------------------------------------------------------
    //! @brief  マトリクスを取得します。
    //! @param[in] matrix     TBD
    //! @param[in] idx     TBD
    //---------------------------------------------------------------------------
    virtual void GetMatrix( nw::math::MTX34* matrix, s32 idx )
    {
        EFT_UNUSED_VARIABLE( idx );
        matrix->SetIdentity();
    }

    //---------------------------------------------------------------------------
    //! @brief        描画のOn/Offを状態を取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    virtual bool IsVisible() const { return true; }

    //@}

    //----------------------------------------
    //! @name 定期処理
    //@{

    //---------------------------------------------------------------------------
    //! @brief  計算処理です。
    //! @param[in] bPause     TBD
    //! @param[in] frameRate     TBD
    //! @param[in] centerMatrix     TBD
    //! @param[in] viewMatrix     TBD
    //---------------------------------------------------------------------------
    virtual void Calc( bool bPause, f32 frameRate, const nw::math::MTX34& centerMatrix, const nw::math::MTX34& viewMatrix );

    //@}

    //----------------------------------------
    //! @name プレビューリスト操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief  プレビューをリストに追加します。
    //! @param[in] preview     TBD
    //---------------------------------------------------------------------------
    void AddPreview( Preview* preview );

    //---------------------------------------------------------------------------
    //! @brief  このプレビューをリストから削除します。
    //---------------------------------------------------------------------------
    void RemovePreview();

    //---------------------------------------------------------------------------
    //! @brief  NextPreviewを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Preview* GetNextPreview() const { return mNext; }

    //---------------------------------------------------------------------------
    //! @brief  Guidを指定してプレビューを取得します。
    //! @param[in] guid     TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    Preview* GetPreview( eftcom::Guid guid );

    //---------------------------------------------------------------------------
    //! @brief  最後尾のPreviewを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Preview* GetTailPreview();

    //---------------------------------------------------------------------------
    //! @brief  先頭のPreviewを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Preview* GetHeadPreview();

    //@}

    //----------------------------------------
    //! @name プレビュー操作
    //@{

    //---------------------------------------------------------------------------
    //! @brief        プレビューをリセットします。
    //! @param[in] bFade     TBD
    //---------------------------------------------------------------------------
    virtual void ResetPreview( bool bFade ) { EFT_UNUSED_VARIABLE( bFade ); }

    //---------------------------------------------------------------------------
    //! @brief  描画のOn/Offを設定します。
    //! @param[in] visible     TBD
    //---------------------------------------------------------------------------
    virtual void SetVisible( bool visible ) { EFT_UNUSED_VARIABLE( visible ); }

    //@}

private:
    //---------------------------------------------------------------------------
    //! @brief        自動移動を計算します。
    //---------------------------------------------------------------------------
    void CalcAutoMove();

protected:
    nw::eft2::Heap*		    mHeap;              //!< ヒープ
    eftcom::Guid            mGuId;              //!< guid
    EftPreviewType          mPreviewType;       //!< プレビュータイプ
    Preview*                mPrev;              //!< プレビュー:Prev
    Preview*                mNext;              //!< プレビュー:Next
    f32                     mTime;              //!< プレビュータイム
    nw::math::MTX34         mResMatrix;         //!< リソースマトリクス
    bool                    mIsSetResPreview;   //!< プレビューリソースがセットされたか？
    ResPreview              mResPreview;        //!< プレビューリソース
    nw::math::MTX34         mAutoMoveMtx;       //!< 自動移動計算結果
    nw::math::MTX34         mDrawMatrix;        //!< 最終描画マトリクス

    //---------------------------------------------------------------------------
    //! @brief        破棄コールバックをセットします。
    //! @param[in] callback     TBD
    //---------------------------------------------------------------------------
    void SetDestroyPreviewCallback( DestroyPreviewCallback callback )
    {
        mDestroyCallback = callback;
    }

    //---------------------------------------------------------------------------
    //! @brief        破棄コールバックを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    DestroyPreviewCallback GetDestroyPreviewCallback() const
    {
        return mDestroyCallback;
    }

    DestroyPreviewCallback      mDestroyCallback;   //!< TBD

#if 0
    //---------------------------------------------------------------------------
    //! @brief  プレビュー名を取得します。
    //---------------------------------------------------------------------------
    virtual const char* GetPreviewName() const { return mPreviewName.Cstr(); }

    //---------------------------------------------------------------------------
    //! @brief        カラーを設定します。
    //---------------------------------------------------------------------------
    virtual void SetColor( nw::ut::Color4f* color )
    {
        EFT_UNUSED_VARIABLE( color );
    }

    //---------------------------------------------------------------------------
    //! @brief        アニメーションの再生速度を取得します。
    //---------------------------------------------------------------------------
    virtual f32 GetAnimationSpeed()
    {
        return mAnimSpeed;
    }

    //---------------------------------------------------------------------------
    //! @brief        アニメーションの再生速度を設定します。
    //---------------------------------------------------------------------------
    virtual void SetAnimationSpeed( f32 speed )
    {
        mAnimSpeed = speed;
    }

    //---------------------------------------------------------------------------
    //! @brief        アニメーションのタイプを取得します。
    //---------------------------------------------------------------------------
    virtual u32 GetAnimationIndex()
    {
        return mAnimIndex;
    }

    //---------------------------------------------------------------------------
    //! @brief        アニメーションのインデックスを設定します。
    //---------------------------------------------------------------------------
    virtual void SetAnimationIndex( u32 index )
    {
        mAnimIndex = index;
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルスケールを設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleScale( nw::math::VEC2* particle, nw::math::VEC2* emisison )
    {
        EFT_UNUSED_VARIABLE( particle );
        EFT_UNUSED_VARIABLE( emisison );
    }

    //---------------------------------------------------------------------------
    //! @brief        エミッタスケールを設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetEmitterScale( nw::math::VEC3* emitter )
    {
        EFT_UNUSED_VARIABLE( emitter );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルライフスケールを設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleLifeScale( f32 lifeScale )
    {
        EFT_UNUSED_VARIABLE( lifeScale );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル初速度スケールを設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleVelocityScale( f32 allDirectionalVec, f32 directionalVec )
    {
        EFT_UNUSED_VARIABLE( allDirectionalVec );
        EFT_UNUSED_VARIABLE( directionalVec );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルワールド加算速度を設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleAddWorldVelocity( nw::math::VEC3* velocity )
    {
        EFT_UNUSED_VARIABLE( velocity );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクルランダム速度のスケール値を設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleRandomVelocity( f32 velScale )
    {
        EFT_UNUSED_VARIABLE( velScale );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル指定方向速度のスケール値を設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    void SetParticleDirectionalVel( f32 velScale )
    {
        EFT_UNUSED_VARIABLE( velScale );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル放出スケールを設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleEmissionScale( f32 emissionRate, f32 emissionInterval )
    {
        EFT_UNUSED_VARIABLE( emissionRate );
        EFT_UNUSED_VARIABLE( emissionInterval );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル エミッタカラー０を設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleEmitterColor0( nw::ut::Color4f* color )
    {
        EFT_UNUSED_VARIABLE( color );
    }

    //---------------------------------------------------------------------------
    //! @brief        パーティクル エミッタカラー０を設定します。(エフェクトプレビュー専用の操作メソッド)
    //---------------------------------------------------------------------------
    virtual void SetParticleEmitterColor1( nw::ut::Color4f* color )
    {
        EFT_UNUSED_VARIABLE( color );
    }

    //---------------------------------------------------------------------------
    //! @brief        ファンクションのパラメータを設定します。
    //---------------------------------------------------------------------------
    //void SetFunctionParameter( EftFuncParamHeader* header, u32 size )
    //{
    //    mFunction.SetFunctionParam( header, size );
    //}

    //---------------------------------------------------------------------------
    //! @brief        指定IDのファンクションを保持するか？
    //---------------------------------------------------------------------------
    bool HasFunction( u32 functionId )
    {
        return mFunction.HasFunction( functionId );
    }

    //---------------------------------------------------------------------------
    //! @brief        指定名のプレビューを検索して取得します。
    //---------------------------------------------------------------------------
    Preview* GetPreview( const char* name );

    //---------------------------------------------------------------------------
    //! @brief        指定名＆指定IDでプレビューを検索して取得します。
    //---------------------------------------------------------------------------
    Preview* GetPreview( const char* name, s32 id );



    //---------------------------------------------------------------------------
    //! @brief        NextPreviewを設定します。
    //---------------------------------------------------------------------------
    void SetNextPreview( Preview* next ) const { mNext = next; }


    //---------------------------------------------------------------------------
    //! @brief        コンストレイン 状態を設定する
    //---------------------------------------------------------------------------
    void SetConstrainMode( bool enable )
    {
        if ( enable )
        {
            mFlag |= EFT_VWR_PREV_CONSTRAIN;
        }
        else
        {
            mFlag &= ~EFT_VWR_PREV_CONSTRAIN;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        コンストレイン 状態を取得する
    //---------------------------------------------------------------------------
    bool IsConstrainMode()
    {
        return 0 != ( mFlag & EFT_VWR_PREV_CONSTRAIN );
    }

    //---------------------------------------------------------------------------
    //! @brief        ファンクション 状態を設定する
    //---------------------------------------------------------------------------
    void SetFunctionEnable( bool enable )
    {
        if ( enable )
        {
            mFlag |= EFT_VWR_PREV_FUNCTION;
        }
        else
        {
            mFlag &= ~EFT_VWR_PREV_FUNCTION;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        ファンクション 状態を取得する
    //---------------------------------------------------------------------------
    bool IsFunctionEnable()
    {
        return 0 != ( mFlag & EFT_VWR_PREV_FUNCTION );
    }

#endif

};

} // namespace eftvw2
} // namespace nw
