﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/typeDef2.h>
#include <nw/eft/eft2_Enum.h>

namespace nw   {
namespace eft2 {

struct ResEmitter;
struct ResFieldCustom;

#ifdef EFT_OGL

//---------------------------------------------------------------------------
//! @brief  64Bitフラグ管理クラス
//---------------------------------------------------------------------------
struct Flag64
{
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //---------------------------------------------------------------------------
    void Initialize()
    {
        flag0 = 0;
        flag1 = 0;
//      for ( s32 i=0; i<64; i++ ) flag[i] = 0;
    }

    //---------------------------------------------------------------------------
    //! @brief  ビット桁を取得します。
    //! @param[in] bits TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    u32 GetCarryCount( u64 bits )
    {
        u32 num  = 63;
        u64 mask = 0x8000000000000000;
        for( ; mask != 0 ; mask = mask >> 1 )
        {
            if( bits & mask ){ break; }
            num--;
        }
        return num;
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグを立てます。
    //! @param[in] bit TBD
    //---------------------------------------------------------------------------
    void Set( u64 bit )
    {
        if ( bit > ((u64)0x01 << (u64)31) ) { flag1 |= (bit >> (u64)32); }
        else                                { flag0 |= bit;              }
//      flag[getCarryCount(bit)] = true;
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグをおろします。
    //! @param[in] bit TBD
    //---------------------------------------------------------------------------
    void Cancel( u64 bit )
    {
        if ( bit > ((u64)0x01 << (u64)31) ) { flag1 &= ~(bit >> (u64)32); }
        else                                { flag0 &= ~bit;              }
        //      flag[getCarryCount(bit)] = false;
    }

    //---------------------------------------------------------------------------
    //! @brief  指定ビットのフラグが立っているか取得します。
    //! @param[in] bit TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool Is( u64 bit ) const
    {
        if ( bit > ((u64)0x01 << (u64)31) ) { return ( flag1 & (bit >> (u64)32) ) ? 1 : 0; }
        else                                { return ( flag0 & bit ) ? 1 : 0; }
    }

    //---------------------------------------------------------------------------
    //! @brief  代入
    //! @param[in] flag TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool operator== ( Flag64 flag )
    {
        if ( ( flag.flag0 == flag0 ) &&
             ( flag.flag1 == flag1 ) )
        {
            return true;
        }
        return false;
    }

private:
    u32 flag0;  //!< TBD
    u32 flag1;  //!< TBD
//  u8 flag[64];
};

//---------------------------------------------------------------------------
//! @brief  シェーダキー生成の為のビットフラグ
//---------------------------------------------------------------------------
#define EFT_SHADER_KEY_FIELD_RANDOM                     ( (u64)0x01 << (u64)0  )   //!< フィールドランダム

//---------------------------------------------------------------------------
//! @brief  シェーダキー
//---------------------------------------------------------------------------
struct ShaderKey
{
    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] resEmitter TBD
    //! @param[in] emitterPluginIndex TBD
    //! @param[in] emitterPluginData TBD
    //! @param[in] fieldFlag TBD
    //! @param[in] customFieldFlag TBD
    //! @param[in] userDefine TBD
    //---------------------------------------------------------------------------
    void Initialize( ResEmitter* resEmitter, u32 emitterPluginIndex, void* emitterPluginData, u32 fieldFlag, ResFieldCustom* resCustomField, const char* userDefine=NULL );

    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを取得します。
    //! @param[in] isCafe TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    const char* GetCompileSetting()
    {
        Generate();
        return g_CompileSetting;
    }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexKey() const { return m_VtxFlag; }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキー2を取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexKey2() const { return m_VtxFlag2; }

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetFragKey() const { return m_FragFlag; }

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetFragKey2() const { return m_FragFlag2; }

    //---------------------------------------------------------------------------
    //! @brief  頂点シェーダキーを取得します。（EmitterPlugin版）
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetVertexKeyEP() const { return m_VtxFlagEP; }

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダキーを取得します。（EmitterPlugin版）
    //! @return TBD
    //---------------------------------------------------------------------------
    Flag64 GetFragKeyEP() const { return m_FragFlagEP; }

    //---------------------------------------------------------------------------
    //! @brief  描画パスを取得します。
    //! @return 描画パス
    //---------------------------------------------------------------------------
    u32 GetDrawPath() const { return m_DrawPath; }

    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u8 GetColor0AnimKeyNum() const { return m_VtxColor0AnimKey; }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u8 GetColor1AnimKeyNum() const { return m_VtxColor1AnimKey; }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u8 GetAlpha0AnimKeyNum() const { return m_VtxAlpha0AnimKey; }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u8 GetAlpha1AnimKeyNum() const { return m_VtxAlpha1AnimKey; }
    //---------------------------------------------------------------------------
    //! @brief  アニメーションキーを取得します。
    //! @return TBD
    //---------------------------------------------------------------------------
    u8 GetScaleAnimKeyNum() const { return m_VtxScaleAnimKey; }

    //---------------------------------------------------------------------------
    //! @brief  キーを比較します。
    //! @param[in] shaderKey TBD
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsEqual( const ShaderKey* shaderKey )
    {
        if ( ( m_VtxFlag             == shaderKey->GetVertexKey() ) &&
             ( m_VtxFlag2            == shaderKey->GetVertexKey2() ) &&
             ( m_FragFlag            == shaderKey->GetFragKey() )  &&
             ( m_FragFlag2           == shaderKey->GetFragKey2() )  &&
             ( m_VtxFlagEP           == shaderKey->GetVertexKeyEP() )  &&
             ( m_FragFlagEP          == shaderKey->GetFragKeyEP() )  &&
             ( m_DrawPath            == shaderKey->GetDrawPath() )  &&
             ( m_VtxColor0AnimKey    == shaderKey->GetColor0AnimKeyNum() )  &&
             ( m_VtxColor1AnimKey    == shaderKey->GetColor1AnimKeyNum() )  &&
             ( m_VtxAlpha0AnimKey    == shaderKey->GetAlpha0AnimKeyNum() )  &&
             ( m_VtxAlpha1AnimKey    == shaderKey->GetAlpha1AnimKeyNum() )  &&
             ( m_VtxScaleAnimKey     == shaderKey->GetScaleAnimKeyNum() )  &&
             ( m_CustomShaderIndex   == shaderKey->m_CustomShaderIndex ) &&
             ( m_CustomShaderFlag    == shaderKey->m_CustomShaderFlag ) &&
             ( m_CustomFieldFlag     == shaderKey->m_CustomFieldFlag ) &&
             ( m_CustomShaderSwitch  == shaderKey->m_CustomShaderSwitch ) &&
             ( m_CombinerShaderIndex == shaderKey->m_CombinerShaderIndex ) &&
             ( strcmp(m_CustomShaderDefine, shaderKey->m_CustomShaderDefine ) == 0 ) )
        {
            return true;
        }
        return false;
    }

    //---------------------------------------------------------------------------
    //! @brief  キーを比較します。
    //! @return TBD
    //---------------------------------------------------------------------------
    bool IsStreamOut() const;

public:
    //---------------------------------------------------------------------------
    //! @brief  キーを無効化します。
    //---------------------------------------------------------------------------
    void Invalidate()
    {
        m_VtxFlag.Initialize();
        m_VtxFlag2.Initialize();
        m_FragFlag.Initialize();
        m_FragFlag2.Initialize();
        m_VtxFlagEP.Initialize();
        m_FragFlagEP.Initialize();
        m_DrawPath           = 0;
        m_CustomShaderIndex  = 0;
        m_CustomShaderFlag   = 0;
        m_CustomFieldFlag    = 0;
        m_CustomFieldEnabled = 0;
        m_CustomShaderSwitch = 0;
        m_CombinerShaderIndex = static_cast< u32 >( -1 );
        m_VtxColor0AnimKey   = 0;
        m_VtxColor1AnimKey   = 0;
        m_VtxAlpha0AnimKey   = 0;
        m_VtxAlpha1AnimKey   = 0;
        m_VtxScaleAnimKey    = 0;
        memset(m_CustomShaderDefine, 0, 16);
    }

    //---------------------------------------------------------------------------
    //! @brief  描画パスを設定します。
    //! @param[in] drawPath 描画パス
    //---------------------------------------------------------------------------
    void SetDrawPath( u32 drawPath );

    //---------------------------------------------------------------------------
    //! @brief  エミッタ挙動タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetEmitterCalcType( EmitterCalcType type );

    //---------------------------------------------------------------------------
    //! @brief  パーティクル形状タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetFigureType( ParticleFigureType type );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルタイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetParticleType( ParticleType type );

    //---------------------------------------------------------------------------
    //! @brief  世界座標系で重力を適用 を設定します。
    //! @param[in] worldGravity TBD
    //---------------------------------------------------------------------------
    void SetParticleWorldGravity( bool worldGravity );

    //---------------------------------------------------------------------------
    //! @brief  パーティクル回転タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetParticleRotateType( ParticleRotType type );

    //---------------------------------------------------------------------------
    //! @brief  エミッタ追従タイプを設定します。
    //! @param[in] type TBD
    //---------------------------------------------------------------------------
    void SetEmitterFollowType( EmitterFollowType type );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルカラー計算タイプを設定します。
    //! @param[in] color0Type TBD
    //! @param[in] color1Type TBD
    //---------------------------------------------------------------------------
    void SetParticleColorCalcType( ParticleColorCalcType color0Type, ParticleColorCalcType color1Type );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルアルファ計算タイプを設定します。
    //! @param[in] alpha0Type TBD
    //! @param[in] alpha1Type TBD
    //---------------------------------------------------------------------------
    void SetParticleAlphaCalcType( ParticleColorCalcType alpha0Type, ParticleColorCalcType alpha1Type );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルテクスチャパターンアニメ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleTexturePatternAnim( bool texture0, bool texture1, bool texture2 );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルテクスチャ座標アニメ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleTextureShiftAnim( bool texture0, bool texture1, bool texture2 );

    //---------------------------------------------------------------------------
    //! @brief  パーティクルスフィアマップ有効無効を設定します。
    //! @param[in] texture0 TBD
    //! @param[in] texture1 TBD
    //! @param[in] texture2 TBD
    //---------------------------------------------------------------------------
    void SetParticleSphereMap( bool texture0, bool texture1, bool texture2 );

    //---------------------------------------------------------------------------
    //! @brief  フラグメントシェーダを標準設定にします。
    //---------------------------------------------------------------------------
    void SetDefaultFragmentShaderSetting();

    //---------------------------------------------------------------------------
    //! @brief  color/alpha 8keyアニメを設定します。
    //! @param[in] color0Anim TBD
    //! @param[in] color1Anim TBD
    //---------------------------------------------------------------------------
    void SetColor8KeyAnim( u8 color0Anim, u8 color1Anim );

    //---------------------------------------------------------------------------
    //! @brief 揺らぎを設定します。
    //! @param[in] fluctuation TBD
    //---------------------------------------------------------------------------
    void SetFluctuation( u32 fluctuation );

    //---------------------------------------------------------------------------
    //! @brief 全てのパーティクルタイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllParticleType();

    //---------------------------------------------------------------------------
    //! @brief 全ての追従タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllFollowType();

    //---------------------------------------------------------------------------
    //! @brief 全ての挙動計算タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllCalcType();

    //---------------------------------------------------------------------------
    //! @brief 全ての挙動計算タイプをクリアする。
    //---------------------------------------------------------------------------
    void ClearAllFluctuation();


private:
    //---------------------------------------------------------------------------
    //! @brief  コンパイルセッティングを生成します。
    //! @param[in] isCafe TBD
    //---------------------------------------------------------------------------
    void Generate();

    Flag64      m_VtxFlag;                                                  //!< 頂点シェーダフラグ
    Flag64      m_VtxFlag2;                                                 //!< 頂点シェーダフラグ（2本目）
    Flag64      m_FragFlag;                                                 //!< フラグメントシェーダフラグ
    Flag64      m_FragFlag2;                                                //!< フラグメントシェーダフラグ（2本目）
    Flag64      m_VtxFlagEP;                                                //!< Emitter Plugin 頂点シェーダフラグ
    Flag64      m_FragFlagEP;                                               //!< Emitter Plugin フラグメントシェーダフラグ
    u64         m_CustomShaderFlag;                                         //!< カスタムシェーダフラグ
    u64         m_CustomShaderSwitch;                                       //!< カスタムシェーダスイッチ
    u32         m_DrawPath;                                                 //!< 描画パス
    u32         m_CustomShaderIndex;                                        //!< カスタムシェーダインデックス
    u32         m_CustomFieldFlag;                                          //!< カスタムフィールドフラグ
    u32         m_CombinerShaderIndex;                                      //!< コンバイナシェーダインデックス
    char        m_CustomShaderDefine[16];                                   //!< カスタムシェーダユーザー定義
    u8          m_CustomFieldEnabled;                                       //!< カスタムフィールドの有無
    u8          m_VtxColor0AnimKey;                                         //!< カラー0アニメ
    u8          m_VtxColor1AnimKey;                                         //!< カラー1アニメ
    u8          m_VtxAlpha0AnimKey;                                         //!< アルファ0アニメ
    u8          m_VtxAlpha1AnimKey;                                         //!< アルファ1アニメ
    u8          m_VtxScaleAnimKey;                                          //!< スケールアニメ
    enum { EFT_SHADER_COMPILE_SETTING_BUFFER_SIZE = 8192 };                 //<! シェーダコンパイル定義用バッファサイズ
    static char g_CompileSetting[EFT_SHADER_COMPILE_SETTING_BUFFER_SIZE];   //<! シェーダコンパイル定義用バッファ
};

#endif

} // namespace nw
} // namespace eft2
