﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/eft/eft2_EmitterRes.h>

namespace nw    {
namespace eft2  {

//---------------------------------------------------
//! @brief  シェーダフラグ。
//---------------------------------------------------
struct ShaderFlag
{
    //---------------------------------------------------------------------------
    //! @brief  無効化
    //---------------------------------------------------------------------------
    void Invalidate() { m_Flag0  = 0; m_Flag1  = 0; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダフラグ0を取得します。
    //! @return シェーダフラグ0
    //---------------------------------------------------------------------------
    u32 GetFlag0() const { return m_Flag0; }

    //---------------------------------------------------------------------------
    //! @brief  シェーダフラグ1を取得します。
    //! @return シェーダフラグ1
    //---------------------------------------------------------------------------
    u32 GetFlag1() const { return m_Flag1; }

    #define BIT_FLAG_FLUCTUATION_TYPE_SIN           ( 0x01 << 0 )       //!< TBD
    #define BIT_FLAG_FLUCTUATION_TYPE_SAW_TOOTH     ( 0x01 << 1 )       //!< TBD
    #define BIT_FLAG_FLUCTUATION_TYPE_RECT          ( 0x01 << 2 )       //!< TBD
    #define EFT_WORLD_GRAVITY_ENABLED               ( 0x01 << 3 )       //!< TBD
    #define BIT_FLAG_TEX0_PTN_LIFE_FIT              ( 0x01 << 4 )       //!< TBD
    #define BIT_FLAG_TEX0_PTN_LIFE_CLAMP            ( 0x01 << 5 )       //!< TBD
    #define BIT_FLAG_TEX0_PTN_LIFE_LOOP             ( 0x01 << 6 )       //!< TBD
    #define BIT_FLAG_TEX0_PTN_LIFE_RANDOM           ( 0x01 << 7 )       //!< TBD
    #define BIT_FLAG_TEX1_PTN_LIFE_FIT              ( 0x01 << 8 )       //!< TBD
    #define BIT_FLAG_TEX1_PTN_LIFE_CLAMP            ( 0x01 << 9 )       //!< TBD
    #define BIT_FLAG_TEX1_PTN_LIFE_LOOP             ( 0x01 << 10 )      //!< TBD
    #define BIT_FLAG_TEX1_PTN_LIFE_RANDOM           ( 0x01 << 11 )      //!< TBD
    #define BIT_FLAG_TEX2_PTN_LIFE_FIT              ( 0x01 << 12 )      //!< TBD
    #define BIT_FLAG_TEX2_PTN_LIFE_CLAMP            ( 0x01 << 13 )      //!< TBD
    #define BIT_FLAG_TEX2_PTN_LIFE_LOOP             ( 0x01 << 14 )      //!< TBD
    #define BIT_FLAG_TEX2_PTN_LIFE_RANDOM           ( 0x01 << 15 )      //!< TBD
    #define BIT_FLAG_ROTATE_DIR_RANDOM_X            ( 0x01 << 16 )      //!< TBD
    #define BIT_FLAG_ROTATE_DIR_RANDOM_Y            ( 0x01 << 17 )      //!< TBD
    #define BIT_FLAG_ROTATE_DIR_RANDOM_Z            ( 0x01 << 18 )      //!< TBD
    #define BIT_FLAG_TEX0_INV_RANDOM_U              ( 0x01 << 19 )      //!< TBD
    #define BIT_FLAG_TEX0_INV_RANDOM_V              ( 0x01 << 20 )      //!< TBD
    #define BIT_FLAG_TEX1_INV_RANDOM_U              ( 0x01 << 21 )      //!< TBD
    #define BIT_FLAG_TEX1_INV_RANDOM_V              ( 0x01 << 22 )      //!< TBD
    #define BIT_FLAG_TEX2_INV_RANDOM_U              ( 0x01 << 23 )      //!< TBD
    #define BIT_FLAG_TEX2_INV_RANDOM_V              ( 0x01 << 24 )      //!< TBD
    #define BIT_FLAG_TEX0_PTN_LOOP_RANDOM           ( 0x01 << 25 )      //!< TBD
    #define BIT_FLAG_TEX1_PTN_LOOP_RANDOM           ( 0x01 << 26 )      //!< TBD
    #define BIT_FLAG_TEX2_PTN_LOOP_RANDOM           ( 0x01 << 27 )      //!< TBD
    #define BIT_FLAG_PRIMITIVE_SCALE_Y_TO_Z         ( 0x01 << 28 )      //!< TBD
    #define BIT_FLAG_OFFSET_TYPE_DEPTH              ( 0x01 << 29 )      //!< TBD
    #define BIT_FLAG_OFFSET_TYPE_CAMERA             ( 0x01 << 30 )      //!< TBD

    // フラグ[1]
    #define BIT_FLAG_TEX_ANIM_FIXED_RANDOM          ( 0x01 << 0 )       //!< TBD
    #define BIT_FLAG_FIELD_RANDOM                   ( 0x01 << 1 )       //!< TBD
    #define BIT_FLAG_FIELD_POSADD                   ( 0x01 << 2 )       //!< TBD
    #define BIT_FLAG_FIELD_MAGNET                   ( 0x01 << 3 )       //!< TBD
    #define BIT_FLAG_FIELD_CONVERGENCE              ( 0x01 << 4 )       //!< TBD
    #define BIT_FLAG_FIELD_SPIN                     ( 0x01 << 5 )       //!< TBD
    #define BIT_FLAG_FIELD_COLLISION                ( 0x01 << 6 )       //!< TBD
    #define BIT_FLAG_FIELD_CURLNOISE                ( 0x01 << 7 )       //!< TBD
    #define BIT_FLAG_FIELD_RANDOM_FE1               ( 0x01 << 8 )       //!< TBD

    #define BIT_FLAG_EMITTER_FOLLOW_ALL             ( 0x01 << 9 )       //!< TBD
    #define BIT_FLAG_EMITTER_FOLLOW_POS             ( 0x01 << 10 )      //!< TBD
    #define BIT_FLAG_EMITTER_FOLLOW_NONE            ( 0x01 << 11 )      //!< TBD


    //---------------------------------------------------------------------------
    //! @brief  初期化
    //! @param[in] emitterRes EmitterResourceへのポインタ
    //---------------------------------------------------------------------------
    void Initialize( EmitterResource* emitterRes )
    {
        ResEmitter* emitterData = emitterRes->emitterData;

        // 重力世界方向
        if ( emitterData->emission.isWorldGravity ) m_Flag0 |= EFT_WORLD_GRAVITY_ENABLED;

        // 波形タイプ
        FluctuationWaveType waveType = emitterData->ptclFluctuation.isWaveType >> 4;
        if ( waveType == EFT_FLUCTUATION_WAVE_TYPE_SIN )        m_Flag0 |= BIT_FLAG_FLUCTUATION_TYPE_SIN;
        if ( waveType == EFT_FLUCTUATION_WAVE_TYPE_SAW_TOOTH )  m_Flag0 |= BIT_FLAG_FLUCTUATION_TYPE_SAW_TOOTH;
        if ( waveType == EFT_FLUCTUATION_WAVE_TYPE_RECT )       m_Flag0 |= BIT_FLAG_FLUCTUATION_TYPE_RECT;

        // テクスチャ パタンアニメタイプ
        if ( emitterData->textureAnim0.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LIFE_FIT ) m_Flag0 |= BIT_FLAG_TEX0_PTN_LIFE_FIT;
        if ( emitterData->textureAnim0.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_CLAMP )    m_Flag0 |= BIT_FLAG_TEX0_PTN_LIFE_CLAMP;
        if ( emitterData->textureAnim0.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LOOP )     m_Flag0 |= BIT_FLAG_TEX0_PTN_LIFE_LOOP;
        if ( emitterData->textureAnim0.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_RANDOM )
        {
            for ( u32 i = 0; i < emitterData->staticUbo.texPtnAnim0.ptnNum; i++ )
            {
                emitterData->staticUbo.texPtnAnim0.ptnTable[i] = i;
            }
            emitterData->staticUbo.texPtnAnim0.ptnTableNum =
                emitterData->staticUbo.texPtnAnim0.ptnNum;
            m_Flag0 |= BIT_FLAG_TEX0_PTN_LIFE_RANDOM;
        }

        if ( emitterData->textureAnim1.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LIFE_FIT ) m_Flag0 |= BIT_FLAG_TEX1_PTN_LIFE_FIT;
        if ( emitterData->textureAnim1.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_CLAMP )    m_Flag0 |= BIT_FLAG_TEX1_PTN_LIFE_CLAMP;
        if ( emitterData->textureAnim1.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LOOP )     m_Flag0 |= BIT_FLAG_TEX1_PTN_LIFE_LOOP;
        if ( emitterData->textureAnim1.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_RANDOM )
        {
            for ( u32 i = 0; i < emitterData->staticUbo.texPtnAnim1.ptnNum; i++ )
            {
                emitterData->staticUbo.texPtnAnim1.ptnTable[i] = i;
            }
            emitterData->staticUbo.texPtnAnim1.ptnTableNum =
                emitterData->staticUbo.texPtnAnim1.ptnNum;
            m_Flag0 |= BIT_FLAG_TEX1_PTN_LIFE_RANDOM;
        }

        if ( emitterData->textureAnim2.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LIFE_FIT ) m_Flag0 |= BIT_FLAG_TEX2_PTN_LIFE_FIT;
        if ( emitterData->textureAnim2.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_CLAMP )    m_Flag0 |= BIT_FLAG_TEX2_PTN_LIFE_CLAMP;
        if ( emitterData->textureAnim2.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_LOOP )     m_Flag0 |= BIT_FLAG_TEX2_PTN_LIFE_LOOP;
        if ( emitterData->textureAnim2.patternAnimType == EFT_TEXTURE_PTN_ANIM_TYPE_RANDOM )
        {
            for ( u32 i = 0; i < emitterData->staticUbo.texPtnAnim2.ptnNum; i++ )
            {
                emitterData->staticUbo.texPtnAnim2.ptnTable[i] = i;
            }
            emitterData->staticUbo.texPtnAnim2.ptnTableNum =
                emitterData->staticUbo.texPtnAnim2.ptnNum;
            m_Flag0 |= BIT_FLAG_TEX2_PTN_LIFE_RANDOM;
        }

        // 回転方向ランダム
        if ( emitterData->ptcl.rotRevRandX ) m_Flag0 |=  BIT_FLAG_ROTATE_DIR_RANDOM_X;
        if ( emitterData->ptcl.rotRevRandY ) m_Flag0 |=  BIT_FLAG_ROTATE_DIR_RANDOM_Y;
        if ( emitterData->ptcl.rotRevRandZ ) m_Flag0 |=  BIT_FLAG_ROTATE_DIR_RANDOM_Z;

        // テクスチャ反転ランダム
        if ( emitterData->textureAnim0.invRandU ) m_Flag0 |=  BIT_FLAG_TEX0_INV_RANDOM_U;
        if ( emitterData->textureAnim0.invRandV ) m_Flag0 |=  BIT_FLAG_TEX0_INV_RANDOM_V;
        if ( emitterData->textureAnim1.invRandU ) m_Flag0 |=  BIT_FLAG_TEX1_INV_RANDOM_U;
        if ( emitterData->textureAnim1.invRandV ) m_Flag0 |=  BIT_FLAG_TEX1_INV_RANDOM_V;
        if ( emitterData->textureAnim2.invRandU ) m_Flag0 |=  BIT_FLAG_TEX2_INV_RANDOM_U;
        if ( emitterData->textureAnim2.invRandV ) m_Flag0 |=  BIT_FLAG_TEX2_INV_RANDOM_V;

        // テクスチャパタン　ループランダム
        if ( emitterData->textureAnim0.isPatAnimLoopRandom ) m_Flag0 |=  BIT_FLAG_TEX0_PTN_LOOP_RANDOM;
        if ( emitterData->textureAnim1.isPatAnimLoopRandom ) m_Flag0 |=  BIT_FLAG_TEX1_PTN_LOOP_RANDOM;
        if ( emitterData->textureAnim2.isPatAnimLoopRandom ) m_Flag0 |=  BIT_FLAG_TEX2_PTN_LOOP_RANDOM;

        // プリミティブスケール適用
        if ( emitterData->ptcl.primitiveScaleType ) m_Flag0 |= BIT_FLAG_PRIMITIVE_SCALE_Y_TO_Z;

        // オフセットタイプ
        if ( emitterData->ptcl.offsetType == EFT_PARTICLE_OFFSET_TYPE_CAMERA ) m_Flag0 |= BIT_FLAG_OFFSET_TYPE_CAMERA;
        if ( emitterData->ptcl.offsetType == EFT_PARTICLE_OFFSET_TYPE_DEPTH )  m_Flag0 |= BIT_FLAG_OFFSET_TYPE_DEPTH;

        // flag1
        // テクスチャ共通ランダム
        if ( emitterData->ptcl.isTextureCommonRandom ) m_Flag1 |= BIT_FLAG_TEX_ANIM_FIXED_RANDOM;

        // フィールド
        if ( emitterRes->fieldRandomData )          m_Flag1 |= BIT_FLAG_FIELD_RANDOM;
        if ( emitterRes->fieldRandomDataFe1 )       m_Flag1 |= BIT_FLAG_FIELD_RANDOM_FE1;
        if ( emitterRes->fieldPosAddData )          m_Flag1 |= BIT_FLAG_FIELD_POSADD;
        if ( emitterRes->fieldMagnetData )          m_Flag1 |= BIT_FLAG_FIELD_MAGNET;
        if ( emitterRes->fieldConvergenceData )     m_Flag1 |= BIT_FLAG_FIELD_CONVERGENCE;
        if ( emitterRes->fieldSpinData )            m_Flag1 |= BIT_FLAG_FIELD_SPIN;
        if ( emitterRes->fieldCollisionData )       m_Flag1 |= BIT_FLAG_FIELD_COLLISION;
        if ( emitterRes->fieldCurlNoiseData )       m_Flag1 |= BIT_FLAG_FIELD_CURLNOISE;

        // エミッタ追従
        switch( emitterData->emitter.followType )
        {
        case EFT_EMITTER_FOLLOW_TYPE_ALL:   m_Flag1 |= BIT_FLAG_EMITTER_FOLLOW_ALL;
            break;
        case EFT_EMITTER_FOLLOW_TYPE_POS:   m_Flag1 |= BIT_FLAG_EMITTER_FOLLOW_POS;
            break;
        case EFT_EMITTER_FOLLOW_TYPE_NONE:  m_Flag1 |= BIT_FLAG_EMITTER_FOLLOW_NONE;
            break;
        }
    }

private:
    u32     m_Flag0;    //!< TBD
    u32     m_Flag1;    //!< TBD
};

} // namespace eft2
} // namespace nw
