﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

//--------------------------------------------------------------------------------------
// Includes
//--------------------------------------------------------------------------------------
#include <nw/config.h>
#include <nw/eft/typeDef2.h>
#ifndef _EFT_TOOL
#include <math.h>
#endif

namespace nw   {
namespace eft2 {

enum EndianDefinition
{
    Little = 0,
    Big = 1,
};

//--------------------------------------------------------------------------------------
//! @brief      エンディアンユーティリティークラスです.
//--------------------------------------------------------------------------------------
class EndianUtil
{
public:
    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @tparam T                   フリップする値の型
    //! @param [in,out]      ptr    フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    template< typename T >
    static void Flip( T* ptr )
    {
#ifdef  EFT_ENDIAN_LITTLE
        T orig = (*ptr);
        char *src = reinterpret_cast<char*>( &orig );
        char *dst = reinterpret_cast<char*>( ptr );

        for( int i=0 ; i<sizeof(T); i++ )
        {
            dst[i] = src[ sizeof(T) - 1 - i ];
        }
#else
        (void)(&ptr);
#endif
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in,out]      ptr     フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void Flip( nw::math::VEC2* ptr )
    {
        Flip( &ptr->x );
        Flip( &ptr->y );
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in,out]      ptr     フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void Flip( nw::math::VEC3* ptr )
    {
        Flip( &ptr->x );
        Flip( &ptr->y );
        Flip( &ptr->z );
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in,out]      ptr     フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void Flip( nw::math::VEC4* ptr )
    {
        Flip( &ptr->x );
        Flip( &ptr->y );
        Flip( &ptr->z );
        Flip( &ptr->w );
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in,out]      ptr     フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void Flip( nw::math::MTX34* ptr )
    {
        Flip( &ptr->m[0][0] );
        Flip( &ptr->m[0][1] );
        Flip( &ptr->m[0][2] );
        Flip( &ptr->m[0][3] );

        Flip( &ptr->m[1][0] );
        Flip( &ptr->m[1][1] );
        Flip( &ptr->m[1][2] );
        Flip( &ptr->m[1][3] );

        Flip( &ptr->m[2][0] );
        Flip( &ptr->m[2][1] );
        Flip( &ptr->m[2][2] );
        Flip( &ptr->m[2][3] );
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in,out]      ptr     フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void Flip( nw::math::MTX44* ptr )
    {
        Flip( &ptr->m[0][0] );
        Flip( &ptr->m[0][1] );
        Flip( &ptr->m[0][2] );
        Flip( &ptr->m[0][3] );

        Flip( &ptr->m[1][0] );
        Flip( &ptr->m[1][1] );
        Flip( &ptr->m[1][2] );
        Flip( &ptr->m[1][3] );

        Flip( &ptr->m[2][0] );
        Flip( &ptr->m[2][1] );
        Flip( &ptr->m[2][2] );
        Flip( &ptr->m[2][3] );

        Flip( &ptr->m[3][0] );
        Flip( &ptr->m[3][1] );
        Flip( &ptr->m[3][2] );
        Flip( &ptr->m[3][3] );
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @tparam T                           フリップする値の型
    //! @param [in]          arraySize      配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    template< typename T >
    static void FlipArray( const u32 arraySize, T* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, nw::math::VEC2* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, nw::math::VEC3* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, nw::math::VEC4* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, nw::math::MTX34* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray         フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, nw::math::MTX44* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            Flip( &pArray[i] );
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //!
    //! @param [in]          arraySize     配列サイズ.
    //! @param [in,out]      pArray        フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void FlipArray( const u32 arraySize, u8* pArray )
    {
        u32* pTmpArray = reinterpret_cast<u32*>( pArray );
        u32    tmpSize = arraySize / 4;
        for( u32 i=0; i<tmpSize; ++i )
        {
            Flip( &pTmpArray[i] );
        }
    }

public:
    //----------------------------------------------------------------------------------
    //! @brief          エンディアン設定に関係なく、エンディアンをフリップします.
    //!
    //! @tparam         T           フリップする値の型
    //! @param [in,out] ptr         フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    template< typename T >
    static void ForceFlip( T* ptr )
    {
        T orig = (*ptr);
        char *src = reinterpret_cast<char*>( &orig );
        char *dst = reinterpret_cast<char*>( ptr );

        for( int i=0 ; i<sizeof(T); i++ )
        {
            dst[i] = src[ sizeof(T) - 1 - i ];
        }
    }

    //----------------------------------------------------------------------------------
    //! @brief          エンディアン設定に関係なく、エンディアンをフリップします.
    //!
    //! @param [in,out] ptr         フリップする値のポインタ.
    //----------------------------------------------------------------------------------
    static void ForceFlip( nw::math::VEC4* ptr )
    {
        ForceFlip( &ptr->x );
        ForceFlip( &ptr->y );
        ForceFlip( &ptr->z );
        ForceFlip( &ptr->w );
    }

    //----------------------------------------------------------------------------------
    //! @brief          エンディアン設定に関係なく、エンディアンをフリップします.
    //!
    //! @param [in]     arraySize   配列サイズ.
    //! @param [in,out] pArray      フリップする配列の先頭ポインタ.
    //----------------------------------------------------------------------------------
    static void ForceFlipArray( const u32 arraySize, nw::math::VEC4* pArray )
    {
        for( u32 i=0; i<arraySize; ++i )
        {
            ForceFlip( &pArray[i] );
        }
    }
};

} // namespace eft2
} // namespace nw
