﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once



#include <nw/eft/eft2_Enum.h>
#ifndef _EFT_TOOL
#include <nw/math.h>
#endif//_EFT_TOOL

#include <nw/eft/eft2_EndianUtil.h>

namespace nw   {
namespace eft2 {

//------------------------------------------------------------------------------
//! @brief      リソースヘッダ
//------------------------------------------------------------------------------
struct ResHeader
{
    u32                                               tag;                                                      //!< タグです
    u32                                               version;                                                  //!< バイナリバージョンです
    u8                                                name[32];                                                 //!< バイナリ名です
    u8                                                cpuEndian;                                                //!< CPUのエンディアンです(0:Little/1:Big)
    u8                                                gpuEndian;                                                //!< GPUのエンディアンです(0:Little/1:Big)
    u8                                                targetApi;                                                //!< ターゲットAPIを表す定数値です
    u8                                                targetAddressSize;                                        //!< ターゲットアドレスサイズ(sizeof(void*)で比較可能な値)です
    u8                                                extraTag[4];                                              //!< 拡張タグです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &tag );
        nw::eft2::EndianUtil::Flip( &version );

    }
};

//------------------------------------------------------------------------------
//! @brief      共通データ
//------------------------------------------------------------------------------
struct ResCommon
{
    u32                                               flag;                                                     //!< 汎用フラグ
    u32                                               randomSeed;                                               //!< 乱数シード
    u32                                               dummy[2];                                                 //!< シェーダ用アライメント
    char                                              name[64];                                                 //!< アドレス解決後の名前

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &flag );
        nw::eft2::EndianUtil::Flip( &randomSeed );
        EndianUtil::FlipArray( 2, dummy );

    }
};

//------------------------------------------------------------------------------
//! @brief      描画設定データ
//------------------------------------------------------------------------------
struct ResRenderState
{
    u8                                                isBlendEnable;                                            //!< ブレンド
    u8                                                isDepthTest;                                              //!< デプステスト
    u8                                                depthFunc;                                                //!< デプステスト通過条件
    u8                                                isDepthMask;                                              //!< デプスマスク
    u8                                                isAlphaTest;                                              //!< アルファテスト
    u8                                                alphaFunc;                                                //!< アルファテスト通過条件
    u8                                                blendType;                                                //!< フレームバッファとのブレンドタイプ
    u8                                                displaySide;                                              //!< 表示面
    f32                                               alphaThreshold;                                           //!< アルファ閾値
    u8                                                dummy[4];                                                 //!< アライメント

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &alphaThreshold );

    }
};

//------------------------------------------------------------------------------
//! @brief      テクスチャデータ
//------------------------------------------------------------------------------
struct ResTexture
{
    u16                                               width;                                                    //!< 幅
    u16                                               height;                                                   //!< 高さ
    u32                                               depth;                                                    //!< 配列数
    u32                                               compSel;                                                  //!< コンポーネントスワップ
    u32                                               mipLevel;                                                 //!< ミップレベル数
    u32                                               nativeFormat;                                             //!< 機種固有フォーマット
    u32                                               nativeTileMode;                                           //!< 機種固有タイルモード
    u32                                               nativeSwizzle;                                            //!< 機種固有スウィズル
    u32                                               swizzle;                                                  //!< スウィズル
    u64                                               guid;                                                     //!< テクスチャのGUID
    u8                                                format;                                                   //!< フォーマット
    u8                                                dummy[7];                                                 //!< ダミー

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &width );
        nw::eft2::EndianUtil::Flip( &height );
        nw::eft2::EndianUtil::Flip( &depth );
        nw::eft2::EndianUtil::Flip( &compSel );
        nw::eft2::EndianUtil::Flip( &mipLevel );
        nw::eft2::EndianUtil::Flip( &nativeFormat );
        nw::eft2::EndianUtil::Flip( &nativeTileMode );
        nw::eft2::EndianUtil::Flip( &nativeSwizzle );
        nw::eft2::EndianUtil::Flip( &swizzle );
        nw::eft2::EndianUtil::Flip( &guid );

    }
};

//------------------------------------------------------------------------------
//! @brief      テクスチャサンプラデータ
//------------------------------------------------------------------------------
struct ResTextureSampler
{
    u64                                               guid;                                                     //!< 利用するテクスチャのguid
    u8                                                wrapU;                                                    //!< ラップモードU
    u8                                                wrapV;                                                    //!< ラップモードV
    u8                                                filter;                                                   //!< フィルタモード
    u8                                                isSphereMap;                                              //!< スフィアマップを利用するかどうか
    f32                                               mipLevel;                                                 //!< ミップレベルの有効値( 0.0 ～ 15.99 )
    f32                                               mipMapBias;                                               //!< ミップレベルのバイアス
    u8                                                mipLevelLimit;                                            //!< ミップマップレベルを制限する
    u8                                                isDensityFixedU;                                          //!< テクスチャ密度固定オプションU
    u8                                                isDensityFixedV;                                          //!< テクスチャ密度固定オプションV
    u8                                                isSquareRgb;                                              //!< テクスチャRGB値を二乗して取得するかどうか（リニアの近似処理）
    u8                                                isOnAnotherBinary;                                        //!< サブバイナリ側で省略されたテクスチャであることを示す
    u8                                                dummy[7];                                                 //!< ダミー

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &guid );
        nw::eft2::EndianUtil::Flip( &mipLevel );
        nw::eft2::EndianUtil::Flip( &mipMapBias );

    }
};

//------------------------------------------------------------------------------
//! @brief      8キーアニメリソース
//------------------------------------------------------------------------------
struct ResAnim8KeyParam
{
    nw::math::VEC4                                    value[8];                                                 //!< 値( x, y, z, time )

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        EndianUtil::FlipArray( 8, value );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタリソース
//------------------------------------------------------------------------------
struct ResEmitterSet : public ResCommon
{
    u32                                               emitterAllNum;                                            //!< 総エミッタ数(チャイルドを含める).
    u16                                               UserDataBit;                                              //!< ユーザーデータビット.
    u16                                               dummy;                                                    //!< パディング領域
    u32                                               UserDataValue1;                                           //!< ユーザーデータ値1.
    u32                                               UserDataValue2;                                           //!< ユーザーデータ値2.

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &emitterAllNum );
        nw::eft2::EndianUtil::Flip( &UserDataBit );
        nw::eft2::EndianUtil::Flip( &dummy );
        nw::eft2::EndianUtil::Flip( &UserDataValue1 );
        nw::eft2::EndianUtil::Flip( &UserDataValue2 );

    }
};

//------------------------------------------------------------------------------
//! @brief      テクスチャ座標アニメ
//------------------------------------------------------------------------------
struct ResTexUvShiftAnim
{
    nw::math::VEC2                                    scrollAdd;                                                //!< スクロール加算値
    nw::math::VEC2                                    scrollInit;                                               //!< スクロール初期値
    nw::math::VEC2                                    scrollInitRand;                                           //!< スクロール初期値ランダム
    nw::math::VEC2                                    scaleAdd;                                                 //!< スケール加算値
    nw::math::VEC2                                    scaleInit;                                                //!< スケール初期値
    nw::math::VEC2                                    scaleInitRand;                                            //!< スケール初期値ランダム
    f32                                               rotAdd;                                                   //!< 回転加算値
    f32                                               rotInit;                                                  //!< 回転初期値
    f32                                               rotInitRand;                                              //!< 回転初期値ランダム
    f32                                               randomType;                                               //!< 乱数タイプ
    nw::math::VEC2                                    uvScale;                                                  //!< UVスケール値
    nw::math::VEC2                                    uvDiv;                                                    //!< 縦横分割数

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &scrollAdd );
        nw::eft2::EndianUtil::Flip( &scrollInit );
        nw::eft2::EndianUtil::Flip( &scrollInitRand );
        nw::eft2::EndianUtil::Flip( &scaleAdd );
        nw::eft2::EndianUtil::Flip( &scaleInit );
        nw::eft2::EndianUtil::Flip( &scaleInitRand );
        nw::eft2::EndianUtil::Flip( &rotAdd );
        nw::eft2::EndianUtil::Flip( &rotInit );
        nw::eft2::EndianUtil::Flip( &rotInitRand );
        nw::eft2::EndianUtil::Flip( &randomType );
        nw::eft2::EndianUtil::Flip( &uvScale );
        nw::eft2::EndianUtil::Flip( &uvDiv );

    }
};

//------------------------------------------------------------------------------
//! @brief      テクスチャパターンアニメ
//------------------------------------------------------------------------------
struct ResTexPtnAnim
{
    f32                                               ptnTableNum;                                              //!< 利用するパターンテーブル数
    f32                                               ptnFreq;                                                  //!< パターン周期
    f32                                               ptnNum;                                                   //!< ランダム設定時のパターン数
    f32                                               dummy;                                                    //!< ダミー
    s32                                               ptnTable[EFT_TEXTURE_PATTERN_NUM];    //!< パターンテーブル

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &ptnTableNum );
        nw::eft2::EndianUtil::Flip( &ptnFreq );
        nw::eft2::EndianUtil::Flip( &ptnNum );
        nw::eft2::EndianUtil::Flip( &dummy );
        EndianUtil::FlipArray( EFT_TEXTURE_PATTERN_NUM, ptnTable );

    }
};

//------------------------------------------------------------------------------
//! @brief      揺らぎ
//------------------------------------------------------------------------------
struct FluctuationData
{
    nw::math::VEC2                                    amplitude;                                                //!< 振幅
    nw::math::VEC2                                    cycle;                                                    //!< 周期
    nw::math::VEC2                                    phaseRnd;                                                 //!< 位相ランダム
    nw::math::VEC2                                    phaseInit;                                                //!< 初期位相

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &amplitude );
        nw::eft2::EndianUtil::Flip( &cycle );
        nw::eft2::EndianUtil::Flip( &phaseRnd );
        nw::eft2::EndianUtil::Flip( &phaseInit );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタ 静的ユニフォームブロック
//------------------------------------------------------------------------------
struct EmitterStaticUniformBlock
{
    s32                                               flag[4];                                                  //!< 各種フラグ
    s32                                               color0AnimKeynum;                                         //!< カラー０アニメキー数
    s32                                               alpha0AnimKeyNum;                                         //!< アルファ0アニメキー数
    s32                                               color1AnimKeynum;                                         //!< カラー１アニメキー数
    s32                                               alpha1AnimKeyNum;                                         //!< アルファ1アニメキー数
    s32                                               scaleAnimKeyNum;                                          //!< スケールアニメキー数
    s32                                               shaderAnimKeyNum;                                         //!< シェーダ係数アニメキー数
    s32                                               dummy0[2];                                                //!< パディング領域
    f32                                               color0AnimLoopRate;                                       //!< color0のアニメのループフレーム比率（寿命に対する一周期の％）
    f32                                               alpha0AnimLoopRate;                                       //!< alpha0のアニメのループフレーム比率（寿命に対する一周期の％）
    f32                                               color1AnimLoopRate;                                       //!< color1のアニメのループフレーム比率（寿命に対する一周期の％）
    f32                                               alpha1AnimLoopRate;                                       //!< alpha1のアニメのループフレーム比率（寿命に対する一周期の％）
    f32                                               scaleAnimLoopRate;                                        //!< scaleのアニメのループフレーム比率（寿命に対する一周期の％）
    f32                                               color0AnimIsLoopInitRandom;                               //!< color0のアニメの初期位置ランダムの有効／無効フラグ
    f32                                               alpha0AnimIsLoopInitRandom;                               //!< alpha0のアニメの初期位置ランダムの有効／無効フラグ
    f32                                               color1AnimIsLoopInitRandom;                               //!< color1のアニメの初期位置ランダムの有効／無効フラグ
    f32                                               alpha1AnimIsLoopInitRandom;                               //!< alpha1のアニメの初期位置ランダムの有効／無効フラグ
    f32                                               scaleAnimIsLoopInitRandom;                                //!< scaleのアニメの初期位置ランダムの有効／無効フラグ
    f32                                               dummy6[2];                                                //!< パディング領域
    nw::math::VEC3                                    gravity;                                                  //!< 重力( x / y / z )
    f32                                               gravityScale;                                             //!< 重力スケール
    f32                                               airRegist;                                                //!< 重力スケール
    f32                                               dummy1[3];                                                //!< パディング領域
    nw::math::VEC2                                    center;                                                   //!< パーティクル中心
    f32                                               offset;                                                   //!< パーティクルオフセット
    f32                                               dummy2;                                                   //!< パディング領域
    nw::eft2::FluctuationData                  fluctuationData;                                          //!< 揺らぎデータ
    f32                                               coefficient0;                                             //!< シェーダ 係数0
    f32                                               coefficient1;                                             //!< シェーダ 係数1
    f32                                               dummy3[2];                                                //!< パディング領域
    nw::eft2::ResTexPtnAnim                    texPtnAnim0;                                              //!< テクスチャ0パターンアニメ
    nw::eft2::ResTexPtnAnim                    texPtnAnim1;                                              //!< テクスチャ1パターンアニメ
    nw::eft2::ResTexPtnAnim                    texPtnAnim2;                                              //!< テクスチャ2パターンアニメ
    nw::eft2::ResTexUvShiftAnim                texShiftAnim0;                                            //!< テクスチャ0座標アニメ
    nw::eft2::ResTexUvShiftAnim                texShiftAnim1;                                            //!< テクスチャ1座標アニメ
    nw::eft2::ResTexUvShiftAnim                texShiftAnim2;                                            //!< テクスチャ2座標アニメ
    f32                                               colorScale;                                               //!< カラースケール
    f32                                               dummy4[3];                                                //!< パディング領域
    nw::eft2::ResAnim8KeyParam                 color0Anim;                                               //!< カラー8キーアニメ
    nw::eft2::ResAnim8KeyParam                 alpha0Anim;                                               //!< アルファ8キーアニメ
    nw::eft2::ResAnim8KeyParam                 color1Anim;                                               //!< カラー１アニメパラメータ
    nw::eft2::ResAnim8KeyParam                 alpha1Anim;                                               //!< アルファ１アニメパラメータ
    nw::math::VEC2                                    softEdgeParam;                                            //!< ソフトパーティクル
    nw::math::VEC2                                    fresnelAlphaParam;                                        //!< フレネルアルファ
    nw::math::VEC2                                    nearDistAlphaParam;                                       //!< 近距離アルファ
    nw::math::VEC2                                    farDistAlphaParam;                                        //!< 遠距離アルファ
    nw::math::VEC2                                    decalParam;                                               //!< デカール
    f32                                               alphaThreshold;                                           //!< アルファテストのしきい値です。
    f32                                               dummy7;                                                   //!< パディング領域
    f32                                               addVelToScale;                                            //!< 速度をスケールに加算
    f32                                               softPartcileDist;                                         //!< フェード開始距離
    f32                                               softParticleVolume;                                       //!< ソフトパーティクルボリューム値
    f32                                               dummy5;                                                   //!< パディング領域
    nw::eft2::ResAnim8KeyParam                 scale8keyAnim;                                            //!< スケール8キーアニメ
    nw::eft2::ResAnim8KeyParam                 shaderAnim;                                               //!< シェーダ係数アニメ
    nw::math::VEC4                                    rotateInit;                                               //!< 初期回転値( xyz / empty )
    nw::math::VEC4                                    rotateInitRand;                                           //!< 初期ランダム回転( xyz / empty)
    nw::math::VEC3                                    rotateAdd;                                                //!< 回転速度
    f32                                               rotateRegist;                                             //!< 回転減衰率
    nw::math::VEC3                                    rotateAddRand;                                            //!< 回転速度ランダム( xyz / empty )
    f32                                               dummy8;                                                   //!< パディング領域
    f32                                               scaleLimitDistNear;                                       //!< カメラ前でのスケール抑制距離（近）
    f32                                               scaleLimitDistFar;                                        //!< カメラ前でのスケール抑制距離（遠）
    f32                                               dummy9[2];                                                //!< パディング領域

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        EndianUtil::FlipArray( 4, flag );
        nw::eft2::EndianUtil::Flip( &color0AnimKeynum );
        nw::eft2::EndianUtil::Flip( &alpha0AnimKeyNum );
        nw::eft2::EndianUtil::Flip( &color1AnimKeynum );
        nw::eft2::EndianUtil::Flip( &alpha1AnimKeyNum );
        nw::eft2::EndianUtil::Flip( &scaleAnimKeyNum );
        nw::eft2::EndianUtil::Flip( &shaderAnimKeyNum );
        EndianUtil::FlipArray( 2, dummy0 );
        nw::eft2::EndianUtil::Flip( &color0AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &alpha0AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &color1AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &alpha1AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &scaleAnimLoopRate );
        nw::eft2::EndianUtil::Flip( &color0AnimIsLoopInitRandom );
        nw::eft2::EndianUtil::Flip( &alpha0AnimIsLoopInitRandom );
        nw::eft2::EndianUtil::Flip( &color1AnimIsLoopInitRandom );
        nw::eft2::EndianUtil::Flip( &alpha1AnimIsLoopInitRandom );
        nw::eft2::EndianUtil::Flip( &scaleAnimIsLoopInitRandom );
        EndianUtil::FlipArray( 2, dummy6 );
        nw::eft2::EndianUtil::Flip( &gravity );
        nw::eft2::EndianUtil::Flip( &gravityScale );
        nw::eft2::EndianUtil::Flip( &airRegist );
        EndianUtil::FlipArray( 3, dummy1 );
        nw::eft2::EndianUtil::Flip( &center );
        nw::eft2::EndianUtil::Flip( &offset );
        nw::eft2::EndianUtil::Flip( &dummy2 );
        nw::eft2::EndianUtil::Flip( &coefficient0 );
        nw::eft2::EndianUtil::Flip( &coefficient1 );
        EndianUtil::FlipArray( 2, dummy3 );
        nw::eft2::EndianUtil::Flip( &colorScale );
        EndianUtil::FlipArray( 3, dummy4 );
        nw::eft2::EndianUtil::Flip( &softEdgeParam );
        nw::eft2::EndianUtil::Flip( &fresnelAlphaParam );
        nw::eft2::EndianUtil::Flip( &nearDistAlphaParam );
        nw::eft2::EndianUtil::Flip( &farDistAlphaParam );
        nw::eft2::EndianUtil::Flip( &decalParam );
        nw::eft2::EndianUtil::Flip( &alphaThreshold );
        nw::eft2::EndianUtil::Flip( &dummy7 );
        nw::eft2::EndianUtil::Flip( &addVelToScale );
        nw::eft2::EndianUtil::Flip( &softPartcileDist );
        nw::eft2::EndianUtil::Flip( &softParticleVolume );
        nw::eft2::EndianUtil::Flip( &dummy5 );
        nw::eft2::EndianUtil::Flip( &rotateInit );
        nw::eft2::EndianUtil::Flip( &rotateInitRand );
        nw::eft2::EndianUtil::Flip( &rotateAdd );
        nw::eft2::EndianUtil::Flip( &rotateRegist );
        nw::eft2::EndianUtil::Flip( &rotateAddRand );
        nw::eft2::EndianUtil::Flip( &dummy8 );
        nw::eft2::EndianUtil::Flip( &scaleLimitDistNear );
        nw::eft2::EndianUtil::Flip( &scaleLimitDistFar );
        EndianUtil::FlipArray( 2, dummy9 );

        fluctuationData.FlipEndian();
        texPtnAnim0.FlipEndian();
        texPtnAnim1.FlipEndian();
        texPtnAnim2.FlipEndian();
        texShiftAnim0.FlipEndian();
        texShiftAnim1.FlipEndian();
        texShiftAnim2.FlipEndian();
        color0Anim.FlipEndian();
        alpha0Anim.FlipEndian();
        color1Anim.FlipEndian();
        alpha1Anim.FlipEndian();
        scale8keyAnim.FlipEndian();
        shaderAnim.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      8キーアニメ情報
//------------------------------------------------------------------------------
struct ResAnim8KeyParamSet
{
    u32                                               enable;                                                   //!< 有効無効
    u32                                               loop;                                                     //!< ループ再生
    u32                                               startRandom;                                              //!< 開始位置ランダム
    s32                                               keyNum;                                                   //!< キー数
    s32                                               loopNum;                                                  //!< ループフレーム数
    nw::math::VEC4                                    keyValue[8];                                              //!< キーデータです。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &enable );
        nw::eft2::EndianUtil::Flip( &loop );
        nw::eft2::EndianUtil::Flip( &startRandom );
        nw::eft2::EndianUtil::Flip( &keyNum );
        nw::eft2::EndianUtil::Flip( &loopNum );
        EndianUtil::FlipArray( 8, keyValue );

    }
};

//------------------------------------------------------------------------------
//! @brief      新ランダム フィールド情報
//------------------------------------------------------------------------------
struct ResFieldRandom
{
    u8                                                enableUnifiedPhase;                                       //!< 固定された乱数の種を使うかどうかのフラグ
    u8                                                enableDetailedOption;                                     //!< 詳細設定を有効
    u8                                                enableAirRegist;                                          //!< 空気抵抗の有無
    u8                                                dummy;                                                    //!< パディング領域
    nw::math::VEC3                                    randomVel;                                                //!< blank毎に加算するランダム速度
    s32                                               blank;                                                    //!< ランダムをかけるタイミング
    f32                                               unifiedPhaseSpeed;                                        //!< 統一位相の変化速度。
    f32                                               unifiedPhaseDistribution;                                 //!< 統一位相の幅
    f32                                               waveParam0;                                               //!< 波形重み0
    f32                                               waveParam1;                                               //!< 波形重み1
    f32                                               waveParam2;                                               //!< 波形重み2
    f32                                               waveParam3;                                               //!< 波形重み3
    f32                                               waveParamHzRate0;                                         //!< 成分0の周波数の、基本周波数に対する比率
    f32                                               waveParamHzRate1;                                         //!< 成分1の周波数の、基本周波数に対する比率
    f32                                               waveParamHzRate2;                                         //!< 成分2の周波数の、基本周波数に対する比率
    f32                                               waveParamHzRate3;                                         //!< 成分3の周波数の、基本周波数に対する比率
    nw::eft2::ResAnim8KeyParamSet              animParam;                                                //!< パーティクル時間アニメのデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &randomVel );
        nw::eft2::EndianUtil::Flip( &blank );
        nw::eft2::EndianUtil::Flip( &unifiedPhaseSpeed );
        nw::eft2::EndianUtil::Flip( &unifiedPhaseDistribution );
        nw::eft2::EndianUtil::Flip( &waveParam0 );
        nw::eft2::EndianUtil::Flip( &waveParam1 );
        nw::eft2::EndianUtil::Flip( &waveParam2 );
        nw::eft2::EndianUtil::Flip( &waveParam3 );
        nw::eft2::EndianUtil::Flip( &waveParamHzRate0 );
        nw::eft2::EndianUtil::Flip( &waveParamHzRate1 );
        nw::eft2::EndianUtil::Flip( &waveParamHzRate2 );
        nw::eft2::EndianUtil::Flip( &waveParamHzRate3 );

        animParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタ基本情報
//------------------------------------------------------------------------------
struct ResEmitterFnd
{
    u8                                                isParticleDraw;                                           //!< パーティクルを描画する
    u8                                                sortType;                                                 //!< パーティクルソートタイプ
    u8                                                calcType;                                                 //!< 挙動計算タイプ
    u8                                                followType;                                               //!< エミッタ追従タイプ
    u8                                                isFadeEmit;                                               //!< 終了処理で放出停止するか
    u8                                                isFadeAlphaFade;                                          //!< 終了処理でアルファフェードアウトするか
    u8                                                isScaleFade;                                              //!< スケールフェードアウトの有無
    u8                                                randomSeedType;                                           //!< 乱数シードのタイプ
    u8                                                isUpdateMatrixByEmit;                                     //!< 放出ごとにマトリクスを更新
    u8                                                testAlways;                                               //!< 常時判定するか
    u8                                                interpolateEmissionAmount;                                //!< 放出量を補間するか
    u8                                                isAlphaFadeIn;                                            //!< αフェードインの有無
    u8                                                isScaleFadeIn;                                            //!< スケールフェードインの有無
    u8                                                dummy[3];                                                 //!< パディング領域
    u32                                               randomSeed;                                               //!< 乱数の種
    u32                                               drawPath;                                                 //!< 描画パス
    u32                                               alphaFadeTime;                                            //!< アルファフェードアウト時間
    u32                                               fadeInTime;                                               //!< フェードイン時間
    nw::math::VEC3                                    trans;                                                    //!< エミッタ位置値
    nw::math::VEC3                                    transRand;                                                //!< マトリクス移動量ランダム
    nw::math::VEC3                                    rotate;                                                   //!< エミッタ回転値
    nw::math::VEC3                                    rotateRand;                                               //!< マトリクス回転ランダム
    nw::math::VEC3                                    scale;                                                    //!< エミッタスケール値
    nw::math::VEC4                                    color0;                                                   //!< エミッタカラー0
    nw::math::VEC4                                    color1;                                                   //!< エミッタカラー1
    f32                                               emissionRangeNear;                                        //!< 放出範囲近距離
    f32                                               emissionRangeFar;                                         //!< 放出範囲遠距離
    s32                                               emissionRatioFar;                                         //!< 遠距離での放出割合

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &randomSeed );
        nw::eft2::EndianUtil::Flip( &drawPath );
        nw::eft2::EndianUtil::Flip( &alphaFadeTime );
        nw::eft2::EndianUtil::Flip( &fadeInTime );
        nw::eft2::EndianUtil::Flip( &trans );
        nw::eft2::EndianUtil::Flip( &transRand );
        nw::eft2::EndianUtil::Flip( &rotate );
        nw::eft2::EndianUtil::Flip( &rotateRand );
        nw::eft2::EndianUtil::Flip( &scale );
        nw::eft2::EndianUtil::Flip( &color0 );
        nw::eft2::EndianUtil::Flip( &color1 );
        nw::eft2::EndianUtil::Flip( &emissionRangeNear );
        nw::eft2::EndianUtil::Flip( &emissionRangeFar );
        nw::eft2::EndianUtil::Flip( &emissionRatioFar );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタ形状情報
//------------------------------------------------------------------------------
struct ResEmitterVolume
{
    u8                                                volumeType;                                               //!< ボリュームタイプ
    u8                                                sweepStartRandom;                                         //!< 弧の幅 ランダム
    u8                                                arcType;                                                  //!< 弧のタイプ
    u8                                                isVolumeLatitudeEnabled;                                  //!< 未使用
    u8                                                volumeTblIndex;                                           //!< 球形状ボリュームテーブルインデックス
    u8                                                volumeTblIndex64;                                         //!< 球64形状ボリュームテーブルインデックス
    u8                                                volumeLatitudeDir;                                        //!< 球の緯度の方向
    u8                                                isGpuEmitter;                                             //!< 仮住まいです。GPUエミッタの有効無効。
    f32                                               sweepLongitude;                                           //!< 弧の計算に使う値
    f32                                               sweepLatitude;                                            //!< 弧の計算に使う緯度
    f32                                               sweepStart;                                               //!< 弧の幅(開始)
    f32                                               volumeSurfacePosRand;                                     //!< エミッタ形状表面上のランダム位置
    f32                                               caliberRatio;                                             //!< 内径（比率）
    f32                                               lineCenter;                                               //!< ラインセンター
    f32                                               lineLength;                                               //!< ライン長
    nw::math::VEC3                                    volumeRadius;                                             //!< ボリューム半径
    nw::math::VEC3                                    volumeFormScale;                                          //!< エミッタ形状スケール
    u32                                               primEmitType;                                             //!< プリミティブ指定時の放出タイプ
    u64                                               primitiveIndex;                                           //!< プリミティブインデックスです.
    u32                                               numDivideCircle;                                          //!< 円等分割の分割数
    u32                                               numDivideCircleRandom;                                    //!< 分割数ランダム（円等分割）
    u32                                               numDivideLine;                                            //!< 線等分割の分割数
    u32                                               numDivideLineRandom;                                      //!< 分割数ランダム（線等分割）
    u8                                                isOnAnotherBinaryVolumePrimitive;                         //!< エミッタ形状プリミティブが他のバイナリにあるかどうか
    u8                                                dummy[7];                                                 //!< パディング

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &sweepLongitude );
        nw::eft2::EndianUtil::Flip( &sweepLatitude );
        nw::eft2::EndianUtil::Flip( &sweepStart );
        nw::eft2::EndianUtil::Flip( &volumeSurfacePosRand );
        nw::eft2::EndianUtil::Flip( &caliberRatio );
        nw::eft2::EndianUtil::Flip( &lineCenter );
        nw::eft2::EndianUtil::Flip( &lineLength );
        nw::eft2::EndianUtil::Flip( &volumeRadius );
        nw::eft2::EndianUtil::Flip( &volumeFormScale );
        nw::eft2::EndianUtil::Flip( &primEmitType );
        nw::eft2::EndianUtil::Flip( &primitiveIndex );
        nw::eft2::EndianUtil::Flip( &numDivideCircle );
        nw::eft2::EndianUtil::Flip( &numDivideCircleRandom );
        nw::eft2::EndianUtil::Flip( &numDivideLine );
        nw::eft2::EndianUtil::Flip( &numDivideLineRandom );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタ放出情報
//------------------------------------------------------------------------------
struct ResEmission
{
    u8                                                isOneTime;                                                //!< ワンタイム
    u8                                                isWorldGravity;                                           //!< ワールド座標系で重力を適用する
    u8                                                isEmitDistEnabled;                                        //!< 距離放出を使うかどうか
    u8                                                isWorldOrientedVelocity;                                  //!< 指定方向初速をワールド座標系で適用
    u32                                               start;                                                    //!< 放出開始フレーム
    u32                                               emitTiming;                                               //!< 放出開始タイミング
    u32                                               emitDuration;                                             //!< 放出時間
    f32                                               rate;                                                     //!< 放出レート
    s32                                               rateRandom;                                               //!< 放出レートランダム
    s32                                               interval;                                                 //!< 放出間隔
    s32                                               intervalRandom;                                           //!< 放出間隔ランダム
    f32                                               posRand;                                                  //!< 初期位置のランダム量
    f32                                               gravityScale;                                             //!< 重力スケール
    nw::math::VEC3                                    gravityDir;                                               //!< 重力方向
    f32                                               emitDistUnit;                                             //!< 放出間隔（距離）
    f32                                               emitDistMin;                                              //!< 1フレームに認める最小移動距離
    f32                                               emitDistMax;                                              //!< 1フレームに認める最大移動距離
    f32                                               emitDistMargin;                                           //!< 移動距離切り捨ての閾値
    s32                                               emitDistParticleMax;                                      //!< 距離放出の場合の、パーティクル放出の最大数

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &start );
        nw::eft2::EndianUtil::Flip( &emitTiming );
        nw::eft2::EndianUtil::Flip( &emitDuration );
        nw::eft2::EndianUtil::Flip( &rate );
        nw::eft2::EndianUtil::Flip( &rateRandom );
        nw::eft2::EndianUtil::Flip( &interval );
        nw::eft2::EndianUtil::Flip( &intervalRandom );
        nw::eft2::EndianUtil::Flip( &posRand );
        nw::eft2::EndianUtil::Flip( &gravityScale );
        nw::eft2::EndianUtil::Flip( &gravityDir );
        nw::eft2::EndianUtil::Flip( &emitDistUnit );
        nw::eft2::EndianUtil::Flip( &emitDistMin );
        nw::eft2::EndianUtil::Flip( &emitDistMax );
        nw::eft2::EndianUtil::Flip( &emitDistMargin );
        nw::eft2::EndianUtil::Flip( &emitDistParticleMax );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクル情報
//------------------------------------------------------------------------------
struct ResPtclFnd
{
    u8                                                isLifeInfinity;                                           //!< 無限寿命
    u8                                                isTriming;                                                //!< トリミング
    u8                                                billboardType;                                            //!< ビルボードタイプ
    u8                                                rotType;                                                  //!< 回転タイプ
    u8                                                offsetType;                                               //!< カメラ・デプスオフセットタイプ
    u8                                                rotRevRandX;                                              //!< 回転方向ランダムX
    u8                                                rotRevRandY;                                              //!< 回転方向ランダムY
    u8                                                rotRevRandZ;                                              //!< 回転方向ランダムZ
    u8                                                isRotateX;                                                //!< 回転Xを利用するか
    u8                                                isRotateY;                                                //!< 回転Yを利用するか
    u8                                                isRotateZ;                                                //!< 回転Zを利用するか
    u8                                                primitiveScaleType;                                       //!< プリミティブスケール適用タイプ
    u8                                                isTextureCommonRandom;                                    //!< テクスチャのランダマイズを共通化する
    u8                                                connectPtclScaleAndZOffset;                               //!< パーティクルスケールとZオフセットを連動させる
    u8                                                enableAvoidZFighting;                                     //!< Z-Fighting 回避のためのオフセットを入れるかどうか
    u8                                                dummy[1];                                                 //!< パディング領域
    s32                                               life;                                                     //!< ライフ
    s32                                               lifeRandom;                                               //!< ライフランダム
    f32                                               dynamicsRand;                                             //!< 運動量ランダム
    u32                                               primitiveVertexInfoFlags;                                 //!< 使用しているプリミティブの頂点が持つデータの種類を示すビットフラグです。下位ビットから順に、Position/Normal/Tangent/Color/UV0/UV1を表します。
    u64                                               primitiveIndex;                                           //!< 利用するプリミティブインデックスです。
    u64                                               trimmingPrimitiveIndex;                                   //!< 利用するトリミングプリミティブインデックスです。
    u8                                                color0AnimIsLoop;                                         //!< color0のアニメのループフラグ
    u8                                                alpha0AnimIsLoop;                                         //!< alpha0のアニメのループフラグ
    u8                                                color1AnimIsLoop;                                         //!< color1のアニメのループフラグ
    u8                                                alpha1AnimIsLoop;                                         //!< alpha1のアニメのループフラグ
    u8                                                scaleAnimIsLoop;                                          //!< scaleのアニメのループフラグ
    u8                                                color0AnimIsLoopInitRandom;                               //!< color0のアニメの初期位置ランダムの有効／無効フラグ
    u8                                                alpha0AnimIsLoopInitRandom;                               //!< alpha0のアニメの初期位置ランダムの有効／無効フラグ
    u8                                                color1AnimIsLoopInitRandom;                               //!< color1のアニメの初期位置ランダムの有効／無効フラグ
    u8                                                alpha1AnimIsLoopInitRandom;                               //!< alpha1のアニメの初期位置ランダムの有効／無効フラグ
    u8                                                scaleAnimIsLoopInitRandom;                                //!< scaleのアニメの初期位置ランダムの有効／無効フラグ
    u8                                                isOnAnotherBinaryPrimitive;                               //!< プリミティブが他のバイナリにあるかどうか
    u8                                                isOnAnotherBinaryTrimmingPrimitive;                       //!< トリミングプリミティブが他のバイナリにあるかどうか
    s32                                               color0AnimLoopRate;                                       //!< color0のアニメのループフレーム比率（寿命に対する一周期の％）
    s32                                               alpha0AnimLoopRate;                                       //!< alpha0のアニメのループフレーム比率（寿命に対する一周期の％）
    s32                                               color1AnimLoopRate;                                       //!< color1のアニメのループフレーム比率（寿命に対する一周期の％）
    s32                                               alpha1AnimLoopRate;                                       //!< alpha1のアニメのループフレーム比率（寿命に対する一周期の％）
    s32                                               scaleAnimLoopRate;                                        //!< scaleのアニメのループフレーム比率（寿命に対する一周期の％）

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &life );
        nw::eft2::EndianUtil::Flip( &lifeRandom );
        nw::eft2::EndianUtil::Flip( &dynamicsRand );
        nw::eft2::EndianUtil::Flip( &primitiveVertexInfoFlags );
        nw::eft2::EndianUtil::Flip( &primitiveIndex );
        nw::eft2::EndianUtil::Flip( &trimmingPrimitiveIndex );
        nw::eft2::EndianUtil::Flip( &color0AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &alpha0AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &color1AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &alpha1AnimLoopRate );
        nw::eft2::EndianUtil::Flip( &scaleAnimLoopRate );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタコンバイナ情報
//------------------------------------------------------------------------------
struct ResCombiner
{
    u8                                                colorCombinerProcess;                                     //!< カラー計算式タイプ
    u8                                                alphaCombinerProcess;                                     //!< アルファ計算式タイプ
    u8                                                texture1ColorBlend;                                       //!< テクスチャ1カラーの上段との合成
    u8                                                texture2ColorBlend;                                       //!< テクスチャ2カラーの上段との合成
    u8                                                primitiveColorBlend;                                      //!< プリミティブカラーの上段との合成
    u8                                                texture1AlphaBlend;                                       //!< テクスチャ1アルファの上段との合成
    u8                                                texture2AlphaBlend;                                       //!< テクスチャ2アルファの上段との合成
    u8                                                primitiveAlphaBlend;                                      //!< プリミティブアルファの上段との合成
    u8                                                texColor0InputType;                                       //!< テクスチャカラー0入力タイプ
    u8                                                texColor1InputType;                                       //!< テクスチャカラー1入力タイプ
    u8                                                texColor2InputType;                                       //!< テクスチャカラー2入力タイプ
    u8                                                texAlpha0InputType;                                       //!< テクスチャアルファ0入力タイプ
    u8                                                texAlpha1InputType;                                       //!< テクスチャアルファ1入力タイプ
    u8                                                texAlpha2InputType;                                       //!< テクスチャアルファ2入力タイプ
    u8                                                primitiveColorInputType;                                  //!< プリミティブカラー入力タイプ
    u8                                                primitiveAlphaInputType;                                  //!< プリミティブアルファ入力タイプ
    u8                                                shaderType;                                               //!< シェーダタイプ
    u8                                                applyAlpha;                                               //!< 屈折シェーダ/アルファ値を反映
    u8                                                isDistortionByCameraDistance;                             //!< カメラ距離によって屈折強度を強くするかどうかのフラグ
    u8                                                dummy[5];                                                 //!< アライメント

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

    }
};

//------------------------------------------------------------------------------
//! @brief      シェーダ情報
//------------------------------------------------------------------------------
struct ResShader
{
    u8                                                shaderType;                                               //!< シェーダタイプ
    u8                                                dummy[3];                                                 //!< ダミー
    u32                                               vertexShaderIndex;                                        //!< 利用するシェーダインデックス
    u32                                               computeShaderIndex;                                       //!< 利用するコンピュートシェーダインデックス
    u32                                               userVertexShaderIndex1;                                   //!< ユーザー頂点シェーダインデックス1
    u32                                               userPixelShaderIndex1;                                    //!< ユーザーピクセルシェーダインデックス1 (eft2専用)
    u32                                               userVertexShaderIndex2;                                   //!< ユーザー頂点シェーダインデックス2
    u32                                               userPixelShaderIndex2;                                    //!< ユーザーピクセルシェーダインデックス2 (eft2専用)
    u32                                               customShaderIndex;                                        //!< カスタムシェーダインデックス
    u64                                               customShaderFlag;                                         //!< カスタムシェーダフラグ
    u64                                               customShaderSwitch;                                       //!< ラジオボタンによるスイッチ選択状態です。
    u32                                               eftCombinerShaderIndex;                                   //!< エフェクトコンバイナが生成したシェーダのインデックス
    u32                                               reservedShaderIndex;                                      //!< Reserved shader (emitter plugin) index.
    char                                              userShaderDefine1[16];                                    //!< ユーザーシェーダ定義1
    char                                              userShaderDefine2[16];                                    //!< ユーザーシェーダ定義2

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &vertexShaderIndex );
        nw::eft2::EndianUtil::Flip( &computeShaderIndex );
        nw::eft2::EndianUtil::Flip( &userVertexShaderIndex1 );
        nw::eft2::EndianUtil::Flip( &userPixelShaderIndex1 );
        nw::eft2::EndianUtil::Flip( &userVertexShaderIndex2 );
        nw::eft2::EndianUtil::Flip( &userPixelShaderIndex2 );
        nw::eft2::EndianUtil::Flip( &customShaderIndex );
        nw::eft2::EndianUtil::Flip( &customShaderFlag );
        nw::eft2::EndianUtil::Flip( &customShaderSwitch );
        nw::eft2::EndianUtil::Flip( &eftCombinerShaderIndex );
        nw::eft2::EndianUtil::Flip( &reservedShaderIndex );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクル初速情報
//------------------------------------------------------------------------------
struct ResPtclVel
{
    f32                                               allDirection;                                             //!< 全方向初速
    f32                                               designatedDirScale;                                       //!< 指定方向速度
    nw::math::VEC3                                    designatedDir;                                            //!< 指定方向
    f32                                               diffusionDirAngle;                                        //!< 指定方向拡散角度
    f32                                               xzDiffusion;                                              //!< Y軸拡散速度
    nw::math::VEC3                                    diffusion;                                                //!< 拡散初速
    f32                                               velRandom;                                                //!< 速度ランダム
    f32                                               emVelInherit;                                             //!< エミッタ速度継承率

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &allDirection );
        nw::eft2::EndianUtil::Flip( &designatedDirScale );
        nw::eft2::EndianUtil::Flip( &designatedDir );
        nw::eft2::EndianUtil::Flip( &diffusionDirAngle );
        nw::eft2::EndianUtil::Flip( &xzDiffusion );
        nw::eft2::EndianUtil::Flip( &diffusion );
        nw::eft2::EndianUtil::Flip( &velRandom );
        nw::eft2::EndianUtil::Flip( &emVelInherit );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクルカラー情報
//------------------------------------------------------------------------------
struct ResPtclColor
{
    u8                                                isSoftParticle;                                           //!< ソフトパーティクル
    u8                                                isFresnelAlpha;                                           //!< フレネルアルファ
    u8                                                isNearDistAlpha;                                          //!< 近距離アルファ
    u8                                                isFarDistAlpha;                                           //!< 遠距離アルファ
    u8                                                isDecal;                                                  //!< デカール
    u8                                                dummy[3];                                                 //!< ダミー
    u8                                                color0Type;                                               //!< カラー0挙動タイプ
    u8                                                color1Type;                                               //!< カラー1挙動タイプ
    u8                                                alpha0Type;                                               //!< アルファ0挙動タイプ
    u8                                                alpha1Type;                                               //!< アルファ1挙動タイプ
    nw::math::VEC3                                    color0;                                                   //!< カラー0
    f32                                               alpha0;                                                   //!< アルファ0
    nw::math::VEC3                                    color1;                                                   //!< カラー1
    f32                                               alpha1;                                                   //!< アルファ1

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &color0 );
        nw::eft2::EndianUtil::Flip( &alpha0 );
        nw::eft2::EndianUtil::Flip( &color1 );
        nw::eft2::EndianUtil::Flip( &alpha1 );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクルスケール情報
//------------------------------------------------------------------------------
struct ResPtclScale
{
    nw::math::VEC3                                    base;                                                     //!< 基準スケール
    nw::math::VEC3                                    baseRandom;                                               //!< 基準スケールランダム
    u8                                                enableScalingByCameraDistNear;                            //!< カメラ距離スケール「近」のON/OFF
    u8                                                enableScalingByCameraDistFar;                             //!< カメラ距離スケール「遠」のON/OFF
    u8                                                enableAddScaleY;                                          //!< 速度スケールY
    u8                                                enableLinkFovyToScaleValue;                               //!< スケール制限に画角を連動させる
    f32                                               scaleLimitDistNear;                                       //!< パーティクルスケール制限距離（近）
    f32                                               scaleLimitDistFar;                                        //!< パーティクルスケール制限距離（遠）

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &base );
        nw::eft2::EndianUtil::Flip( &baseRandom );
        nw::eft2::EndianUtil::Flip( &scaleLimitDistNear );
        nw::eft2::EndianUtil::Flip( &scaleLimitDistFar );

    }
};

//------------------------------------------------------------------------------
//! @brief      パーティクル揺らぎ情報
//------------------------------------------------------------------------------
struct ResPtclFluctuation
{
    u8                                                isApplyAlpha;                                             //!< アルファに適用するかどうか
    u8                                                isApplayScale;                                            //!< スケールに適用するかどうか
    u8                                                isApplayScaleY;                                           //!< Y軸を個別に設定する
    u8                                                isWaveType;                                               //!< 揺らぎ波形タイプ
    u8                                                isPhaseRandomX;                                           //!< 依存ランダム X
    u8                                                isPhaseRandomY;                                           //!< 位相ランダム X
    u8                                                dummy[6];                                                 //!< ダミー

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

    }
};

//------------------------------------------------------------------------------
//! @brief      テクスチャアニメ情報

//------------------------------------------------------------------------------
struct ResTextureAnim
{
    u8                                                patternAnimType;                                          //!< パターンアニメタイプ
    u8                                                isScroll;                                                 //!< UVスクロールアニメ有効無効
    u8                                                isRotate;                                                 //!< UV回転アニメ有効無効
    u8                                                isScale;                                                  //!< UVスケールアニメ有効無効
    u8                                                repeat;                                                   //!< 繰り返し数
    u8                                                invRandU;                                                 //!< U 反転ランダム
    u8                                                invRandV;                                                 //!< V 反転ランダム
    u8                                                isPatAnimLoopRandom;                                      //!< テクスチャパタンアニメ ループ開始ランダム
    u8                                                uvChannel;                                                //!< プリミティブのUVチャンネル
    u8                                                dummy[7];                                                 //!< アライン用

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

    }
};

//------------------------------------------------------------------------------
//! @brief      継承パラメータ
//------------------------------------------------------------------------------
struct ResInherit
{
    u8                                                velocity;                                                 //!< 速度継承
    u8                                                scale;                                                    //!< スケール継承
    u8                                                rotate;                                                   //!< 回転継承
    u8                                                colorScale;                                               //!< カラースケール継承
    u8                                                color0;                                                   //!< カラー0継承
    u8                                                color1;                                                   //!< カラー1継承
    u8                                                alpha0;                                                   //!< アルファ0継承
    u8                                                alpha1;                                                   //!< アルファ1継承
    u8                                                drawPath;                                                 //!< 描画パスを継承
    u8                                                preDraw;                                                  //!< 継承先より先に描画
    u8                                                alpha0EachFrame;                                          //!< アルファ0を毎フレーム継承するかどうか
    u8                                                alpha1EachFrame;                                          //!< アルファ1を毎フレーム継承するかどうか
    u8                                                enableEmitterParticle;                                    //!< パーティクルごとにエミッタを生成するかどうか
    u8                                                dummy[3];                                                 //!< パディング領域
    f32                                               velocityRate;                                             //!< 速度継承率
    f32                                               scaleRate;                                                //!< スケール継承

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &velocityRate );
        nw::eft2::EndianUtil::Flip( &scaleRate );

    }
};

//------------------------------------------------------------------------------
//! @brief      Holds custom action data.
//------------------------------------------------------------------------------
struct ResAction
{
    s32                                               customActionIndex;                                        //!< The selected custom action index.

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &customActionIndex );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタリソース
//------------------------------------------------------------------------------
struct ResEmitter : public ResCommon
{
    nw::eft2::EmitterStaticUniformBlock        staticUbo;                                                //!< GPUに転送されるデータ( TODO:先頭を 0x100 アラインする )
    nw::eft2::ResEmitterFnd                    emitter;                                                  //!< エミッタ基本
    nw::eft2::ResInherit                       inherit;                                                  //!< 継承パラメータ
    nw::eft2::ResEmission                      emission;                                                 //!< エミッタ放出
    nw::eft2::ResEmitterVolume                 volume;                                                   //!< エミッタボリューム
    nw::eft2::ResRenderState                   renderState;                                              //!< 描画設定
    nw::eft2::ResPtclFnd                       ptcl;                                                     //!< パーティクル基本情報
    nw::eft2::ResCombiner                      combiner;                                                 //!< コンバイナ情報
    nw::eft2::ResShader                        shader;                                                   //!< シェーダ情報
    nw::eft2::ResAction                        action;                                                   //!< The custom action data.
    nw::eft2::ResPtclVel                       ptclVel;                                                  //!< パーティクル速度情報
    nw::eft2::ResPtclColor                     ptclColor;                                                //!< パーティクルカラー情報
    nw::eft2::ResPtclScale                     ptclScale;                                                //!< パーティクルスケール情報
    nw::eft2::ResPtclFluctuation               ptclFluctuation;                                          //!< 揺らぎ設定
    nw::eft2::ResTextureSampler                textureSampler0;                                          //!< テクスチャサンプラー
    nw::eft2::ResTextureSampler                textureSampler1;                                          //!< テクスチャサンプラ
    nw::eft2::ResTextureSampler                textureSampler2;                                          //!< テクスチャサンプラ
    nw::eft2::ResTextureAnim                   textureAnim0;                                             //!< テクスチャアニメ0
    nw::eft2::ResTextureAnim                   textureAnim1;                                             //!< テクスチャアニメ1
    nw::eft2::ResTextureAnim                   textureAnim2;                                             //!< テクスチャアニメ2

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        /* DO_NOTHING */

        staticUbo.FlipEndian();
        emitter.FlipEndian();
        inherit.FlipEndian();
        emission.FlipEndian();
        volume.FlipEndian();
        renderState.FlipEndian();
        ptcl.FlipEndian();
        combiner.FlipEndian();
        shader.FlipEndian();
        action.FlipEndian();
        ptclVel.FlipEndian();
        ptclColor.FlipEndian();
        ptclScale.FlipEndian();
        ptclFluctuation.FlipEndian();
        textureSampler0.FlipEndian();
        textureSampler1.FlipEndian();
        textureSampler2.FlipEndian();
        textureAnim0.FlipEndian();
        textureAnim1.FlipEndian();
        textureAnim2.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      アニメーションのキー
//------------------------------------------------------------------------------
struct ResAnimKey
{
    nw::math::VEC3                                    value;                                                    //!< 値
    f32                                               time;                                                     //!< 時間

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &value );
        nw::eft2::EndianUtil::Flip( &time );

    }
};

//------------------------------------------------------------------------------
//! @brief      マグネット フィールド情報
//------------------------------------------------------------------------------
struct ResFieldMagnet
{
    u8                                                isFollowEmitter;                                          //!< エミッタに追従するか？
    u8                                                fieldMagnetFlagX;                                         //!< フラグX
    u8                                                fieldMagnetFlagY;                                         //!< フラグY
    u8                                                fieldMangetFlagZ;                                         //!< フラグZ
    f32                                               fieldMagnetPower;                                         //!< 磁力
    nw::math::VEC3                                    fieldMagnetPos;                                           //!< 磁力位置
    nw::eft2::ResAnim8KeyParamSet              animParam;                                                //!< パーティクル時間アニメのデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldMagnetPower );
        nw::eft2::EndianUtil::Flip( &fieldMagnetPos );

        animParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      スピン フィールド情報
//------------------------------------------------------------------------------
struct ResFieldSpinData
{
    f32                                               fieldSpinRotate;                                          //!< 回転力
    u32                                               fieldSpinAxis;                                            //!< 軸
    f32                                               fieldSpinOuter;                                           //!< 外に向かう速度
    nw::eft2::ResAnim8KeyParamSet              rotateAnimParam;                                          //!< 回転のパーティクル時間アニメデータです
    nw::eft2::ResAnim8KeyParamSet              outerAnimParam;                                           //!< 拡散のパーティクル時間アニメデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldSpinRotate );
        nw::eft2::EndianUtil::Flip( &fieldSpinAxis );
        nw::eft2::EndianUtil::Flip( &fieldSpinOuter );

        rotateAnimParam.FlipEndian();
        outerAnimParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      コリジョン フィールド情報
//------------------------------------------------------------------------------
struct ResFieldCollisionData
{
    u8                                                fieldCollisionType;                                       //!< タイプ（0:反射  1:消滅)
    u8                                                fieldCollisionIsWorld;                                    //!< ワールドで処理するかどうか
    u8                                                fieldCollisionIsCommonCoord;                              //!< 共有平面を利用するか？
    u8                                                dummy;                                                    //!< パディング領域
    f32                                               fieldCollisionCoord;                                      //!< 座標
    f32                                               fieldCollisionCoef;                                       //!< 反射率
    s32                                               fieldCollisionCnt;                                        //!< 反射回数
    f32                                               fieldCollisionRegist;                                     //!< 摩擦係数

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldCollisionCoord );
        nw::eft2::EndianUtil::Flip( &fieldCollisionCoef );
        nw::eft2::EndianUtil::Flip( &fieldCollisionCnt );
        nw::eft2::EndianUtil::Flip( &fieldCollisionRegist );

    }
};

//------------------------------------------------------------------------------
//! @brief      収束 フィールド情報
//------------------------------------------------------------------------------
struct ResFieldConvergenceData
{
    u8                                                fieldConvergenceType;                                     //!< 収束タイプ
    u8                                                dummy[3];                                                 //!< パディング領域
    nw::math::VEC3                                    fieldConvergencePos;                                      //!< 位置
    f32                                               fieldConvergenceRatio;                                    //!< 割合
    nw::eft2::ResAnim8KeyParamSet              animParam;                                                //!< パーティクル時間アニメのデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldConvergencePos );
        nw::eft2::EndianUtil::Flip( &fieldConvergenceRatio );

        animParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      フィールド：カールノイズ
//------------------------------------------------------------------------------
struct ResFieldCurlNoiseData
{
    u8                                                isFieldCurlNoiseInterpolation;                            //!< 補間するか？
    u8                                                isFieldCurlNoiseBaseRandom;                               //!< ノイズのオフセットランダム
    u8                                                isWorldCoordinate;                                        //!< カールノイズの影響をワールド座標で受けさせるかどうか
    u8                                                dummy;                                                    //!< パディング領域
    nw::math::VEC3                                    fieldCurlNoiseSpeed;                                      //!< 渦アニメ速度
    nw::math::VEC3                                    fieldCurlNoiseInfluence;                                  //!< 影響の強さ
    f32                                               fieldCurlNoiseScale;                                      //!< 渦テーブルの枠サイズ
    f32                                               fieldCurlNoiseBase;                                       //!< ノイズのオフセット

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldCurlNoiseSpeed );
        nw::eft2::EndianUtil::Flip( &fieldCurlNoiseInfluence );
        nw::eft2::EndianUtil::Flip( &fieldCurlNoiseScale );
        nw::eft2::EndianUtil::Flip( &fieldCurlNoiseBase );

    }
};

//------------------------------------------------------------------------------
//! @brief      フィールド：位置に加算
//------------------------------------------------------------------------------
struct ResFieldPosAddData
{
    u8                                                isFieldPosAddGlobal;                                      //!< ワールド座標系で加算するか？
    u8                                                dummy[3];                                                 //!< パディング領域
    nw::math::VEC3                                    fieldPosAdd;                                              //!< 位置加算値
    nw::eft2::ResAnim8KeyParamSet              animParam;                                                //!< パーティクル時間アニメのデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &fieldPosAdd );

        animParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      履歴式ストライプデータです。
//------------------------------------------------------------------------------
struct ResStripeHistory
{
    s32                                               calcType;                                                 //!< 計算方法です。
    s32                                               emitterFollow;                                            //!< エミッタに完全追従かどうかのフラグです。
    s32                                               option;                                                   //!< オプションです。
    s32                                               texturing;                                                //!< テクスチャリングです。
    f32                                               numDivide;                                                //!< 分割数です。
    f32                                               numHistory;                                               //!< 履歴数(ストライプの長さ)です。
    f32                                               interval;                                                 //!< 履歴ポリゴン化の間隔です。
    f32                                               headAlpha;                                                //!< 先端αです。
    f32                                               tailAlpha;                                                //!< 末端αです。
    f32                                               histInterpolate;                                          //!< 履歴補間パラメータです。
    f32                                               dirInterpolate;                                           //!< 方向補間率です。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &calcType );
        nw::eft2::EndianUtil::Flip( &emitterFollow );
        nw::eft2::EndianUtil::Flip( &option );
        nw::eft2::EndianUtil::Flip( &texturing );
        nw::eft2::EndianUtil::Flip( &numDivide );
        nw::eft2::EndianUtil::Flip( &numHistory );
        nw::eft2::EndianUtil::Flip( &interval );
        nw::eft2::EndianUtil::Flip( &headAlpha );
        nw::eft2::EndianUtil::Flip( &tailAlpha );
        nw::eft2::EndianUtil::Flip( &histInterpolate );
        nw::eft2::EndianUtil::Flip( &dirInterpolate );

    }
};

//------------------------------------------------------------------------------
//! @brief      連結式ストライプです。
//------------------------------------------------------------------------------
struct ResStripeConnection
{
    s32                                               calcType;                                                 //!< 計算方法です。
    s32                                               emitterFollow;                                            //!< エミッタに完全追従するかどうかのフラグです。
    s32                                               option;                                                   //!< オプションです。
    s32                                               texturing;                                                //!< テクスチャリングです。
    s32                                               numDivide;                                                //!< 分割数
    s32                                               connectionType;                                           //!< 接続タイプです。
    f32                                               headAlpha;                                                //!< 先端αです。
    f32                                               tailAlpha;                                                //!< 末端αです。
    f32                                               histInterpolate;                                          //!< 履歴補間パラメータです。
    f32                                               dirInterpolate;                                           //!< 方向補間率です。

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &calcType );
        nw::eft2::EndianUtil::Flip( &emitterFollow );
        nw::eft2::EndianUtil::Flip( &option );
        nw::eft2::EndianUtil::Flip( &texturing );
        nw::eft2::EndianUtil::Flip( &numDivide );
        nw::eft2::EndianUtil::Flip( &connectionType );
        nw::eft2::EndianUtil::Flip( &headAlpha );
        nw::eft2::EndianUtil::Flip( &tailAlpha );
        nw::eft2::EndianUtil::Flip( &histInterpolate );
        nw::eft2::EndianUtil::Flip( &dirInterpolate );

    }
};

//------------------------------------------------------------------------------
//! @brief      8キーアニメのパラメータ
//------------------------------------------------------------------------------
struct ResAnimEmitterKeyParamSet
{
    u8                                                enable;                                                   //!< アニメの有効無効フラグです。
    u8                                                loop;                                                     //!< アニメ再生のループフラグです。
    u8                                                startRandom;                                              //!< 再生開始フレームをランダマイズするか
    u8                                                dummy0;                                                   //!< ダミー0
    s32                                               keyNum;                                                   //!< キー数
    s32                                               loopNum;                                                  //!< ループ数
    nw::math::VEC4                                    keyValue[1];                                              //!< キー配列の先頭

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &keyNum );
        nw::eft2::EndianUtil::Flip( &loopNum );
        EndianUtil::FlipArray( 1, keyValue );

    }
};

//------------------------------------------------------------------------------
//! @brief      ランダム フィールド情報
//------------------------------------------------------------------------------
struct ResFieldRandomFe1
{
    nw::math::VEC3                                    randomVel;                                                //!< ランダム幅
    s32                                               blank;                                                    //!< ランダム適用間隔(フレーム)
    nw::eft2::ResAnim8KeyParamSet              animParam;                                                //!< パーティクル時間アニメのデータです

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &randomVel );
        nw::eft2::EndianUtil::Flip( &blank );

        animParam.FlipEndian();
    }
};

//------------------------------------------------------------------------------
//! @brief      スーパーストライプです。
//------------------------------------------------------------------------------
struct ResStripeSuper
{
    s32                                               calcType;                                                 //!< 計算方法です。
    s32                                               emitterFollow;                                            //!< エミッタに完全追従するかどうかのフラグです。
    s32                                               option;                                                   //!< オプションです。（通常／クロス）
    s32                                               texturing0;                                               //!< UV0のテクスチャリング設定
    s32                                               texturing1;                                               //!< UV1のテクスチャリング設定
    s32                                               texturing2;                                               //!< UV2のテクスチャリング設定
    f32                                               numHistory;                                               //!< 履歴の総数
    s32                                               connectionType;                                           //!< 接続タイプです。
    f32                                               headAlpha;                                                //!< 先端αです。
    f32                                               tailAlpha;                                                //!< 末端αです。
    s32                                               numDivide;                                                //!< 分割数です。
    f32                                               histInterpolate;                                          //!< 履歴補間パラメータです。
    f32                                               dirInterpolate;                                           //!< 方向補間率です。
    f32                                               historyAirRegist;                                         //!< [仮]履歴点の受ける空気抵抗
    nw::math::VEC3                                    historyAcceleration;                                      //!< [仮]履歴点の加速度
    f32                                               historyVecRegulation;                                     //!< [仮]
    f32                                               historyVecInitSpeed;                                      //!< [仮]
    nw::math::VEC3                                    historyInitVecRotateCycle;                                //!< [仮]
    s32                                               textureUvMapType;                                         //!< テクスチャUVマッピングタイプ
    f32                                               headScale;                                                //!< 先端スケール
    f32                                               tailScale;                                                //!< 末端スケール
    f32                                               test00;                                                   //!< [仮]実験用パラメータ
    f32                                               test01;                                                   //!< [仮]実験用パラメータ
    f32                                               test02;                                                   //!< [仮]実験用パラメータ
    f32                                               test03;                                                   //!< [仮]実験用パラメータ
    f32                                               test04;                                                   //!< [仮]実験用パラメータ
    f32                                               test05;                                                   //!< [仮]実験用パラメータ
    f32                                               test06;                                                   //!< [仮]実験用パラメータ
    f32                                               test07;                                                   //!< [仮]実験用パラメータ

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &calcType );
        nw::eft2::EndianUtil::Flip( &emitterFollow );
        nw::eft2::EndianUtil::Flip( &option );
        nw::eft2::EndianUtil::Flip( &texturing0 );
        nw::eft2::EndianUtil::Flip( &texturing1 );
        nw::eft2::EndianUtil::Flip( &texturing2 );
        nw::eft2::EndianUtil::Flip( &numHistory );
        nw::eft2::EndianUtil::Flip( &connectionType );
        nw::eft2::EndianUtil::Flip( &headAlpha );
        nw::eft2::EndianUtil::Flip( &tailAlpha );
        nw::eft2::EndianUtil::Flip( &numDivide );
        nw::eft2::EndianUtil::Flip( &histInterpolate );
        nw::eft2::EndianUtil::Flip( &dirInterpolate );
        nw::eft2::EndianUtil::Flip( &historyAirRegist );
        nw::eft2::EndianUtil::Flip( &historyAcceleration );
        nw::eft2::EndianUtil::Flip( &historyVecRegulation );
        nw::eft2::EndianUtil::Flip( &historyVecInitSpeed );
        nw::eft2::EndianUtil::Flip( &historyInitVecRotateCycle );
        nw::eft2::EndianUtil::Flip( &textureUvMapType );
        nw::eft2::EndianUtil::Flip( &headScale );
        nw::eft2::EndianUtil::Flip( &tailScale );
        nw::eft2::EndianUtil::Flip( &test00 );
        nw::eft2::EndianUtil::Flip( &test01 );
        nw::eft2::EndianUtil::Flip( &test02 );
        nw::eft2::EndianUtil::Flip( &test03 );
        nw::eft2::EndianUtil::Flip( &test04 );
        nw::eft2::EndianUtil::Flip( &test05 );
        nw::eft2::EndianUtil::Flip( &test06 );
        nw::eft2::EndianUtil::Flip( &test07 );

    }
};

//------------------------------------------------------------------------------
//! @brief      テスト
//------------------------------------------------------------------------------
struct ResEPTest
{
    u32                                               param;                                                    //!< テスト

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &param );

    }
};

//------------------------------------------------------------------------------
//! @brief      エリア内ループ
//------------------------------------------------------------------------------
struct ResEPAreaLoop
{
    nw::math::VEC3                                    repeatOffsetPos;                                          //!< リピート時の位置オフセット
    f32                                               repeatNum;                                                //!< リピート回数
    nw::math::VEC3                                    areaSize;                                                 //!< 範囲（箱）の大きさ
    f32                                               clippingWorldHeight;                                      //!< ワールド空間のクリッピング平面の高さ
    nw::math::VEC3                                    areaPos;                                                  //!< 範囲（箱）の位置
    s32                                               clippingType;                                             //!< クリッピングタイプ
    nw::math::VEC3                                    alphaRatio;                                               //!< 端でアルファを薄くする割合
    f32                                               isCameraLoop;                                             //!< 範囲（箱）をカメラ前に固定するかどうか
    nw::math::VEC3                                    areaRotate;                                               //!< 範囲（箱）の回転量（Radian）
    f32                                               dummy2;                                                   //!< パディング領域

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &repeatOffsetPos );
        nw::eft2::EndianUtil::Flip( &repeatNum );
        nw::eft2::EndianUtil::Flip( &areaSize );
        nw::eft2::EndianUtil::Flip( &clippingWorldHeight );
        nw::eft2::EndianUtil::Flip( &areaPos );
        nw::eft2::EndianUtil::Flip( &clippingType );
        nw::eft2::EndianUtil::Flip( &alphaRatio );
        nw::eft2::EndianUtil::Flip( &isCameraLoop );
        nw::eft2::EndianUtil::Flip( &areaRotate );
        nw::eft2::EndianUtil::Flip( &dummy2 );

    }
};

//------------------------------------------------------------------------------
//! @brief      カスタムフィールドデータ
//------------------------------------------------------------------------------
struct ResFieldCustom
{
    u32                                               flags;                                                    //!< フラグ
    f32                                               value0;                                                   //!< 数値0
    f32                                               value1;                                                   //!< 数値1
    f32                                               value2;                                                   //!< 数値2
    f32                                               value3;                                                   //!< 数値3
    f32                                               value4;                                                   //!< 数値4
    f32                                               value5;                                                   //!< 数値5
    f32                                               value6;                                                   //!< 数値6
    f32                                               value7;                                                   //!< 数値7
    f32                                               value8;                                                   //!< 数値8
    f32                                               value9;                                                   //!< 数値9
    f32                                               value10;                                                  //!< 数値10
    f32                                               value11;                                                  //!< 数値11
    f32                                               value12;                                                  //!< 数値12
    f32                                               value13;                                                  //!< 数値13
    f32                                               value14;                                                  //!< 数値14
    f32                                               value15;                                                  //!< 数値15

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &flags );
        nw::eft2::EndianUtil::Flip( &value0 );
        nw::eft2::EndianUtil::Flip( &value1 );
        nw::eft2::EndianUtil::Flip( &value2 );
        nw::eft2::EndianUtil::Flip( &value3 );
        nw::eft2::EndianUtil::Flip( &value4 );
        nw::eft2::EndianUtil::Flip( &value5 );
        nw::eft2::EndianUtil::Flip( &value6 );
        nw::eft2::EndianUtil::Flip( &value7 );
        nw::eft2::EndianUtil::Flip( &value8 );
        nw::eft2::EndianUtil::Flip( &value9 );
        nw::eft2::EndianUtil::Flip( &value10 );
        nw::eft2::EndianUtil::Flip( &value11 );
        nw::eft2::EndianUtil::Flip( &value12 );
        nw::eft2::EndianUtil::Flip( &value13 );
        nw::eft2::EndianUtil::Flip( &value14 );
        nw::eft2::EndianUtil::Flip( &value15 );

    }
};

//------------------------------------------------------------------------------
//! @brief      名前テーブルのレコード
//------------------------------------------------------------------------------
struct ResNameTableRecord
{
    u64                                               guid;                                                     //!< ハッシュコードと衝突カウンタからなるGUID
    u32                                               nextOffset;                                               //!< 次のレコードへのオフセット
    u32                                               nameLength;                                               //!< 名前の長さ
    char                                              name[8];                                                  //!< 名前

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &guid );
        nw::eft2::EndianUtil::Flip( &nextOffset );
        nw::eft2::EndianUtil::Flip( &nameLength );

    }
};

//------------------------------------------------------------------------------
//! @brief      g3dプリミティブテーブルのレコード
//------------------------------------------------------------------------------
struct ResG3dPrimitiveTableRecord
{
    u64                                               guid;                                                     //!< ハッシュコードと衝突カウンタからなるGUID
    u32                                               nextOffset;                                               //!< 次のレコードへのオフセット
    u32                                               nameLength;                                               //!< 名前の長さ
    u8                                                positionIndex;                                            //!< posデータのインデックス
    u8                                                normalIndex;                                              //!< 法線データのインデックス
    u8                                                tangentIndex;                                             //!< 接線データのインデックス
    u8                                                colorIndex;                                               //!< 頂点カラーのインデックス
    u8                                                uv0Index;                                                 //!< UV0のインデックス
    u8                                                uv1Index;                                                 //!< UV1のインデックス
    char                                              dummy[2];                                                 //!< アライメント

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &guid );
        nw::eft2::EndianUtil::Flip( &nextOffset );
        nw::eft2::EndianUtil::Flip( &nameLength );

    }
};

//------------------------------------------------------------------------------
//! @brief      エミッタセットのファイルパステーブルのレコード
//------------------------------------------------------------------------------
struct ResEmitterSetFilePathTableRecord
{
    u8                                                guid[16];                                                 //!< エミッタセットのguid
    u32                                               nextOffset;                                               //!< 次のレコードへのオフセット
    u32                                               filePathLength;                                           //!< ファイルパスの長さ
    char                                              filePath[8];                                              //!< ファイルパス

    //--------------------------------------------------------------------------
    //! @brief      エンディアンをフリップします.
    //--------------------------------------------------------------------------
    void FlipEndian()
    {
        nw::eft2::EndianUtil::Flip( &nextOffset );
        nw::eft2::EndianUtil::Flip( &filePathLength );

    }
};


} // namespace eft2
} // namespace nw

