﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef DW_SELECTABLE_PARAMETER_LIST_ITEM_H_
#define DW_SELECTABLE_PARAMETER_LIST_ITEM_H_

#include <nw/dw/control/dw_ParameterListItem.h>
#include <nw/dw/control/dw_Enums.h>

namespace nw {
namespace internal {
namespace dw {

//---------------------------------------------------------------------------
//! @brief     選択可能なパラメータリストアイテムクラスです。
//---------------------------------------------------------------------------
template<class TModel, typename TValue>
class SelectableParameterListItem : public ModelParameterListItem<TModel>
{
private:
    typedef ModelParameterListItem<TModel> Base;

public:
    typedef TValue (*GetValueHandler)(TModel* pModel);
    typedef void   (*SetValueHandler)(TModel* pModel, TValue value);
    typedef void   (*PrintValueHandler)(TModel* pModel, char* pText, u32 bufferLength, TValue value);


    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    defaultValue デフォルト値です。
    //---------------------------------------------------------------------------
    /* implicit */ SelectableParameterListItem(TValue defaultValue);

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~SelectableParameterListItem() {}

    //---------------------------------------------------------------------------
    //! @brief        初期化処理を行います。
    //!
    //! @param[in]    enumList           選択肢リストです。
    //! @param[in]    pGetValueHandler   値取得ハンドラです。
    //! @param[in]    pSetValueHandler   値設定ハンドラです。
    //! @param[in]    pPrintValueHandler 描画ハンドラです。
    //---------------------------------------------------------------------------
    void Initialize(
        const IEnums* enumList,
        GetValueHandler pGetValueHandler,
        SetValueHandler pSetValueHandler,
        PrintValueHandler pPrintValueHandler
    );

    //---------------------------------------------------------------------------
    //! @brief        値を取得します。
    //!
    //! @return       TValue 型の値を返します。
    //---------------------------------------------------------------------------
    TValue GetValue() const;

    //---------------------------------------------------------------------------
    //! @brief        値を設定します。
    //!
    //! @param[in]    value     TValue 型の値です。
    //---------------------------------------------------------------------------
    void SetValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        デフォルトの値を取得します。
    //!
    //! @return       TValue 型の値を返します。
    //---------------------------------------------------------------------------
    TValue GetDefaultValue() const;

    //---------------------------------------------------------------------------
    //! @brief        デフォルトの値を設定します。
    //!
    //! @param[in]    value     TValue 型の値です。
    //---------------------------------------------------------------------------
    void SetDefaultValue(TValue value);

    //---------------------------------------------------------------------------
    //! @brief        値をリセットします。
    //---------------------------------------------------------------------------
    /*override*/ void ResetValue();


protected:
    //! @brief 次の値にします。
    /*override*/ void NextValue(bool isLargeChange);
    //! @brief 前の値にします。
    /*override*/ void PreviousValue(bool isLargeChange);
    //! @brief 値を表示します。
    /*override*/ void PrintValue(char* pText, u32 bufferLength);

    const IEnums* m_Enums;
    TValue m_DefaultValue;

    GetValueHandler   m_pGetValueHandler;
    SetValueHandler   m_pSetValueHandler;
    PrintValueHandler m_pPrintValueHandler;

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
public:
    virtual const char* ToString() const
    {
        return "SelectableParameterListItem";
    }
#endif
};

} /* namespace dw */
} /* namespace internal */
} /* namespace nw */


#include "dw_SelectableParameterListItem.hpp"

#endif //  DW_SELECTABLE_PARAMETER_LIST_ITEM_H_
