﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PADPROCESSOR_H_
#define NW_DEV_PADPROCESSOR_H_

#include <nw/types.h>

namespace nw  {
namespace dev {

//---------------------------------------------------------------------------
//! @brief     パッドのキー状態管理クラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class PadProcessor
{
public:
    //! @brief 処理可能な最大のキー数です。
    enum
    {
        MAX_KEY = 32    //!< 処理可能な最大のキー数です。
    };

    //---------------------------------------------------------------------------
    //! @brief PadProcessor の設定です。
    //---------------------------------------------------------------------------
    struct Setting
    {
        u32 repeatStart;    //!< キーリピート開始フレームです。
        u32 repeatSpan;     //!< キーリピート間隔です。
        u32 holdCount;      //!< 長押し判定時間です。
        u32 clickCount;     //!< クリック判定時間です。
        u32 exclusiveMask;  //!< 排他するキーのマスクです。

        //! @param 各種デフォルト設定です。
        enum
        {
            DEFAULT_REPEAT_START = 25,  //!< キーリピート開始フレームのデフォルト設定です。
            DEFAULT_REPEAT_SPAN  = 8,   //!< キーリピート間隔のデフォルト設定です。
            DEFAULT_HOLD_COUNT   = 90,  //!< 長押し判定時間のデフォルト設定です。
            DEFAULT_CLICK_COUNT  = 20   //!< クリック判定時間のデフォルト設定です。
        };

        //---------------------------------------------------------------------------
        //! @brief        デフォルトコンストラクタです。
        //---------------------------------------------------------------------------
        /* ctor */ Setting()
         : repeatStart( DEFAULT_REPEAT_START )
         , repeatSpan ( DEFAULT_REPEAT_SPAN )
         , holdCount  ( DEFAULT_HOLD_COUNT )
         , clickCount ( DEFAULT_CLICK_COUNT )
         , exclusiveMask( 0 )
        {}
    };


#if defined(NW_DEV_PAD_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        デフォルトコンストラクタです。
    //---------------------------------------------------------------------------
    /* ctor */ PadProcessor()
      : m_Setting()
    {
        this->Reset();
    }

    //---------------------------------------------------------------------------
    //! @brief        設定付きのコンストラクタです。
    //! @param[in]    setting 初期化に用いる設定です。
    //---------------------------------------------------------------------------
    /* ctor */ explicit PadProcessor( Setting setting )
      : m_Setting( setting )
    {
        this->Reset();
    }


    //----------------------------------------
    //! @name 更新／設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        キー情報を更新します。
    //!
    //! @param[in]    key  新しい キー押下状態の入力です。
    //---------------------------------------------------------------------------
    void Update(u32 key);

    //---------------------------------------------------------------------------
    //! @brief        PadProcessor の設定を変更します。
    //!
    //! @param[in]    setting 設定です。
    //---------------------------------------------------------------------------
    void SetSetting( Setting& setting ) { m_Setting = setting; }

    //---------------------------------------------------------------------------
    //! @brief        すべてのキー状態をリセットします。
    //---------------------------------------------------------------------------
    void Reset()
    {
        m_PressPrev = 0x0;
        m_Press     = 0x0;
        m_PressExclusive = 0x0;
        m_Trigger = 0x0;
        m_Release = 0x0;
        m_Repeat = 0x0;
        m_RepeatAccel = 0x0;
        m_LastPressedExclusive = 0x0;
        m_LastRepeatedExclusive = 0x0;
        m_LastRepeatedAccelExclusive = 0x0;
        m_Click = 0x0;
        m_ClickIgnore = 0x0;
        m_LongPress = 0x0;

        for (int i = 0; i < MAX_KEY; ++i)
        {
            m_RepeateCount[ i ]        = 0;
            m_RepeateAccelCount[ i ]   = 0;
            m_RepeateTriggerCount[ i ] = 0;
            m_ClickHold[ i ]           = 0;
            m_Counter[ i ]             = 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        現在押されているキーのクリック判定状態をリセットします。
    //---------------------------------------------------------------------------
    void ResetClickState()
    {
        m_ClickIgnore = 0xFFFFFFFF;
    }

    //  @}

    //----------------------------------------
    //! @name 操作情報の取得
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        キーが押されている状態かをマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetPressedMask() const { return m_Press; }

    //---------------------------------------------------------------------------
    //! @brief        斜め入力禁止の十字ボタン処理済みのフラグをマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetPressedExclusiveMask() const { return m_PressExclusive; }

    //---------------------------------------------------------------------------
    //! @brief        キーのトリガ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetTriggeredMask() const { return m_Trigger; }

    //---------------------------------------------------------------------------
    //! @brief        キーのリリーストリガ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetReleasedMask() const { return m_Release; }

    //---------------------------------------------------------------------------
    //! @brief        キーのリピートトリガ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetRepeatedMask() const { return m_Repeat; }

    //---------------------------------------------------------------------------
    //! @brief        キーの加速リピートトリガ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetRepeatedAccelMask() const { return m_RepeatAccel; }

    //---------------------------------------------------------------------------
    //! @brief        キーのクリックフラグ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetClickedMask() const { return m_Click; }

    //---------------------------------------------------------------------------
    //! @brief        キーの長押しフラグ判定をマスク値で取得します。
    //!
    //! @return       マスク値を返します。
    //---------------------------------------------------------------------------
    u32 GetLongPressedMask() const { return m_LongPress; }

    //---------------------------------------------------------------------------
    //! @brief        キーが連続で押されているフレーム数を取得します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       フレーム数を返します。
    //---------------------------------------------------------------------------
    const u32 GetPressCount( u32 key ) const { return m_Counter[ key ]; }

    //  @}

    //----------------------------------------
    //! @name ボタンの押下判定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        キーが押されている状態かどうかのフラグを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsPressed( u32 key ) const { return m_Press & key; }

    //---------------------------------------------------------------------------
    //! @brief        斜め入力禁止の十字ボタン処理済みのフラグを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsPressedExclusive( u32 key ) const { return m_PressExclusive & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーのトリガを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsTriggered( u32 key ) const { return m_Trigger & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーのリリーストリガを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsReleased( u32 key ) const { return m_Release & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーのリピートトリガを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsRepeated( u32 key ) const { return m_Repeat & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーのリピートトリガを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //! @param[in]    start   リピート開始フレームを指定します。
    //! @param[in]    span    リピート間隔を指定します。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsRepeated( u32 key, u32 start, u32 span ) const;

    //---------------------------------------------------------------------------
    //! @brief        キーの加速リピートトリガを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsRepeatedAccel( u32 key ) const { return m_RepeatAccel & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーのクリックフラグを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsClicked( u32 key ) const { return m_Click & key; }

    //---------------------------------------------------------------------------
    //! @brief        キーの長押しフラグを判定します。
    //!
    //! @param[in]    key     判定するキー情報です。
    //!
    //! @return       キーのフラグ値を返します。
    //---------------------------------------------------------------------------
    u32 IsLongPressed( u32 key ) const { return m_LongPress & key ; }

    //  @}

private:
    u32     m_PressPrev;                        //!< 前回の入力値
    u32     m_Press;                            //!< 入力値
    u32     m_PressExclusive;                   //!< 排他的な入力値
    u32     m_Trigger;                          //!< 押しトリガ入力
    u32     m_Release;                          //!< 離しトリガ入力
    u32     m_Repeat;                           //!< 押し維持リピート入力
    u32     m_RepeatAccel;                      //!< 加速リピート入力
    u32     m_LastPressedExclusive;             //!< 排他的な入力判定用の最後に押されたキーです。
    u32     m_LastRepeatedExclusive;            //!< 排他的な入力判定用の最後にリピートされたキーです。
    u32     m_LastRepeatedAccelExclusive;       //!< 排他的な入力判定用の最後に加速リピートされたキーです。
    u32     m_Click;                            //!< クリック判定の値です。
    u32     m_ClickIgnore;                      //!< クリック無視フラグです。
    u32     m_LongPress;                        //!< 長押し判定の値です。

    s32     m_RepeateCount[ MAX_KEY ];          //!< キーリピートのカウンタです。
    s32     m_RepeateAccelCount[ MAX_KEY ];     //!< 加速付きキーリピートのカウンタです。
    s32     m_RepeateTriggerCount[ MAX_KEY ];   //!< リピート用の入力トリガ時点からのカウンタです。
    u32     m_ClickHold[ MAX_KEY ];             //!< クリック＆長押し用のカウンタです。
    u32     m_Counter[ MAX_KEY ];               //!< 押している長さのカウンタです。

    Setting m_Setting;                          //!< PadProcessor の設定です。

    void UpdateRepeate_();
    u32  UpdateLastRepeatKey_( u32 lastPressedExclusive );
    void UpdateRepeateAccel_();
    void UpdateClick_();
#else

    /* ctor */ PadProcessor() {}

    /* ctor */ explicit PadProcessor( Setting ) {}

    void Update(u32) {}

    void SetSetting( Setting& ) {}

    void Reset() {}

    void ResetClickState() {}

    u32 GetPressedMask() const { return 0; }

    u32 GetPressedExclusiveMask() const { return 0; }

    u32 GetTriggeredMask() const { return 0; }

    u32 GetReleasedMask() const { return 0; }

    u32 GetRepeatedMask() const { return 0; }

    u32 GetRepeatedAccelMask() const { return 0; }

    u32 GetClickedMask() const { return 0; }

    u32 GetLongPressedMask() const { return 0; }

    const u32 GetPressCount( u32 ) const { return 0; }

    u32 IsPressed( u32 ) const { return 0; }

    u32 IsPressedExclusive( u32 ) const { return 0; }

    u32 IsTriggered( u32 ) const { return 0; }

    u32 IsReleased( u32 ) const { return 0; }

    u32 IsRepeated( u32 ) const { return 0; }

    u32 IsRepeated( u32, u32, u32 ) const { return 0; }

    u32 IsRepeatedAccel( u32 ) const { return 0; }

    u32 IsClicked( u32 ) const { return 0; }

    u32 IsLongPressed( u32 ) const { return 0; }

#endif // NW_DEV_PAD_ENABLE
};

} // namespace dev
} // nemaspace nw

#endif // NW_DEV_PADPROCESSOR_H_


