﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_PAD_BASE_H_
#define NW_DEV_PAD_BASE_H_

#include <nw/dev/dev_PadProcessor.h>
#include <nw/math.h>
#include <nw/ut/ut_BitFlag.h>
#include <nw/ut/ut_Memory.h>
#include <nw/ut/ut_RuntimeTypeInfo.h>

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief        devPad 基底クラスです。
//!
//! @details :category     入力デバイス
//---------------------------------------------------------------------------
class PadBase
{
public:
    NW_UT_RUNTIME_TYPEINFO_ROOT()

#if defined(NW_DEV_PAD_ENABLE)
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //!
    //! @param[in]    padBitMax               パッドのビット値の最大値です。
    //! @param[in]    leftStickCrossStartBit  左スティックを十字ボタンとして扱うとき、何ビット目から始まるかです。存在しないときは -1 を指定します。
    //! @param[in]    rightStickCrossStartBit 右スティックを十字ボタンとして扱うとき、何ビット目から始まるかです。存在しないときは -1 を指定します。
    //! @param[in]    touchKeyBit             タッチキーが何ビット目かです。存在しないときは -1 を指定します。
    //---------------------------------------------------------------------------
    PadBase(int padBitMax, int leftStickCrossStartBit, int rightStickCrossStartBit, int touchKeyBit);

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    virtual ~PadBase() {}


    //----------------------------------------
    //! @name 操作情報の取得／設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        現在押されているボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetHoldMask() const { return m_PadProcessor.GetPressedMask(); }

    //---------------------------------------------------------------------------
    //! @brief        斜め入力禁止の十字ボタン処理済みのフラグをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetPressExclusiveMask() const { return m_PadProcessor.GetPressedExclusiveMask(); }

    //---------------------------------------------------------------------------
    //! @brief        このフレームに押されたボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetTrigMask() const { return m_PadProcessor.GetTriggeredMask(); }

    //---------------------------------------------------------------------------
    //! @brief        このフレームに離されたボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetReleaseMask() const { return m_PadProcessor.GetReleasedMask(); }

    //---------------------------------------------------------------------------
    //! @brief        このフレームにリピートがかかっているボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetRepeatMask() const { return m_PadProcessor.GetRepeatedMask(); }

    //---------------------------------------------------------------------------
    //! @brief        このフレームに加速リピートがかかっているボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetRepeatAccelMask() const { return m_PadProcessor.GetRepeatedAccelMask(); }

    //---------------------------------------------------------------------------
    //! @brief        クリックされたボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetClickMask() const { return m_PadProcessor.GetClickedMask(); }

    //---------------------------------------------------------------------------
    //! @brief        クリックされたボタンをマスク値で取得します。
    //!
    //! @return       ボタンをマスク値で返します。
    //---------------------------------------------------------------------------
    u32 GetLongPressMask() const { return m_PadProcessor.GetLongPressedMask(); }

    //---------------------------------------------------------------------------
    //! @brief        指定したビット番号のボタンが何フレーム連続でホールド状態になっているかを取得します。
    //!
    //! @param[in]    bit       ビット番号です。
    //!
    //! @return       何フレーム押し続けているかを返します。
    //---------------------------------------------------------------------------
    u32 GetPadHoldCount(int bit) const { return m_PadProcessor.GetPressCount(bit); }

    //---------------------------------------------------------------------------
    //! @brief        左アナログスティックの値を取得します。
    //!
    //! @return       左アナログスティックの値を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetLeftStick() const { return m_LeftStick; }

    //---------------------------------------------------------------------------
    //! @brief        右アナログスティックの値を取得します。
    //!
    //! @return       右アナログスティックの値を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetRightStick() const { return m_RightStick; }

    //---------------------------------------------------------------------------
    //! @brief        左アナログトリガーの値を取得します。
    //!
    //! @return       左アナログトリガーの値を返します。
    //---------------------------------------------------------------------------
    float GetLeftAnalogTrigger() const { return m_LeftAnalogTrigger; }

    //---------------------------------------------------------------------------
    //! @brief        右アナログトリガーの値を取得します。
    //!
    //! @return       右アナログトリガーの値を返します。
    //---------------------------------------------------------------------------
    float GetRightAnalogTrigger() const { return m_RightAnalogTrigger; }

    //---------------------------------------------------------------------------
    //! @brief        ポインタの指し示す位置を取得します。
    //!
    //! @return       ポインタの指し示す位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetPointer() const { return m_Pointer; }

    //---------------------------------------------------------------------------
    //! @brief        前フレームのポインタの位置を取得します。
    //!
    //! @return       前フレームのポインタの位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetPointerPrev() const { return m_PointerPrev; }

    //---------------------------------------------------------------------------
    //! @brief        前フレームからのポインタの移動量を取得します。
    //!
    //! @return       ポインタの移動量を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2 GetPointerDiff() const { return m_Pointer - m_PointerPrev; }

    //---------------------------------------------------------------------------
    //! @brief        ポインタが画面内に入っているかを取得します。
    //!
    //! @return       ポインタが画面内に入っている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPointerOn() const { return m_PointerFlag.IsMaskOn( POINTER_ON ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタがこのフレームに画面内に入ったかを取得します。
    //!
    //! @return       ポインタが画面内に入っている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPointerOnNow() const { return m_PointerFlag.IsMaskOn( POINTER_ON_NOW ); }

    //---------------------------------------------------------------------------
    //! @brief        ポインタがこのフレームに画面内から出たかを取得します。
    //!
    //! @return       ポインタが画面内から出た場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPointerOffNow() const { return m_PointerFlag.IsMaskOn( POINTER_OFF_NOW ); }

    //---------------------------------------------------------------------------
    //! @brief        PadProcessor の設定を変更します。
    //!
    //! @param[in]    setting    変更する設定です。
    //---------------------------------------------------------------------------
    void SetPadProcessorSetting( PadProcessor::Setting& setting )
    {
        m_PadProcessor.SetSetting( setting );
    }

    //  @}

    //----------------------------------------
    //! @name ボタンの押下判定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかが押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsHold(u32 mask) const { return (mask & GetHoldMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの斜め入力禁止の十字ボタン処理済みのフラグを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsPressExclusive(u32 mask) const { return (mask & GetPressExclusiveMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームに押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsTrig(u32 mask) const { return (mask & GetTrigMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンが全て押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       全て押されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsHoldAll(u32 mask) const { return (mask & GetHoldMask()) == mask; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームで離されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       離されている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsRelease(u32 mask) const { return (mask & GetReleaseMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームでリピートしているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       リピートしている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsRepeat(u32 mask) const { return (mask & GetRepeatMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがこのフレームで加速リピートしているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       加速リピートしている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsRepeatAccel(u32 mask) const { return (mask & GetRepeatAccelMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかがクリックを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       クリックされた場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsClick(u32 mask) const { return (mask & GetClickMask()) != 0; }

    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンの何れかが長押しされているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       長押しされている場合、 true を返します。
    //---------------------------------------------------------------------------
    bool IsLongPress(u32 mask) const { return (mask & GetLongPressMask()) != 0; }


    //---------------------------------------------------------------------------
    //! @brief        マスク値で与えたボタンが全て押されていて、かつ何れかがこのフレームに
    //!               押されているかを判定します。
    //!
    //! @param[in]    mask       判定するボタンのマスク値です。
    //!
    //! @return       全て押されていて、かつ何れかがこのフレームに押されている場合、true を返します。
    //---------------------------------------------------------------------------
    bool IsHoldAllAndTrigAny(u32 mask) const { return (IsHoldAll(mask) && IsTrig(mask)); }

    //  @}

    //----------------------------------------
    //! @name パッドの設定
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ボタンリピートの設定を行います。
    //!
    //! @param[in]    mask       リピートを設定するボタンをマスク値で設定します。
    //! @param[in]    delayFrame ボタンが押されてから何フレーム後に最初のリピートが発生するかを指定します。
    //! @param[in]    pulseFrame 最初にリピートになった後、何フレーム毎にリピートが発生するかを指定します。
    //---------------------------------------------------------------------------
    void SetPadRepeat(u32 mask, u8 delayFrame, u8 pulseFrame);

    //---------------------------------------------------------------------------
    //! @brief        左アナログスティックを十字ボタンとみなしたときの閾値を設定します。
    //!
    //! @param[in]    hold       押したと判定する閾値を設定します。
    //! @param[in]    release    離したと判定する閾値を設定します。
    //---------------------------------------------------------------------------
    void SetLeftStickCrossThreshold(f32 hold, f32 release);

    //---------------------------------------------------------------------------
    //! @brief        右アナログスティックを十字ボタンとみなしたときの閾値を設定します。
    //!
    //! @param[in]    hold       押したと判定する閾値を設定します。
    //! @param[in]    release    離したと判定する閾値を設定します。
    //---------------------------------------------------------------------------
    void SetRightStickCrossThreshold(f32 hold, f32 release);

    //  @}

    //----------------------------------------
    //! @name ポインタの範囲
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        ポインタの範囲を設定します。
    //!
    //!               ポインタの範囲を指定したい場合に設定します。
    //!               範囲を設定する必要が無い場合には、 min、max にゼロベクトルを指定してください。
    //!
    //! @param[in]    min        ポインタの範囲の最小位置です。
    //! @param[in]    max        ポインタの範囲の最大位置です。
    //---------------------------------------------------------------------------
    void SetPointerBound(const nw::math::VEC2& min, const nw::math::VEC2& max)
    {
        m_PointerBoundMin = min;
        m_PointerBoundMax = max;
    }

    //---------------------------------------------------------------------------
    //! @brief        SetPointerBound で設定したポインタの範囲の最小位置を取得します。
    //!
    //! @return       ポインタの範囲の最小位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetPointerBoundMin() const { return m_PointerBoundMin; }

    //---------------------------------------------------------------------------
    //! @brief        SetPointerBound で設定したポインタの範囲の最大位置を取得します。
    //!
    //! @return       ポインタの範囲の最大位置を返します。
    //---------------------------------------------------------------------------
    const nw::math::VEC2& GetPointerBoundMax() const { return m_PointerBoundMax; }

    //  @}

    //----------------------------------------
    //! @name パッドの状態の取得
    //  @{

    //---------------------------------------------------------------------------
    //! @brief        パッドが接続状態にあるか否かを返します。
    //!
    //! @return       接続されている場合、 true を返します。
    //---------------------------------------------------------------------------
    virtual bool IsConnected() const { return true; }

    //  @}


protected:
    //! @brief ユーザが操作を行っていないかを返します。
    //! @return 操作を行っていない場合は true を返します。
    virtual bool IsIdle();

    //! @brief 入力値を何も操作が行われていない状態にします。
    virtual void SetIdle();

    //! @brief ポインタの範囲を適用しつつ、値を設定します。
    //!
    //! @param[in]    isOn         ポインタが画面に入っているかです。
    //! @param[in]    touchkeyHold タッチしたことにするボタンが押されているかどうかです。
    //! @param[in]    pos          ポインタの座標です。
    void SetPointerWithBound(bool isOn, bool touchkeyHold, const nw::math::VEC2& pos);

    //! @brief m_PadHold 等の値から、m_PadTrig 等の派生パラメータを求めます。
    //!
    //! @param[in]    prevHold      前のフレームの m_PadHold の値です。
    //! @param[in]    prevPointerOn 前のフレームに POINTER_ON が on だったかを表します。
    void UpdateDerivativeParams(u32 prevHold, bool prevPointerOn);

    //! @brief アナログスティックを十字ボタンとみなしたときのマスク値を取得します。
    //!
    //! @param[in]    prevHold         1フレーム前のホールドの状態です。
    //! @param[in]    stick            アナログスティックの状態です。
    //! @param[in]    holdThreshold    押したと判定される閾値です。
    //! @param[in]    releaseThreshold 離したと判定される閾値です。
    //! @param[in]    startBit         十字ボタンの enum 値の最初のビットを指定します。
    //! @return       マスク値です。
    u32 GetStickHold(u32 prevHold, const nw::math::VEC2& stick, float holdThreshold, float releaseThreshold, int startBit);

    //! @brief スティックを十字ボタンとみなしたときの全てのキーのフラグが立ったマスク値を作成します。
    //!
    //! @return       マスク値です。
    u32 CreateStickCrossMask();
#else
    PadBase(int, int, int, int) {}

    virtual ~PadBase() {}

    u32 GetHoldMask() const { return 0; }

    u32 GetPressExclusiveMask() const { return 0; }

    u32 GetTrigMask() const { return 0; }

    u32 GetReleaseMask() const { return 0; }

    u32 GetRepeatMask() const { return 0; }

    u32 GetRepeatAccelMask() const { return 0; }

    u32 GetClickMask() const { return 0; }

    u32 GetLongPressMask() const { return 0; }

    u32 GetPadHoldCount(int) const { return 0; }

    const nw::math::VEC2& GetLeftStick() const { return m_LeftStick; }

    const nw::math::VEC2& GetRightStick() const { return m_RightStick; }

    float GetLeftAnalogTrigger() const { return 0.f; }

    float GetRightAnalogTrigger() const { return 0.f; }

    const nw::math::VEC2& GetPointer() const { return m_Pointer; }

    const nw::math::VEC2& GetPointerPrev() const { return m_PointerPrev; }

    const nw::math::VEC2 GetPointerDiff() const { return m_Pointer; }

    bool IsPointerOn() const { return false; }

    bool IsPointerOnNow() const { return false; }

    bool IsPointerOffNow() const { return false; }

    void SetPadProcessorSetting( PadProcessor::Setting& ) {}

    bool IsHold(u32) const { return false; }

    bool IsPressExclusive(u32) const { return false; }

    bool IsTrig(u32) const { return false; }

    bool IsHoldAll(u32) const { return false; }

    bool IsRelease(u32) const { return false; }

    bool IsRepeat(u32) const { return false; }

    bool IsRepeatAccel(u32) const { return false; }

    bool IsClick(u32) const { return false; }

    bool IsLongPress(u32) const { return false; }

    bool IsHoldAllAndTrigAny(u32) const { return false; }

    void SetPadRepeat(u32, u8, u8) {}

    void SetLeftStickCrossThreshold(f32, f32) {}

    void SetRightStickCrossThreshold(f32, f32) {}

    void SetPointerBound(const nw::math::VEC2&, const nw::math::VEC2&) {}

    const nw::math::VEC2& GetPointerBoundMin() const { return m_PointerBoundMin; }

    const nw::math::VEC2& GetPointerBoundMax() const { return m_PointerBoundMax; }

    virtual bool IsConnected() const { return false; }

protected:
    virtual bool IsIdle() { return false; }

    virtual void SetIdle() {}

    void SetPointerWithBound(bool, bool, const nw::math::VEC2&) {}

    void UpdateDerivativeParams(u32, bool) {}

    u32 GetStickHold(u32, const nw::math::VEC2&, float, float, int) { return 0; }

    u32 CreateStickCrossMask() { return 0; }
#endif // NW_DEV_PAD_ENABLE

    //! @param パッドで表すことができるボタンの最大値です。
    enum
    {
        PAD_IDX_MAX_BASE = 32   //!< パッドで表すことができるボタンの最大値です。
    };

    //! @brief ポインタの位置についての情報です。
    enum PointerFlagMask
    {
        POINTER_ON           = 1,       //!< ポインタが画面内に入っている
        POINTER_ON_NOW       = 1 << 1,  //!< ポインタがこのフレームに画面内に入った
        POINTER_OFF_NOW      = 1 << 2,  //!< ポインタがこのフレームに画面内から出た
    };

    //! @brief 十字ボタン単体のボタンの定義の順番です。
    enum
    {
        CROSS_UP,
        CROSS_DOWN,
        CROSS_LEFT,
        CROSS_RIGHT
    };

    nw::dev::PadProcessor m_PadProcessor;   //!< PadProcessor です。
    nw::ut::BitFlag32 m_PadHold;            //!< 今押されているボタンです。
    nw::ut::BitFlag32 m_PointerFlag;        //!< ポインタの状態を保持するフラグです。
    float m_LeftStickHoldThreshold;         //!< 左アナログスティックを十字ボタンとみなしたとき、押したことにする閾値です。
    float m_RightStickHoldThreshold;        //!< 右アナログスティックを十字ボタンとみなしたとき、押したことにする閾値です。
    float m_LeftStickReleaseThreshold;      //!< 左アナログスティックを十字ボタンとみなしたとき、押したことにする閾値です。
    float m_RightStickReleaseThreshold;     //!< 右アナログスティックを十字ボタンとみなしたとき、押したことにする閾値です。

    int m_PadBitMax;                        //!< パッドのビット値の最大値です。
    int m_LeftStickCrossStartBit;           //!< 左スティックを十字ボタンとして扱うとき、何ビット目から始まるか。存在しないときは-1。
    int m_RightStickCrossStartBit;          //!< 左スティックを十字ボタンとして扱うとき、何ビット目から始まるか。存在しないときは-1。
    int m_TouchKeyBit;                      //!< タッチキーが何ビット目か。存在しないときは-1。

    nw::math::VEC2 m_Pointer;               //!< ポインティングデバイスが指し示している位置です。
    nw::math::VEC2 m_PointerPrev;           //!< 前フレームのポインティングデバイスの位置です。
    nw::math::VEC2 m_PointerBoundMin;       //!< ポインタの範囲の最小位置です。
    nw::math::VEC2 m_PointerBoundMax;       //!< ポインタの範囲の最大位置です。
    nw::math::VEC2 m_LeftStick;             //!< 左アナログスティックの状態。値は[-1.f,1.f]の範囲となり、右および上の方向が正。
    nw::math::VEC2 m_RightStick;            //!< 右アナログスティックの状態。値は[-1.f,1.f]の範囲となり、右および上の方向が正。
    float m_LeftAnalogTrigger;              //!< 左トリガーの状態(GCコンのLボタンを想定)。値は[0.f,1.f]の範囲となり、もっとも押し込まれたときが1.f。
    float m_RightAnalogTrigger;             //!< 右トリガーの状態(GCコンのRボタンを想定)。値は[0.f,1.f]の範囲となり、もっとも押し込まれたときが1.f。

    static const f32 STICK_HOLD_THRESHOLD_DEFAULT;      //!< アナログスティックを十字ボタンとみなしたとき、どこまで傾けたら押したことにするかのデフォルト値です。
    static const f32 STICK_RELEASE_THRESHOLD_DEFAULT;   //!< アナログスティックを十字ボタンとみなしたとき、どこまで戻したら離したことにするかのデフォルト値です。
};

} // namespace dev
} // namespace nw

#endif // NW_DEV_PAD_BASE_H_
