﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_DEV_MEMORY_H_
#define NW_DEV_MEMORY_H_

#include <nw/types.h>
#include <nw/dev/dev_Config.h>

#if defined(NW_DEV_ENABLED)

#if defined(NW_PLATFORM_WIN32)
    #include <nn/lmem/lmem_ExpHeap.h>
#elif defined(NW_PLATFORM_CAFE)
    #include <cafe/mem.h>
#endif

#if defined(NW_COMPILER_CLANG)
#include <stdlib.h>
#endif

namespace nw
{
namespace dev
{

//---------------------------------------------------------------------------
//! @brief malloc を利用した標準ライブラリ用のアロケータです。
//---------------------------------------------------------------------------
template <typename T>
class StdAllocator
{
public:
    typedef size_t    size_type;
    typedef ptrdiff_t difference_type;
    typedef T*        pointer;
    typedef const T*  const_pointer;
    typedef T&        reference;
    typedef const T&  const_reference;
    typedef T         value_type;

    StdAllocator() {}
    StdAllocator(const StdAllocator&) {}

    pointer allocate(size_type n, const void * = 0)
    {
        T* t = reinterpret_cast<T*>(malloc(n * sizeof(T)));
        return t;
    }

    void deallocate(void* p, size_type)
    {
        if (p) { free(p); }
    }

    pointer address(reference x) const { return &x; }
    const_pointer address(const_reference x) const { return &x; }
    StdAllocator<T>& operator=(const StdAllocator&) { return *this; }
    void construct(pointer p, const T& val) { new ((T*) p) T(val); }
    void destroy(pointer p) { p->~T(); }

    size_type max_size() const { return size_t(-1); }

    template <typename U>
    struct rebind { typedef StdAllocator<U> other; };

    template <typename U>
    StdAllocator(const StdAllocator<U>&) {}

    template <typename U>
    StdAllocator& operator=(const StdAllocator<U>&) { return *this; }
};

#if defined(NW_PLATFORM_WIN32)

//---------------------------------------------------------------------------
//! @brief 拡張ヒープ を利用した標準ライブラリ用のアロケータです。
//---------------------------------------------------------------------------
template <typename T>
class HeapAllocator
{
public:
    typedef size_t    size_type;
    typedef ptrdiff_t difference_type;
    typedef T*        pointer;
    typedef const T*  const_pointer;
    typedef T&        reference;
    typedef const T&  const_reference;
    typedef T         value_type;

    HeapAllocator(void* memory, size_t capacity)
    {
        m_Heap = nn::lmem::CreateExpHeap(memory, capacity, nn::lmem::CreationOption_ZeroClear);
    }
    HeapAllocator(const HeapAllocator& allocator) : m_Heap(allocator.m_Heap) {}

    pointer allocate(size_type n, const void * = 0)
    {
        T* t = reinterpret_cast<T*>(nn::lmem::AllocateFromExpHeap(m_Heap, n * sizeof(T), 4));
        return t;
    }

    void deallocate(void* p, size_type)
    {
        if (p) { nn::lmem::FreeToExpHeap(m_Heap, p); }
    }

    pointer address(reference x) const { return &x; }
    const_pointer address(const_reference x) const { return &x; }
    HeapAllocator<T>& operator=(const HeapAllocator&) { return *this; }
    void construct(pointer p, const T& val) { new ((T*) p) T(val); }
    // TODO: p が未使用変数となる。
    void destroy(pointer p) { NN_UNUSED(p); p->~T(); }

    size_type max_size() const
    {
        //u32 allocatableSize = MEMGetAllocatableSizeForExpHeap(m_Heap);
        //NW_LOG("#### %08x, allocatable size = %d\n", reinterpret_cast<u32>(m_Heap), allocatableSize);
        return nn::lmem::GetExpHeapAllocatableSize(m_Heap, 4) / sizeof(T);
    }

    template <typename U>
    struct rebind { typedef HeapAllocator<U> other; };

    template <typename U>
    HeapAllocator(const HeapAllocator<U>& allocator) : m_Heap(allocator.GetHeap()) {}

    template <typename U>
    HeapAllocator& operator=(const HeapAllocator<U>&) { return *this; }

    nn::lmem::HeapHandle GetHeap() const { return m_Heap; }

private:
    nn::lmem::HeapHandle m_Heap;
};

#elif defined(NW_PLATFORM_CAFE)
//---------------------------------------------------------------------------
//! @brief 拡張ヒープ を利用した標準ライブラリ用のアロケータです。
//---------------------------------------------------------------------------
template <typename T>
class HeapAllocator
{
public:
    typedef size_t    size_type;
    typedef ptrdiff_t difference_type;
    typedef T*        pointer;
    typedef const T*  const_pointer;
    typedef T&        reference;
    typedef const T&  const_reference;
    typedef T         value_type;

    HeapAllocator(void* memory, size_t capacity)
    {
        m_Heap = MEMCreateExpHeap(memory, capacity);
    }
    HeapAllocator(const HeapAllocator& allocator) : m_Heap(allocator.m_Heap) {}

    pointer allocate(size_type n, const void * = 0)
    {
        T* t = reinterpret_cast<T*>(MEMAllocFromExpHeap(m_Heap, n * sizeof(T)));
        return t;
    }

    void deallocate(void* p, size_type)
    {
        if (p) { MEMFreeToExpHeap(m_Heap, p); }
    }

    pointer address(reference x) const { return &x; }
    const_pointer address(const_reference x) const { return &x; }
    HeapAllocator<T>& operator=(const HeapAllocator&) { return *this; }
    void construct(pointer p, const T& val) { new ((T*) p) T(val); }
    void destroy(pointer p) { p->~T(); }

    size_type max_size() const
    {
        //u32 allocatableSize = MEMGetAllocatableSizeForExpHeap(m_Heap);
        //NW_LOG("#### %08x, allocatable size = %d\n", reinterpret_cast<u32>(m_Heap), allocatableSize);
        return MEMGetAllocatableSizeForExpHeap(m_Heap) / sizeof(T);
    }

    template <typename U>
    struct rebind { typedef HeapAllocator<U> other; };

    template <typename U>
    HeapAllocator(const HeapAllocator<U>& allocator) : m_Heap(allocator.GetHeap()) {}

    template <typename U>
    HeapAllocator& operator=(const HeapAllocator<U>&) { return *this; }

    MEMHeapHandle GetHeap() const { return m_Heap; }

private:
    MEMHeapHandle m_Heap;
};

#endif // NW_PLATFORM

} // namespace nw::dev
} // namespace nw

#endif // NW_DEV_ENABLED

#endif // NW_DEV_MEMORY_H_
