﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_CTRL_CHECK_BUTTON_H_
#define NW_CTRL_CHECK_BUTTON_H_

#include <nw/ctrl/ctrl_AnimButton.h>

namespace nw
{

namespace ctrl
{

//---------------------------------------------------------------------------
//! @brief ボタンが押されるたびにトグルで特定のアニメの状態が切り替わるボタンです。
//!
//! @details
//! NormalButtonとほぼ同じ挙動ですが、いわゆるチェックボックスのようなUIを実現するための
//! ボタンです。
//!
//! このボタンには、ON/OFFとDOWNの他にもう一つ、Checkアニメと呼ばれるアニメが必要です。
//! Cancelアニメはパラメータアニメの3つ目に指定してください。
//! ボタンが押されると、Checkアニメが順再生されます。この状態でもう一度ボタンが押されると、
//! 今度はCheckアニメが逆再生されます。
//!
//! CheckButtonのSTATE_DOWNは、Checkアニメが再生終了しない限り終了しませんのでご注意ください。
//!
//! また、CheckアニメはDownアニメと同時に再生しますので、同時に再生しても問題ないアニメに
//! してください。例えばDownアニメではスケールを使用して、Checkアニメではマテリアルカラー
//! を使用する、あるいはCheckアニメの対象のペインはDownアニメとは全く別のペインにする
//! 等の方法があります。
//!
//! チェックマークが入った状態か否かは、IsChecked()メソッドで取得することができます。
//!
//! CheckButtonは、NormalButtonと同様にデフォルトではIsExcludeDownがfalseになっています。
//!
//---------------------------------------------------------------------------
class CheckButton : public AnimButton
{
public:
    //! 実行時型情報です。
    NW_UT_RUNTIME_TYPEINFO(AnimButton);

    //! @brief コンストラクタです。
    //!
    CheckButton() : AnimButton(), m_IsChecked(false), m_CheckAnimator(NULL) {}

    //! @brief ボタンの構築を行います。
    //!
    //! AnimButtonの標準の動作に加えて、パラメータアニメの3つ目に指定された
    //! Checkアニメを作成します。
    //!
    //! @param[in] controlSrc   コントロールの情報
    //! @param[in,out] layout   所属するレイアウト
    //!
    void Build(const lyt::ControlSrc& controlSrc, lyt::Layout* layout);

    //----------------------------------------
    //! @name アニメの管理
    //@{
    //! @brief ボタンのCheckアニメを取得します。
    //!
    //! @return アニメ
    //!
    lyt::Animator* GetCheckAnimator() const { return m_CheckAnimator; }

    //! @brief ボタンのCheckアニメを設定します。
    //!
    //! @param[in] animator 設定するアニメ
    //!
    void SetCheckAnimator(lyt::Animator* animator) { m_CheckAnimator = animator; }
    //@}

    //----------------------------------------
    //! @name 状態の取得・設定
    //@{
    //! @brief チェック状態か否かを取得します。
    //!
    //! @return チェック状態であればtrue。Build直後はfalseになっています。
    //!
    bool IsChecked() const { return m_IsChecked; }

    //! @brief 強制的にチェック状態を変更し、チェックアニメも指定した状態に設定します。
    //!
    //! @details
    //! レイアウトが表示されているときにこのメソッドを使用すると、急に状態が変わった
    //! ように見えてしまう場合がありますのでご注意ください。
    //!
    //! レイアウトを表示する前に、既にチェックされた状態にしておきたいような場合に
    //! 使うことを想定しています。
    //!
    //! @param[in] checked  チェック状態にするか否か。
    //!
    void ForceSetChecked(bool checked);
    //@}

protected:
    bool m_IsChecked;
    lyt::Animator* m_CheckAnimator;

    //! @brief ボタンの挙動をカスタマイズするためにオーバーライドします。
    //!
    virtual void FinishDown();
    virtual void StartDown();
    virtual bool UpdateDown();

};

} // namespace nw::ctrl
} // namespace nw

#endif // NW_CTRL_SELECT_BUTTON_H_
