﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_ASSERT_H_
#define NW_ASSERT_H_

#include <nw/types.h>

#include <nw/config.h>
#include <limits.h>

#if !defined( NW_RELEASE )
  #define NW_ASSERTION_ENABLE
#endif

#if defined( NW_DEBUG )
  #define NW_INTERNAL_ASSERTION_ENABLE
#endif

namespace nw
{
namespace internal
{
template<bool> struct nw_static_assert;
template<> struct nw_static_assert<true> {};
template<int> struct nw_static_check {};
} // namespace internal
} // namespace nw

//--------------------------------------------------------------------------------
// コンパイル時アサーションです。
//
// マクロの引数にテンプレート引数などのカンマ "," が含まれると
// カンマがマクロ引数と解釈されます。引数の中を括弧で括らないと構文エラーとなります。
// 例： NW_STATIC_ASSERT( (nw::ut::IsSame<Type, int>::value) );

// C でも使えるバージョンの NW_STATIC_ASSERT_C です。
#if defined( __ghs__ )
    #define NW_STATIC_ASSERT_C(expr) \
        extern void nw_compiler_assert( char is[(expr) ? +1 : -1] )
#else
    #define NW_STATIC_ASSERT_C(expr) \
        extern void nw_compiler_assert ## __LINE__ ( char is[(expr) ? +1 : -1] )
#endif

// テンプレート版です。
#define NW_STATIC_ASSERT(expr) \
    typedef nw::internal::nw_static_check<sizeof(nw::internal::nw_static_assert<expr>)> STATIC_ASSERT_FAILED

// 詳しくは NW_STATIC_ASSERT を参照してください。
#define NW_COMPILER_ASSERT NW_STATIC_ASSERT


//--------------------------------------------------------------------------------
//  Assert with message
//
#if defined( NW_ASSERTION_ENABLE )

  #if defined( NW_PLATFORM_CAFE )

    #define NW_TASSERTMSG(exp, ...)                                 \
        (void) ((exp) ||                                            \
                (::nw::ut::TPanic(NW_FILE_NAME, __LINE__, __VA_ARGS__), 0))

    #define NW_ASSERTMSG(exp, ...)                                  \
        (void) ((exp) ||                                            \
                (::nw::ut::Panic(NW_FILE_NAME, __LINE__, __VA_ARGS__), 0))

  #else

    #ifdef NW_FROM_TOOL
      #define NW_TASSERTMSG(exp, ... )     \
          if(!(exp)) { NW_LOG(__VA_ARGS__); NW_HALT; }

      #define NW_ASSERTMSG(exp, ... )     \
          if(!(exp)) { NW_LOG(__VA_ARGS__); NW_HALT; }
    #else
      #define NW_TASSERTMSG(exp, ... )     \
          if(!((void)0, exp)) { ::nw::ut::HaltWithDetail(__FUNCTION__, __LINE__, __VA_ARGS__); }

      #define NW_ASSERTMSG(exp, ... )     \
          if(!((void)0, exp)) { ::nw::ut::HaltWithDetail(__FUNCTION__, __LINE__, __VA_ARGS__); }
    #endif
  #endif

#else

  #if defined( NW_PLATFORM_CAFE )

    #define NW_TASSERTMSG(exp, ...) ((void)0)
    #define NW_ASSERTMSG(exp, ...)  ((void)0)

  #else

    #if defined(_MSC_VER) && _MSC_VER >= 1700
      #define NW_TASSERTMSG(exp, ...) do { (void)(exp); } while((void)0, 0)
      #define NW_ASSERTMSG(exp, ...)  do { (void)(exp); } while((void)0, 0)
    #else
      #define NW_TASSERTMSG(exp, ...) do { (void)sizeof(exp); } while((void)0, 0)
      #define NW_ASSERTMSG(exp, ...)  do { (void)sizeof(exp); } while((void)0, 0)
    #endif

  #endif

#endif


//--------------------------------------------------------------------------------
//  Err
//
#ifndef NW_ERR
#define NW_ERR(...) \
    { NW_TASSERTMSG(false, __VA_ARGS__); }
#endif

//--------------------------------------------------------------------------------
//  Assert
//
#ifndef NW_ASSERT
#define NW_ASSERT(exp) \
            NW_TASSERTMSG( (exp), "NW:Failed assertion " #exp )
#endif

//--------------------------------------------------------------------------------
//  NULL Assert
//
#ifndef NW_ASSERT_NOT_NULL
#define NW_ASSERT_NOT_NULL(exp) \
            NW_TASSERTMSG( (exp) != NULL, "NW:Pointer must not be NULL ("#exp")" )
#endif

//--------------------------------------------------------------------------------
//  NULL Assert
//
#ifndef NW_ASSERT_VALID_POINTER
#define NW_ASSERT_VALID_POINTER(exp) \
            NW_TASSERTMSG( (exp) != NULL, "NW:Pointer must not be NULL ("#exp")" )
#endif

//--------------------------------------------------------------------------------
//  Min Assert
//
#ifndef NW_ASSERT_MIN
#define NW_ASSERT_MIN(exp, min) \
            NW_TASSERTMSG( (exp) >= (min), #exp " is out of bounds(%d)\n%d <= "#exp" not satisfied.", static_cast<int>(exp), static_cast<int>(min) )
#endif

//--------------------------------------------------------------------------------
//  Max Assert
//
#ifndef NW_ASSERT_MAX
#define NW_ASSERT_MAX(exp, max) \
            NW_TASSERTMSG( (exp) <= (max), #exp " is out of bounds(%d)\n"#exp" <= %d not satisfied.", static_cast<int>(exp), static_cast<int>(max) )
#endif

//--------------------------------------------------------------------------------
//  Max Less Than Assert
//
#ifndef NW_ASSERT_MAXLT
#define NW_ASSERT_MAXLT(exp, max) \
            NW_TASSERTMSG( (exp) < (max), #exp " is out of bounds(%d)\n"#exp" < %d not satisfied.", static_cast<int>(exp), static_cast<int>(max) )
#endif

//--------------------------------------------------------------------------------
//  Min Max Assert
//
#ifndef NW_ASSERT_MINMAX
#define NW_ASSERT_MINMAX(exp, min, max) \
            NW_TASSERTMSG( (exp) >= (min) && (exp) <= (max), #exp " is out of bounds(%d)\n%d <= "#exp" <= %d not satisfied.", static_cast<int>(exp), static_cast<int>(min), static_cast<int>(max))
#endif

//--------------------------------------------------------------------------------
//  Min Max Less Than Assert
//
#ifndef NW_ASSERT_MINMAXLT
#define NW_ASSERT_MINMAXLT(exp, min, max) \
            NW_TASSERTMSG( (exp) >= (min) && (exp) < (max), #exp " is out of bounds(%d)\n%d <= "#exp" < %d not satisfied.", static_cast<int>(exp), static_cast<int>(min), static_cast<int>(max))
#endif

//--------------------------------------------------------------------------------
//  Min Assert for floating point value
//
#ifndef NW_ASSERT_FMIN
#define NW_ASSERT_FMIN(exp, min) \
            NW_ASSERTMSG( (exp) >= (min), #exp " is out of bounds(%f)\n%f <= "#exp" not satisfied.", static_cast<double>(exp), static_cast<double>(min) )
#endif

//--------------------------------------------------------------------------------
//  Max Assert for floating point value
//
#ifndef NW_ASSERT_FMAX
#define NW_ASSERT_FMAX(exp, max) \
            NW_ASSERTMSG( (exp) <= (max), #exp " is out of bounds(%f)\n"#exp" <= %f not satisfied.", static_cast<double>(exp), static_cast<double>(max) )
#endif

//--------------------------------------------------------------------------------
//  Min Max Assert for floating point value
//
#ifndef NW_ASSERT_FMINMAX
#define NW_ASSERT_FMINMAX(exp, min, max) \
            NW_ASSERTMSG( (exp) >= (min) && (exp) <= (max), #exp " is out of bounds(%f)\n%f <= "#exp" <= %f not satisfied.", static_cast<double>(exp), static_cast<double>(min), static_cast<double>(max))
#endif

//--------------------------------------------------------------------------------
//  Fatal error
//
#ifndef NW_FATAL_ERROR
#ifdef  NW_FROM_TOOL
#define NW_FATAL_ERROR    true ? 0:
#else
#define NW_FATAL_ERROR(...) \
            NW_TASSERTMSG( false, "NW:Fatal Error\n" __VA_ARGS__ )
#endif
#endif

//--------------------------------------------------------------------------------
//  Internal error
//
#ifndef NW_INTERNAL_ERROR
#ifdef  NW_FROM_TOOL
#define NW_INTERNAL_ERROR true ? 0:
#else
#define NW_INTERNAL_ERROR(...) \
            NW_TASSERTMSG( false, "NW Internal error\n" __VA_ARGS__ )
#endif
#endif


//--------------------------------------------------------------------------------
//  Cache Alignment error
//
#ifndef NW_ASSERT_ALIGN_CACHE
#if defined( NW_PLATFORM_CAFE )
  #define NW_ASSERT_ALIGN_CACHE     NW_ASSERT_ALIGN64
#else
  #define NW_ASSERT_ALIGN_CACHE     NW_ASSERT_ALIGN32
#endif
#endif


//--------------------------------------------------------------------------------
//  Alignment error
//
#ifndef NW_ASSERT_ALIGN
#define NW_ASSERT_ALIGN(exp,align) \
    NW_TASSERTMSG( (NW_ANY_TO_PTR_VALUE(exp) & ((align) - 1)) == 0, "NW:Alignment Error(0x%x)\n"#exp" must be aligned to %d bytes boundary.", exp, align )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(128 bytes)
//
#ifndef NW_ASSERT_ALIGN128
#define NW_ASSERT_ALIGN128(exp)     NW_ASSERT_ALIGN( (exp), 128 )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(64 bytes)
//
#ifndef NW_ASSERT_ALIGN64
#define NW_ASSERT_ALIGN64(exp)      NW_ASSERT_ALIGN( (exp), 64 )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(32 bytes)
//
#ifndef NW_ASSERT_ALIGN32
#define NW_ASSERT_ALIGN32(exp)      NW_ASSERT_ALIGN( (exp), 32 )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(8 bytes)
//
#ifndef NW_ASSERT_ALIGN8
#define NW_ASSERT_ALIGN8(exp)       NW_ASSERT_ALIGN( (exp), 8 )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(4 bytes)
//
#ifndef NW_ASSERT_ALIGN4
#define NW_ASSERT_ALIGN4(exp)       NW_ASSERT_ALIGN( (exp), 4 )
#endif

//--------------------------------------------------------------------------------
//  Alignment error(2 bytes)
//
#ifndef NW_ASSERT_ALIGN2
#define NW_ASSERT_ALIGN2(exp)       NW_ASSERT_ALIGN( (exp), 2 )
#endif

//--------------------------------------------------------------------------------
//  Range error
//

#ifndef NW_ASSERT_RANGE_U8
#define NW_ASSERT_RANGE_U8(exp)     NW_ASSERT_MINMAX(exp, 0, UCHAR_MAX)
#endif

#ifndef NW_ASSERT_RANGE_S8
#define NW_ASSERT_RANGE_S8(exp)     NW_ASSERT_MINMAX(exp, SCHAR_MIN, SCHAR_MAX)
#endif

#ifndef NW_ASSERT_RANGE_U16
#define NW_ASSERT_RANGE_U16(exp)    NW_ASSERT_MINMAX(exp, 0, USHRT_MAX)
#endif

#ifndef NW_ASSERT_RANGE_S16
#define NW_ASSERT_RANGE_S16(exp)    NW_ASSERT_MINMAX(exp, SHRT_MIN, SHRT_MAX)
#endif

#ifndef NW_ASSERT_RANGE_U32
#define NW_ASSERT_RANGE_U32(exp)    NW_ASSERT_MINMAX(exp, 0, UINT_MAX)
#endif

#ifndef NW_ASSERT_RANGE_S32
#define NW_ASSERT_RANGE_S32(exp)    NW_ASSERT_MINMAX(exp, INT_MIN, INT_MAX)
#endif

//--------------------------------------------------------------------------------
//  Floating point value error
//

#ifndef NW_ASSERT_VALID_FLOAT
#ifdef  NW_FROM_TOOL
#define NW_ASSERT_VALID_FLOAT   true ? 0:
#else
#define NW_ASSERT_VALID_FLOAT(exp) \
        NW_ASSERTMSG((::nw::isfinite(exp) && !::nw::isnan(exp)), "NW:Floating Point Value Error(%f)\n"#exp" is infinite or nan.", exp)
#endif
#endif

//--------------------------------------------------------------------------------
//  Index Bounds error
//

#ifndef NW_ASSERT_INDEX_BOUNDS
#define NW_ASSERT_INDEX_BOUNDS(index, size)                                            \
    NW_TASSERTMSG(static_cast<uint>(index) < static_cast<uint>(size),                   \
    "NW: " #index " is out of bounds. (%d)\n"                                          \
    "0 <= " #index " < %d is not satisfied.\n",                                        \
    static_cast<int>(index), static_cast<int>(size))
#endif

//--------------------------------------------------------------------------------
//  Not Implemented error
//
#ifndef NW_NOT_IMPLEMENTED
#define NW_NOT_IMPLEMENTED() \
    NW_TASSERTMSG(0, "NW: Not implemented yet.\n")
#endif


#if !defined( NW_ASSERTION_ENABLE ) && !defined(NW_GL_ASSERT)
#define NW_GL_ASSERT() ((void)0)
#else

inline const char*
nwGlErrorString( u32 err )
{
    typedef struct
    {
        u32 value;
        const char* name;
    } EnumString;

    static const EnumString ENUM_STRINGS[] =
    {
        { 0,      "GL_NO_ERROR"      },
        { 0x0500, "GL_INVALID_ENUM"  },
        { 0x0501, "GL_INVALID_VALUE" },
        { 0x0502, "GL_INVALID_OPERATION" },
        { 0x0505, "GL_OUT_OF_MEMORY" },
    };

    const int NUM_ENUM_STRING = sizeof(ENUM_STRINGS) / sizeof(ENUM_STRINGS[0]);

    for (int idx = 0; idx < NUM_ENUM_STRING; ++idx)
    {
        if ( ENUM_STRINGS[ idx ].value == err )
        {
            return ENUM_STRINGS[ idx ].name;
        }
    }

    return "Unknown";
}

#if defined(NW_PLATFORM_WIN32) && !defined(NW_GL_ASSERT)
#define NW_GL_ASSERT()                                              \
    do {                                                            \
        GLuint err = glGetError();                                  \
        NW_TASSERTMSG( err == GL_NO_ERROR, "GL_ERROR : %s (0x%x)", nwGlErrorString( err ), err ); \
    } while ((void)0, 0)
#else
  #define NW_GL_ASSERT() ((void)0)
#endif

#endif

//--------------------------------------------------------------------------------
//  Warning
//

#if defined( NW_ASSERTION_ENABLE )
#define NW_WARNING(exp, ...)  if(!((void)0, exp)) { NW_LOG("%s:%d Warning:", NW_FILE_NAME, __LINE__); NW_LOG(__VA_ARGS__); NW_LOG("\n"); }
#define NW_TWARNING(exp, ...) if(!((void)0, exp)) { NW_TLOG("%s:%d Warning:", NW_FILE_NAME, __LINE__); NW_TLOG(__VA_ARGS__); NW_TLOG("\n"); }
#else
#if defined( NW_PLATFORM_CAFE )
#define NW_WARNING(exp, ...)   ((void)0)
#define NW_TWARNING(exp, ...)  ((void)0)
#else
#if defined(_MSC_VER) && _MSC_VER >= 1700
#define NW_WARNING(exp, ...) do { (void)(exp); } while((void)0, 0)
#define NW_TWARNING(exp, ...)  do { (void)(exp); } while((void)0, 0)
#else
#define NW_WARNING(exp, ...) do { (void)sizeof(exp); } while((void)0, 0)
#define NW_TWARNING(exp, ...)  do { (void)sizeof(exp); } while((void)0, 0)
#endif
#endif
#endif

//--------------------------------------------------------------------------------
//  Fail safe
//

#if defined( NW_ASSERTION_ENABLE )
#define NW_FAILSAFE_IF(exp) if (exp) { NW_FATAL_ERROR(#exp); } if ((void)0, 0)
#else
#define NW_FAILSAFE_IF(exp) if (exp)
#endif

//--------------------------------------------------------------------------------
//  デバッグ時のみ有効で Development ではチェックしない ASSERT
//

#if defined( NW_INTERNAL_ASSERTION_ENABLE )
#define NW_INTERNAL_ASSERTMSG           NW_ASSERTMSG
#define NW_INTERNAL_ASSERT              NW_ASSERT
#define NW_INTERNAL_ASSERT_NOT_NULL     NW_ASSERT_NOT_NULL
#else
#define NW_INTERNAL_ASSERTMSG(exp, ...)  ((void)0)
#define NW_INTERNAL_ASSERT(exp)          ((void)0)
#define NW_INTERNAL_ASSERT_NOT_NULL(exp) ((void)0)
#endif

#define NW_ASSERT_ALIGNMENT NW_ASSERT_ALIGN


// 旧互換の為の定義。

#if 1

#define NW_NULL_ASSERT          NW_ASSERT_NOT_NULL
#define NW_POINTER_ASSERT       NW_ASSERT_VALID_POINTER
#define NW_MIN_ASSERT           NW_ASSERT_MIN
#define NW_MAX_ASSERT           NW_ASSERT_MAX
#define NW_MINMAX_ASSERT        NW_ASSERT_MINMAX
#define NW_MINMAXLT_ASSERT      NW_ASSERT_MINMAXLT
#define NW_FMIN_ASSERT          NW_ASSERT_FMIN
#define NW_FMAX_ASSERT          NW_ASSERT_FMAX
#define NW_FMINMAX_ASSERT       NW_ASSERT_FMINMAX
#define NW_CACHE_ALIGN_ASSERT   NW_ASSERT_ALIGN_CACHE
#define NW_ALIGN_ASSERT         NW_ASSERT_ALIGN
#define NW_ALIGN2_ASSERT        NW_ASSERT_ALIGN2
#define NW_ALIGN4_ASSERT        NW_ASSERT_ALIGN4
#define NW_ALIGN32_ASSERT       NW_ASSERT_ALIGN32
#define NW_ALIGN128_ASSERT      NW_ASSERT_ALIGN128
#define NW_U8_RANGE_ASSERT      NW_ASSERT_RANGE_U8
#define NW_S8_RANGE_ASSERT      NW_ASSERT_RANGE_S8
#define NW_U16_RANGE_ASSERT     NW_ASSERT_RANGE_U16
#define NW_S16_RANGE_ASSERT     NW_ASSERT_RANGE_S16
#define NW_U32_RANGE_ASSERT     NW_ASSERT_RANGE_U32
#define NW_S32_RANGE_ASSERT     NW_ASSERT_RANGE_S32
#define NW_FLOAT_ASSERT         NW_ASSERT_VALID_FLOAT


// g3d のビルドを通す為の定義
#include <cstdarg>

namespace nw {
namespace ut {
void VPrintf(const char *fmt, std::va_list vlist);
}

//! @briefprivate
//! @param[in] format 書式付き文字列です。
inline void
DebugPrint(const char* format, ...)
{
    va_list arglist;
    va_start(arglist, format);
    ut::VPrintf(format, arglist);
    va_end(arglist);
}

}

#endif

/* NW_ASSERT_H_ */
#endif
