﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief シェアページを開くのに利用される引数の宣言(非公開)
 */

#pragma once

#include <nn/capsrv.h>
#include <nn/applet/applet.h>
#include <nn/account/account_Types.h>
#include <nn/web/web_ShareTypesPrivate.h>

namespace nn { namespace web {

/**
 * @brief シェアページを開くための情報を構築するクラスです。
 */
class ShowSharePageArg
{
public:
    /**
     * @brief NULL 終端を含むコールバック可能として指定できる URL の最大バイト数です。
     */
    static const size_t CallbackableUrlMaxLength = 1024;

    /**
     * @brief NULL 終端を含むシェアページに受け渡す文字列の最大バイト数です。
     */
    static const size_t AdditionalCommentTextMaxLength = 256;

    /**
     * @brief シェアページに受け渡すメディアファイルの追加情報の最大バイト数です。
     */
    static const size_t AdditionalMediaDataMaxLength = 16;

    /**
     * @brief シェアページに受け渡すことができるアルバム内のファイル情報の最大個数です。
     */
    static const size_t AlbumEntryMaxCount = 4;

    /**
     * @brief シェアページを開くための最低限の情報を持つクラスを作成します。
     *
     * @param[in]   shareStartPage  シェアページの開始ページを指定してください。
     */
    explicit ShowSharePageArg(ShareStartPage shareStartPage) NN_NOEXCEPT;

    /**
     * @brief アルバム内のファイル情報を設定します。
     *
     * @param[in]   albumEntry  アルバム内のファイル情報を指定してください。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     */
    void SetAlbumEntry(const nn::capsrv::AlbumEntry& albumEntry) NN_NOEXCEPT;

    /**
     * @brief アルバム内のファイル情報を設定します。
     *
     * @param[in]   applicationAlbumEntry   アルバム内のファイル情報を指定してください。
     *
     * @details nn::album を経由した一般アプリケーションからの利用を想定しています
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetApplicationAlbumEntry(const nn::capsrv::ApplicationAlbumEntry& applicationAlbumEntry) NN_NOEXCEPT;

    /**
     * @brief シェアページに受け渡す文字列を設定します。
     *
     * @param[in]   pAdditionalCommentText   受け渡す UTF-8 文字列。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     *          AdditionalCommentTextMaxLength -1 文字以下にする必要があります。
     *          入力文字列はクエリーパラメータの 'text' として渡され、その際にパーセントエンコーディングされます。
     *          サロゲートペアについては非対応となります。
     */
    void SetAdditionalCommentText(const char* pAdditionalCommentText) NN_NOEXCEPT;

    /**
     * @brief シェアページに受け渡すメディアファイルの追加情報を設定します。
     *
     * @param[in]   pAdditionalMediaData   メディアファイルの追加情報を表すバイナリ。
     * @param[in]   dataLength             バイナリの大きさ。バイト単位で指定します。
     *
     * @details 二度呼び出した場合、値は上書きされます。
     *          dataLength は AdditionalMediaDataMaxLength 以下にする必要があります。
     *          入力したバイナリはクエリーパラメータの 'media_data' として渡され、その際に 16 進数変換されます。
     */
    void SetAdditionalMediaData(const uint8_t* pAdditionalMediaData, size_t dataLength) NN_NOEXCEPT;

    /**
     * @brief アルバム内のファイル情報およびそのメディアファイルの追加情報を併せて設定します。
     *
     * @param[in]   albumEntry        アルバム内のファイル情報を指定してください。
     * @param[in]   pMediaData        メディアファイルの追加情報を表すバイナリ。
     * @param[in]   mediaDataLength   メディアファイルの追加情報を表すバイナリの大きさ。バイト単位で指定します。
     *
     * @details 複数ファイルの情報を設定するときに 1 ファイルにつき 1 回呼び出します。
     *          複数回呼び出しても、値は上書きされません。
     *          最大 MaxAlbumEntryCount 回呼び出すことができ、呼び出した回数分の情報が蓄積されます。
     *          メディアファイルの追加情報を付加しない場合は pMediaData と mediaDataLength を指定せずに呼び出してください。
     */
    void AddAlbumEntryAndMediaData(const nn::capsrv::AlbumEntry& albumEntry, const uint8_t* pMediaData = nullptr, size_t mediaDataLength = 0) NN_NOEXCEPT;

    /**
     * @brief Uid を設定します。
     *
     * @param[in]   uid アカウントの Uid を指定してください。
     *
     * @details 指定する値は、アカウントシステムに登録されているユーザーの Uid または
     *          nn::account::InvalidUid である必要があります。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetUid(const nn::account::Uid& uid) NN_NOEXCEPT;

    /**
     * @brief シェアページに受け渡すメディアファイルのレーティング情報をもとにペアレンタルコントロールの制限判定を行うようにします。
     *
     * @param[in]   ratingAge メディアファイルのレーティング情報。
     *
     * @details 本関数による判定はペアレンタルコントロールが一時解除状態であるかどうかも加味します。
     *          現状ではメディアファイルが動画の場合のみ判定が実行されます。
     *          二度呼び出した場合、値は上書きされます。
     */
    void SetMediaCreatorApplicationRatingAge(const int8_t(&ratingAge)[32]) NN_NOEXCEPT;

    /**
     * @internal
     *
     * @brief   シェアページを開く際に必要なデータを一つのストレージにまとめ、ライブラリアプレットの入力チャンネルにプッシュします。
     *
     * @param[in]   libraryAppletHandle     対象のライブラリアプレットのハンドルを指定します。
     */
    void PushToInChannel(nn::applet::LibraryAppletHandle handle) const NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];

private:
    /**
     * @internal
     */
    ShowSharePageArg() NN_NOEXCEPT;
};

//--------------------------------------------------------------------------
/**
 * @brief シェアページを開いた結果の情報を提供するクラスです。
 */
class SharePageReturnValue
{
public:
    /**
     * @brief NULL 終端を含む最後に開いていたページの URL の最大バイト数です。
     */
    static const size_t LastUrlMaxLength = 4096;
    /**
     * @brief NULL 終端を含む画像投稿した Web サービスサイト名の最大バイト数です。
     */
    static const size_t PostServiceNameMaxLength = 16;
    /**
     * @brief NULL 終端を含む画像投稿 ID の最大バイト数です。
     */
    static const size_t PostIdMaxLength = 256;

    /**
     * @brief 有効な情報を持たないクラスを作成します。
     */
    SharePageReturnValue() NN_NOEXCEPT;

    /**
     * @brief 終了理由を返します。
     *
     * @return 終了理由を表す定数(@ref ShareExitReason)
     */
    ShareExitReason GetShareExitReason() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL を取得します。
     *
     * @return ページの URL を返します。
     *
     * @pre
     *     GetShareExitReason() == ShareExitReason_CallbackUrlReached
     */
    const char* GetLastUrl() const NN_NOEXCEPT;

    /**
     * @brief 最後に開いていたページの URL のバイト数を取得します。
     *
     * @return URL のサイズを返します。
     *
     * @pre
     *     GetShareExitReason() == ShareExitReason_CallbackUrlReached
     */
    size_t GetLastUrlSize() const NN_NOEXCEPT;

    /**
     * @brief 画像投稿の成否結果を返します。
     *
     * @return 画像投稿の成否結果を表す定数(@ref SharePostResult)
     */
    SharePostResult GetSharePostResult() const NN_NOEXCEPT;

    /**
     * @brief 画像投稿した Web サービスサイト名を取得します。
     *
     * @return Web サービスサイト名を返します。
     */
    const char* GetPostServiceName() const NN_NOEXCEPT;

    /**
     * @brief 画像投稿した Web サービスサイト名のバイト数を取得します。
     *
     * @return Webサービスサイト名のサイズを返します。
     */
    size_t GetPostServiceNameSize() const NN_NOEXCEPT;

    /**
     * @brief 画像投稿 ID を取得します。
     *
     * @return 画像投稿 ID を返します。
     */
    const char* GetPostId() const NN_NOEXCEPT;

    /**
     * @brief 画像投稿 ID のバイト数を取得します。
     *
     * @return 画像投稿 ID のサイズを返します。
     */
    size_t GetPostIdSize() const NN_NOEXCEPT;

protected:
    /**
     * @internal
     * @brief 情報を格納するために使用するデータのサイズ(バイト)です。
     */
    static const size_t DataSize = 8192;

    /**
     * @internal
     */
    NN_ALIGNAS(4) uint8_t m_Data[DataSize];
};

}} // namespace nn::web
